/*
   Program: LiteMenu.PRG
   Author:  Greg Lief
   Date:    March 1991
   Modified By: James J. Orlowski, M.D.
   Modified Date: 7/14/91
   Dialect: Clipper 5.01
   Purpose: Alternate Menuing System
   Compile: clipper LiteMenu /n/w/a/m
   Link:    rtlink /free fi LiteMenu

   Copyright (c) 1990-91 Greg Lief
                         Grumpfish, Inc.
                         P. O. Box 17761
                         Salem, OR 97305
                         Tel 503.588.1815
                         Fax 503.588.1980

   For use by attendees of Nantucket Developers Conference
   Palm Desert, CA -- June 1991

   NOTE: this is excerpted from the Grumpfish Library.  The Grumpfish
         Library version of this includes a continuous ticking clock,
         integrated screen saver in the event of keypress time-out,
         and a WHEN clause (for security levels) for each menu option
*/

#include "Inkey.ch"
#xcommand DEFAULT <param> TO <val> [, <paramn> TO <valn> ]  ;
=>    <param>  := IIF(<param>  == NIL, <val>,  <param>  )   ;
   [; <paramn> := IIF(<paramn> == NIL, <valn>, <paramn> ) ]

STATIC nMenuWait := 0
STATIC bMenuFunc

// these manifest constants are for easy identification
// of levels in the multi-dimensional array
#define  XROW            1
#define  YCOL            2
#define  PROMPT          3
#define  MESSAGE         4
#define  NAME            5
#define  WHEN            6
#define  ACTION          7

/*
   LiteMenu() -- alternate menu system
*/
FUNCTION LiteMenu(aOptions, cSel, cVarname, nRowStart, nRowSkip)
   LOCAL nElements := len(aOptions)
   LOCAL nX, nPtr, cTemp
   // set default colors for unselected and selected options
   LOCAL cOldColor  := SetColor()
   LOCAL aColors    := ColorSplit(cOldColor)
   LOCAL cColorUnsl := aColors[1]
   LOCAL cColorSel  := aColors[2]
   LOCAL cColorHiLt := aColors[5]
   // set other locals and defaults
   LOCAL nSel       := 0
   LOCAL nKey       := 0
   LOCAL cTriggers  := []
   LOCAL lFallout   := .f.
   LOCAL lWhen      := .t.
   LOCAL lOldmsgctr := SET(_SET_MCENTER, .T.)
   LOCAL nMessrow   := SET(_SET_MESSAGE)
   LOCAL lOldcursor := SETCURSOR(0)
   LOCAL lHoriz     := (nRowStart == NIL .AND. nRowSkip == NIL) .AND. ;
                       (aOptions[1, XROW] == aOptions[2, XROW])

   // if MESSAGE row was never set, use the bottom row of screen
   IF nMessrow == 0
      nMessrow := maxrow()
   ENDIF

   // clear MESSAGE line before start
   SetColor("W/N")
   @ 24,0

   // space messages (and center if vertical) if YCOL == 0
   // or space rows if vertical and either nRowStart Or nRowSkip Passed
   IF aOptions[1, YCOL] == 0 .OR. (nRowStart <> NIL .OR. nRowSkip <> NIL)
      aOptions := IF(lHoriz, HorizSpace(aOptions), ;
                             VertSpace(aOptions, nRowStart, nRowSkip) )
   ENDIF

   // set initial value for nSel if VALTYPE(cSel) == "N"
   IF VALTYPE(cSel) == "N" .AND. cSel >= 0 .AND. cSel <= nElements
      nSel := cSel
   ENDIF

   // build the string containing available letters for nSel
   // also check number of initial highlighted item 
   FOR nX = 1 TO nElements
      // the default is to add the first non-space character.
      // However, if there is a tilde embedded in this menu
      // option, use the letter directly following it.
      IF (nPtr := AT("~", aOptions[nX, PROMPT]) ) > 0
         cTriggers += UPPER(SUBSTR(aOptions[nX, PROMPT], nPtr + 1, 1))
         IF aOptions[nX, NAME] == NIL
            cTemp  := aOptions[nX, PROMPT]
            aOptions[nX, NAME] := ;
               UPPER(LEFT(cTemp,1)) + SUBSTR(cTemp, 2, nPtr-2) + ;
               LOWER(SUBSTR(cTemp, nPtr+1, 1)) + SUBSTR(cTemp, nPtr+2)
         ENDIF
      ELSE
         cTriggers += UPPER(LEFT(aOptions[nX, PROMPT], 1))
      ENDIF
      ShowOption(aOptions[nX], cColorUnsl, cColorHiLt)

      // check if cSel is within selected Names/Prompts
      IF cSel <> NIL .AND. VALTYPE(cSel) == "C" .AND. nSel == 0
         IF cSel == aOptions[nX, IF(aOptions[nX, NAME] <> NIL, NAME, PROMPT)]
            nSel := nX
         ENDIF
      ENDIF
   NEXT
   // default nSel to 1 if first selection not found
   nSel := IF(nSel == 0, 1, nSel)

   // commence main key-grabbing loop
   DO WHILE nKey != K_ENTER .AND. nKey != K_ESC

      // display current option in highlight color
      @ aOptions[nSel, XROW], aOptions[nSel, YCOL] SAY ;
                    STRTRAN(aOptions[nSel, PROMPT], "~", "") COLOR cColorSel

      // display corresponding message if there is one
      SetColor(cColorUnsl)
      IF aOptions[nSel, MESSAGE] == NIL
         SCROLL(nMessrow, 0, nMessrow, MAXCOL(), 0)
      ELSE
         @ nMessrow, 0 SAY PADC(aOptions[nSel, MESSAGE], MAXCOL() + 1)
      ENDIF

      IF lFallout
         EXIT
      ELSE
         nKey := inKey(nMenuWait)
         IF nKey == 0 .AND. VALTYPE(bMenuFunc) == "B"
            IF .NOT. EVAL( bMenuFunc )
               EXIT
            ENDIF
         ENDIF

         DO CASE

            /*
               use SETKEY() to see if an action block attached to the last
               keypress -- if it returns anything other than NIL, then you
               know that the answer is a resounding YES!
            */
            CASE SETKEY(nKey) != NIL
               /*
                  pass action block the name of the previous procedure,
                  along with the name of the variable referenced in the
                  MENU TO statement and the current highlighted menu
                  option (this means that you can tie a help screen to
                  each individual menu option; try that with MENU TO)
               */
               EVAL(SETKEY(nKey), PROCNAME(1), PROCLINE(1), cVarname + ;
                          "[" + ltrim(str(nSel)) + "]")

            // go down one line if vertical menu or
            // go right one item if horizontal menu
            // always observe wrap-around conventions
            case (nKey == K_DOWN  .AND. !lHoriz) ;
            .OR. (nKey == K_RIGHT .AND.  lHoriz)
               ShowOption(aOptions[nSel], cColorUnsl, cColorHiLt)
               nSel := IF(nSel == nElements, 1, ++nSel )

            // go up one line if vertical menu or
            // go left one item if horizontal menu
            // always observe wrap-around conventions
            CASE (nKey == K_UP   .AND. !lHoriz) ;
            .OR. (nKey == K_LEFT .AND.  lHoriz)
               ShowOption(aOptions[nSel], cColorUnsl, cColorHiLt)
               nSel := IF(nSel == 1, nElements, --nSel )

            // jump to top option
            CASE nKey == K_HOME
               // no point in changing color if we're already there
               IF nSel != 1
                  ShowOption(aOptions[nSel], cColorUnsl, cColorHiLt)
                  nSel := 1
               ENDIF

            // jump to bottom option
            CASE nKey == K_END
               // no point in changing color if we're already there
               IF nSel != nElements
                  ShowOption(aOptions[nSel], cColorUnsl, cColorHiLt)
                  nSel := nElements
               ENDIF

            // first letter - jump to appropriate option
            CASE upper(chr(nKey)) $ cTriggers
               ShowOption(aOptions[nSel], cColorUnsl, cColorHiLt)
               nSel := at(upper(chr(nKey)), cTriggers)
               lFallout := .t.

         ENDCASE
      ENDIF
   ENDDO  // WHILE nKey != K_ENTER .AND. nKey != K_ESC

   // clear message line before continuing
   SetColor("W/N")
   SCROLL(nMessrow, 0, nMessrow, MAXCOL(), 0)

   // if there is an action block attached to this nSel, run it
   IF nKey != 0 .AND. nKey != K_ESC
      IF aOptions[nSel, ACTION] != NIL
         lWhen := .t.
         IF aOptions[nSel, WHEN] != NIL
            lWhen := EVAL(aOptions[nSel, WHEN])
         ENDIF
         IF lWhen
            EVAL(aOptions[nSel, ACTION])
         ENDIF
      ENDIF
   ELSE
      nSel := 0                    // since they Esc'd out, return a zero
   ENDIF
   SETCURSOR(lOldcursor)
   SET(_SET_MCENTER, lOldmsgctr)   // reset SET MESSAGE CENTER

   // clear MESSAGE line before continue Then reset to original color
   SetColor("W/N")
   @ 24,0
   SetColor(cOldColor)

   IF VALTYPE(cSel) == "N"  // Note: <Esc> returns 0
      cSel := nSel
   ELSEIF nSel == 0
      cSel := "Quit"
   ELSE  // return name of selected option of VALTYPE(cSel) is character 
      cSel := TRIM(aOptions[nSel, ;
         IF(aOptions[nSel, NAME] <> NIL, NAME, PROMPT)] )
   ENDIF
RETURN cSel
// end function LiteMenu()

/*
  Function: ShowOption()
  Purpose:  Display current prompt in mixed colors
*/
STATIC FUNCTION ShowOption(aLine, cColorUnSl, cColorHiLt)
   LOCAL nPtr := at("~", aLine[PROMPT])
   cColorHiLt := IIF(cColorHiLt == NIL .OR. LEFT(cColorHiLt,3) == "N/N", ;
      "+" + cColorUnSl, cColorHiLt )
   IF nPtr > 0
      @ aLine[XROW], aLine[YCOL] SAY STRTRAN(aLine[PROMPT], "~", "") ;
         COLOR cColorUnSl
      @ aLine[XROW], aLine[YCOL] + nPtr - 1 SAY ;
        SUBSTR(aLine[PROMPT], nPtr + 1, 1) COLOR cColorHiLt
        
   ELSE
      @ aLine[XROW], aLine[YCOL] SAY left(aLine[PROMPT], 1) COLOR cColorHiLt
      DEVOUT(SUBSTR(aLine[PROMPT], 2), cColorUnSl)
   ENDIF
RETURN nil
// end static function ShowOption()

FUNCTION SetMnuWait( nSec )
   LOCAL xOld := nMenuWait
   nMenuWait  := IF(nSec == NIL, xOld, nSec)
RETURN( xOld )
// end function SetMnuWait()

FUNCTION SetMnuFunc( bFunc )
   LOCAL xOld := bMenuFunc
   bMenuFunc  := IF(bFunc == NIL, xOld, bFunc)
RETURN( xOld )
// end function SetMnuFunc()

STATIC FUNCTION ColorSplit(cColor)
   LOCAL aColor[5], i, nAt
   FOR i = 1 TO 5
      nAt := AT(",", cColor)
      IF nAt == 0
         aColor[i] := cColor
         EXIT
      ENDIF
      aColor[i] := LEFT(cColor, nAt - 1)
      cColor    := RTRIM(LTRIM(SUBSTR(cColor, nAt + 1)))
   NEXT i
RETURN aColor
// end static function ColorSplit()

/*
  Function: HorizSpace()
  Purpose:  Set Up Spacing For Horizontal Menu Using Array aMenu[]
*/
FUNCTION HorizSpace(aMenu)
   * Syntax: HorizSpace( aMenu )
   * Return: Array with proper spacing for horizontal menus
   LOCAL nLenArray, nSpaceLeft, i, nSpaceAmt, nStart, cTemp

   // calculate spacing between prompts
   nLenArray  := LEN(aMenu)
   nSpaceLeft := MaxCol() - 1
   FOR i = 1 TO nLenArray
      cTemp := aMenu[i, PROMPT]
      nSpaceLeft -= ( LEN(cTemp) - IF("~" $ cTemp, 1, 0) )
   NEXT i

   // use full amt of space at each end if space left over > 25% of line
   // otherwise if space at premium, use 1/2 amt of space at each end
   nSpaceAmt := INT( nSpaceLeft / (nLenArray + ;
      IF(nSpaceLeft > (0.25 * (MaxCol()+1)), 1, 0) ) )
   nStart := INT( (1 + nSpaceLeft - ( nSpaceAmt * (nLenArray - 1) ) ) / 2 )

   IF nSpaceAmt < 1 .OR. nStart < 0
      @ MaxRow()-1,0 SAY ;
         PADC("ERROR - Menu Prompts Won't Fit On One Line", MAXCOL()+1)
      @ MaxRow(),0 SAY ;
         PADC("Press Any Key To Continue", MAXCOL()+1)
      INKEY(60)      
      RETURN(NIL)
   ENDIF

   // set up YCOL based on new calculated start positions
   FOR i = 1 TO nLenArray
      aMenu[i, YCOL] := nStart
      cTemp  := aMenu[i, PROMPT]
      nStart += ( LEN(cTemp) + nSpaceAmt - IF("~" $ cTemp, 1, 0) )
   NEXT i
RETURN( aMenu )
// end function HorizSpace()

/*
  Function: VertSpace()
  Purpose:  Set Up Spacing For Vertical Menu Using Array aMenu[]
*/
FUNCTION VertSpace(aMenu, nRowStart, nRowSkip)
   * Syntax: VertSpace( aMenu )
   * Return: Array with proper spacing for centered vertical menus
   LOCAL i, nColStart
   LOCAL nWidth     := 0
   LOCAL nLenArray  := LEN(aMenu)
   LOCAL lRowChange := ! (nRowStart == NIL .OR. nRowSkip == NIL)

   DEFAULT nRowStart TO 5, ;
           nRowSkip  To 1

   // find maximum length of prompt
   FOR i = 1 TO nLenArray
      nWidth := MAX(nWidth, LEN(aMenu[i, PROMPT]) )
   NEXT i

   // set up same maximum width for all vertical prompts
   // set up YCOL based on new centered calculated start positions
   nColStart := ((MaxCol() + 1 - nWidth) / 2)
   FOR i = 1 TO nLenArray
      aMenu[i, PROMPT]  := PADR( aMenu[i, PROMPT], nWidth )
      aMenu[i, YCOL]    := nColStart
      IF lRowChange
         aMenu[i, XROW] := nRowStart
         nRowStart      += nRowSkip
      ENDIF
   NEXT i
RETURN( aMenu )
// end function VertSpace()

// end of file LiteMenu.PRG
