/****************************************************************
*  LISTSTRU.C                                                   *
*                                                               *
*  By        : Leonard Zerman                                   *
*  Notice    : Placed into the Public Domain by Leonard Zerman  *
*  Date      : 04/01/1990                                       *
*  Version   : 1.00                                             *
*  Revision  : None                                             *
*  Syntax    : LISTSTRU <filespec[.dbf]>                        *
*  Operation : List the structure of a dBASE III and dBASE IV   *
*              .DBF file.                                       *
*  Notes     :                                                  *
****************************************************************/
#include <stdio.h>      /* declares for file functions */
#include <stdlib.h>     /* declares for malloc and free */
#include <string.h>     /* declares for strcpy, strchr, strupr */
#include <fcntl.h>      /* attributes for file functions */

/* dbf defines */
#define DB3FILE         0x03    /* dbase III file */
#define DB3WITHMEMO     0x83    /* dbase III file w/memo */
#define DB4FILE         0x04    /* dbase IV file */
#define DB4WITHMEMO     0x8b    /* dbase IV file w/memo */
#define FIELD_REC_LEN   32      /* feild record len */
#define HEADER_PROLOG   32      /* fixed header len */
#define MAX_PATH        80      /* max path + file length */
#define MAX_FIELDS      256     /* max fields in a file */
#define MAX_RECORD      4000    /* largest record size */

/*----------------* STRUCTURE DEFINITIONS *--------------------*/
typedef struct
{
   unsigned char dbf_version;   /* version character */
   unsigned char update_yr;     /* last update - year (-1900) */
   unsigned char update_mo;     /* date of last update - month */
   unsigned char update_day;    /* date of last update - day */
   unsigned long records;       /* number of records in dbf */
   unsigned int header_length;  /* length of header structure */
   unsigned int record_length;  /* length of a record */
}PROLOG;

typedef struct
{
   char name[11];               /* field name */
   char type;                   /* C, N, D, L, M */
   char * field_data_address;   /* pointer into record buffer */
#ifdef __SMALL__
   int filler;                  /* filler for small model */
#endif
   unsigned char len;           /* length of field */
   unsigned char dec;           /* decimal places in numeric */
   unsigned char reserved_bytes[14];  /* reserved for future */
}FIELD_RECORD;

/*----------------------* PROTOTYPES *-------------------------*/
char * num2date(int day, int month, int year);
void output_dbf_stru(char * fname, PROLOG * header,
                     FIELD_RECORD * frptr);

/*---------------------* GLOBAL DATA *-------------------------*/
char open_error[]   = {"ERROR: Opening file %s\n"};
char read_error[]   = {"ERROR: Reading file %s\n"};
char seek_error[]   = {"ERROR: Seeking file %s\n"};
char alloc_error[]  = {"ERROR: Allocation memory\n"};
char dbf_error[]    = {"ERROR: %s is not a DBF file.\n"};
char usage_error[]  =
{
   "Usage: LISTSTRU <filespec[.dbf]>\n"
};

char copyright[]   =
{
   "Copyright (c) 1990 Leonard Zerman, All Rights Reserved\n"
};

/*-------------------------* MAIN *----------------------------*/
int main(int argc, char * argv[])
{
   int nbr_of_fields;
   int length;
   FILE * infile;
   PROLOG inprolog;
   FIELD_RECORD * fields_ptr;
   char infname[MAX_PATH];

   /* no input files passed, print copyright, usage */
   if(argc < 2)
   {
      fprintf(stderr, copyright);
      fprintf(stderr, usage_error);
      return(1);                /* DOS return value 1 */
   }

   strcpy(infname, argv[1]);    /* copy to buffer */
   if(!strchr(infname, '.'))    /* if no extention */
      strcat(infname, ".DBF");  /* concat extention */
   strupr(infname);

   /* open input file read only in raw mode */
   if((infile = fopen(infname, "rb")) == (FILE *)0)
   {                            /* print error message */
      fprintf(stderr, open_error, infname);
      return(2);                /* DOS return value 2 */
   }

   /* read file prolog */
   fread(&inprolog, sizeof(inprolog), 1, infile);
   if(ferror(infile))           /* if read error */
   {
      fclose(infile);           /* close file */
      fprintf(stderr, read_error, infname);
      return(3);                /* DOS return value 3 */
   }

   /* Is this a dBASE file ? */
   switch(inprolog.dbf_version)
   {
      case DB3FILE:
      case DB3WITHMEMO:
      case DB4FILE:
      case DB4WITHMEMO:
         break;
      default:
         fclose(infile);        /* close file */
         fprintf(stderr, dbf_error, infname);
         return(4);             /* DOS return value 4 */
   }

   /* position at field descriptions */
   if(fseek(infile, (long)HEADER_PROLOG, SEEK_SET) != 0)
   {
      fclose(infile);           /* close file */
      fprintf(stderr, seek_error, infname);
      return(5);                /* DOS return value 5 */
   }

   /* calculate the number of fields */
   nbr_of_fields = ((inprolog.header_length -
                    (FIELD_REC_LEN + 1)) / HEADER_PROLOG);
   length = nbr_of_fields * sizeof(FIELD_RECORD);

   if((fields_ptr = (FIELD_RECORD *)malloc(length)) ==
                    (FIELD_RECORD *)0)
   {
      fclose(infile);           /* close file */
      fprintf(stderr, alloc_error);
      return(6);                /* DOS return value 6 */
   }

   /* read field records */
   fread(fields_ptr, sizeof(FIELD_RECORD), nbr_of_fields,
                                                 infile);
   if(ferror(infile))           /* if read error */
   {
      free(fields_ptr);         /* release memory */
      fclose(infile);           /* close file */
      fprintf(stderr, read_error, infname);
      return(7);                /* DOS return value 7 */
   }
   output_dbf_stru(infname, &inprolog, fields_ptr);
   free(fields_ptr);
   fclose(infile);
   return(0);
}

/*------------------* SUPPORT FUNCTIONS *----------------------*/
void output_dbf_stru(char * fname, PROLOG * header,
                                   FIELD_RECORD * frptr)
{
   static char field_line[] =
   {
      "%5d  %-10s  %-10s  %3d    %-2d\n"
   };

   static char output_header[] =
   {
      "Structure for database: %s\n"
      "Number of data records: %ld\n"
      "Date of last update   : %s\n"
      "Field  Field Name  Type      Width    Dec\n"
   };

   int field_nbr;
   int nbr_of_fields;
   char * ptr;

   /* calculate the number of fields */
   nbr_of_fields = ((header->header_length -
                    (FIELD_REC_LEN + 1)) / HEADER_PROLOG);

   /* make date string */
   ptr = num2date((int)header->update_day,
                  (int)header->update_mo,
                  (int)header->update_yr);

   printf(output_header, fname, header->records, ptr);

   /* process all fields in for loop. fields start with 1 */
   for(field_nbr = 1; field_nbr <= nbr_of_fields;
                      field_nbr++, frptr++)
   {
      switch(frptr->type)
      {
         case 'C':
            ptr = "Character";
            break;
         case 'N':
            ptr = "Numeric";
            break;
         case 'F':
            ptr = "Float";
            break;
         case 'D':
            ptr = "Date";
            break;
         case 'L':
            ptr = "Logical";
            break;
         case 'M':
            ptr = "Memo";
            break;
         default:
            ptr = "";
      }
      printf(field_line, field_nbr, frptr->name, ptr,
                                    frptr->len, frptr->dec);
   }
   printf("**Total**%25d\n", header->record_length);
}

char * num2date(int day, int month, int year)
{
   static char retbuff[9];
   static char zero = '0';

   char * ptr;
   ptr = retbuff;

   *ptr++ = zero + month / 10;
   *ptr++ = zero + month % 10;
   *ptr++ = '/';
   *ptr++ = zero + day   / 10;
   *ptr++ = zero + day   % 10;
   *ptr++ = '/';
   *ptr++ = zero + year  / 10;
   *ptr++ = zero + year  % 10;
   *ptr++ = '\0';
   return(retbuff);
}
/*-------------------* EOF LISTSTRU.C *------------------------*/

