;****************************************************************************
; PROGRAM ----: CLIPTERM.ASM
; AUTHOR -----: Kevin E. Saffer 
; COPYRIGHT --: None, placed into the public domain
; CREATED ----: 10/2/1991 at 9:07
;****************************************************************************
; Implements a limited ANSI terminal emulator for Clipper
;****************************************************************************

.RADIX 10               ;use decimal values

PUSH_REGS MACRO         ;register push and pop macros for clipper interface
        PUSH    BP
        MOV     BP,SP
        PUSH    DI      
        PUSH    SI
        PUSH    DS
        PUSH    ES
        PUSH    SS
ENDM

POP_REGS MACRO
        POP     SS
        POP     ES
        POP     DS
        POP     SI
        POP     DI
        POP     BP
ENDM

;declare clipper parameter routines
EXTRN   __PARINFO:FAR	;get number of parameters or type of one
EXTRN   __PARINFA:FAR	;get size of array parameter or element type
EXTRN   __PARCSIZ:FAR	;get size of memory allocated for string parameter
EXTRN   __PARC:FAR	;get CHARACTER string, segment:offset in DX:AX
EXTRN   __RETC:FAR	;return string, push seg:off onto stack
EXTRN   __PARCLEN:FAR	;get length of a string parameter into ax
EXTRN   __RETCLEN:FAR	;return string of x length, push length:seg:off
EXTRN   __PARDS:FAR	;get DATE string, segment:offset in DX:AX
EXTRN   __RETDS:FAR	;return date string, push seg:off onto stack
EXTRN   __PARL:FAR	;get LOGICAL, value in AX
EXTRN   __RETL:FAR	;return logical, push 1 register onto stack
EXTRN   __PARND:FAR	;get numeric DOUBLE, segment:offset in DX:AX
EXTRN   __RETND:FAR	;return double, push 4 registers onto stack
EXTRN   __PARNI:FAR	;get numeric INTEGER, value in AX
EXTRN   __RETNI:FAR	;return integer, push 1 register onto stack
EXTRN   __PARNL:FAR	;get numeric LONG, value in DX:AX
EXTRN   __RETNL:FAR	;return long, push 2 registers onto stack

;declare clipper callable functions
PUBLIC CMASTERM         ;terminal emulator
PUBLIC CMROW            ;returns actual screen row
PUBLIC CMCOL            ;returns actual screen column
PUBLIC CMCAPTURE        ;sets the terminal emulation capture file on/off
PUBLIC CMPRINTER        ;set the printer output on/off
PUBLIC CMWRAP           ;set the terminal line wrap on/off
PUBLIC CMBEEP           ;set the terminal beep supression on/off
PUBLIC CMCEXIT          ;set the terminal to exit if carrier is lost

;declare external routines
EXTRN   CM_ACCESS:FAR   ;access point for the resident routines

;declare function number equates for calls to the communication modules
FN_OPENPORT     EQU     1       ;opens a port and starts input
FN_OUTSTR       EQU     2       ;start buffered output
FN_OUTCHR       EQU     3       ;send unbuffered character
FN_TDCOUNT      EQU     4       ;return TD buffer count   
FN_TDFLUSH      EQU     5       ;flush TD buffer
FN_TDXOFF       EQU     6       ;return TD XOFF status           
FN_GETSTR       EQU     7       ;return input buffer with optional flush
FN_GETCHR       EQU     8       ;return input char with flush
FN_RDCOUNT      EQU     9       ;return RD buffer count   
FN_RDFLUSH      EQU     10      ;flush RD buffer
FN_RDXOFF       EQU     11      ;return TD XOFF status           
FN_SETXOFF      EQU     12      ;change flow control
FN_STATUS       EQU     13      ;return LSR+MSR in AX     
FN_BREAK        EQU     14      ;send BREAK signal to modem/terminal
FN_CHECKPORT    EQU     15      ;port availability check
FN_SETPORT      EQU     16      ;port address/interrupt change
FN_TOGGLEDTR    EQU     17      ;toggles DTR to force carrier break
FN_CLOSEPORT    EQU     18      ;closes port 

PROGSEG SEGMENT BYTE 'CODE'     ;set up our code segment, and
        ASSUME CS:PROGSEG       ;let MASM know of our intentions

;****************************************************************************
; CMASTERM - assembler terminal emulator
;****************************************************************************
; Syntax: CMASTERM(<portnbr>)
;
; Passed: <portnbr> - 1,2,3, or 4 
;
; Returns: numeric code corresponding to Clipper function key F1 - F10
;****************************************************************************
CMASTERM PROC FAR

        JMP SHORT CMASTERM_START           ;jump over the local data area

        CMASTERM_PORTNBR        DB      0       ;passed port number
        CMASTERM_HANDLE         DW      0       ;capture file handle
        CMASTERM_FCHAR          DB      0       ;received character storage
        CMASTERM_PRINTER        DB      0       ;printer on or off (0=.F.=off)
        CMASTERM_WRAP           DB      1       ;console wrap on/off (1=.T.=on)
        CMASTERM_BEEP           DB      0       ;allow beeps (0=.F.=no)
        CMASTERM_CEXIT          DB      0       ;exit of carrier lost (0=.F.=no)
        CMASTERM_ATTRIBUTE      DB      7       ;attribute byte for screen color
        CMASTERM_ROW            DB      0       ;current row value
        CMASTERM_COL            DB      0       ;current column value
        CMASTERM_INESC          DB      0       ;flag for escape sequence processing

CMASTERM_START:
        PUSH_REGS                       ;save clipper registers
        MOV     AX,1                    ;get the port number
        PUSH    AX
        CALL    __PARNI                 ;integer now in AX
        ADD     SP,2                    ;fix the stack pointer
        MOV     CS:CMASTERM_PORTNBR,AL  ;save port number for later use

        PUSH    DS                      ;save DS for later
        PUSH    CS                      ;set DS to point to our code segment
        POP     DS

        MOV     AH,8                    ;read char and attribute
        XOR     BX,BX
        INT     10H                     ;attribute byte in AH
        MOV     CMASTERM_ATTRIBUTE,AH   ;store it for future use

CMASTERM_KEYCHECK:
        CALL    CMASTERM_KPROCESS   
        CMP     BX,1                    ;exit code?
        JE      CMASTERM_RETURN         ;yes, return it to clipper
        CMP     AX,0                    ;character waiting?
        JE      CMASTERM_CHARCHECK      ;no, check for incoming
        XOR     BH,BH
        MOV     BL,AL                   ;keystroke to bl
        MOV     AL,CS:CMASTERM_PORTNBR  ;retrieve saved port number
        MOV     AH,FN_OUTCHR            ;function number to transmit ASCII char
        CALL    CM_ACCESS               ;call communications module

CMASTERM_CHARCHECK:
        MOV     AL,CS:CMASTERM_PORTNBR  ;retrieve saved port number
        MOV     AH,FN_GETCHR            ;function number to get ASCII char
        CALL    CM_ACCESS
        CMP     AX,0
        JL      CMASTERM_CARRIERCHECK   ;no input waiting, check for carrier loss
        CALL    CMASTERM_CPROCESS       ;call character processing routine

CMASTERM_CARRIERCHECK:
        CMP     CMASTERM_CEXIT,0        ;checking for carrier?
        JE      CMASTERM_KEYCHECK       ;no, continue processing
        MOV     AL,CMASTERM_PORTNBR     ;port number to AL
        MOV     AH,FN_STATUS            ;function number to AH
        CALL    CM_ACCESS               ;call communications module
        CMP     AX,0                    ;error?
        JE      CMASTERM_KEYCHECK       ;yes, continue processing
        TEST    AL,10000000b            ;bit 7 set? (carrier high)
        JZ      CMASTERM_CARRIERCHECK1  ;no, exit with a zero code
        JMP     CMASTERM_KEYCHECK       ;get next keystroke

CMASTERM_CARRIERCHECK1:
        XOR     AX,AX

CMASTERM_RETURN:
        POP     DS
        POP_REGS                        
        PUSH    AX
        CALL    __RETNI                 ;return it to clipper
        ADD     SP,2
        RET

CMASTERM ENDP

;****************************************************************************
; CMASTERM_KPROCESS - used by CMASTERM to check for and process a keystroke
;
; On Entry: Nothing
;
;  On Exit: AL -> character received or Clipper function key
;           BX -> 0 if normal character or 1 if function key pressed
;****************************************************************************
CMASTERM_KPROCESS PROC NEAR

        MOV     AH,6                    ;keyboard check function to AH
        MOV     DL,0FFH
        INT     21H                     ;call DOS to check for keystroke waiting
        JNZ     CMASTERM_KPROCESS1      ;one is waiting, continue processing
        XOR     AX,AX                   ;no key code to AX
        XOR     BX,BX                   ;no exit code to BX
        RET

CMASTERM_KPROCESS1:
        CMP     AL,0                    ;extended key?
        JE      CMASTERM_KPROCESS2      ;yes, process it
        XOR     AH,AH                   ;clear high byte
        XOR     BX,BX                   ;no exit code to BX
        RET                             ;return to process character

CMASTERM_KPROCESS2:

        MOV     AH,6                    ;keyboard check function to AH
        MOV     DL,0FFH
        INT     21H                     ;call DOS
        XOR     AH,AH                   ;clear AH register

CMASTERM_KPROCESS3:
        CMP     AL,77                   ;right arrow key?
        JNE     CMASTERM_KPROCESS4      ;no, continue checking
        MOV     AL,CS:CMASTERM_PORTNBR  ;retrieve saved port number
        MOV     AH,FN_OUTCHR            ;function number to transmit ASCII char
        XOR     BH,BH                   ;clear BH register
        MOV     BL,27                   ;escape character to AL
        CALL    CM_ACCESS               ;call communications module
        XOR     AH,AH                   ;clear high byte
        MOV     AL,'C'                  ;set routine to send character
        XOR     BX,BX                   ;no exit code to BX
        RET                             ;return to process character

CMASTERM_KPROCESS4:
        CMP     AL,80                   ;down arrow key?
        JNE     CMASTERM_KPROCESS5      ;no, continue checking
        MOV     AL,CS:CMASTERM_PORTNBR  ;retrieve saved port number
        MOV     AH,FN_OUTCHR            ;function number to transmit ASCII char
        XOR     BH,BH                   ;clear BH register
        MOV     BL,27                   ;escape character to AL
        CALL    CM_ACCESS               ;call communications module
        XOR     AH,AH                   ;clear high byte
        MOV     AL,'B'                  ;set routine to send character
        XOR     BX,BX                   ;no exit code to BX
        RET                             ;return to process character

CMASTERM_KPROCESS5:
        CMP     AL,75                   ;left arrow key?
        JNE     CMASTERM_KPROCESS6      ;no, continue checking
        MOV     AL,CS:CMASTERM_PORTNBR  ;retrieve saved port number
        MOV     AH,FN_OUTCHR            ;function number to transmit ASCII char
        XOR     BH,BH                   ;clear BH register
        MOV     BL,27                   ;escape character to AL
        CALL    CM_ACCESS               ;call communications module
        XOR     AH,AH                   ;clear high byte
        MOV     AL,'D'                  ;set routine to send character
        XOR     BX,BX                   ;no exit code to BX
        RET                             ;return to process character

CMASTERM_KPROCESS6:
        CMP     AL,72                   ;up arrow key?
        JNE     CMASTERM_KPROCESS7      ;no, continue checking
        MOV     AL,CS:CMASTERM_PORTNBR  ;retrieve saved port number
        MOV     AH,FN_OUTCHR            ;function number to transmit ASCII char
        XOR     BH,BH                   ;clear BH register
        MOV     BL,27                   ;escape character to AL
        CALL    CM_ACCESS               ;call communications module
        XOR     AH,AH                   ;clear high byte
        MOV     AL,'A'                  ;set routine to send character
        XOR     BX,BX                   ;no exit code to BX
        RET                             ;return to process character

CMASTERM_KPROCESS7:
        CMP     AL,59                   ;F1 key?
        JNE     CMASTERM_KPROCESS8      ;no, continue checking
        MOV     AL,28                   ;clipper F1 key to AL
        MOV     BX,1                    ;program exit code to BX
        RET

CMASTERM_KPROCESS8:
        CMP     AL,60                   ;greater than or = F2 key?
        JGE     CMASTERM_KPROCESS9
        XOR     AX,AX                   ;no key code to AX
        XOR     BX,BX                   ;no exit code to BX
        RET

CMASTERM_KPROCESS9:
        CMP     AL,68                   ;less than or = F10 key?
        JLE     CMASTERM_KPROCESS10
        XOR     AX,AX                   ;no key code to AX
        XOR     BX,BX                   ;no exit code to BX
        RET

CMASTERM_KPROCESS10:
        XOR     BH,BH           ;translate the code into a Clipper Fkey
        MOV     BL,AL
        SUB     BL,59
        XOR     AX,AX
        SUB     AX,BX                   
        MOV     BX,1            ;exit code to BX
        RET

CMASTERM_KPROCESS ENDP

;****************************************************************************
; CMASTERM_CPROCESS - processes an incoming character
;
; On Entry: AL -> character received from communication port
;
;  On Exit: Nothing
;****************************************************************************
CMASTERM_CPROCESS PROC NEAR

        MOV     CMASTERM_FCHAR,AL       ;save the character for later use
        CMP     CMASTERM_INESC,1        ;are we processing an escape sequence?
        JE      CMASTERM_CPROCESS11     ;yes, transfer to esc processor

CMASTERM_CPROCESS10:
        CMP     AL,27                   ;is it an esc character?
        JNE     CMASTERM_CPROCESS20     ;no, process normally
        MOV     AH,1                    ;set flag for esc processor
        MOV     CMASTERM_INESC,AH       ;store it
        RET                             ;return for the next character

CMASTERM_CPROCESS11:
        CALL    CMASTERM_ESCPROCESS     ;call ANSI sequence processor
        CMP     BX,1                    ;print current character?
        JE      CMASTERM_CPROCESS20     ;yes, continue normally
        RET                             ;return for the next character

CMASTERM_CPROCESS20:
        MOV     AH,3                    ;read cursor position from screen
        XOR     BH,BH                   ;page 0
        INT     10H                     ;DH -> row, DL -> column
        MOV     CMASTERM_ROW,DH         ;save row for later
        MOV     CMASTERM_COL,DL         ;save column for later
        MOV     AL,CMASTERM_FCHAR       ;retrieve character

CMASTERM_CPROCESS21:
        CMP     AL,7                    ;char a bell?
        JNE     CMASTERM_CPROCESS22     ;no, continue checking
        CMP     CMASTERM_BEEP,0         ;allowing beeps?
        JNE     CMASTERM_CPROCESS211    ;no, do not print it
        CALL    CMASTERM_CWRITE         ;write using Int 21 call

CMASTERM_CPROCESS211:
        CALL    CMASTERM_PWRITE         ;write char to the printer/file
        RET

CMASTERM_CPROCESS22:
        CMP     AL,8                    ;char a backspace?
        JNE     CMASTERM_CPROCESS23     ;no, continue checking
        CALL    CMASTERM_CWRITE         ;write using Int 21 call
        CALL    CMASTERM_PWRITE         ;write char to the printer/file
        RET

CMASTERM_CPROCESS23:
        CMP     AL,9                    ;char a tab?
        JNE     CMASTERM_CPROCESS24     ;no, continue checking
        CMP     CMASTERM_COL,70         ;print the tab without scrolling?
        JG      CMASTERM_CPROCESS232    ;no, proceed to printer write
        MOV     CX,8                    ;loop eight times
        MOV     DL,32                   ;space character to DL
        MOV     AH,6                    ;print function to AH

CMASTERM_CPROCESS231:
        INT     21H                     ;call DOS
        LOOP    CMASTERM_CPROCESS231    ;loop until CX = 0

CMASTERM_CPROCESS232:
        CALL    CMASTERM_PWRITE         ;write char to the printer/file
        RET

CMASTERM_CPROCESS24:
        CMP     AL,10                   ;char a linefeed?
        JNE     CMASTERM_CPROCESS25     ;no, continue checking
        CMP     DH,23                   ;at the last row?
        JL      CMASTERM_CPROCESS241    ;no, print the linefeed normally
        CALL    CMASTERM_SCROLL         ;scroll the window up one line

CMASTERM_CPROCESS241:
        CALL    CMASTERM_CWRITE         ;write using Int 21 call
        CALL    CMASTERM_PWRITE         ;write char to the printer/file
        RET

CMASTERM_CPROCESS25:
        CMP     AL,12                   ;is it a FF character?
        JNE     CMASTERM_CPROCESS26     ;no, continue checking
        CALL    CMASTERM_CLEAR          ;clear the screen and return
        RET

CMASTERM_CPROCESS26:
        CMP     AL,13                   ;char a carriage return?
        JNE     CMASTERM_CPROCESS27     ;no, write the character and attribute
        CALL    CMASTERM_CWRITE         ;write using Int 21 call
        CALL    CMASTERM_PWRITE         ;write char to the printer/file
        RET

CMASTERM_CPROCESS27:
        MOV     AH,9                    ;write character and attribute func
        MOV     AL,CMASTERM_FCHAR       ;character to al
        XOR     BH,BH                   ;select page 0
        MOV     BL,CMASTERM_ATTRIBUTE   ;retrieve current attribute
        MOV     CX,1                    ;write one character
        INT     10H                     ;call DOS
        CMP     DL,79                   ;at the last column?
        JE      CMASTERM_CPROCESS271    ;check for console wrap
        INC     DL                      ;move column over 1 char
        MOV     AH,2                    ;re-position cursor function
        INT     10H                     ;call DOS
        CALL    CMASTERM_PWRITE         ;write char to the printer/file
        RET

CMASTERM_CPROCESS271:
        CMP     CMASTERM_WRAP,0         ;wrap active?
        JE      CMASTERM_CPROCESS28     ;no, proceed to printer check 
        CMP     DH,23                   ;at the last row?
        JL      CMASTERM_CPROCESS272    ;no, print cr/lf normally
        CALL    CMASTERM_SCROLL         ;scroll the window up one line

CMASTERM_CPROCESS272:
        MOV     AH,2                    ;print character function
        MOV     DL,13                   ;carriage return to DL
        INT     21H                     ;call DOS
        MOV     AH,2                    ;print character function
        MOV     DL,10                   ;line feed to DL
        INT     21H                     ;call DOS

CMASTERM_CPROCESS28:
        CALL    CMASTERM_PWRITE         ;write char to the printer/file
        RET

CMASTERM_CPROCESS ENDP 

;****************************************************************************
; CMASTERM_ESCPROCESS - processor for ANSI escape sequences
;
; On entry - CMASTERM_INESC = 1 
;            AL = received character
;
;  On exit - BX = 0 retrieve next character
;            BX = 1 print current character
;****************************************************************************
CMASTERM_ESCPROCESS PROC NEAR

        JMP     CMASTERM_ESCPROCESS1

        ANSI_VALUES     DB      '[0123456789;ABCDHJKfnsumhlp'    ;ANSI delimiters
        ANSI_WORK       DB      20 DUP(0)         ;workspace
        ANSI_PTR        DW      0                 ;pointer into workspace
        ANSI_ROW        DB      0                 ;for saved positions
        ANSI_COL        DB      0                 

        ANSI_DEVSTAT    DB      '[6n'             ;device status report 
        ANSI_CLRSCR     DB      '[2J'             ;clear screen sequence
        ANSI_CLREOL     DB      '[K'              ;clear to end of line
        ANSI_SAVEPOS    DB      '[s'              ;save current position
        ANSI_RESTPOS    DB      '[u'              ;save current position

CMASTERM_ESCPROCESS1:
        PUSH    ES                      ;save ES for later
        PUSH    CS
        POP     ES
        MOV     DI,OFFSET ANSI_VALUES   ;load ES:DI for string scan
        MOV     CX,27                   ;scan for len of values
        CLD                             ;clear direction flag
        REPNZ   SCASB                   ;perform the scan until a match
        POP     ES                      ;restore ES register
        JE      CMASTERM_ESCPROCESS2    ;if found, continue with AL -> value

        CALL    CMASTERM_ANSICLEAR      ;clear ansi process flags
        RET

CMASTERM_ESCPROCESS2:
        CMP     AL,'['                  ;ANSI start value?
        JNE     CMASTERM_ESCPROCESS4    ;no, continue processing
        MOV     DI,OFFSET ANSI_WORK     ;load offset to the workspace
        MOV     BYTE PTR DS:[DI],AL     ;save the start value
        INC     DI                      ;move to the next byte
        MOV     ANSI_PTR,DI             ;store input pointer for later
        MOV     CX,19                   ;repeat for len of string
        XOR     AX,AX                   ;set zero value for copying

CMASTERM_ESCPROCESS3:
        MOV     DS:[DI],AL              ;clear workspace byte
        INC     DI                      ;point to next byte
        DEC     CX                      ;decrement counter
        CMP     CX,0                    ;check if done
        JNE     CMASTERM_ESCPROCESS3    ;no, loop back
        XOR     BX,BX                   ;have terminal ignore character
        RET

CMASTERM_ESCPROCESS4:
        MOV     DI,ANSI_PTR             ;retrieve pointer to workspace for bounds ckeck
        CMP     DI,0                    ;zero pointer?
        JE      CMASTERM_ESCPROCESS5    ;out of bounds, exit
        MOV     SI,OFFSET ANSI_WORK     ;retrieve workspace offset
        ADD     SI,20                   ;bump to ending byte + 1
        CMP     SI,DI                   ;check limits
        JLE     CMASTERM_ESCPROCESS5    ;out of bounds, exit
        JMP     CMASTERM_ESCPROCESS6    ;continue processing

CMASTERM_ESCPROCESS5:
        CALL    CMASTERM_ANSICLEAR      ;clear ansi process flags
        RET

CMASTERM_ESCPROCESS6:
        MOV     BYTE PTR DS:[DI],AL     ;store character into workspace
        INC     DI                      ;move pointer up to the next byte
        MOV     ANSI_PTR,DI             ;store it
        CMP     AL,';'                  ;numeric value or delimiter?
        JG      CMASTERM_ESCPROCESS7    ;no, it's a command character
        RET

CMASTERM_ESCPROCESS7:   
        CMP     AL,'m'                  ;set graphics rendition?
        JNE     CMASTERM_ESCPROCESS8    ;no, try next command   

        CALL    CMASTERM_ANSICLEAR      ;this command is disabled for now
        RET

        MOV     SI,OFFSET ANSI_WORK     ;point to the workspace
        XOR     BX,BX                   ;clear BX attribute register

CMASTERM_ESCPROCESS7_1:   
        INC     SI                      ;point to where next number should be
        CMP     BYTE PTR DS:[SI],'0'    ;within lower limit?
        JL      CMASTERM_ESCPROCESS7_2  ;no, set current attribute
        CMP     BYTE PTR DS:[SI],'9'    ;within upper limit?
        JG      CMASTERM_ESCPROCESS7_2  ;no, set current attribute
        PUSH    BX                      ;save current attribute
        CALL    CMASTERM_ASCIITONUM     ;call ASCII - numeric conversion routine
        MOV     AX,BX                   ;move attribute to AX
        POP     BX                      ;restore existing attribute
        ADD     BX,AX                   ;add in new 
        JMP     CMASTERM_ESCPROCESS7_1  ;loop back for more

CMASTERM_ESCPROCESS7_2:   
        CMP     BX,0                    ;check for error
        JG      CMASTERM_ESCPROCESS7_3  ;attribute OK, move on
        MOV     BX,7                    ;set default attribute

CMASTERM_ESCPROCESS7_3:   
        MOV     CMASTERM_ATTRIBUTE,BL   ;store new attribute
        CALL    CMASTERM_ANSICLEAR      ;clear ansi process flags
        RET

CMASTERM_ESCPROCESS8:   
        CMP     AL,'H'                  ;full cursor position command?
        JE      CMASTERM_ESCPROCESS8_1  ;yes, process it
        CMP     AL,'f'                  ;cursor position command?
        JNE     CMASTERM_ESCPROCESS9    ;no, try next command

CMASTERM_ESCPROCESS8_1:
        MOV     SI,OFFSET ANSI_WORK     ;point to the workspace
        INC     SI                      ;point to where number should be
        CALL    CMASTERM_ASCIITONUM     ;call ASCII - numeric conversion routine
        CMP     BX,0                    ;check for error or default
        JNE     CMASTERM_ESCPROCESS8_2
        MOV     BX,1                    ;set default

CMASTERM_ESCPROCESS8_2:
        DEC     BX                      ;adjust for the dos call later
        PUSH    BX                      ;save result
        INC     SI                      ;point to where next number should be
        CALL    CMASTERM_ASCIITONUM     ;call ASCII - numeric conversion routine
        CMP     BX,0                    ;check for error or default
        JNE     CMASTERM_ESCPROCESS8_3
        MOV     BX,1                    ;set default

CMASTERM_ESCPROCESS8_3:
        DEC     BX                      ;adjust for the dos call
        MOV     DL,BL                   ;column to DL
        POP     BX                      ;retrieve saved row value
        MOV     DH,BL                   ;row to DH
        CMP     DL,79                   ;check limits
        JG      CMASTERM_ESCPROCESS8_4
        CMP     DL,0                    ;check limits
        JL      CMASTERM_ESCPROCESS8_4
        CMP     DH,23                   ;check limits
        JG      CMASTERM_ESCPROCESS8_4
        CMP     DH,0                    ;check limits
        JL      CMASTERM_ESCPROCESS8_4
        MOV     AH,2                    ;re-position cursor function
        XOR     BX,BX                   ;select page 0     
        INT     10H                     ;call DOS

CMASTERM_ESCPROCESS8_4:
        CALL    CMASTERM_ANSICLEAR      ;clear ansi process flags
        RET

CMASTERM_ESCPROCESS9:
        CMP     AL,'A'                  ;cursor up command?
        JNE     CMASTERM_ESCPROCESS10   ;no, try next command
        MOV     SI,OFFSET ANSI_WORK     ;point to the workspace
        INC     SI                      ;point to where number should be
        CALL    CMASTERM_ASCIITONUM     ;call ASCII - numeric conversion routine
        CMP     BX,0                    ;check result
        JNE     CMASTERM_ESCPROCESS9_1  ;set it
        MOV     BX,1                    ;set default

CMASTERM_ESCPROCESS9_1:
        PUSH    BX                      ;save BX for later
        MOV     AH,3                    ;read cursor position from screen
        XOR     BH,BH                   ;page 0
        INT     10H                     ;DH -> row, DL -> column
        POP     BX
        SUB     DH,BL                   ;move cursor up
        CMP     DH,0
        JGE     CMASTERM_ESCPROCESS9_2  ;set up new cursor position
        MOV     DH,0                    ;correct overflow

CMASTERM_ESCPROCESS9_2:
        MOV     AH,2                    ;re-position cursor function
        XOR     BX,BX                   ;page 0            
        INT     10H                     ;call DOS
        CALL    CMASTERM_ANSICLEAR      ;clear escape processor
        RET

CMASTERM_ESCPROCESS10:
        CMP     AL,'B'                  ;cursor down command?
        JNE     CMASTERM_ESCPROCESS11   ;no, try next command
        MOV     SI,OFFSET ANSI_WORK     ;point to the workspace
        INC     SI                      ;point to where number should be
        CALL    CMASTERM_ASCIITONUM     ;call ASCII - numeric conversion routine
        CMP     BX,0                    ;check result
        JNE     CMASTERM_ESCPROCESS10_1 ;set it
        MOV     BX,1                    ;set default

CMASTERM_ESCPROCESS10_1:
        PUSH    BX                      ;save BX for later
        MOV     AH,3                    ;read cursor position from screen
        XOR     BH,BH                   ;page 0
        INT     10H                     ;DH -> row, DL -> column
        POP     BX
        ADD     DH,BL                   ;move cursor down
        CMP     DH,23
        JLE     CMASTERM_ESCPROCESS10_2 ;set up new cursor position
        MOV     DH,23                   ;correct overflow

CMASTERM_ESCPROCESS10_2:
        MOV     AH,2                    ;re-position cursor function
        XOR     BX,BX                   ;page 0            
        INT     10H                     ;call DOS
        CALL    CMASTERM_ANSICLEAR      ;clear escape processor
        RET

CMASTERM_ESCPROCESS11:
        CMP     AL,'C'                  ;cursor right command?
        JNE     CMASTERM_ESCPROCESS12   ;no, try next command
        MOV     SI,OFFSET ANSI_WORK     ;point to the workspace
        INC     SI                      ;point to where number should be
        CALL    CMASTERM_ASCIITONUM     ;call ASCII - numeric conversion routine
        CMP     BX,0                    ;check result
        JNE     CMASTERM_ESCPROCESS11_1 ;set it
        MOV     BX,1                    ;set default

CMASTERM_ESCPROCESS11_1:
        PUSH    BX                      ;save BX for later
        MOV     AH,3                    ;read cursor position from screen
        XOR     BH,BH                   ;page 0
        INT     10H                     ;DH -> row, DL -> column
        POP     BX
        ADD     DL,BL                   ;move cursor right
        CMP     DL,79
        JLE     CMASTERM_ESCPROCESS11_2 ;set up new cursor position
        MOV     DL,79                   ;correct overflow

CMASTERM_ESCPROCESS11_2:
        MOV     AH,2                    ;re-position cursor function
        XOR     BX,BX                   ;page 0            
        INT     10H                     ;call DOS
        CALL    CMASTERM_ANSICLEAR      ;clear escape processor
        RET

CMASTERM_ESCPROCESS12:
        CMP     AL,'D'                  ;cursor down command?
        JNE     CMASTERM_ESCPROCESS13   ;no, try next command
        MOV     SI,OFFSET ANSI_WORK     ;point to the workspace
        INC     SI                      ;point to where number should be
        CALL    CMASTERM_ASCIITONUM     ;call ASCII - numeric conversion routine
        CMP     BX,0                    ;check result
        JNE     CMASTERM_ESCPROCESS12_1 ;set it
        MOV     BX,1                    ;set default

CMASTERM_ESCPROCESS12_1:
        PUSH    BX                      ;save BX for later
        MOV     AH,3                    ;read cursor position from screen
        XOR     BH,BH                   ;page 0
        INT     10H                     ;DH -> row, DL -> column
        POP     BX
        SUB     DL,BL                   ;move cursor left
        CMP     DL,0
        JGE     CMASTERM_ESCPROCESS12_2 ;set up new cursor position
        MOV     DL,0                    ;correct overflow

CMASTERM_ESCPROCESS12_2:
        MOV     AH,2                    ;re-position cursor function
        XOR     BX,BX                   ;page 0            
        INT     10H                     ;call DOS
        CALL    CMASTERM_ANSICLEAR      ;clear escape processor
        RET

CMASTERM_ESCPROCESS13:
        PUSH    ES                      ;save ES for later
        PUSH    CS                      ;set ES to our code segment for
        POP     ES                      ;static command checks
        CLD                             ;check for device status report 
        MOV     DI,OFFSET ANSI_WORK     ;load ES:DI with workspace 
        MOV     SI,OFFSET ANSI_DEVSTAT  ;load DS:SI with first ANSI command
        MOV     CX,3                    ;length of command to CX
        REPZ    CMPSB                   ;check for the command
        JNE     CMASTERM_ESCPROCESS14   ;not found, check for next
        POP     ES                      ;restore ES
        CALL    CMASTERM_DEVSTAT        ;output device status report
        CALL    CMASTERM_ANSICLEAR      ;clear ansi process flags
        RET

CMASTERM_ESCPROCESS14:
        INC     SI                      ;check for clear screen sequence
        MOV     DI,OFFSET ANSI_WORK     ;reset ES:DI with workspace 
        MOV     SI,OFFSET ANSI_CLRSCR   ;load DS:SI with ANSI command
        MOV     CX,3                    ;length of command to CX
        REPZ    CMPSB                   ;check for the command
        JNE     CMASTERM_ESCPROCESS15   ;not found, check for next
        POP     ES                      ;restore ES
        CALL    CMASTERM_CLEAR          ;clear the screen and home the cursor
        CALL    CMASTERM_ANSICLEAR      ;clear ansi process flags
        RET

CMASTERM_ESCPROCESS15:
        INC     SI                      ;check for clear to end of line
        MOV     DI,OFFSET ANSI_WORK     ;reset ES:DI with workspace 
        MOV     SI,OFFSET ANSI_CLREOL   ;load DS:SI with ANSI command
        MOV     CX,2                    ;length of command to CX
        REPZ    CMPSB                   ;check for the command
        JNE     CMASTERM_ESCPROCESS16   ;not found, check for next
        POP     ES                      ;restore ES
        CALL    CMASTERM_CLEAREOL       ;clear the screen and home the cursor
        CALL    CMASTERM_ANSICLEAR      ;clear ansi process flags
        RET

CMASTERM_ESCPROCESS16:
        INC     SI                      ;check for save cursor position
        MOV     DI,OFFSET ANSI_WORK     ;reset ES:DI with workspace 
        MOV     SI,OFFSET ANSI_SAVEPOS  ;load DS:SI with ANSI command
        MOV     CX,2                    ;length of command to CX
        REPZ    CMPSB                   ;check for the command
        JNE     CMASTERM_ESCPROCESS17   ;not found, check for next
        POP     ES                      ;restore ES
        MOV     AH,3                    ;read cursor position from screen
        XOR     BH,BH                   ;page 0
        INT     10H                     ;DH -> row, DL -> column
        MOV     ANSI_ROW,DH
        MOV     ANSI_COL,DL
        CALL    CMASTERM_ANSICLEAR      ;clear ansi process flags
        RET

CMASTERM_ESCPROCESS17:
        INC     SI                      ;check for restore cursor position
        MOV     DI,OFFSET ANSI_WORK     ;reset ES:DI with workspace 
        MOV     SI,OFFSET ANSI_RESTPOS  ;load DS:SI with ANSI command
        MOV     CX,2                    ;length of rest screen position cmd to CX
        REPZ    CMPSB                   ;check for the command
        JNE     CMASTERM_ESCPROCESS18   ;not found, exit sequence
        POP     ES                      ;restore ES
        MOV     AH,2                    ;re-position cursor function
        XOR     BX,BX                   ;page 0            
        MOV     DH,ANSI_ROW
        MOV     DL,ANSI_COL
        INT     10H                     ;call DOS
        CALL    CMASTERM_ANSICLEAR      ;clear ansi process flags
        RET

CMASTERM_ESCPROCESS18:
        POP     ES                      ;restore ES
        CALL    CMASTERM_ANSICLEAR
        RET

CMASTERM_ESCPROCESS ENDP

;****************************************************************************
; CMASTERM_CWRITE - prints character to console using DOS int 21
;****************************************************************************
CMASTERM_CWRITE PROC NEAR

        MOV     AL,CMASTERM_FCHAR       ;retrieve character
        CMP     AL,0FFH                 ;FF character?
        JE      CMASTERM_CWRITE1        ;do not attempt to write it
        MOV     AH,6                    ;print character function to AH
        MOV     DL,AL                   ;character to dl
        INT     21H                     ;call DOS

CMASTERM_CWRITE1:
        RET

CMASTERM_CWRITE ENDP

;****************************************************************************
; CMASTERM_PWRITE - prints character to printer ands capture file
;****************************************************************************
CMASTERM_PWRITE PROC NEAR

        MOV     AH,CMASTERM_PRINTER     ;check for printer toggle
        CMP     AH,0
        JE      CMASTERM_PWRITE1        ;no print, check for capture write
        MOV     DL,CMASTERM_FCHAR       ;character to dl
        MOV     AH,5                    ;print the character
        INT     21H

CMASTERM_PWRITE1:
        MOV     BX,CMASTERM_HANDLE      ;check for valid file handle
        CMP     BX,0
        JNE     CMASTERM_PWRITE2
        RET                             ;all done, exit

CMASTERM_PWRITE2:
        MOV     AL,CMASTERM_FCHAR       ;retrieve character 
        CMP     AL,7                    ;char a bell?
        JNE     CMASTERM_PWRITE3        ;no continue checking
        RET                             ;do not capture it

CMASTERM_PWRITE3:
        CMP     AL,8                    ;char a backspace?
        JNE     CMASTERM_PWRITE4        ;no continue checking
        RET                             ;do not capture it

CMASTERM_PWRITE4:
        CMP     AL,10                   ;line feed?
        JNE     CMASTERM_PWRITE5        ;no continue checking
        RET                             ;do not capture it

CMASTERM_PWRITE5:
        CALL    CMASTERM_FWRITE         ;write the character to the capture file
        MOV     AL,CMASTERM_FCHAR       ;retrieve character
        CMP     AL,13                   ;carriage return?
        JE      CMASTERM_PWRITE6        ;yes, add a linefeed in the proper place
        RET

CMASTERM_PWRITE6:
        MOV     AL,10                   ;line feed to AH
        MOV     CMASTERM_FCHAR,AL       ;store it 
        CALL    CMASTERM_FWRITE         ;write it
        RET                             ;get next keystroke

CMASTERM_PWRITE ENDP

;****************************************************************************
; CMASTERM_FWRITE - writes stored byte to file handle
;
; On Entry: CMASTERM_FCHAR - character to write
;           CMASTERM_HANDLE - handle returned by Clipper FOPEN
;
;  On Exit: Nothing
;****************************************************************************
CMASTERM_FWRITE PROC NEAR

        MOV     CX,1                            ;write one byte
        MOV     DX,OFFSET CMASTERM_FCHAR        ;offset of buffer to DX
        MOV     AH,40H                          ;write to file function
        INT     21H
        RET

CMASTERM_FWRITE ENDP

;****************************************************************************
; CMASTERM_ANSICLEAR - clear the esc processing flags
;****************************************************************************
CMASTERM_ANSICLEAR PROC NEAR

        XOR     AX,AX                   ;error, clear flag for esc processing
        MOV     CMASTERM_INESC,AH       ;save it 
        MOV     ANSI_PTR,AX             ;reset the workspace pointer
        XOR     BX,BX                   ;have terminal ignore current character
        RET

CMASTERM_ANSICLEAR ENDP

;****************************************************************************
; CMASTERM_SCROLL - scrolls the screen up one line, positions cursor to 22,0
;
; On Entry: CMASTERM_ATTRIBUTE - color attribute to be used for new line
;
;  On Exit: Nothing
;****************************************************************************
CMASTERM_SCROLL PROC NEAR

        XOR     BL,BL                   ;select page 0
        MOV     BH,CMASTERM_ATTRIBUTE   ;load current screen attribute
        MOV     AH,6                    ;scroll window function 
        MOV     AL,1                    ;scroll one line
        XOR     CX,CX                   ;top row and column to zero
        MOV     DH,23                   ;to row 23, column 79
        MOV     DL,79
        INT     10H                     ;call DOS

        MOV     AH,2                    ;re-position cursor function
        XOR     BX,BX                   ;page 0  
        MOV     DH,22                   ;row 22, column 0
        MOV     DL,0
        INT     10H                     ;call DOS
        RET

CMASTERM_SCROLL ENDP

;****************************************************************************
; CMASTERM_CLEAR - clears the screen and home the cursor
;
; On Entry: CMASTERM_ATTRIBUTE - color attribute to be used for screen
;
;  On Exit: Nothing
;****************************************************************************
CMASTERM_CLEAR PROC NEAR

        MOV     BH,CMASTERM_ATTRIBUTE   ;load current screen attribute
        MOV     AH,6                    ;scroll window function
        MOV     AL,0                    ;0 lines will clear the area
        XOR     CX,CX                   ;top row and column to zero
        MOV     DH,23                   ;to row 23, column 79
        MOV     DL,79                   
        INT     10H                     ;call DOS

        MOV     AH,2                    ;re-position cursor function
        XOR     BX,BX                   ;                  
        XOR     DX,DX                   ;row and column to zero
        INT     10H                     ;call DOS
        RET

CMASTERM_CLEAR ENDP

;****************************************************************************
; CMASTERM_DEVSTAT - outputs a cursor position report to the port
;****************************************************************************
CMASTERM_DEVSTAT PROC NEAR

        JMP     CMASTERM_DEVSTAT1               ;jump over data area

        CMASTERM_DEVSTAT_STR    DB 20 DUP(0)    ;string storage space

CMASTERM_DEVSTAT1:

        MOV     AH,3                    ;read cursor position from screen
        XOR     BH,BH                   ;page 0
        INT     10H                     ;DH -> row, DL -> column
        MOV     CMASTERM_ROW,DH         ;save row for later
        MOV     CMASTERM_COL,DL         ;save column for later

        MOV     DI,OFFSET CMASTERM_DEVSTAT_STR
        MOV     CX,0
        MOV     BYTE PTR DS:[DI],27     ;load the escape character
        INC     DI
        INC     CX
        MOV     BYTE PTR DS:[DI],'['    ;load the [ character
        INC     DI
        INC     CX

        MOV     AL,CMASTERM_ROW         ;retrieve row to AL
        INC     AL                      ;adjust for ANSI row
        CALL    CMASTERM_NUMTOASCII     ;convert to ASCII string
        MOV     BYTE PTR DS:[DI],';'    ;store a separator
        INC     DI
        INC     CX

        MOV     AL,CMASTERM_COL         ;retrieve col to AL
        INC     AL                      ;adjust for ANSI row
        CALL    CMASTERM_NUMTOASCII     ;convert to ASCII string
        MOV     BYTE PTR DS:[DI],'R'    ;store the delimiter
        INC     DI
        INC     CX
        MOV     AH,FN_OUTSTR            ;function number to sent string
        MOV     AL,CS:CMASTERM_PORTNBR  ;retrieve saved port number
        MOV     BX,CS                   ;segment to BX
        MOV     DX,CX                   ;length to DX
        MOV     CX,OFFSET CMASTERM_DEVSTAT_STR  ;offset to CX
        CALL    CM_ACCESS               ;send it
        RET

CMASTERM_DEVSTAT ENDP

;****************************************************************************
; CMASTERM_CLEAREOL - clears from the current position to end of line
;****************************************************************************
CMASTERM_CLEAREOL PROC NEAR

        MOV     AH,3                    ;read cursor position from screen
        XOR     BH,BH                   ;page 0
        INT     10H                     ;DH -> row, DL -> column
        MOV     BH,CMASTERM_ATTRIBUTE   ;load current screen attribute
        MOV     AH,6                    ;scroll window function
        MOV     AL,0                    ;0 lines will clear the area
        MOV     CH,DH                   ;current row to CH
        MOV     CL,DL                   ;current column to CH
        MOV     DL,79                   
        INT     10H                     ;call DOS

CMASTERM_CLEAREOL ENDP

;****************************************************************************
; CMASTERM_NUMTOASCII - converts number in AL to ASCII bytes, stores in DS:DI, inc CX
;****************************************************************************
CMASTERM_NUMTOASCII PROC NEAR

        MOV     BYTE PTR DS:[DI],'0'    ;assume zero hundreds
        CMP     AL,100 
        JL      CMASTERM_NUMTOASCII1
        MOV     BL,100
        XOR     AH,AH                   ;clear AH register
        DIV     BL
        ADD     AL,48
        MOV     BYTE PTR DS:[DI],AL     ;store the number
        MOV     AL,AH                   ;remainder to AL

CMASTERM_NUMTOASCII1:
        INC     DI
        INC     CX
        MOV     BYTE PTR DS:[DI],'0'    ;assume zero tens
        CMP     AL,10 
        JL      CMASTERM_NUMTOASCII2
        MOV     BL,10
        XOR     AH,AH                   ;clear AH register
        DIV     BL
        ADD     AL,48
        MOV     BYTE PTR DS:[DI],AL     ;store the number
        MOV     AL,AH                   ;remainder to AL

CMASTERM_NUMTOASCII2:
        INC     DI
        INC     CX
        MOV     BYTE PTR DS:[DI],'0'    ;assume zero ones
        CMP     AL,1 
        JL      CMASTERM_NUMTOASCII3
        ADD     AL,48
        MOV     BYTE PTR DS:[DI],AL     ;store the number

CMASTERM_NUMTOASCII3:
        INC     DI
        INC     CX
        RET

CMASTERM_NUMTOASCII ENDP

;****************************************************************************
; CMASTERM_ASCIITONUM - converts ASCII string at DS:[SI] to AX numeric
;****************************************************************************
CMASTERM_ASCIITONUM PROC NEAR

        XOR     BX,BX                   ;clear accumulator
        PUSH    SI                      ;save pointer to ASCII number

CMASTERM_ASCIITONUM1:
        CMP     BYTE PTR DS:[SI],'0'    ;within lower limit?
        JL      CMASTERM_ASCIITONUM2    ;no, begin calculation 
        CMP     BYTE PTR DS:[SI],'9'    ;within upper limit?
        JG      CMASTERM_ASCIITONUM2    ;no, begin calculation 
        INC     SI
        JMP     CMASTERM_ASCIITONUM1

CMASTERM_ASCIITONUM2:
        MOV     CX,SI                   ;end of string to CX
        POP     SI                      ;restore SI pointer
        SUB     CX,SI                   ;length of string now in CX

CMASTERM_ASCIITONUM3:
        CMP     CX,0                    ;all done?
        JNE     CMASTERM_ASCIITONUM4    ;no, build a multiplier
        RET                             ;return to caller

CMASTERM_ASCIITONUM4:
        PUSH    CX                      ;save CX for later
        MOV     AX,1                    ;ones multiplier to AL

CMASTERM_ASCIITONUM5:
        CMP     CX,1                    ;done multiplying ?
        JE      CMASTERM_ASCIITONUM6    ;yes, continue calulation
        MOV     DL,10                   ;set up multiplier     
        MUL     DL                      ;multiply AL by 10
        DEC     CX
        JMP     CMASTERM_ASCIITONUM5

CMASTERM_ASCIITONUM6:
        POP     CX                      ;restore and decrement CX
        DEC     CX
        PUSH    BX                      ;save previous value
        MOV     BX,AX                   ;move multiplier in BX
        XOR     AX,AX                   ;clear AX
        XOR     DX,DX                   ;clear DX
        MOV     AL,BYTE PTR DS:[SI]     ;retrieve ASCII number
        SUB     AL,48                   ;convert to value
        MUL     BX                      ;multiply value by multiplier
        POP     BX                      ;retrieve saved accumulator
        ADD     BX,AX                   ;add value
        INC     SI                      ;point to next character
        JMP     CMASTERM_ASCIITONUM3    ;loop back for the next ASCII number

CMASTERM_ASCIITONUM ENDP

;****************************************************************************
; CMROW - returns actual row value to clipper
;****************************************************************************
; Syntax: m_result = CMROW()
;****************************************************************************
CMROW PROC FAR

        PUSH_REGS
        MOV     AH,3
        XOR     BX,BX
        INT     10H

        POP_REGS
        XOR     DL,DL
        XCHG    DL,DH
        PUSH    DX
        CALL    __RETNI
        ADD     SP,2
        RET

CMROW  ENDP

;****************************************************************************
; CMCOL - returns actual column value to clipper
;****************************************************************************
; Syntax: m_result = CMCOL()
;****************************************************************************
CMCOL PROC FAR

        PUSH_REGS
        MOV     AH,3
        XOR     BX,BX
        INT     10H

        POP_REGS
        XOR     DH,DH
        PUSH    DX
        CALL    __RETNI
        ADD     SP,2
        RET

CMCOL  ENDP

;*****************************************************************************
; CMCAPTURE - sets the terminal emulation capture file on/off
;****************************************************************************
; Syntax: CMCAPTURE(<capture file handle>)
; Passed: <capture file handle> - handle returned by FOPEN() or 0 if no capture
; Returns: numeric code, where:  0 = successful
;****************************************************************************
CMCAPTURE PROC FAR

        PUSH_REGS
        MOV     AX,1                    ;get the capture file handle
        PUSH    AX
        CALL    __PARNI                 ;handle now in AX
        ADD     SP,2                    ;fix the stack pointer

        MOV     CS:CMASTERM_HANDLE,AX   ;save for later

        POP_REGS
        XOR     AX,AX
        PUSH    AX
        CALL    __RETNI
        ADD     SP,2
        RET
CMCAPTURE ENDP

;*****************************************************************************
; CMPRINTER - sets the terminal print function on/off
;****************************************************************************
; Syntax: CMPRINTER(<switch>)
; Passed: <switch> - logical .T. or .F. to control printer switch
; Returns: numeric code, where:  0 = successful
;****************************************************************************
CMPRINTER PROC FAR

        PUSH_REGS
        MOV     AX,1                    ;get the printer toggle
        PUSH    AX
        CALL    __PARL                  ;logical now in AX
        ADD     SP,2                    ;fix the stack pointer

        MOV     CS:CMASTERM_PRINTER,AL  ;save for later

        POP_REGS
        XOR     AX,AX
        PUSH    AX
        CALL    __RETNI
        ADD     SP,2
        RET

CMPRINTER ENDP

;*****************************************************************************
; CMWRAP - sets the terminal to wrap lines or ignore screen overflow
;****************************************************************************
; Syntax: CMWRAP(<switch>)
; Passed: <switch> - logical .T. or .F. to control line wrap
; Returns: numeric code, where:  0 = successful
;****************************************************************************
CMWRAP PROC FAR

        PUSH_REGS
        MOV     AX,1                    ;get the printer toggle
        PUSH    AX
        CALL    __PARL                  ;logical now in AX
        ADD     SP,2                    ;fix the stack pointer

        MOV     CS:CMASTERM_WRAP,AL     ;save for later

        POP_REGS
        XOR     AX,AX
        PUSH    AX
        CALL    __RETNI
        ADD     SP,2
        RET

CMWRAP  ENDP

;*****************************************************************************
; CMCEXIT - sets the terminal to exit if carrier signal is lost
;****************************************************************************
; Syntax: CMCEXIT<switch>)
; Passed: <switch> - logical .T. or .F.
; Returns: numeric code, where:  0 = successful
;****************************************************************************
CMCEXIT PROC FAR

        PUSH_REGS
        MOV     AX,1                    ;get the printer toggle
        PUSH    AX
        CALL    __PARL                  ;logical now in AX
        ADD     SP,2                    ;fix the stack pointer

        MOV     CS:CMASTERM_CEXIT,AL    ;save for later

        POP_REGS
        XOR     AX,AX
        PUSH    AX
        CALL    __RETNI
        ADD     SP,2
        RET

CMCEXIT ENDP

;*****************************************************************************
; CMWRAP - sets the terminal to wrap lines or ignore screen overflow
;****************************************************************************
; Syntax: CMBEEP(<switch>)
; Passed: <switch> - logical .T. or .F. to control speaker beeps
; Returns: numeric code, where:  0 = successful
;****************************************************************************
CMBEEP PROC FAR

        PUSH_REGS
        MOV     AX,1                    ;get the beep toggle
        PUSH    AX
        CALL    __PARL                  ;logical now in AX
        ADD     SP,2                    ;fix the stack pointer
        MOV     CS:CMASTERM_BEEP,AL     ;save for later

        POP_REGS
        XOR     AX,AX
        PUSH    AX
        CALL    __RETNI
        ADD     SP,2
        RET

CMBEEP  ENDP

PROGSEG ENDS    ;end the code segment
END             ;end the program
