;****************************************************************************
; PROGRAM ----: CLIPFACE.ASM
; AUTHOR -----: Kevin E. Saffer 
; COPYRIGHT --: None, placed into the public domain
; CREATED ----: 10/2/1991 at 9:07
;****************************************************************************
; Implements the Clipper interface to the async routines
;****************************************************************************

.RADIX 10               ;use decimal values

PUSH_REGS MACRO         ;register push and pop macros for clipper interface
        PUSH    BP
        MOV     BP,SP
        PUSH    DI      
        PUSH    SI
        PUSH    DS
        PUSH    ES
        PUSH    SS
ENDM

POP_REGS MACRO
        POP     SS
        POP     ES
        POP     DS
        POP     SI
        POP     DI
        POP     BP
ENDM

;declare clipper parameter routines
EXTRN   __PARINFO:FAR	;get number of parameters or type of one
EXTRN   __PARINFA:FAR	;get size of array parameter or element type
EXTRN   __PARCSIZ:FAR	;get size of memory allocated for string parameter
EXTRN   __PARC:FAR	;get CHARACTER string, segment:offset in DX:AX
EXTRN   __RETC:FAR	;return string, push seg:off onto stack
EXTRN   __PARCLEN:FAR	;get length of a string parameter into ax
EXTRN   __RETCLEN:FAR	;return string of x length, push length:seg:off
EXTRN   __PARDS:FAR	;get DATE string, segment:offset in DX:AX
EXTRN   __RETDS:FAR	;return date string, push seg:off onto stack
EXTRN   __PARL:FAR	;get LOGICAL, value in AX
EXTRN   __RETL:FAR	;return logical, push 1 register onto stack
EXTRN   __PARND:FAR	;get numeric DOUBLE, segment:offset in DX:AX
EXTRN   __RETND:FAR	;return double, push 4 registers onto stack
EXTRN   __PARNI:FAR	;get numeric INTEGER, value in AX
EXTRN   __RETNI:FAR	;return integer, push 1 register onto stack
EXTRN   __PARNL:FAR	;get numeric LONG, value in DX:AX
EXTRN   __RETNL:FAR	;return long, push 2 registers onto stack

;declare clipper callable functions
PUBLIC CMOPEN           ;opens the comm port and installs an interrupt handler
PUBLIC CMCLOSE          ;remove the above handler      
PUBLIC CMSENDSTR        ;sends a clipper string to a port
PUBLIC CMSENDCHR        ;sends an ASCII character value to a port
PUBLIC CMTDCOUNT        ;count of bytes remaining in the transmit buffer
PUBLIC CMTDFLUSH        ;halts tranmitter and flushes the transmit buffer
PUBLIC CMTDXOFF         ;has transmitter halted due to received XOFF?
PUBLIC CMGETSTR         ;read data and optionally flush port buffer
PUBLIC CMGETCHR         ;reads an ASCII character from a port
PUBLIC CMRDCOUNT        ;count of bytes in the receive buffer
PUBLIC CMRDFLUSH        ;flushes the receive buffer
PUBLIC CMRDXOFF         ;has int handler sent an XOFF?
PUBLIC CMSTATUS         ;returns coded 16 byte string derived from LSR & MSR
PUBLIC CMSETXOFF        ;sets flow control on or off for download procedures
PUBLIC CMBREAK          ;sends BREAK to the modem
PUBLIC CMTOGDTR         ;toggles DTR to force carrier break
PUBLIC CMCHKSUM         ;returns checksum of passed string
PUBLIC CMCRC            ;returns cyclic redundancy check of passed string
PUBLIC CMSETPORT        ;changes the address/interrupt for a channel

;declare external routines
EXTRN   CM_ACCESS:FAR   ;access point for the resident routines

;declare function number equates for calls to the communication modules
FN_OPENPORT     EQU     1       ;opens a port and starts input
FN_OUTSTR       EQU     2       ;start buffered output
FN_OUTCHR       EQU     3       ;send unbuffered character
FN_TDCOUNT      EQU     4       ;return TD buffer count   
FN_TDFLUSH      EQU     5       ;flush TD buffer
FN_TDXOFF       EQU     6       ;return TD XOFF status           
FN_GETSTR       EQU     7       ;return input buffer with optional flush
FN_GETCHR       EQU     8       ;return input char with flush
FN_RDCOUNT      EQU     9       ;return RD buffer count   
FN_RDFLUSH      EQU     10      ;flush RD buffer
FN_RDXOFF       EQU     11      ;return TD XOFF status           
FN_SETXOFF      EQU     12      ;change flow control
FN_STATUS       EQU     13      ;return LSR+MSR in AX     
FN_BREAK        EQU     14      ;send BREAK signal to modem/terminal
FN_CHECKPORT    EQU     15      ;port availability check
FN_SETPORT      EQU     16      ;port address/interrupt change
FN_TOGGLEDTR    EQU     17      ;toggles DTR to force carrier break
FN_CLOSEPORT    EQU     18      ;closes port 

PROGSEG SEGMENT BYTE 'CODE'     ;set up our code segment, and
        ASSUME CS:PROGSEG       ;let MASM know of our intentions

;****************************************************************************
; CMOPEN - opens a communications port
;****************************************************************************
; Syntax: Result = CMOPEN(<port>,<baudrate>,<parity>,<databits>,<stopbits>)
; Passed: 5 numeric values and as follows:
; 
;         <portnbr> - 1,2,3, or 4 
;
;         <baudrate> - One of the following speeds:
;                           150
;                           300
;                           600
;                           1200
;                           2400
;                           4800
;                           9600
;                           19200
;                           38400
;                           57600
;                           0 = maximum (115,200)
;                                     
;         <parity> - where: 0 = none
;                           1 = odd
;                           2 = even
;
;         <databits> - 7 or 8
;         <stopbits> - 1 or 2
;
; Returns: numeric code, where:  0 = successful
;                               -1 = passed incorrect port number
;                               -2 = passed incorrect baudrate
;                               -3 = passed incorrect parity
;                               -4 = passed incorrect databits
;                               -5 = passed incorrect stopbits
;                               -6 = port not physically installed
;****************************************************************************
CMOPEN PROC FAR

        JMP SHORT CMOPEN_START            ;jump over the local data area

        CMOPEN_PORTNBR          DB      ?       
        CMOPEN_BAUDRATE         DW      ?       
        CMOPEN_PARITY           DB      ?       
        CMOPEN_DATABITS         DB      ?       
        CMOPEN_STOPBITS         DB      ?       

CMOPEN_START:
        PUSH_REGS                       ;save clipper registers
        MOV     AX,1                    ;get the port number
        PUSH    AX
        CALL    __PARNI                 ;integer now in AX
        ADD     SP,2                    ;fix the stack pointer
        MOV     CS:CMOPEN_PORTNBR,AL    ;store port number for for later

        MOV     AX,2                    ;get the baudrate
        PUSH    AX
        CALL    __PARNI                 ;integer now in AX
        ADD     SP,2                    ;fix the stack pointer

        CMP     AX,150                  ;compare against each acceptable value
        JZ      CMOPEN_BAUD_OK
        CMP     AX,300          
        JZ      CMOPEN_BAUD_OK
        CMP     AX,600          
        JZ      CMOPEN_BAUD_OK
        CMP     AX,1200         
        JZ      CMOPEN_BAUD_OK
        CMP     AX,2400         
        JZ      CMOPEN_BAUD_OK
        CMP     AX,4800         
        JZ      CMOPEN_BAUD_OK
        CMP     AX,9600         
        JZ      CMOPEN_BAUD_OK
        CMP     AX,19200        
        JZ      CMOPEN_BAUD_OK
        CMP     AX,38400        
        JZ      CMOPEN_BAUD_OK
        CMP     AX,57600        
        JZ      CMOPEN_BAUD_OK
        CMP     AX,0        
        JZ      CMOPEN_BAUD_OK

        POP_REGS                        
        XOR     AX,AX                   ;AX -> error code 
        SUB     AX,2
        PUSH    AX
        CALL    __RETNI                 ;return it to clipper
        ADD     SP,2
        RET

CMOPEN_BAUD_OK:
        MOV     CS:CMOPEN_BAUDRATE,AX   ;store it for later 

        MOV     AX,3                    ;get the parity value
        PUSH    AX
        CALL    __PARNI                 ;integer now in AX
        ADD     SP,2                    ;fix the stack pointer
        CMP     AX,0                    ;compare against each acceptable value
        JZ      CMOPEN_PARITY_OK      
        CMP     AX,1                    
        JZ      CMOPEN_PARITY_OK      
        CMP     AX,2                    
        JZ      CMOPEN_PARITY_OK      

        POP_REGS                        
        XOR     AX,AX                   ;AX -> error code 
        SUB     AX,3
        PUSH    AX
        CALL    __RETNI                 ;return it to clipper
        ADD     SP,2
        RET

CMOPEN_PARITY_OK:     
        MOV     CS:CMOPEN_PARITY,AL     ;store it for later 

        MOV     AX,4                    ;get the databits value
        PUSH    AX
        CALL    __PARNI                 ;integer now in AX
        ADD     SP,2                    ;fix the stack pointer
        CMP     AX,7                    ;compare against each acceptable value
        JZ      CMOPEN_DATABITS_OK      
        CMP     AX,8                    
        JZ      CMOPEN_DATABITS_OK      

        POP_REGS                        
        XOR     AX,AX                   ;AX -> error code 
        SUB     AX,4
        PUSH    AX
        CALL    __RETNI                 ;return it to clipper
        ADD     SP,2
        RET

CMOPEN_DATABITS_OK:
        MOV     CS:CMOPEN_DATABITS,AL   ;store it for later 

        MOV     AX,5                    ;get the stopbits value
        PUSH    AX
        CALL    __PARNI                 ;integer now in AX
        ADD     SP,2                    ;fix the stack pointer
        CMP     AX,1                    ;compare against each acceptable value
        JZ      CMOPEN_STOPBITS_OK      
        CMP     AX,2                    
        JZ      CMOPEN_STOPBITS_OK      

        POP_REGS                        ;return error code
        XOR     AX,AX                   ;AX -> error code 
        SUB     AX,5
        PUSH    AX
        CALL    __RETNI                 ;return it to clipper
        ADD     SP,2
        RET

CMOPEN_STOPBITS_OK:     
        MOV     CS:CMOPEN_STOPBITS,AL   ;store it for later 

CMOPEN_CHECKPORT:     
        MOV     AL,CS:CMOPEN_PORTNBR    ;retrieve port number
        MOV     AH,FN_CHECKPORT         ;check port function number             
        CALL    CM_ACCESS               ;call comm routines
        CMP     AX,0                    ;port available?
        JE      CMOPEN_OPENPORT         ;yes, open it

        CMP     CS:CMOPEN_PORTNBR,3     ;port 3 or 4?
        JL      CMOPEN_OPENPORT         ;no, go ahead and fail
        MOV     AL,CS:CMOPEN_PORTNBR    ;retrieve port number
        MOV     AH,FN_SETPORT           ;set port function number             
        CMP     AL,3                    ;port 3?
        JNE     CMOPEN_CHECKPORT1       ;no, set up 4
        MOV     BX,0C94H                ;PS/2 port 3 address
        MOV     CX,4                    ;port 3 interrupt
        JMP     CMOPEN_CHECKPORT2       ;check for it

CMOPEN_CHECKPORT1:     
        MOV     BX,0C9CH                ;PS/2 port 4 address
        MOV     CX,3                    ;port 4 interrupt

CMOPEN_CHECKPORT2:     
        CALL    CM_ACCESS               ;set the new address/int

CMOPEN_OPENPORT:
        MOV     AH,FN_OPENPORT          ;open port function number             
        MOV     AL,CS:CMOPEN_PORTNBR    ;retrieve port number
        MOV     BX,CS:CMOPEN_BAUDRATE   ;retrieve baudrate
        XOR     CX,CX                   ;clear CX for the line settings

CMOPEN_ADD_DATABITS:
        CMP     CS:CMOPEN_DATABITS,7    ;7 data bits?
        JNE     CMOPEN_ADD_DATABITS_8   ;no, switch to 8
        ADD     CL,00000010b            ;bit map for 7 data bits
        JMP     CMOPEN_ADD_STOPBITS     ;continue processing

CMOPEN_ADD_DATABITS_8:
        ADD     CL,00000011b            ;bit map for 8 data bits

CMOPEN_ADD_STOPBITS:
        CMP     CS:CMOPEN_STOPBITS,1    ;1 stop bit?         
        JNE     CMOPEN_ADD_STOPBITS_2   ;no, switch to 2
        JMP     CMOPEN_ADD_PARITY       ;leave bit off for 1 stop bit

CMOPEN_ADD_STOPBITS_2:
        ADD     CL,00000100b            ;bit map for 2 stop bits

CMOPEN_ADD_PARITY:
        CMP     CS:CMOPEN_PARITY,0      ;no parity?
        JNE     CMOPEN_ADD_PARITY_1     ;no, check for odd
        JMP     CMOPEN_SETUP_LINE       ;leave bit off for no parity

CMOPEN_ADD_PARITY_1:              
        CMP     CS:CMOPEN_PARITY,1      ;odd parity?
        JNE     CMOPEN_ADD_PARITY_2     ;no, check for even
        ADD     CL,00001000b            ;bit map for odd parity
        JMP     CMOPEN_SETUP_LINE

CMOPEN_ADD_PARITY_2:
        ADD     CL,00011000b            ;bit map for even parity

CMOPEN_SETUP_LINE:
        CALL    CM_ACCESS               ;call communications routine
        CMP     AX,0                    ;routines install correctly?
        JE      CMOPEN_RETURN           ;yes, return sucess code
        XOR     AX,AX
        SUB     AX,6                    ;port error code to AX

CMOPEN_RETURN:
        POP_REGS                        
        PUSH    AX
        CALL    __RETNI                 ;return it to clipper
        ADD     SP,2
        RET

CMOPEN ENDP

;****************************************************************************
; CMCLOSE - closes a comunications port
;****************************************************************************
;  Syntax: Result = CMCLOSE(<port>)
;  Passed: <portnbr> - 1,2,3, or 4 
; Returns: numeric code, where:  0 = successful
;                               -1 = incorrect port number
;****************************************************************************
CMCLOSE PROC FAR

        PUSH_REGS                       ;save clipper registers
        MOV     AX,1                    ;get the port number
        PUSH    AX
        CALL    __PARNI                 ;integer now in AX
        ADD     SP,2                    ;fix the stack pointer

        CMP     AX,1                    ;compare against each acceptable value
        JZ      CMCLOSE_PORT_OK      
        CMP     AX,2                    
        JZ      CMCLOSE_PORT_OK      
        CMP     AX,3                    
        JZ      CMCLOSE_PORT_OK      
        CMP     AX,4                    
        JZ      CMCLOSE_PORT_OK      
        POP_REGS                        
        XOR     AX,AX                   ;AX -> port error code 
        SUB     AX,1
        PUSH    AX
        CALL    __RETNI                 ;return it to clipper
        ADD     SP,2
        RET

CMCLOSE_PORT_OK:
        MOV     AH,FN_CLOSEPORT         ;close port function to AH
        CALL    CM_ACCESS               ;call communication routines
        POP_REGS                        
        PUSH    AX                      
        CALL    __RETNI                 ;return it to clipper
        ADD     SP,2
        RET

CMCLOSE ENDP

;****************************************************************************
; CMSENDSTR - send string to a comm port 
;****************************************************************************
;  Syntax: CMSENDSTR(<port>,<string>)
;  Passed: <port> = a previously opened port number
;          <string> = Any character string or character memvar 
; Returns: numeric code, where:  0 = successful
;                               -1 = incorrect port number
;                               -2 = string error 
;****************************************************************************
CMSENDSTR PROC FAR

        JMP SHORT CMSENDSTR_START         ;jump over the data area

        CMSENDSTR_PORTNBR       DB      ?       
        CMSENDSTR_STRLEN        DW      ?

CMSENDSTR_START:
        PUSH_REGS                       ;save vital registers
        MOV     AX,1                    ;get the port number
        PUSH    AX
        CALL    __PARNI                
        ADD     SP,2                    ;fix the stack pointer
        MOV     CS:CMSENDSTR_PORTNBR,AL ;save for later    
        MOV     AX,2                    ;get the length of the string
        PUSH    AX
        CALL    __PARCLEN              
        ADD     SP,2                    ;fix the stack pointer
        CMP     AX,0
        JNZ     CS:CMSENDSTR_LEN_OK
        POP_REGS                        ;return error code
        XOR     AX,AX                   ;AX -> error code 
        SUB     AX,2
        PUSH    AX
        CALL    __RETNI                 ;return it to clipper
        ADD     SP,2
        RET

CMSENDSTR_LEN_OK:
        MOV     CS:CMSENDSTR_STRLEN,AX  ;save for later
        MOV     AX,2                    ;get the pointer to the string
        PUSH    AX
        CALL    __PARC              
        ADD     SP,2                    ;fix the stack pointer
        PUSH    DX                      ;push the pointers to the string, and
        POP     BX                      ;pop them into required registers
        PUSH    AX
        POP     CX                      
        MOV     DX,CS:CMSENDSTR_STRLEN  ;retrieve string length
        MOV     AL,CS:CMSENDSTR_PORTNBR ;retrieve port number into AL
        MOV     AH,FN_OUTSTR            ;function number to AH
        CALL    CM_ACCESS

        POP_REGS                        ;restore caller's registers
        PUSH    AX                      ;push return value onto stack
        CALL    __RETNI                 ;return it to clipper
        ADD     SP,2                    ;fix up the stack
        RET                             ;return

CMSENDSTR ENDP

;******************************************************************************
; CMSENDCHR - sends an ASCII value to the port
;******************************************************************************
;  Syntax: m_result = CMSENDCHR(<port>,<ASCII value>)
;  Passed: <port> - port number to send to
; Returns: numeric code, where:  0 = successful
;                               -1 = incorrect port number
;****************************************************************************
CMSENDCHR PROC FAR

        JMP SHORT CMSENDCHR_START         ;jump over the data area

        CMSENDCHR_PORTNBR        DB      ?       

CMSENDCHR_START:
        PUSH_REGS                       ;save vital registers
        MOV     AX,1                    ;get the port number
        PUSH    AX
        CALL    __PARNI                
        ADD     SP,2                    ;fix the stack pointer

        MOV     CS:CMSENDCHR_PORTNBR,AL ;save for later use
        MOV     AX,2                    ;get the character to send
        PUSH    AX
        CALL    __PARNI                
        ADD     SP,2                    ;fix the stack pointer
        XOR     BX,BX
        MOV     BL,AL                   ;character to bl
        MOV     AH,FN_OUTCHR 
        MOV     AL,CS:CMSENDCHR_PORTNBR
        CALL    CM_ACCESS

        POP_REGS                        
        PUSH    AX
        CALL    __RETNI                 ;return value to clipper
        ADD     SP,2
        RET

CMSENDCHR ENDP

;****************************************************************************
; CMTDCOUNT - returns the number of characters left to transmit
;  Syntax: m_result = CMTDCOUNT(<port>)
;  Passed: <port> - port number to read
; Returns: numeric code, where:  0-? = count of chars in buffer
;                               -1 = incorrect port number
;****************************************************************************
CMTDCOUNT PROC FAR

        PUSH_REGS                       ;save clipper registers
        MOV     AX,1                    ;get the port number
        PUSH    AX
        CALL    __PARNI                
        ADD     SP,2                    ;fix the stack pointer

        MOV     AH,FN_TDCOUNT           ;function number to AH
        CALL    CM_ACCESS               ;call communications module

        POP_REGS                        ;return result to clipper
        PUSH    AX
        CALL    __RETNI
        ADD     SP,2
        RET

CMTDCOUNT ENDP

;****************************************************************************
; CMTDFLUSH - halts transmitter and flushes output buffer
;  Syntax: m_result = CMTDFLUSH(<port>)
;  Passed: <port> - port number to halt
; Returns: numeric code, where:  0 = successful
;                               -1 = incorrect port number
;****************************************************************************
CMTDFLUSH PROC FAR

        PUSH_REGS                       ;save clipper registers
        MOV     AX,1                    ;get the port number
        PUSH    AX
        CALL    __PARNI                
        ADD     SP,2                    ;fix the stack pointer

        MOV     AH,FN_TDFLUSH           ;function number to AH
        CALL    CM_ACCESS               ;call communications module

        POP_REGS                        ;return result to clipper
        PUSH    AX
        CALL    __RETNI
        ADD     SP,2
        RET

CMTDFLUSH ENDP

;****************************************************************************
; CMTDXOFF - returns .T. if tranmitter has been halted by an XOFF
;  Syntax: m_result = CMTDXOFF(<port>)
;  Passed: <port> - port number to check
; Returns: numeric code, where:  0 = running, 1 = halted
;                               -1 = incorrect port number
;****************************************************************************
CMTDXOFF PROC FAR

        PUSH_REGS                       ;save clipper registers
        MOV     AX,1                    ;get the port number
        PUSH    AX
        CALL    __PARNI                
        ADD     SP,2                    ;fix the stack pointer

        MOV     AH,FN_TDXOFF            ;function number to AH
        CALL    CM_ACCESS               ;call communications module

        POP_REGS                        ;return result to clipper
        PUSH    AX
        CALL    __RETNI
        ADD     SP,2
        RET

CMTDXOFF ENDP

;******************************************************************************
; CMGETSTR - read a string from the comm port 
;******************************************************************************
;  Syntax: m_result = CMGETSTR(<port>,<killbuff>)
;  Passed: <port> - port number to read
; Returns: All characters received or a null string if empty/bad port
;****************************************************************************
CMGETSTR PROC FAR

        JMP SHORT CMGETSTR1         ;jump over the data area

        CMGETSTR_PORTNBR        DB      ?       
        CMGETSTR_NULLSTR        DB      0
        CMGETSTR_KILLBUFF       DB      0

CMGETSTR1:
        PUSH_REGS                       ;save vital registers
        MOV     AX,1                    ;get the port number
        PUSH    AX
        CALL    __PARNI                
        ADD     SP,2                    ;fix the stack pointer
        MOV     CS:CMGETSTR_PORTNBR,AL ;save parameter for later    

        MOV     AX,2                    ;get the kill buffer flag
        PUSH    AX
        CALL    __PARL                 
        ADD     SP,2                            ;fix the stack pointer
        MOV     CS:CMGETSTR_KILLBUFF,AL         ;save it for later

        MOV     AL,CS:CMGETSTR_PORTNBR  ;retrieve port number
        MOV     AH,FN_GETSTR            ;function number to AH
        XOR     BX,BX
        MOV     BL,CS:CMGETSTR_KILLBUFF ;buffer clear flag to BX
        CALL    CM_ACCESS               ;call communications module
        CMP     AX,1                    ;result >= 1?       
        JGE     CMGETSTR2               ;yes, return string to clipper

        POP_REGS                        ;return a null string to clipper
        PUSH    CS
        MOV     AX,OFFSET CS:CMGETSTR_NULLSTR
        PUSH    AX
        CALL    __RETC                  ;return it to clipper
        ADD     SP,4
        RET

CMGETSTR2:
        POP_REGS                        ;return string to clipper
        PUSH    AX                      ;push length of buffer 
        PUSH    DX                      ;push segment of buffer
        PUSH    BX                      ;push it onto the stack
        CALL    __RETCLEN               ;return to clipper
        ADD     SP,6                    ;fix up the stack
        RET

CMGETSTR ENDP

;******************************************************************************
; CMGETCHR - reads an ASCII value from the input buffer
;******************************************************************************
;  Syntax: m_result = CMGETCHR(<port>)
;  Passed: <port> - port number to read
; Returns: numeric code, where:  0-255 = ASCII code of character
;                               -1 = incorrect port number
;                               -2 = no character awaiting
;****************************************************************************
CMGETCHR PROC FAR

        PUSH_REGS                       ;save vital registers
        MOV     AX,1                    ;get the port number into AX
        PUSH    AX
        CALL    __PARNI                
        ADD     SP,2                    ;fix the stack pointer

        MOV     AH,FN_GETCHR            ;function number to AH
        CALL    CM_ACCESS               ;call communications module

        POP_REGS                        
        PUSH    AX
        CALL    __RETNI                 ;return it to clipper
        ADD     SP,2
        RET

CMGETCHR ENDP

;****************************************************************************
; CMRDCOUNT - returns the number of characters left in the receive buffer
;  Syntax: m_result = CMRDCOUNT(<port>)
;  Passed: <port> - port number to read
; Returns: numeric code, where:  0-? = number of characters
;                               -1 = incorrect port number
;****************************************************************************
CMRDCOUNT PROC FAR

        PUSH_REGS                       ;save clipper registers
        MOV     AX,1                    ;get the port number
        PUSH    AX
        CALL    __PARNI                
        ADD     SP,2                    ;fix the stack pointer

        MOV     AH,FN_RDCOUNT           ;function number to AH
        CALL    CM_ACCESS               ;call communications module

        POP_REGS                        ;return result to clipper
        PUSH    AX
        CALL    __RETNI
        ADD     SP,2
        RET

CMRDCOUNT ENDP

;****************************************************************************
; CMRDFLUSH - flushes the receive buffer
;  Syntax: m_result = CMRDFLUSH(<port>)
;  Passed: <port> - port number to read
; Returns: numeric code, where:  0 = successful
;                               -1 = incorrect port number
;****************************************************************************
CMRDFLUSH PROC FAR

        PUSH_REGS                       ;save clipper registers
        MOV     AX,1                    ;get the port number
        PUSH    AX
        CALL    __PARNI                
        ADD     SP,2                    ;fix the stack pointer

        MOV     AH,FN_RDFLUSH           ;function number to AH
        CALL    CM_ACCESS               ;call communications module

        POP_REGS                        ;return result to clipper
        PUSH    AX
        CALL    __RETNI
        ADD     SP,2
        RET

CMRDFLUSH ENDP

;****************************************************************************
; CMRDXOFF - returns .T. if int handler has issued an XOFF
;  Syntax: m_result = CMRDXOFF(<port>)
;  Passed: <port> - port number to check
; Returns: numeric code, where:  0 = running, 1 = halted
;                               -1 = incorrect port number
;****************************************************************************
CMRDXOFF PROC FAR

        PUSH_REGS                       ;save clipper registers
        MOV     AX,1                    ;get the port number
        PUSH    AX
        CALL    __PARNI                
        ADD     SP,2                    ;fix the stack pointer

        MOV     AH,FN_RDXOFF            ;function number to AH
        CALL    CM_ACCESS               ;call communications module

        POP_REGS                        ;return result to clipper
        PUSH    AX
        CALL    __RETNI
        ADD     SP,2
        RET

CMRDXOFF ENDP

;****************************************************************************
; CMSTATUS - returns coded 16 byte string derived from LSR & MSR
;  Syntax: m_result = CMSTATUS(<port>)
;  Passed: <port> = port number to check   
; Returns: Null if port error or 16 byte string encoded with "0" or "1" as follows:
;
; Line Control Register                 Modem Control Register
; Pos Description                       Pos Description
; --- --------------------------------  --- --------------------------------
;   1 (Always Zero)                       9 Rec. Line Signal Detect (CD)
;   2 Transmitter Holding Reg. Empty     10 Ring Indicator (RI)
;   3 Transmitter Shift Register Empty   11 Data Set Ready (DSR)
;   4 Break Interrupt                    12 Clear To Send (CTS)
;   5 Framing Error                      13 Delta Rec. Line Signal Detect (CD)
;   6 Parity Error                       14 Trailing Edge Ring Indicator (RI)
;   7 Overrun Error                      15 Delta Data Set Ready (DSR)
;   8 Data Ready                         16 Delta Clear To Send (CTS)
;****************************************************************************
CMSTATUS PROC FAR

        JMP SHORT CMSTATUS1

        CMSTATUS_STRING        DB      16 DUP('0')      ;return string
        CMSTATUS_NULL          DB      0                ;error return string

CMSTATUS1:
        PUSH_REGS                       ;save clipper registers
        MOV     AX,1                    ;get the port number
        PUSH    AX
        CALL    __PARNI                
        ADD     SP,2                    ;fix the stack pointer

        MOV     AH,FN_STATUS            ;function number to AH
        CALL    CM_ACCESS               ;call communications module
        CMP     AX,0                    ;error?
        JG      CMSTATUS2               ;no, continue processing

        POP_REGS                        ;return null string to clipper
        PUSH    CS                      
        MOV     AX,OFFSET CS:CMSTATUS_NULL
        PUSH    AX
        CALL    __RETC                  
        ADD     SP,4
        RET

CMSTATUS2:
        PUSH    AX                              ;save LSR/MSR
        MOV     AX,08000H                       ;bit mask for bit 15
        MOV     BX,2                            ;divisor to BX
        MOV     DI,OFFSET CS:CMSTATUS_STRING    ;point to the string
        MOV     CX,16                           ;16 repititions
        XOR     DX,DX                           ;clear DX

CMSTATUS3:
        POP     DX                      ;restore LSR/MSR
        TEST    AX,DX                   ;bit set?
        JNZ     CMSTATUS4               ;yes, continue processing
        MOV     BYTE PTR CS:[DI],'0'    ;update string with 0
        JMP     CMSTATUS5

CMSTATUS4:
        MOV     BYTE PTR CS:[DI],'1'    ;update string with 1

CMSTATUS5:
        INC     DI                      ;point to next byte
        DEC     CX                      ;decrement counter
        CMP     CX,0                    ;all done?
        JE      CMSTATUS6               ;yes, continue processing
        PUSH    DX                      ;save LSR/MSR
        XOR     DX,DX                   ;clear DX for the division
        DIV     BX                      ;set mask to check next bit
        JMP     CMSTATUS3               ;loop back

CMSTATUS6:
        POP_REGS                        
        MOV     AX,16                   ;return 16 bytes
        PUSH    AX
        PUSH    CS                      ;return coded string to clipper
        MOV     AX,OFFSET CS:CMSTATUS_STRING
        PUSH    AX
        CALL    __RETCLEN               
        ADD     SP,6
        RET

CMSTATUS ENDP

;****************************************************************************
; CMSETXOFF - sets flow control on or off for download procedures
;  Syntax: m_result = CMSETXOFF(<port>,<flag>)
;  Passed: <port>    - port number to monitor
;          <flag> - .T. to start, .F. to halt flow control
; Returns: numeric code, where:  0 = successful
;                               -1 = incorrect port number
;****************************************************************************
CMSETXOFF PROC FAR

        JMP SHORT CMSETXOFF1    ;jump over the data area

        CMSETXOFF_PORTNBR          DB      ?       ;passed port number

CMSETXOFF1:
        PUSH_REGS                       ;save clipper registers
        MOV     AX,1                    ;get the port number
        PUSH    AX
        CALL    __PARNI                
        ADD     SP,2                            ;fix the stack pointer
        MOV     CS:CMSETXOFF_PORTNBR,AL         ;save parameter for later    

        MOV     AX,2                            ;get the flag .T. or .F.
        PUSH    AX
        CALL    __PARL                 
        ADD     SP,2                            ;fix the stack pointer

        MOV     BX,AX                           ;value to BX (false = 0)
        MOV     AL,CS:CMSETXOFF_PORTNBR         ;restore port number
        MOV     AH,FN_SETXOFF                   ;function code to AH
        CALL    CM_ACCESS                       ;call communications module

        POP_REGS                ;return result to clipper 
        XOR     AX,AX
        PUSH    AX              
        CALL    __RETNI                 
        ADD     SP,2
        RET

CMSETXOFF ENDP

;****************************************************************************
; CMBREAK - sends a BREAK to the passed port
;  Syntax: m_result = CMBREAK(<port>)
;  Passed: <port>    - port number to send to
; Returns: numeric code, where:  0 = successful
;                               -1 = incorrect port number
;****************************************************************************
CMBREAK PROC FAR

        PUSH_REGS                       ;save clipper registers
        MOV     AX,1                    ;get the port number
        PUSH    AX
        CALL    __PARNI                
        ADD     SP,2                    ;fix the stack pointer

        MOV     AH,FN_BREAK             ;break function number to AH
        CALL    CM_ACCESS               ;call communication module

        POP_REGS                        ;return result to clipper
        PUSH    AX
        CALL    __RETNI                 
        ADD     SP,2
        RET

CMBREAK ENDP

;****************************************************************************
; CMTOGDTR - toggles the DTR line to force carrier break
;  Syntax: m_result = CMTOGDTR(<port>)
;  Passed: <port> - port number to read
; Returns: numeric code, where:  0 = successful
;                               -1 = incorrect port number
;****************************************************************************
CMTOGDTR PROC FAR

        PUSH_REGS                       ;save clipper registers
        MOV     AX,1                    ;get the port number
        PUSH    AX
        CALL    __PARNI                
        ADD     SP,2                    ;fix the stack pointer

        MOV     AH,FN_TOGGLEDTR         ;function number to AH
        CALL    CM_ACCESS               ;call communications module

        POP_REGS                        ;return result to clipper
        PUSH    AX
        CALL    __RETNI
        ADD     SP,2
        RET

CMTOGDTR ENDP

;****************************************************************************
; CMCHKSUM - returns checksum of passed string
;  Syntax: m_result = CMCHKSUM(<string>)
;  Passed: <string>  - string up to 1024 characters to be processed
; Returns: ASCII value of the checksum byte
;****************************************************************************
CMCHKSUM PROC FAR

        JMP     CMCHKSUM_START

        CMCHKSUM_STRLEN DW      ?       ;saved string length

CMCHKSUM_START:
        PUSH_REGS                       ;save clipper registers
        MOV     AX,1                    ;get the length of the string into CX
        PUSH    AX
        CALL    __PARCLEN              
        ADD     SP,2                    ;fix the stack pointer
        MOV     CS:CMCHKSUM_STRLEN,AX   ;save for later

        MOV     AX,1                    ;get the str seg:off into DX:AX
        PUSH    AX
        CALL    __PARC              
        ADD     SP,2                    ;fix the stack pointer

        PUSH    DS                      ;save DS for later
        PUSH    DX                      ;set DS to string segment 
        POP     DS
        MOV     SI,AX                   ;offset of string to SI
        XOR     AX,AX                   ;clear AX
        XOR     BX,BX                   ;clear AX
        XOR     DX,DX                   ;clear DX
        MOV     CX,CS:CMCHKSUM_STRLEN   ;count to CX

CMCHKSUM_LOOP:
        CMP     CX,0                    ;all done?
        JE      CMCHKSUM_DIVIDE
        MOV     BL,BYTE PTR DS:[SI]     ;get the byte to BL
        INC     SI                      ;point to the next byte
        ADC     AX,BX                   ;add the byte to AX, check for overflow
        JNC     CMCHKSUM_NOCARRY        ;no overflow, continue
        INC     DX                      ;increment DX

CMCHKSUM_NOCARRY:
        DEC     CX
        JMP     CMCHKSUM_LOOP

CMCHKSUM_DIVIDE:
        MOV     BX,256
        DIV     BX                     ;divide sum in DX:AX by 256

        POP     DS                      ;restore data segment
        POP_REGS
        PUSH    DX                      ;return remainder to clipper
        CALL    __RETNI                 
        ADD     SP,2
        RET

CMCHKSUM ENDP

;****************************************************************************
; CMCRC - returns cyclic redundancy chech of passed string
;  Syntax: m_result = CMCRC(<string>)
;  Passed: <string>  - string up to 1024 characters to be processed
; Returns: 2 byte string containing CrcHi, CrcLo
;****************************************************************************
CMCRC PROC FAR

        JMP     CMCRC_START

        CMCRC_STRLEN DW      ?          ;saved string length
        CMCRC_RETSTR DB      3 DUP(0)   ;return string

CMCRC_START:
        PUSH_REGS                       ;save clipper registers
        MOV     AX,1                    ;get the length of the string into CX
        PUSH    AX
        CALL    __PARCLEN              
        ADD     SP,2                    ;fix the stack pointer
        MOV     CS:CMCRC_STRLEN,AX      ;save for later

        MOV     AX,1                    ;get the str seg:off into DX:AX
        PUSH    AX
        CALL    __PARC              
        ADD     SP,2                    ;fix the stack pointer

        PUSH    DS                      ;save DS for later
        PUSH    DX                      ;set DS to string segment 
        POP     DS
        MOV     SI,AX                   ;offset of string to SI
        MOV     CX,CS:CMCRC_STRLEN      ;count to CX

CMCRC_CALC_XOR:
        MOV     BX,0                    ;initialize CRC byte to 0
        MOV     DI,1021h                ;CRC xor value

CMCRC_BYTE_LOOP:
        PUSH    CX                      ;Save the data string length
        LODSB                           ;Move data byte at [SI] to AL
        MOV     CX,8                    ;Loop 8 bits of each data byte

CMCRC_BIT_LOOP:
        MOV     AH,AL                   ;Move data byte to work reg
        AND     AH,80H                  ;Isolate Data(8) MSB bit
        MOV     DX,BX                   ;Save CRC word
        SHL     DX,1                    ;Left shift the CRC word
        AND     BH,80H                  ;Isolate CRC(1) LSB bit
        XOR     BH,AH                   ;Flag = Data(8) Xor CRC(1)
        MOV     BX,DX                   ;Restore CRC word
        JZ      CMCRC_NEXT_BIT          ;ZFlag is set if Flag = 0
        XOR     BX,DI                   ;CRC bits 16,12,5 Xor 1

CMCRC_NEXT_BIT:
        SHL     AL,1                    ;Shift thru each data bit
        LOOP    CMCRC_BIT_LOOP          ;Loop 8 times thru data byte
        POP     CX                      ;Restore data string length
        LOOP    CMCRC_BYTE_LOOP         ;Loop thru data string

        MOV     DI,OFFSET CS:CMCRC_RETSTR       ;Move string bytes to storage
        MOV     CS:[DI],BH  
        INC     DI
        MOV     CS:[DI],BL  
        POP     DS

        POP_REGS                        ;set up to return string to clipper
        MOV     AX,2
        PUSH    AX
        PUSH    CS
        MOV     AX,OFFSET CS:CMCRC_RETSTR
        PUSH    AX
        CALL    __RETCLEN
        ADD     SP,6

        RET
         
CMCRC ENDP

;****************************************************************************
; CMSETPORT - changes a port's address and interrupt, use VERY carefully
;****************************************************************************
; Syntax: Result = CMSETPORT(<port>,<address>,<interrupt>)
; 
; Passed: <portnbr>   - 1,2,3, or 4 
;         <address>   - DECIMAL port address
;         <interrupt) - interrupt level, 2 through 7 supported
;
; Returns: numeric code, where:  0 = successful
;                               -1 = passed incorrect port number
;                               -2 = port not physically installed
;****************************************************************************
CMSETPORT PROC FAR

        JMP SHORT CMSETPORT_START       ;jump over the local data area

        CMSETPORT_PORTNBR       DB      ?       
        CMSETPORT_ADDRESS       DW      ?       

CMSETPORT_START:
        PUSH_REGS                       ;save clipper registers
        MOV     AX,1                    ;get the port number
        PUSH    AX
        CALL    __PARNI                 ;integer now in AX
        ADD     SP,2                    ;fix the stack pointer
        MOV     CS:CMSETPORT_PORTNBR,AL ;store port number for for later

        MOV     AX,2                    ;get the address
        PUSH    AX
        CALL    __PARNI                 ;integer now in AX
        ADD     SP,2                    ;fix the stack pointer
        MOV     CS:CMSETPORT_ADDRESS,AX ;store address for for later

        MOV     AX,3                    ;get the interrupt level
        PUSH    AX
        CALL    __PARNI                 ;integer now in AX
        ADD     SP,2                    ;fix the stack pointer

        XOR     CX,CX                   ;clear CX
        MOV     CL,AL                   ;int to CL
        MOV     BX,CS:CMSETPORT_ADDRESS ;address to BX
        MOV     AL,CS:CMSETPORT_PORTNBR ;port number to AL
        MOV     AH,FN_SETPORT           ;set port function number
        CALL    CM_ACCESS               ;call comm routine

        POP_REGS                        
        PUSH    AX
        CALL    __RETNI                 ;return result to clipper
        ADD     SP,2
        RET

CMSETPORT ENDP

PROGSEG ENDS    ;end the code segment
END             ;end the program
