;*****************************************************************************
; PROGRAM ----: ASYNC.ASM 
; AUTHOR -----: Kevin E. Saffer 
; COPYRIGHT --: None, placed into the public domain
; CREATED ----: 3/6/1993 at 10:35
;*****************************************************************************
; This program implements the low-level async routines.
;*****************************************************************************

.RADIX 10               ;use decimal values

PUSH_REGS MACRO
	PUSH	BP
	PUSH	SI
	PUSH	DI
	PUSH	DS
	PUSH	ES
ENDM

POP_REGS MACRO
	POP	ES                      ;restore registers
	POP	DS
	POP	DI
	POP	SI
	POP	BP
ENDM

PUBLIC CM_ACCESS        ;communications access function

;declare global system structures and equates
;structure for holding the fixed and operating parameters for a port
COM_PORT STRUC
        COM_IH	        DW	?	;(offset) address of interrupt handler
        COM_ADDRESS	DW	?	;i/o address
        COM_IVA	        DW	?	;interrupt vector address
        COM_BIT	        DB	?	;IRQ mask bit
        COM_IVSEG       DW      ?       ;saved int vector segment
        COM_IVOFF       DW      ?       ;saved int vector offset
        COM_OPTIONS	DW	?	;options set for port at configuration
        COM_FLAGS	DB	?	;various bit flags
        COM_OUTFLOW	DB	?	;flow-control character awaiting transmission
        COM_IN_CTR	DW	?	;counter for bytes in input buffer
        COM_IN_LO	DW	?	;low threshold for input flow control
        COM_IN_HI	DW	?	;high threshold for input flow control
        COM_IN_PTR	DW	?	;input pointer (offset)
        COM_IN_SEG	DW	?	;input segment
        COM_RD_PTR	DW	?	;read pointer (offset)
        COM_IN_FIRST	DW	?	;base offset of input buffer
        COM_IN_LIMIT	DW	?	;limit offset of input buffer
        COM_OUT_PTR	DW	?	;output pointer (offset)
        COM_OUT_SEG	DW	?	;output segment
        COM_OUT_LIMIT	DW	?	;output limit (offset)
COM_PORT ENDS
COM_SIZE	=	SIZE COM_PORT

;COM port register offsets
COM_TXB         =	0	;Transmit Buffer
COM_RXB         =	0	;Receive Buffer
COM_DLL         =	0	;Divisor Latch LSB
COM_DLM         =	1	;Divisor Latch MSB
COM_IER         =	1	;Interrupt Enable Register
COM_IIR         =	2	;Interrupt Identification Register
COM_LCR         =	3	;Line-Control Register
COM_MCR         =	4	;Modem Control Register
COM_LSR         =	5	;Line Status Register
COM_MSR         =	6	;Modem Status Register

;definitions of bit flags for the COM_FLAGS field
CF_ISF		=	01H	;input suspended by flow control
CF_OSF		=	02H	;output suspended by flow control
 
;service flags
INPUT_FLOW	=	01H	;handle input flow control (send ^S/^Q)
OUTPUT_FLOW	=	02H	;handle output flow control (receive ^S/^Q)

;misc equates  
TIMER		=	46CH	;low-memory timer word
CTRL_Q		=	11H	;ASCII Control-Q/XON/DC1
CTRL_S		=	13H	;ASCII Control-S/XOFF/DC3
BUFF_SIZE       =       2048    ;use 2k byte input and output buffers

PROGSEG SEGMENT BYTE 'CODE'             ;set up our code segment, and
        ASSUME CS:PROGSEG               ;let MASM know of our intentions

        CM_RX_BUFFER_1  DB BUFF_SIZE DUP(0)     ;receive buffer for channel 1
        CM_RX_BUFFER_2  DB BUFF_SIZE DUP(0)     ;receive buffer for channel 2
        CM_RX_BUFFER_3  DB BUFF_SIZE DUP(0)     ;receive buffer for channel 3
        CM_RX_BUFFER_4  DB BUFF_SIZE DUP(0)     ;receive buffer for channel 4

        ;tables for each comm port
        PORT1	COM_PORT        <CM_INT_HANDLER1, 3F8H, 30H, 10H>       ;COM1
        PORT2	COM_PORT	<CM_INT_HANDLER2, 2F8H, 2CH, 08H>       ;COM2
        PORT3	COM_PORT	<CM_INT_HANDLER3, 3E8H, 30H, 10H>       ;COM3
        PORT4	COM_PORT	<CM_INT_HANDLER4, 2E8H, 2CH, 08H>       ;COM4
        ;PS/2 ports: 3 = 0C94, 4=0C9C, same IRQ's

        ;jump table used by access process
        CM_ACCESS_SWITCH LABEL WORD
                DW      CM_OPENPORT     ;1 - opens a port and starts input
                DW      CM_OUTSTR       ;2 - start buffered output
                DW      CM_OUTCHR       ;3 - send unbuffered character
                DW      CM_TDCOUNT      ;4 - return TD buffer count   
                DW      CM_TDFLUSH      ;5 - flush TD buffer
                DW      CM_TDXOFF       ;6 - return TD XOFF status           
                DW      CM_GETSTR       ;7 - return input buffer with optional flush
                DW      CM_GETCHR       ;8 - return input char with flush
                DW      CM_RDCOUNT      ;9 - return RD buffer count   
                DW      CM_RDFLUSH      ;10 - flush RD buffer
                DW      CM_RDXOFF       ;11 - return TD XOFF status           
                DW      CM_SETXOFF      ;12 - change flow control
                DW      CM_STATUS       ;13 - return LSR+MSR in AX     
                DW      CM_BREAK        ;14 - send break signal to modem
		DW	CM_CHECKPORT    ;15 - port installation check
		DW	CM_SETPORT      ;16 - port address/interrupt change
                DW      CM_TOGGLEDTR    ;17 - toggles DTR to force carrier break
                DW      CM_CLOSEPORT    ;18 - closes port 

        ;jump table used by interrupt handler
        CM_INTSWITCH	DW	CM_MODEM_STATUS_INT
        		DW	CM_TX_INT
        		DW	CM_RX_INT
        		DW	CM_LINE_STATUS_INT

;****************************************************************************
; CM_ACCESS - executes requests for services from other assembler programs
;
; On entry: AH -> function number to be executed  
;           AL -> port number to operate on (1,2,3 or 4)
;           BX -> function specific parameter
;           CX -> function specific parameter
;           DX -> function specific parameter
;
;  On exit: AX -> result code, negative number = error of some type
;           BX -> function specific return value
;           CX -> function specific return value
;           DX -> function specific return value
;
;****************************************************************************
CM_ACCESS PROC FAR
        JMP     CM_ACCESS1              ;jump over the data area

        CM_ACCESS_TABLE         DW      0       ;port table offset
        CM_ACCESS_IOADDR        DW      0       ;port I/O address 
        CM_ACCESS_PARM1         DW      0       ;function/port parameter   
        CM_ACCESS_PARM2         DW      0       ;specialized parameters
        CM_ACCESS_PARM3         DW      0
        CM_ACCESS_PARM4         DW      0

CM_ACCESS1:
	PUSH_REGS	                ;save registers
        PUSH    CS                      ;set data segment to our code segment
        POP     DS
        DEC     AL                      ;normalize port number to 0-3
        MOV     CM_ACCESS_PARM1,AX      ;save all the parameter registers
        MOV     CM_ACCESS_PARM2,BX         
        MOV     CM_ACCESS_PARM3,CX         
        MOV     CM_ACCESS_PARM4,DX         

        CMP     AL,0                    ;check port number lower limit
        JGE     CM_ACCESS2              ;OK, continue processing
        XOR     AX,AX                   ;return -1, port number error
        DEC     AX
        POP_REGS                        ;restore caller's registers
        RET                             ;return to caller

CM_ACCESS2:
        CMP     AL,3                    ;check port number upper limit
        JLE     CM_ACCESS3              ;OK, continue processing
        XOR     AX,AX                   ;return -1, port number error
        DEC     AX
        POP_REGS                        ;restore caller's registers
        RET                             ;return to caller

CM_ACCESS3:
        CMP     AH,1                    ;check function number lower limit
        JGE     CM_ACCESS4              ;OK, continue processing
        XOR     AX,AX                   ;return -10, function number error
        SUB     AX,10
        POP_REGS                        ;restore caller's registers
        RET                             ;return to caller

CM_ACCESS4:
        CMP     AH,18                   ;check function number upper limit
        JLE     CM_ACCESS5              ;OK, continue processing
        XOR     AX,AX                   ;return -10, function number error
        SUB     AX,10
        POP_REGS                        ;restore caller's registers
        RET                             ;return to caller

CM_ACCESS5:
        MOV     AH,COM_SIZE                     ;multiply port # by port table size
        MUL     AH
        MOV     BX,OFFSET PORT1                 ;load BX with the address of 1st port
	ADD	BX,AX			        ;move to the appropriate port table
        MOV     CM_ACCESS_TABLE,BX              ;save value for later use
	MOV	AX,[BX+COM_ADDRESS]	        ;load AX with the I/O address for the port
        MOV     CM_ACCESS_IOADDR,AX             ;save I/O addr for later use

        MOV     AX,CM_ACCESS_PARM1              ;restore parameter for function number 
        XOR     AL,AL                           ;clear port number
        XCHG    AL,AH                           ;move function number to AL
        DEC     AL                              ;adjust for zero offset 
	MOV	SI,OFFSET CM_ACCESS_SWITCH	;point to the function jump table
	ADD	SI,AX                           ;move to appropriate function
	ADD	SI,AX
	CALL	[SI]	                        ;call selected function
        POP_REGS                                ;restore caller's registers
        RET                                     ;return

CM_ACCESS ENDP
 
;****************************************************************************
; CM_OPENPORT - configures a port and starts input
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;           CM_ACCESS_PARM2  -> port speed in bits per second
;           CM_ACCESS_PARM3  -> port line setting, as follows:
;                               bits 1-0 = 00 - 5 data bits
;                                          01 - 6 data bits
;                                          10 - 7 data bits
;                                          11 - 8 data bits
;                               bit 2    =  0 - 1 stop bit
;                                           1 - 2 stop bits
;                               bits 4-3 = 00 - no parity
;                                          01 - odd parity
;                                          11 - even parity
;                               bits 7-5 = always zero
;
;  On exit: AX -> 0 = port initialized and ready
;                -1 = port not installed in the machine
;****************************************************************************
CM_OPENPORT PROC NEAR
        CALL    CM_CHECKPORT            ;check for the port first
        CMP     AX,-2                   ;port installed?
        JNE     CM_OPENPORT1            ;yes, proceed
        XOR     AX,AX
        DEC     AX                      ;set -1 return value
        RET

CM_OPENPORT1:
        MOV     AX,3                    ;set up full flow control
	MOV	[BX+COM_OPTIONS],AX	;record it in the port table
	ADD	DL,COM_LCR		;move to the Line-Control Register
	XOR	AX,AX                   ;set up a zero value
	OUT	DX,AL			;allow access to Int Enable Register
	MOV	[BX+COM_FLAGS],AL	;clear all flags for the port
	MOV	[BX+COM_OUTFLOW],AL	;and outbound flow control field
	ADD	DL,COM_IER-COM_LCR	;DX -> Interrupt Enable Register
	OUT	DX,AL			;turn off all interrupts
	ADD	DL,COM_MCR-COM_IER	;DX -> Modem-Control Register
        MOV	AL,0BH			;set DTR, RTS, out2 to get ints
	OUT	DX,AL                   ;program the chip
	ADD	DL,COM_TXB-COM_MCR	;DX -> base i/o address
	CALL	CM_SET_SPEED            ;set the port speed
	XOR	AX,AX			;set up new interrupt vector
	MOV	ES,AX                   ;change segment
	MOV	AX,CS                   ;store code segment for later
	MOV	DI,[BX+COM_IVA]		;ES:DI -> interrupt vector
	CLI				;turn interrupts off
        MOV     CX,ES:[DI+2]            ;retrieve old segment
	MOV	[BX+COM_IVSEG],CX       ;store old segment
        MOV     CX,ES:[DI]              ;retrieve old offset 
	MOV	[BX+COM_IVOFF],CX       ;store old offset 
	MOV	ES:[DI+2],AX		;set new segment in vector table
	MOV	AX,[BX+COM_IH]          ;retrieve handler address
	MOV	ES:[DI],AX              ;set new offset in vector table
	STI                             ;start up interrupts
	MOV	AH,[BX+COM_BIT]		;mask on interrupt at controller
	NOT	AH                      ;invert each bit
	CLI                             ;turn interrupts off
	IN	AL,21H                  ;retrieve current int byte
	AND	AL,AH                   ;mask on interrupt
	OUT	21H,AL                  ;activate it
	STI                             ;start up interrupts
	MOV	AX,CM_ACCESS_PARM3      ;retrieve line settings
	ADD	DL,COM_LCR-COM_TXB	;DX -> Line-Control Register
	OUT	DX,AL                   ;program the line control register
	MOV	CL,9    		;set up 1/2 second wait
        CALL    CM_DELAY                ;pause to allow port stabilization
        PUSH    CS                      ;set ES to our code segment
        POP     ES

        MOV     DI,OFFSET CM_RX_BUFFER_1        ;assume channel 1 buffer
        MOV     CX,CM_ACCESS_PARM1              ;retrieve saved parameter
        XOR     CH,CH                           ;clear function number

CMOPENPORT2:
        CMP     CX,0                            ;at the specified port?
        JE      CMOPENPORT3                     ;yes, continue processing
        ADD     DI,BUFF_SIZE                    ;advance to the next buffer
        DEC     CX                              ;decrement the counter
        JMP     CMOPENPORT2                     ;loop back 

CMOPENPORT3:
        MOV     BX,CM_ACCESS_TABLE      ;retrieve port table
        MOV     DX,CM_ACCESS_IOADDR     ;and port address
        MOV     CX,BUFF_SIZE            ;set up buffer size
        CALL    CM_INPUT                ;start the input int routine
        XOR     AX,AX                   ;set success code to AX
	RET                             ;return to caller

CM_OPENPORT ENDP

;****************************************************************************
; CM_SET_SPEED - Sets the line speed for a port, called by CM_OPENPORT
;
; On entry: CM_ACCESS_IOADDR -> port I/O address
;           CM_ACCESS_PARM2  -> port speed
;
;  On exit: Nothing
;****************************************************************************
CM_SET_SPEED PROC NEAR
	PUSH	DX                      ;save DX for later
        MOV     AX,CM_ACCESS_PARM2      ;retrieve port speed
        MOV     DX,CM_ACCESS_IOADDR     ;retrieve port I/O address
	CALL	CM_GET_DIVISOR		;convert speed to divisor
	PUSH	AX                      ;save divisor for later
	ADD	DL,COM_LCR		;DX -> Line-Control Register
	IN	AL,DX			;AL = LCR setting
	OR      AL,80H			;set Divisor Latch Access bit
	JMP	$+2                     ;pause for a cycle
	OUT	DX,AL                   ;allow access    
	POP	AX			;retrieve the divisor
	ADD	DL,COM_DLL-COM_LCR      ;move to Divisor LSB
	OUT	DX,AL                   ;program it
	INC	DX			;move to Divisor MSB
	MOV	AL,AH                   ;adjust register
	OUT	DX,AL                   ;program it
	ADD	DL,COM_LCR-COM_DLM	;move to the Line-Control Register
	JMP	$+2                     ;pause for a cycle
	IN	AL,DX			;read LCR again
	AND	AL,07FH			;clear DLA bit
	JMP	$+2                     ;pause for a cycle
	OUT	DX,AL                   ;program it
	POP	DX                      ;restore DX
	RET                             ;return to caller

CM_SET_SPEED ENDP

;****************************************************************************
; CM_GET_DIVISOR - Convert BPS to divisor, called by CM_SET_SPEED
;
; On entry: AX -> speed in bits per second
;
;  On exit: AX -> divisor needed for rated speed
;****************************************************************************
CM_GET_DIVISOR PROC NEAR
	OR	AX,AX			;special case of zero speed
	JNE	CM_GET_DIVISOR1
	INC	AX			;really means 115,200 bps
	RET				;for which divisor is 1

CM_GET_DIVISOR1:
	PUSH	BX
	CMP	AX,600			;calculation break at 600 bps
	JB	CM_GET_DIVISOR2		;if below 600
	PUSH	DX
	XOR	DX,DX
	MOV	BX,100
	DIV	BX			;divide speed by 100
	MOV	BX,AX
	MOV	AX,1152
	DIV	BX			;then divide (speed/100) into 1152
	POP	DX			;that gives the desired result
	POP	BX
	RET

CM_GET_DIVISOR2:
	MOV	BL,10			;divide the speed by 10
	DIV	BL
	MOV	BL,AL			;BL = speed/10
	MOV	AX,120
	DIV	BL			;divide that into 120
	MOV	BL,96			;BL = divisor for 1,200 bps
	MUL	BL			;this gives the desired result
	POP	BX
	RET

CM_GET_DIVISOR ENDP

;****************************************************************************
; CM_INPUT - programs the 8250 to begin receive interrupts
;****************************************************************************
CM_INPUT PROC NEAR
	MOV	AX,ES			;set up input pointers
	MOV	[BX+COM_IN_SEG],AX
	MOV	[BX+COM_IN_FIRST],DI
	MOV	[BX+COM_RD_PTR],DI
	AND	CX,0FFFEH		;need even length
	ADD	DI,CX
	MOV	[BX+COM_IN_LIMIT],DI
	CALL	CM_RDFLUSH    		;initialize the buffer to receive data
	SHR	CX,1			;convert byte count to word count
	MOV	AX,CX			;calculate thresholds for flow control
	SHR	AX,1
	MOV	[BX+COM_IN_LO],AX	;send Ctrl-Q when half empty
	SHR	AX,1
	SUB	CX,AX
	MOV	[BX+COM_IN_HI],CX	;send Ctrl-S when three-quarters full
	IN	AL,DX			;flush input from port
	IN	AL,DX
	INC	DX			;DX -> COM_IER
	IN	AL,DX			;read Interrupt Enable Register
        OR	AL,01H		        ;enable input-available interrupt
	JMP	$+2			;(this may be more than is needed)
	OUT	DX,AL
	RET

CM_INPUT ENDP

;****************************************************************************
; CM_RESTART_INPUT - Sends a Control-Q to restart input  
; 
; On entry: BX -> port table
;           DX -> I/O address
;           Interrupts off
;****************************************************************************
CM_RESTART_INPUT PROC NEAR
	AND	[BX+COM_FLAGS],NOT CF_ISF	;reset input-off flag
	ADD	DL,COM_LSR		;DX -> Line Status Register
	IN	AL,DX
	TEST	AL,20H			;transmit buffer empty?
	MOV	AL,CTRL_Q
	JZ	CM_RESTART_INPUT1      	;no
	ADD	DL,COM_TXB-COM_LSR	;DX -> Transmit Buffer
	OUT	DX,AL
	XOR	AX,AX

CM_RESTART_INPUT1:
	MOV	[BX+COM_OUTFLOW],AL
	RET

CM_RESTART_INPUT	ENDP

;****************************************************************************
; CM_OUTSTR - starts buffered output
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;           CM_ACCESS_PARM2  -> segment of transmit buffer
;           CM_ACCESS_PARM3  -> offset of the transmit buffer 
;           CM_ACCESS_PARM4  -> length of the transmit buffer 
;
;  On exit: AX -> 0 = transmission started
;                -1 = port not installed in the machine
;****************************************************************************
CM_OUTSTR PROC NEAR
        CALL    CM_CHECKPORT            ;check for the port first
        CMP     AX,-2                   ;port installed?
        JNE     CM_OUTSTR1              ;yes, proceed
        XOR     AX,AX
        DEC     AX                      ;port number error to AX
        RET                             ;return to caller

CM_OUTSTR1:
        MOV     AX,CM_ACCESS_PARM2      ;set ES to buffer segment
        PUSH    AX
        POP     ES
        MOV     DI,CM_ACCESS_PARM3      ;offset of buffer to DI
        MOV     CX,CM_ACCESS_PARM4      ;length to CX
	ADD	DL,COM_LSR		;DX -> Line Status Register
	CLI				;interrupts off
	MOV	[BX+COM_OUT_PTR],DI	;set output pointer
	ADD	DI,CX			;and limit pointer
	MOV	[BX+COM_OUT_LIMIT],DI
	MOV	AX,ES			;buffer assumed to be in one segment
	MOV	[BX+COM_OUT_SEG],AX
	TEST	[BX+COM_FLAGS],CF_OSF	;output flow control on?
	JNZ	CM_OUTSTR2		;if so then a ^Q will start it
	CALL	CM_START_OUTPUT		;else start the transmitter

CM_OUTSTR2:
	STI				;leave with interrupts on
        XOR     AX,AX                   ;clear error register
	RET                             ;return to caller

CM_OUTSTR ENDP

;****************************************************************************
; CM_START_OUTPUT - programs the 8250 for xmit interrupts, called by CM_OUTSTR
;****************************************************************************
CM_START_OUTPUT PROC NEAR
	ADD	DL,COM_IER-COM_LSR	;DX -> Interrupt Enable Register
	IN	AL,DX
	AND	AL,0FDH			;disable transmit interrupts
	JMP	$+2
	OUT	DX,AL
	ADD	DL,COM_LSR-COM_IER	;DX -> Line Status Register

CM_START_OUTPUT_WAIT1:
	IN	AL,DX			;wait until transmitter is idle
	AND	AL,40H
	JZ	CM_START_OUTPUT_WAIT1
	ADD	DL,COM_IIR-COM_LSR	;DX -> Interrupt Identification Reg
	CALL	CM_TX_INT			;dispatch the first/next character
	JC	CM_START_OUTPUT_EMPTY	;if there was nothing to transmit
	ADD	DL,COM_LSR-COM_TXB	;DX -> Line Status Register

CM_START_OUTPUT_WAIT2:
	IN	AL,DX			;wait until the port has picked up
	AND	AL,40H			;this character
	JNZ	CM_START_OUTPUT_WAIT2

CM_START_OUTPUT_WAIT3:
	IN	AL,DX			;then for holding reg to become empty
	AND	AL,20H			;(this should be redundant)
	JZ	CM_START_OUTPUT_WAIT3
	ADD	DL,COM_IER-COM_LSR	;DX -> Interrupt Enable Register
	IN	AL,DX
	OR	AL,02H			;enable transmit interrupts
	JMP	$+2
	OUT	DX,AL
	INC	DX			;DX -> Interrupt Identification Reg
	CALL	CM_TX_INT   		;dispatch the next character

CM_START_OUTPUT_EMPTY:
	RET

CM_START_OUTPUT	ENDP

;****************************************************************************
; CM_OUTCHR - Single-character ASCII output
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;           CM_ACCESS_PARM2  -> ASCII character to send   
;
;  On exit: AX -> 0 = character sent
;                -1 = port not installed in the machine
;****************************************************************************
CM_OUTCHR PROC NEAR
        CALL    CM_CHECKPORT            ;check for the port first
        CMP     AX,-2                   ;port installed?
        JNE     CM_OUTCHR1              ;yes, proceed
        XOR     AX,AX
        DEC     AX                      ;port number error to AX
        RET                             ;return to caller

CM_OUTCHR1:
        MOV     CX,CM_ACCESS_PARM2      ;character to be sent to CL
        XOR     CH,CH

CM_OUTCHR2:
	ADD	DL,COM_LSR-COM_TXB	;DX -> Line Status Register

CM_OUTCHR3:
	IN	AL,DX			;wait until
	TEST	AL,20H			;Transmitter Holding Register empty
	JZ	CM_OUTCHR3
	ADD	DL,COM_TXB-COM_LSR	;DX -> Transmit Buffer
	MOV	AL,CL
	OUT	DX,AL			;transmit it
	XOR	CX,CX
	XCHG	[BX+COM_OUTFLOW],CL	;flow-control character waiting?
	TEST	CL,CL
	JNZ	CM_OUTCHR2 		;if so loop to send it
        XOR     AX,AX                   ;clear error register
	RET                             ;return to caller

CM_OUTCHR ENDP

;****************************************************************************
; CM_TDCOUNT - returns status of output in progress
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;
;  On exit: AX -> 0 - ??? = number of characters left    
;                -1 = port not installed in the machine
;****************************************************************************
CM_TDCOUNT PROC NEAR
        CALL    CM_CHECKPORT            ;check for the port first
        CMP     AX,-2                   ;port installed?
        JNE     CM_TDCOUNT1             ;yes, proceed
        XOR     AX,AX
        DEC     AX                      ;port number error to AX
        RET                             ;return to caller

CM_TDCOUNT1:
	MOV	AX,[BX+COM_OUT_PTR]	;get output pointer
	TEST	AX,AX
	JZ	CM_TDCOUNT2     	;if zero, output is done
	SUB	AX,[BX+COM_OUT_LIMIT]	;else calculate number of characters
	NEG	AX			;left to transmit
	RET

CM_TDCOUNT2:
	ADD	DX,COM_LSR - COM_TXB	;DX -> Line Status Register
	IN	AL,DX
	AND	AL,20H			;Tx Holding Register Empty?
	JNZ	CM_TDCOUNT3     
	INC	AX			;still 1 en route thru int routine
	RET

CM_TDCOUNT3:
	XOR	AX,AX			;only return 0 if ready for more
	RET

CM_TDCOUNT ENDP

;****************************************************************************
; CM_TDFLUSH - Abort output in progress
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;
;  On exit: AX -> 0 = output halted
;                -1 = port not physically installed 
;****************************************************************************
CM_TDFLUSH PROC NEAR
        CALL    CM_CHECKPORT            ;check for the port first
        CMP     AX,-2                   ;port installed?
        JNE     CM_TDFLUSH1             ;yes, proceed
        XOR     AX,AX
        DEC     AX                      ;port number error to AX
        RET                             ;return to caller

CM_TDFLUSH1:
	XOR	AX,AX
	MOV	[BX+COM_OUT_PTR],AX
        XOR     AX,AX
	RET

CM_TDFLUSH ENDP

;****************************************************************************
; CM_TDXOFF - Return status of transmission
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;
;  On exit: AX -> 0 - transmitter running
;                 1 - transmitter halted due to remote sending XOFF
;                -1 = port not physically installed 
;****************************************************************************
CM_TDXOFF PROC NEAR
        CALL    CM_CHECKPORT            ;check for the port first
        CMP     AX,-2                   ;port installed?
        JNE     CM_TDXOFF1              ;yes, proceed
        XOR     AX,AX
        DEC     AX                      ;port number error to AX
        RET                             ;return to caller

CM_TDXOFF1:
        XOR     AX,AX                           ;assume output running
	TEST	[BX+COM_FLAGS],CF_ISF	        ;output flow-controlled off?
	JZ	CM_TDXOFF2                      ;if not...
        MOV     AX,1                            ;1 to AX

CM_TDXOFF2:
        RET

CM_TDXOFF ENDP

;****************************************************************************
; CM_GETCHR - Read a received character and flush buffer
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;
;  On exit: AX -> 0-255 = ASCII character value
;                -1 = port not installed in the machine
;                -2 = no input waiting
;****************************************************************************
CM_GETCHR PROC NEAR
        CALL    CM_CHECKPORT            ;check for the port first
        CMP     AX,-2                   ;port installed?
        JNE     CM_GETCHR1              ;yes, proceed
        XOR     AX,AX
        DEC     AX                      ;port number error to AX
        RET                             ;return to caller

CM_GETCHR1:
        XOR     AX,AX                   ;assume no character waiting code
        SUB     AX,2
	MOV	SI,[BX+COM_RD_PTR]
	CMP	SI,[BX+COM_IN_PTR]	;any input?
	JE	CM_GETCHR4		;no...

        CLI                             ;hold off interrupts for a moment
	DEC	[BX+COM_IN_CTR]		;decrement input counter
	TEST	[BX+COM_FLAGS],CF_ISF	;input flow-controlled off?
	JZ	CM_GETCHR2		;if not...
	MOV	AX,[BX+COM_IN_CTR]	;else see it it's time to resume
	CMP	AX,[BX+COM_IN_LO]
	JA	CM_GETCHR2		;not yet
	CALL	CM_RESTART_INPUT

CM_GETCHR2:
        STI                             ;start interrupts
	MOV	AX,[BX+COM_IN_SEG]	;get buffer segment
	PUSH	DS
	MOV	DS,AX
	LODSB				;load character into AL
        XOR     AH,AH                   ;clear AH
	POP	DS
	CMP	SI,[BX+COM_IN_LIMIT]
	JNE	CM_GETCHR3
	MOV	SI,[BX+COM_IN_FIRST]

CM_GETCHR3:
	MOV	[BX+COM_RD_PTR],SI      ;advance the read pointer

CM_GETCHR4:
	RET     ;AX -> character or -2 if no input waiting
                
CM_GETCHR ENDP
		
;****************************************************************************
; CM_GETSTR - Read a block from the input buffer with optional flush
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;           CM_ACCESS_PARM2  -> buffer clear flag, 1 = killbuffer, 0 = retain
;
;  On exit: AX -> length of read buffer or zero if no input
;                -1 = port not installed in the machine
;           DX -> segment of buffer
;           BX -> offset of buffer
;****************************************************************************
CM_GETSTR PROC NEAR
        JMP     CM_GETSTR1      ;jump over the data area

        CM_GETSTR_BUFF  DB BUFF_SIZE DUP(0)     ;buffer for return string

CM_GETSTR1:
        CALL    CM_CHECKPORT            ;check for the port first
        CMP     AX,-2                   ;port installed?
        JNE     CM_GETSTR2              ;yes, proceed
        XOR     AX,AX
        DEC     AX                      ;port number error to AX
        RET                             ;return to caller

CM_GETSTR2:
	MOV	SI,[BX+COM_RD_PTR]      ;load buffer pointers
	CMP	SI,[BX+COM_IN_PTR]	;any input?
	JNE	CM_GETSTR3       	;yes, copy input buffer to return buffer
        XOR     AX,AX                   ;set no character count to AX
        RET                             ;return to caller

CM_GETSTR3:
	CLI				        ;interrupts off for a moment
        CLD                                     ;clear direction flag
        PUSH    ES                              ;save ES for later
        PUSH    CS                              ;set ES to code sgment
        POP     ES
        MOV     DI,OFFSET CM_GETSTR_BUFF        ;point DI to work buffer
	MOV	SI,[BX+COM_RD_PTR]              ;load buffer read pointer
	CMP	SI,[BX+COM_IN_PTR]      	;check for buffer wrap
        JG      CM_GETSTR4                      ;input ptr has wrapped

        MOV     CX,[BX+COM_IN_PTR]              ;load CX with input pointer
        SUB     CX,SI                           ;subtract from read ptr to get length
        MOV     AX,CX                           ;save length for later
        REP     MOVSB                           ;copy the string
        JMP     CM_GETSTR5                      ;return it to the caller

CM_GETSTR4:
        MOV     CX,[BX+COM_IN_LIMIT]    ;load CX with buffer end
        SUB     CX,SI                   ;subtract from read ptr to get length
        MOV     AX,CX                   ;save length for later
        REP     MOVSB                   ;copy to the end of the buffer
	MOV	SI,[BX+COM_IN_FIRST]    ;load CX with buffer top
        MOV     CX,[BX+COM_IN_PTR]      ;load CX with input pointer
        SUB     CX,SI                   ;subtract from read ptr to get length
        ADD     AX,CX                   ;add length to previous
        REP     MOVSB                   ;copy the rest of the buffer

CM_GETSTR5:
        POP     ES                      ;restore ES segment
        PUSH    AX                      ;save buffer size for later
        MOV     AX,CM_ACCESS_PARM2      ;load the buffer clear flag
        CMP     AX,0                    ;leave buffer alone?
        JE      CM_GETSTR6              ;yes

	MOV	AX,[BX+COM_IN_FIRST]    ;offset of input buffer to AX
	MOV	[BX+COM_RD_PTR],AX      ;reset read pointer
	MOV	[BX+COM_IN_PTR],AX      ;reset input pointer
	XOR	AX,AX
	MOV	[BX+COM_IN_CTR],AX	;zero the counter
	TEST	[BX+COM_FLAGS],CF_ISF	;input flow-controlled off?
	JZ	CM_GETSTR6        	;if not...
        MOV     DX,CM_ACCESS_IOADDR     ;point DX to the I/O address
	CALL	CM_RESTART_INPUT	;send a Control-Q

CM_GETSTR6:
	STI				        ;start interrupts up again
        POP     AX                              ;restore buffer count
        MOV     DX,CS                           ;segment of string to DX
        MOV     BX,OFFSET CM_GETSTR_BUFF        ;offset to BX
        RET

CM_GETSTR ENDP

;****************************************************************************
; CM_RDCOUNT - returns number of characters in the receive buffer
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;
;  On exit: AX -> 0 - ??? = number of characters left    
;                -1 = port not installed in the machine
;****************************************************************************
CM_RDCOUNT PROC NEAR
        CALL    CM_CHECKPORT            ;check for the port first
        CMP     AX,-2                   ;port installed?
        JNE     CM_RDCOUNT1             ;yes, proceed
        XOR     AX,AX
        DEC     AX                      ;port number error to AX
        RET                             ;return to caller

CM_RDCOUNT1:
        XOR     AX,AX                           ;clear AX
        MOV     AX,CS:[BX+COM_IN_CTR]           ;length of the buffer to AX
	RET

CM_RDCOUNT ENDP

;****************************************************************************
; CM_RDFLUSH - Flush pending input
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;
;  On exit: AX -> 0 = input halted
;                -1 = port not physically installed 
;****************************************************************************
CM_RDFLUSH PROC NEAR	       
        CALL    CM_CHECKPORT            ;check for the port first
        CMP     AX,-2                   ;port installed?
        JNE     CM_RDFLUSH1             ;yes, proceed
        XOR     AX,AX
        DEC     AX                      ;port number error to AX
        RET                             ;return to caller

CM_RDFLUSH1:
	CLI				;interrupts off while pointer reset
	MOV	AX,[BX+COM_IN_FIRST]    ;offset of input buffer to AX
	MOV	[BX+COM_RD_PTR],AX      ;reset read pointer
	MOV	[BX+COM_IN_PTR],AX      ;reset input pointer
	XOR	AX,AX
	MOV	[BX+COM_IN_CTR],AX	;zero the counter
	TEST	[BX+COM_FLAGS],CF_ISF	;input flow-controlled off?
	JZ	CM_RDFLUSH2       	;if not...
	CALL	CM_RESTART_INPUT	;send a Control-Q

CM_RDFLUSH2:
	STI
	RET

CM_RDFLUSH ENDP

;****************************************************************************
; CM_RDXOFF - returns status of input flow control
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;
;  On exit: AX -> 0 = input interrupt running
;                 1 - interrupt halted due to buffer full condition
;                -1 = port not physically installed 
;****************************************************************************
CM_RDXOFF PROC NEAR
        CALL    CM_CHECKPORT            ;check for the port first
        CMP     AX,-2                   ;port installed?
        JNE     CM_RDXOFF1              ;yes, proceed
        XOR     AX,AX
        DEC     AX                      ;port number error to AX
        RET                             ;return to caller

CM_RDXOFF1:
	XOR	AX,AX
	XCHG	AL,[BX+COM_OUTFLOW]	;is there a flow-control char waiting?
	TEST	AL,AL
	JNZ	CM_RDXOFF2		;if so set flag and return
        XOR     AX,AX
        JMP     CM_RDXOFF3

CM_RDXOFF2:
        MOV     AX,1    ;set flag to indicate input stanched     

CM_RDXOFF3:
        RET

CM_RDXOFF ENDP

;****************************************************************************
; CM_SETXOFF - starts or cancels flow control for file tranfer operations
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;           CM_ACCESS_PARM2  -> 0 to start, 1 to stop flow control
;
;  On exit: AX -> 0 = flow control set
;                -1 = port not physically installed 
;****************************************************************************
CM_SETXOFF PROC NEAR
        CALL    CM_CHECKPORT            ;check for the port first
        CMP     AX,-2                   ;port installed?
        JNE     CM_SETXOFF1             ;yes, proceed
        XOR     AX,AX
        DEC     AX                      ;port number error to AX
        RET                             ;return to caller

CM_SETXOFF1:
        MOV     AX,CM_ACCESS_PARM2      ;set xoff flag (0 = false)
        CMP     AX,0                    ;false?
        JE      CM_SETXOFF2             ;yes, halt flow control
        MOV     AX,INPUT_FLOW+OUTPUT_FLOW       ;add up flow control bytes
        MOV     [BX+COM_OPTIONS],AX             ;set flag
        JMP     CM_SETXOFF3                     ;return

CM_SETXOFF2:
        XOR     AX,AX                   ;clear flow control bytes
        MOV     [BX+COM_OPTIONS],AX

CM_SETXOFF3:
        XOR     AX,AX
        RET

CM_SETXOFF ENDP

;****************************************************************************
; CM_STATUS - returns LSR+MSR in AX
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;
;  On exit: AX -> Line Status Register/Modem Status Register
;                -1 = port not physically installed 
;****************************************************************************
CM_STATUS PROC NEAR
        CALL    CM_CHECKPORT            ;check for the port first
        CMP     AX,-2                   ;port installed?
        JNE     CM_STATUS1              ;yes, proceed
        XOR     AX,AX
        DEC     AX                      ;port number error to AX
        RET                             ;return to caller

CM_STATUS1:
        ADD     DL,COM_LSR              ;DX -> Line status register
        XOR     AX,AX
        IN      AL,DX
        JMP     $+2                     ;pause for a cycle
        MOV     BL,AL                   ;save byte for later
        ADD     DL,COM_MSR-COM_LSR      ;DX -> Modem status register
        XOR     AX,AX
        IN      AL,DX
        JMP     $+2                     ;pause for a cycle
        MOV     AH,BL                   ;restore saved LSR to AH
        RET

CM_STATUS ENDP

;****************************************************************************
; CM_BREAK - send break signal to modem/terminal
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;
;  On exit: AX -> 0 = break sent 
;                -1 = port not installed in the machine
;****************************************************************************
CM_BREAK PROC NEAR
        CALL    CM_CHECKPORT            ;check for the port first
        CMP     AX,-2                   ;port installed?
        JNE     CM_BREAK1               ;yes, proceed
        XOR     AX,AX
        DEC     AX                      ;port number error to AX
        RET                             ;return to caller

CM_BREAK1:
	ADD	DL,COM_LCR		;DX -> Line Control Register
	CLI				;interrupts off while changing LCR
	IN	AL,DX                   ;read in Line Control Register
	OR	AL,40H			;set BREAK
	JMP	SHORT $+2               ;delay for a cycle
	OUT	DX,AL                   ;program the chip
	STI                             ;start up interrupts
	MOV	CL,7			;pause for approx 385 ms
	CALL	CM_DELAY
	CLI                             ;turn off interrupts again
	IN	AL,DX                   ;read in LCR
	XOR	AL,40H			;reset BREAK
	JMP	SHORT $+2               ;delay for a cycle
	OUT	DX,AL                   ;program the chip
	STI                             ;start up interrupts
        XOR     AX,AX                   ;set success code to AX
        RET                             ;return to caller

CM_BREAK ENDP

;****************************************************************************
; CM_CHECKPORT - Checks for installed port and returns status
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;
;  On exit: AX -> 0 = port available for use
;                -1 = port or interrupt already in use
;                -2 = port not physically installed 
;****************************************************************************
CM_CHECKPORT PROC NEAR
        MOV     DX,CM_ACCESS_IOADDR     ;retrieve port I/O address
        PUSH    DX                      ;save for later
	INC	DX		        ;point DX to the Interrupt Enable Register
	IN	AL,DX                   ;input byte from the port into AL
	CMP	AL,0FFH		        ;is the port installed?
	JE	CM_CHECKPORT_DEAD       ;no, set error and exit
	DEC	AH		        ;AH = 1
	IN	AL,21H		        ;get interrupt mask
	AND	AL,[BX+COM_BIT]	        ;and with IRQ bit
	JZ	CM_CHECKPORT_BUSY       ;if bit not set interrupt is enabled
        POP     DX                      ;restore port I/O address
        XOR     AX,AX                   ;port available code to AX
        RET                             ;return to caller

CM_CHECKPORT_DEAD:
        POP     DX                      ;restore port I/O address
        XOR     AX,AX
        SUB     AX,2                    ;port not installed code to AX
        RET                             ;return to caller

CM_CHECKPORT_BUSY:
        POP     DX                      ;restore port I/O address
        XOR     AX,AX
        SUB     AX,1                    ;port in use code to AX
        RET                             ;return to caller

CM_CHECKPORT ENDP

;****************************************************************************
; CM_SETPORT - Changes a port's I/O address and interrupt level
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;           CM_ACCESS_PARM2  -> decimal port address value
;           CM_ACCESS_PARM3  -> interrupt level
;
;  On exit: AX -> 0 = port available for use and port table modified
;                -1 = port or interrupt already in use
;                -2 = port not physically installed 
;****************************************************************************
CM_SETPORT PROC NEAR
	MOV	AX,[BX+COM_ADDRESS]   	;retrieve old address value
        PUSH    AX                      ;save for later
	MOV	AX,[BX+COM_IVA]   	;retrieve old interrupt vector table address
        PUSH    AX                      ;save for later
        XOR     AH,AH
	MOV	AL,[BX+COM_BIT]   	;retrieve old IRQ mask bit
        PUSH    AX                      ;save for later
        MOV     AX,CM_ACCESS_PARM2      ;retrieve new port address
	MOV	[BX+COM_ADDRESS],AX	;store it in the table
	MOV	CM_ACCESS_IOADDR,AX     ;set port address for checks
        MOV     CX,CM_ACCESS_PARM3      ;retrieve new interrupt level
        MOV     AX,20H                  ;vector location of irq 0

CM_SETPORT1:
        CMP     CX,0                    ;done adding?
        JE      CM_SETPORT2             ;yes, exit
        ADD     AX,4                    ;advance to the next vector location
        DEC     CX                      ;decrement counter
        JMP     CM_SETPORT1             ;loop back

CM_SETPORT2:
	MOV	[BX+COM_IVA],AX	        ;store it in the table
        MOV     CX,CM_ACCESS_PARM3      ;retrieve interrupt level again
        MOV     AX,1                    ;set bit 1 in ax

CM_SETPORT3:
        CMP     CX,0                    ;done shifting?
        JE      CM_SETPORT4             ;yes, exit
        SHL     AX,1                    ;advance to the next vector location
        DEC     CX                      ;decrement counter
        JMP     CM_SETPORT3             ;loop back

CM_SETPORT4:
	MOV	[BX+COM_BIT],AL	        ;store it in the table
        CALL    CM_CHECKPORT            ;check for the port
        CMP     AX,0                    ;port available?
        JNE     CM_SETPORT5             ;no, restore old port values and exit
        POP     BX                      ;dummy pops to fix up the stack
        POP     BX
        POP     BX
        RET                             ;ready to go, return to caller

CM_SETPORT5:
	POP     DX                      ;restore saved IRQ mask bit
        MOV	[BX+COM_BIT],DL	
        POP     DX
	MOV	[BX+COM_IVA],DX   	;restore saved IRQ vector table address
        POP     DX
	MOV	[BX+COM_ADDRESS],DX	;restore saved address value
        RET                             ;return error value

CM_SETPORT ENDP

;****************************************************************************
; CM_TOGGLEDTR - toggles the DTR line to force a modem carrier break
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;
;  On exit: AX -> 0 = DTR has been toggled
;                -1 = port number error
;****************************************************************************
CM_TOGGLEDTR PROC NEAR
        CALL    CM_CHECKPORT            ;check for the port first
        CMP     AX,-2                   ;port installed?
        JNE     CM_TOGGLEDTR1           ;yes, proceed
        XOR     AX,AX
        DEC     AX                      ;port number error to AX
        RET

CM_TOGGLEDTR1:
	ADD	DL,COM_MCR		;DX -> Modem Control Register
	CLI				;interrupts off for a moment
	IN	AL,DX                   ;read in Modem Control Register
        PUSH    AX                      ;save current state of the register
        XOR     AX,AX                   ;clear AX
        OUT     DX,AL                   ;clear DTR, RTS, and OUT2
        STI
	MOV	CL,7			;pause for approx 385 ms
	CALL	CM_DELAY
	CLI				;interrupts off for a moment
        POP     AX                      ;restore previous MCR value
        OUT     DX,AL                   ;reset DTR, RTS, and OUT2
        STI
	MOV	CL,7			;pause for approx 385 ms
	CALL	CM_DELAY
        XOR     AX,AX                   ;set success code to AX
        RET                             ;return to caller

CM_TOGGLEDTR ENDP

;****************************************************************************
; CM_CLOSEPORT - closes a port and removes interrupt routines
;
; On entry: CM_ACCESS_TABLE  -> port table offset in BX
;           CM_ACCESS_IOADDR -> port I/O address in DX
;
;  On exit: AX -> 0 = port closed
;                -1 = port not installed in the machine
;****************************************************************************
CM_CLOSEPORT PROC NEAR
        CALL    CM_CHECKPORT            ;check for the port first
        CMP     AX,-2                   ;port installed?
        JNE     CM_CLOSEPORT1           ;yes, proceed
        XOR     AX,AX
        DEC     AX                      ;port number error to AX
        RET

CM_CLOSEPORT1:
	MOV	AH,[BX+COM_BIT]
	CLI				;mask off interrupt at controller
	IN	AL,21H
	OR	AL,AH
	OUT	21H,AL
        PUSH    ES                      ;save ES for later
        PUSH    DX                      ;save DX for later
	XOR	AX,AX			;clear AX to program ES
	MOV	ES,AX                   ;change segment
	MOV	DI,[BX+COM_IVA]		;ES:DI -> interrupt vector
        MOV     CX,[BX+COM_IVSEG]       ;retrieve saved int vector segment
        MOV     ES:[DI+2],CX            ;restore it
        MOV     CX,[BX+COM_IVOFF]       ;retrieve saved int vector segment
        MOV     ES:[DI],CX              ;restore it
        POP     DX                      ;restore saved registers
        POP     ES
	STI
	ADD	DL,COM_LCR		;DX -> Line-Control Register
	XOR	AX,AX			;allow access to IER
	OUT	DX,AL
	ADD	DL,COM_IER-COM_LCR	;DX -> Interrupt Enable Register
	JMP	$+2
	OUT	DX,AL			;turn off all interrupts
	ADD	DL,COM_MCR-COM_IER	;DX -> Modem-Control Register
	JMP	$+2
	OUT	DX,AL			;turn off OUT2 and all modem controls
        XOR     AX,AX
	RET

CM_CLOSEPORT ENDP

;****************************************************************************
; CM_DELAY - Wait for a given number of clock ticks
;
; On entry: CL = number of 18.2-to-a-second ticks to delay
;****************************************************************************
CM_DELAY PROC NEAR	
	PUSH	DS
	XOR	AX,AX
	MOV	DS,AX			;DS = segment 0
	XOR	CH,CH

CM_DELAY1:	
        CMP	AX,DS:[TIMER]		;check AX against system clock
	JE	CM_DELAY1		;if the same then wait
	MOV	AX,DS:[TIMER]		;else AX = value of system clock
	LOOP	CM_DELAY1       	;decrement tick counter
	POP	DS
	RET

CM_DELAY ENDP

;****************************************************************************
; COM Interrupt Handlers
;****************************************************************************
CM_INT_HANDLER PROC NEAR

CM_INT_HANDLER1:	
        PUSH	BX			        ;COM1
	MOV	BX,OFFSET PORT1
	JMP	SHORT CM_INT_HANDLER_START

CM_INT_HANDLER2:	
       	PUSH	BX			        ;COM2
	MOV	BX,OFFSET PORT2
	JMP	SHORT CM_INT_HANDLER_START

CM_INT_HANDLER3:	
       	PUSH	BX			        ;COM3
	MOV	BX,OFFSET PORT3
	JMP	SHORT CM_INT_HANDLER_START

CM_INT_HANDLER4:	
       	PUSH	BX			        ;COM4
	MOV	BX,OFFSET PORT4

CM_INT_HANDLER_START:
	PUSH	AX			        ;BX -> port table
	PUSH	DX
	PUSH	SI
	PUSH	DS
	PUSH	ES			        ;CX, DI and BP are not used here
	PUSH	CS			        ;set DS to local segment
	POP	DS
	MOV	DX,[BX+COM_ADDRESS]	        ;DX = COM port i/o address
	ADD	DL,COM_IIR		        ;DX -> Interrupt ID Register

CM_INT_HANDLER_PROCESS:
	XOR	AX,AX
	IN	AL,DX			        ;identify the cause of the interrupt
        TEST	AL,01H			        ;check the interrupt pending bit
        JNZ	CM_INT_HANDLER_PROCESS2	        ;if none
	PUSH	DX			        ;save IIR address
	MOV	SI,AX			        ;and branch according to interrupt
	CALL	CM_INTSWITCH[SI]
	POP	DX
	JMP	SHORT CM_INT_HANDLER_PROCESS

CM_INT_HANDLER_PROCESS2:
	MOV	AL,20H		               ;take care of interrupt controller
	OUT	20H,AL
	POP	ES
	POP	DS
	POP	SI
	POP	DX
	POP	AX
	POP	BX
	IRET

CM_INT_HANDLER ENDP
		
;****************************************************************************
; Received Character Interrupt
;****************************************************************************
CM_RX_INT PROC NEAR	
	ADD	DL,COM_RXB-COM_IIR	        ; DX -> Line Status Register
	IN	AL,DX			        ; read character from COM port
	TEST	[BX+COM_OPTIONS],OUTPUT_FLOW	;output flow control on?
	JZ	CM_RX_INT2			;if not...
	CMP	AL,CTRL_S		        ;is character a Control-S?
	JNE	CM_RX_INT1			;no, check for Control-Q
	OR	[BX+COM_FLAGS],CF_OSF	        ;set flag to suspend output
	RET

CM_RX_INT1:
	CMP	AL,CTRL_Q		        ;is character a Control-Q?
	JNE	CM_RX_INT2		        ;if not then it is data
	TEST	[BX+COM_FLAGS],CF_OSF	        ;is output stopped?
	JZ	CM_RX_INT4		        ;if it is not then we need do nothing
	AND	[BX+COM_FLAGS],NOT CF_OSF	;clear suspend-output flag
	ADD	DL,COM_LSR-COM_RXB	        ;DX -> Line Status Register
	IN	AL,DX			        ;read the port status
	TEST	AL,20H			        ;is the transmitter buffer empty?
	JZ	CM_RX_INT4		        ;if not then output will continue
	JMP	CM_START_OUTPUT		        ;else it must be restarted

CM_RX_INT2:
	LES	SI,DWORD PTR [BX+COM_IN_PTR]	;ES:SI = input pointer
	MOV	BYTE PTR ES:[SI],AL		;deposit input in buffer
	INC	SI			        ;bump pointer to next word
	INC	[BX+COM_IN_CTR]		        ;increment counter
	CMP	SI,[BX+COM_IN_LIMIT]	        ;time to wrap?
	JE	CM_RX_INT5			;yes...

CM_RX_INT3:
	MOV	[BX+COM_IN_PTR],SI	        ;update in pointer
	CMP	SI,[BX+COM_RD_PTR]	        ;check for overflow
	JE	CM_RX_INT6			;reset the buffer with error flag
	TEST	[BX+COM_OPTIONS],INPUT_FLOW	;input flow control on?
	JZ	CM_RX_INT4			;if not we are done
	TEST	SI,000EH		        ;else check buffer every 8th char
	JNZ	CM_RX_INT4
	TEST	[BX+COM_FLAGS],CF_ISF	        ;is input staunched?
	JNZ	CM_RX_INT4			;if so all is well
	MOV	AX,[BX+COM_IN_CTR]	        ;AX = input counter
	CMP	AX,[BX+COM_IN_HI]
	JB	CM_RX_INT4
	OR	[BX+COM_FLAGS],CF_ISF	        ;flag input staunched
	ADD	DL,COM_LSR-COM_RXB	        ;DX -> Line Status Register
	IN	AL,DX
	TEST	AL,20H			        ;transmit buffer empty?
	MOV	AL,CTRL_S
	JZ	CM_RX_INT35			;no
	ADD	DL,COM_TXB-COM_LSR	        ;DX -> Transmit Buffer
	OUT	DX,AL			        ;transmit the ^S or ^Q
	RET

CM_RX_INT35:
	MOV	[BX+COM_OUTFLOW],AL

CM_RX_INT4:
	RET

CM_RX_INT5:
	MOV	SI,[BX+COM_IN_FIRST]
	JMP	SHORT	CM_RX_INT3

CM_RX_INT6:
	XOR	AX,AX
	MOV	[BX+COM_IN_CTR],AX	;zero the input counter
	DEC	AX			;store -1 as overflow marker
	JMP	SHORT CM_RX_INT2

CM_RX_INT ENDP
		
;****************************************************************************
; Transmit Character Interrupt
;****************************************************************************
CM_TX_INT PROC NEAR
	ADD	DL,COM_TXB-COM_IIR	        ;DX -> COM port transmit buffer
	XOR	AX,AX
	XCHG	AL,[BX+COM_OUTFLOW]	        ;is there a flow-control char waiting?
	TEST	AL,AL
	JNZ	CM_TX_INT2			;if so go send it

	TEST	[BX+COM_FLAGS],CF_OSF	        ;is output suspended by flow control?
	JNZ	CM_TX_INT4			;if so then we do nothing
	PUSH	ES
	LES	SI,DWORD PTR [BX+COM_OUT_PTR]	;get ES:SI = output pointer
	TEST	SI,SI			        ;is output active?
	JZ	CM_TX_INT3			;no, return

	MOV	AL,ES:[SI]		        ;AL = next character
	INC	SI			        ;bump the pointer
	CMP	SI,[BX+COM_OUT_LIMIT]	        ;time to stop?
	JE	CM_TX_INT5			;yes...

CM_TX_INT1:
	MOV	[BX+COM_OUT_PTR],SI	        ;update pointer
	POP	ES

CM_TX_INT2:
	OUT	DX,AL			        ;transmit the character
	CLC				        ;signal character sent
	RET

CM_TX_INT3:
	POP	ES

CM_TX_INT4:
	STC				        ;signal no character sent
	RET

CM_TX_INT5:
	XOR	SI,SI
	JMP	SHORT	CM_TX_INT1

CM_TX_INT	ENDP

;****************************************************************************
; Status Interrupts
;****************************************************************************
CM_MODEM_STATUS_INT PROC NEAR
	ADD	DL,COM_MSR-COM_IIR
	IN	AL,DX
	RET

CM_MODEM_STATUS_INT ENDP

CM_LINE_STATUS_INT PROC NEAR
	ADD	DL,COM_LSR-COM_IIR
	IN	AL,DX
	RET

CM_LINE_STATUS_INT ENDP

PROGSEG ENDS    ;end the code segment
      END     	;end the program

