*******************************************************************************
*
* Function..: IN_FILE
* Purpose...: Check if the passed value is in the passed File.
* Parameters:
*   Name	Type	Description.
* --------------------------------------------------------------------------
*   pfile	Char  	File to use.
*   pkey	?   	Key to search for.
*			The type is the type of expression in the index
*			you are searching.  For example, to search for
*			mcust_no (type N) where the index is made of
*			STR(CUST_NO), pkey should be str(mcust_no, ?).
*   perror	L 	False is converted to numeric type 2, True to 1.
*		N	Indicates type of message display:
*			0 -- No message, return result of search.
*			1 -- Warning if key is not found, return True.
*			2 -- Error if key is not found, return result of search.
*			Default is 1.
*   pfield	C	Field to display on screen. (Optional)
*			This character expression is evaluated as a macro,
*			therefore, it is possible to use a display formula
*			such as substr() or transform(), etc.
*   prow	N	Row to display pfield on.
*   pcol	N	Column to display pfield on.
*   fattr	L       Use file attributes for protected files.
*   restrec	L	Restore Original Record (default .t.)
* --------------------------------------------------------------------------
*
* Returns...: .t. if exists or perror is .t., .f. if not.
* Notes.....: Intended for use in the VALID clause.  Ex;
*
*	      in_file("CSMA", str(mcust_no, 10)) would search the Customer
*	      Reference file "FHCSMA" for the string of mcust_no.  It will
*	      display an ERROR message and return .f. to the valid so the
*	      field will not be accepted.
*
*	      in_file("CSMA", str(mcust_no, 10), .t.) would do the same as
*	      above with the exception of a WARNING message displaying
*	      instead of an error message, and .t. will return to the valid
*	      so the field will be accepted.
*
*	      in_file("CSMA", str(mcust_no, 10), .f., "NAME", 10, 25) will
*	      perform the same search as the first example.  It will also
*	      put the field "NAME" on row 10, column 25 of the display if
*	      the search is found.
*******************************************************************************

function IN_FILE
parameters pfile, pkey, perror, pfield, prow, pcol, fattr, restrec
private trec				&& Record number on current file.

if (lastkey() = 5)			&& Up pressed, ok.
   return (.t.)
endif

if (type("PERROR") = 'U')		&& Nothing sent, default is error.
   private perror
   perror = 2
elseif (type("PERROR") = 'L')		&& Logical sent, convert to numeric.
   perror = iif(perror, 1, 2)
endif

if (type("RESTREC") != 'L')		&& Restore Original Record Parameter.
   restrec = .t.
endif
if (type("FATTR") != 'L')		&& File attribute parameter.
   fattr = .f.
endif
if (type("PFIELD") != 'C')
   pfield = ''
endif

private inuse, ;			&& Is file already open?
	curscr, ;			&& Current screen.
	oldarea, ;			&& Current work area.
	result, ;			&& Result of "seek".
	trec				&& Current record no if inuse.

inuse = .f.				&& Defa is no.

if (empty(pkey))			&& No value sent, error.
   return (.f.)
endif

oldarea = select()			&& Get current work area.
trec = recno()				&& Save for later.

if (select(pfile) > 0)			&& Already in use.
   select &pfile.
   inuse = .t.
else
   select 0
   if (fattr)
      fattr(pfile, "-r/o -hid")		&& Un-protect the file.
   endif
   use (pfile) index (pfile)
endif

seek pkey				&& Look for the value.
result = found()			&& Was it found?

if (!empty(pfield))			&& Display a field.
   fdisp(pfile, pkey, pfield, prow, pcol)
endif

if (!inuse)
   use					&& Close File.
   if (fattr)
      fattr(pfile, "+r/o +hid")		&& Protect the file.
   endif
endif

select(oldarea)				&& Back to original work area.
if (restrec)
   go trec				&& Back to current record.
endif

if (!result .and. perror > 0)		&& Not Found, display message.
   save screen to curscr
   @ 24,0 say sprint(space(80), n_scolor(24,0), .f.)
   @ 24,25 say sprint(iif(perror = 1, "Warning! ", "Error!!  ") ;
                      + "Entry Does not Exist!", "*"+warning, .f.)
   tone(300, 2)
   tone(800, 2)
   inkey(3)
   restore screen from curscr
endif

return (iif(perror = 1, .t., result)) 	&& Return .t./.f. for Valid Statement.


*******************************************************************************
*
* Function..: FDISP
* Purpose...: Display a field from a database on screen on given row, column.
* Parameters: pfile	File to use.
*	      pkey	Search key.
*	      pfield	Field to display.
*	      prow	Row to display.
*	      pcol	Column to display.
* Returns...: Result of seek.

function FDISP
parameters pfile, pkey, pfield, prow, pcol

private inuse, ;			&& Is file already open?
	oldarea, ;			&& Current work area.
	dfield, ;			&& Contents of "pfield".
	result, ;			&& Result of "seek".
	curcolor			&& Current color.

inuse = .f.				&& Defa is no.

if (lastkey() = 5)			&& Up pressed, ok.
   return (.t.)
endif

if (empty(pkey))			&& No value sent, error.
   return (.f.)
endif

oldarea = select()			&& Get current work area.
curcolor = setcolor()			&& Save current color.

if (select(pfile) > 0)			&& Already in use.
   select &pfile.			&& Go to it.
   inuse = .t.				&& Set inuse flag.
else
   select 0				&& Unused area.
   use (pfile) index (pfile)		&& Open file and index.
endif

seek pkey				&& Look for the value.
result = found()			&& Was it found?
dfield = &pfield.			&& Get the field to display.

if (!inuse)
   use					&& Close File.
endif

select(oldarea)				&& Back to original work area.

if (result)
   setcolor(n_scolor(prow, pcol))	&& Color needed at this position.
   @ prow, pcol say dfield
   setcolor(curcolor)			&& Back to original color.
endif

return (result)				&& Return result of search.

*******************************************************************************
*
* Function..: DISP
* Purpose...: Display on given row, column.
* Parameters: pdisp	Display parameter.
*	      prow	Row to display.
*	      pcol	Column to display.
*	      ppict	Picture to use (optional).
* Returns...: .T.

function DISP
parameters pdisp, prow, pcol, ppict

if (empty(pdisp))			&& No value sent, error.
   return (.t.)
endif

if (type("PPICT") != 'U')		&& Picture sent, use it.
   @ prow, pcol say pdisp picture ppict
else
   @ prow, pcol say pdisp
endif

return (.t.)				&& This function always returns .t.

