#include "extend.h"
#include <ctype.h>

void capit(char *, char *, int);		/* func that does the capitalizing */

/**********************************************************************
* Function   :  capfirst()
*
* Written By :  Richard Horwitz
*
* Copyright  :  Fieldston Consulting Group, Inc. 1990

* Clipper Syntax  :  capfirst(<string>) or capfirst(<array>)
*
*	Where: 
*		<string> 		- the string to capitalize
*		<array> 		- an array of strings to capitalize
*
*
* Description:  This function is called from Clipper and acts as an
*    intermediate function to capit(), the C function that actually
*    does the capitalizing of the string.  It takes one parameter which
*    can be either a string or an array.  If a string is passed it is
*    capitalized and returned unless it has been passed by reference, in which
*    case the value is _stor()'ed back into the parameter.  If an array is 
*    passed each string element is capitalized, and then _stor()'ed back into 
*    the array.
*  
* One of the techniques demonstrated in this function is the use of
*    either a static buffer or allocated memory for the return string
*    depending on the size of the string to capitalize.  The reason
*    you might want to use a static buffer instead of allocating memory
*    is that it's faster to assign a pointer to an existing buffer than
*    to allocate and free memory. 
*  
***********************************************************************/
CLIPPER capfirst()
{
char *_instring;		      /* pointer to the string to capitalize */
char *_outstring;		      /* pointer to the string to return/store */
int _size;			         /* size of string to capitalize */
int _asize;			         /* size of passed array */
char _outbuffer[1024];		/* buffer for return if _size < 1024 chars */

/* do parameter and type checking */
if((PCOUNT < 1) || (!(ISCHAR(1) || ISARRAY(1))))
	{
	_retc("");	/* wrong or no params, so return null */
	return;	/* back to Clipper */
	}

/* if what they passed was a string */
if(ISCHAR(1))
	{
	_size = _parclen(1);	      		/* get size of passed string */

	/* if the size of the passed string is greater than 1024 */
	if (_size > 1024)
		_outstring = _xgrab(_size+1);	/* otherwise allocate the memory */
	else
		_outstring = _outbuffer;		/* use the static buffer for output */

	_instring  = _parc(1);		   	/* set pointer to passed string */

	capit(_instring, _outstring, _size);	/* do the actual capitalizing */

	/* if the string was passed by reference */
	if(ISBYREF(1))
		{
		_storc(_outstring,1);		/* store the output to it */
		_ret();			         	/* return NIL to Clipper */
		}
	else
		_retc(_outstring);		/* return the output string */

	/* if the size of the passed string is greater than 1024 */
	if (_size > 1024)
		_xfree(_outstring);		/* free the allocated buffer */

	}
	else	/* it's an array */
	{

	int _x;					      /* loop counter */
	_asize = ALENGTH(1);			/* get declared size of passed array */

	/* loop around capitalizing each array element */
	for(_x = 1; _x <= _asize; _x++)
		{

		/* make sure current array element is a string */
		if(_parinfa(1,_x) != CHARACTER)
			continue;

		_size = _parclen(1,_x);	   		/* get size of array element */

		/* if the size of the array element is greater than 1024 */
		if (_size > 1024)
			_outstring = _xgrab(_size+1);	/* otherwise allocate the memory */
		else
			_outstring = _outbuffer;		/* use the static buffer for output */

		_instring = _parc(1,_x);			/* set pointer to array element */

		capit(_instring, _outstring, _size);	/* do the actual capitalizing */

		_storc(_outstring,1,_x);			/* stor the output back to array */

		/* if the size of the passed string is greater than 1024 */
		if (_size > 1024)
			_xfree(_outstring);		      /* free the allocated buffer */

		_ret();
		}
	}
}

/**********************************************************************
* Function   :  capit()
*
* Written By :  Richard Horwitz
*
* Copyright  :  Fieldston Consulting Group, Inc. 1990
*
* Description:  This function does the actual capitalizing of the 
*    passed string.  The result is stored into the passed buffer 
*    _outbuf. 
*  
* Note : This function makes use of the isalpha, isupper and isspace
*    macros from ctype.h and requires the linking of the
*    Microsoft Runtime Library LLIBCA. 
***********************************************************************/
void capit(_inbuf,_outbuf, size)
char *_inbuf;
char *_outbuf;
int size;
{
int  _k = 0;                /* loop counter */

/* loop around for each char in string */
for(_k = 0;_k < size; _k++)
	{
	/* if it's an alpha char */
	if(isalpha(_inbuf[_k]))

		/* if we are on the first char (can't check if _k - 1 is space here) */
		if(_k == 0)

			/* make it upper case */
			_outbuf[_k] = (isupper(_inbuf[_k]) ? _inbuf[_k] : _inbuf[_k] - 32 );

		else

			/* if the previous char is a space */
			if(isspace(_inbuf[_k-1]))

				/* make it upper case */
				_outbuf[_k] = (isupper(_inbuf[_k]) ? _inbuf[_k] : _inbuf[_k] - 32 );

			else

				/* make it lower case */
				_outbuf[_k] = (isupper(_inbuf[_k]) ? _inbuf[_k] + 32 : _inbuf[_k]);
	else

		/* not an alpha char, so just move it over */
		_outbuf[_k]=_inbuf[_k];

	}
	_outbuf[size] =0x00;			/* make sure last char is null */
}
