/*
For the sample program:

Compile with "/n /dFT_TEST" switches and link.

Pass "mono" or "MONO" as a command line parameter to force mono mode.

Pass "nosnow" or "NOSNOW" as a command line parameter on a CGA.

Pass "vga" or "VGA" as a command line parameter for 50-line mode.
 */

/*
 * Function..: FT_MENU.PRG
 * Author....: Paul Ferrara [76702,556]
 * Date......: $Date$
 * Revision..: $Revision$
 * Log file..: $Logfile$
 *
 * This function is an original work by Paul Ferrara and is placed in the
 * public domain.
 *
 * Modification history:
 * ---------------------
 *
 * $Log$
 *
 */


/*  $DOC$
 *  $FUNCNAME$
 *    ft_Menu()
 *  $ONELINER$
 *    Fast, pulldown menu system
 *  $SYNTAX$
 *    ft_Menu( <acBarNames>, <acOptions>, <acAction>, <acColors>
 *             [,<nTopRow>], [<lShadow>] ) --> NIL
 *  $ARGUMENTS$
 *    <acBarNames> is a character array containing the names to appear on the
 *    Menu Bar.
 *
 *    <acOptions> is a multi-dimensional array with one element for each
 *    selection to appear on the pulldown menus.
 *
 *    <acColors> is an array containing the colors for the menu groups.
 *
 *    <nTopRow> is a numeric value that determines the row for the menu
 *    bar.  If omitted, it defaults to 0.
 *
 *    <lShadow> is a logical variable.  If true (.T.) or omitted, it
 *    uses ft_Shadow() to add a transparent shadow to the each
 *    pulldown menu.  If false (.F.), the menu is drawn without
 *    the shadow.
 *
 *    All arguments except nTopRow and lShadow are required.
 *
 *  $RETURNS$
 *    NIL
 *  $DESCRIPTION$
 *    FT_MENU() is a function that displays a pulldown menu for each item
 *    on the menu bar and executes the corresponding function for the item
 *    selected.  When a called function returns false, FT_MENU returns
 *    control to the calling program.
 *
 *    Valid keystrokes and their corresponding action:
 *
 *    Home             -  Activates pulldown for first item on the menu bar.
 *    End              -  Activates pulldown for last item on the menu bar.
 *    Left Arrow       -  Activates next pulldown to the left.
 *    Right Arrow      -  Activates next pulldown to the right.
 *    Tab              -  Same as left arrow.
 *    Shift-Tab        -  Same as right arrow.
 *    Page Up          -  Top item on current pulldown menu.
 *    Page Down        -  Bottom item on current pulldown menu.
 *    Enter            -  Selects current item.
 *    Alpha Character  -  Moves to closest match and selects.
 *    Alt-<key>        -  Moves to corresponding menu bar item.
 *    ESCape           -  Prompts for confirmation and either returns to
 *                        the calling routine or resumes.
 *
 *  $EXAMPLES$
 *
 *  // declare arrays
 *  LOCAL aColors  := {}
 *  LOCAL aBar     := { " ENTER/EDIT ", " REPORTS ", " DISPLAY " }
 *
 *  // Include the following two lines of code in your program, as is.
 *  // The first creates aOptions with the same length as aBar.
 *  // The second assigns a three element array to each element of aOptions.
 *  LOCAL aOptions[ LEN( aBar ) ]
 *  AEVAL( aBar, { |x,i| aOptions[i] := { {},{},{} } } )
 *
 *  // fill color array //
 *  // Box Border, Menu Options, Menu Bar, Current Selection, Unselected
 *  aColors := IF( lColor, {"W+/G", "N/G", "N/G", "N/W", "N+/G"}, ;
 *                         {"W+/N", "W+/N", "W/N", "N/W","W/N"} )
 *
 *  // array for first pulldown menu
 *  ft_Fill( aOptions[1], 'A. Execute A Dummy Procedure' , {|| fubar()}, .t. )
 *  ft_Fill( aOptions[1], 'B. Enter Daily Charges'       , {|| .t.},     .f. )
 *  ft_Fill( aOptions[1], 'C. Enter Payments On Accounts', {|| .t.},     .t. )
 *
 *  // array for second pulldown menu
 *  ft_Fill( aOptions[2], 'A. Print Member List'         , {|| .t.},     .t. )
 *  ft_Fill( aOptions[2], 'B. Print Active Auto Charges' , {|| .t.},     .t. )
 *
 *  // array for third pulldown menu
 *  ft_Fill( aOptions[3], 'A. Transaction Totals Display', {|| .t.},     .t. )
 *  ft_Fill( aOptions[3], 'B. Display Invoice Totals'    , {|| .t.},     .t. )
 *  ft_Fill( aOptions[3], 'C. Exit To DOS'               , {|| .f.},     .t. )
 *
 *  Call ft_Fill() once for each item on each pulldown menu, passing it
 *  three parameters:
 *
 *     ft_Fill( <cMenuSelection>, <bCodeBlock>, <lSelectable>
 *
 *  <cMenuSelection> is a character string which will be displayed on the
 *  pulldown menu.
 *
 *  <bCodeBlock> should contain one of the following:
 *
 *     A function name to execute, which in turn should return .t. or .f.
 *     FT_MENU will return control to the calling program if .f. is returned
 *     or continue if .t. is returned.
 *
 *     .f., which will cause FT_MENU to return control to the calling program.
 *
 *     .t., which will do nothing.  This allows the developer to design a
 *     skeleton menu structure prior to completing all of the subroutines.
 *
 *  // Call FT_MENU
 *  FT_MENU( aBar, aOptions, aColors, 0 )
 *
 *  NOTE: FT_MENU() disables Alt-C and Alt-D in order to make them
 *        available for the menu bar.  It enables Alt-D and resets
 *        Alt-C to its previous state prior to calling each function.
 *
 *  $END$
 */

/*  $DOC$
 *  $FUNCNAME$
 *    ft_Fill()
 *  $ONELINER$
 *    Used to declare menu options for FT_MENU()
 *  $SYNTAX$
 *    ft_Fill( <aSubArrayName>, <cMenuSelection>, <bFunction>,
 *             <lSelectable> ) --> NIL
 *  $ARGUMENTS$
 *    <aSubArrayName> is a sub-array of <acOptions> denoting which
 *    group in which to include the selection.  i.e. aOptions[1]
 *
 *    <cMenuSelection> is the character string that will appear on
 *    the menu.
 *
 *    <bFunction> is the code block to be executed when that menu
 *    option is selected.  i.e. {|| MyFunction() } would execute
 *    the function called MyFunction().  {|| .f.} would exit the
 *    FT_MENU and return to the calling routine.  And {|| .t.} would
 *    do nothing.
 *
 *    <lSelectable> is a logical variable that determines whether
 *    the corresponding menu option is selectable or not.
 *
 *  $RETURNS$
 *    NIL
 *  $DESCRIPTION$
 *    FT_FILL() is a function used to set up the menu options prior
 *    to calling FT_MENU().
 *
 *  $EXAMPLES$
 *
 *  ft_Fill( aOptions[1], 'A. Execute A Dummy Procedure' , {|| fubar()}, .t. )
 *
 *  The above would be added to the sub-menu associated with the first menu
 *  bar item, would execute the function FUBAR() when that option was
 *  selected, and would be selectable.
 *
 *  ft_Fill( aOptions[3], 'B. Enter Daily Charges'       , {|| .t.},     .f. )
 *
 *  The above would be added to the sub-menu associated with the third menu
 *  bar item, and would be unselectable.
 *
 *  ft_Fill( aOptions[2], 'C. Enter Payments On Accounts', {|| .t.},     .t. )
 *
 *  The above would be added to the sub-menu associated with the second menu
 *  bar item, and would be selectable, but would do nothing when selected.
 *
 *  ft_Fill( aOptions[4], 'C. Exit'                      , {|| .f.},     .t. )
 *
 *  The above would be added to the sub-menu associated with the fourth menu
 *  bar item, and would be selectable, and would exit FT_MENU() when chosen.
 *
 *  $END$
 */

#include "INKEY.CH"
#include "ACHOICE.CH"
#include "SETCURS.CH"

#define DISABLE     0
#define ENABLE      1

STATIC aStack := {},     ;
       aChoices := {},   ;
       aValidKeys := {}, ;
       nHpos,            ;
       nVpos

/* beginning of demo program */
#ifdef FT_TEST
   PROCEDURE Main( cCmdLine )
      LOCAL sDosScrn, nDosRow, nDosCol, lColor

      /* color memvars */
      STATIC cNormH, cNormN, cNormE, ;
             cWindH, cWindN, cWindE, ;
             cErrH, cErrN, cErrE

      /* options on menu bar */
      LOCAL aColors  := {}
      LOCAL aBar     := { " ENTER/EDIT ", " REPORTS ", " DISPLAY ", " MAINTENANCE ", " QUIT " }
      LOCAL aOptions[ LEN( aBar ) ]
      AEVAL( aBar, { |x,i| aOptions[i] := { {},{},{} } } )

      cCmdLine := IF( cCmdLine == NIL, "", cCmdLine )

      lColor := IF( "MONO" $ UPPER( cCmdLine ), .F., ISCOLOR() )

      * Border, Box, Bar, Current, Unselected
      aColors := IF( lColor, {"W+/G", "N/G", "N/G", "N/W", "N+/G"}, ;
                             {"W+/N", "W+/N", "W/N", "N/W", "W/N"} )

      ft_Fill( aOptions[1], 'A. Execute A Dummy Procedure'        , {|| fubar()}, .t. )
      ft_Fill( aOptions[1], 'B. Enter Daily Charge/Credit Slips'  , {|| .t.}, .t. )
      ft_Fill( aOptions[1], 'C. Enter Payments On Accounts'       , {|| .t.}, .f. )
      ft_Fill( aOptions[1], 'D. Edit Daily Transactions'          , {|| .t.}, .t. )
      ft_Fill( aOptions[1], 'E. Enter/Update Member File'         , {|| .t.}, .t. )
      ft_Fill( aOptions[1], 'F. Update Code File'                 , {|| .t.}, .f. )
      ft_Fill( aOptions[1], 'G. Add/Update Auto Charge File'      , {|| .t.}, .t. )
      ft_Fill( aOptions[1], 'H. Post All Transactions To A/R File', {|| .t.}, .t. )
      ft_Fill( aOptions[1], 'I. Increment Next Posting Date'      , {|| .t.}, .t. )

      ft_Fill( aOptions[2], 'A. Print Member List'                , {|| .t.}, .t. )
      ft_Fill( aOptions[2], 'B. Print Active Auto Charges'        , {|| .t.}, .t. )
      ft_Fill( aOptions[2], 'C. Print Edit List'                  , {|| .t.}, .t. )
      ft_Fill( aOptions[2], 'D. Print Pro-Usage Report'           , {|| .t.}, .t. )
      ft_Fill( aOptions[2], 'E. Print A/R Transaction Report'     , {|| .t.}, .t. )
      ft_Fill( aOptions[2], 'F. Aging Report Preparation'         , {|| .t.}, .t. )
      ft_Fill( aOptions[2], 'G. Add Interest Charges'             , {|| .t.}, .t. )
      ft_Fill( aOptions[2], 'H. Print Aging Report'               , {|| .t.}, .t. )
      ft_Fill( aOptions[2], 'I. Print Monthly Statements'         , {|| .t.}, .t. )
      ft_Fill( aOptions[2], 'J. Print Mailing Labels'             , {|| .t.}, .t. )
      ft_Fill( aOptions[2], 'K. Print Transaction Totals'         , {|| .t.}, .t. )
      ft_Fill( aOptions[2], 'L. Print Transaction Codes File'     , {|| .t.}, .t. )
      ft_Fill( aOptions[2], 'M. Print No-Activity List'           , {|| .t.}, .t. )

      ft_Fill( aOptions[3], 'A. Transaction Totals Display'       , {|| .t.}, .t. )
      ft_Fill( aOptions[3], 'B. Display Invoice Totals'           , {|| .t.}, .t. )
      ft_Fill( aOptions[3], 'C. Accounts Receivable Display'      , {|| .t.}, .t. )

      ft_Fill( aOptions[4], 'A. Backup Database Files'            , {|| .t.}, .t. )
      ft_Fill( aOptions[4], 'B. Reindex Database Files'           , {|| .t.}, .t. )
      ft_Fill( aOptions[4], 'C. Set System Parameters'            , {|| .t.}, .t. )
      ft_Fill( aOptions[4], 'D. This EXITs Too'                   , {|| .f. }, .t. )

      ft_Fill( aOptions[5], 'A. Does Nothing'                     , {|| .t.}, .t. )
      ft_Fill( aOptions[5], 'B. Exit To DOS'                      , {|| .f. }, .t. )

      // main routine starts here
      SET SCOREBOARD OFF

      cNormH := IF( lColor, "W+/G", "W+/N" )
      cNormN := IF( lColor, "N/G" , "W/N"  )
      cNormE := IF( lColor, "N/W" , "N/W"  )
      cWindH := IF( lColor, "W+/B", "W+/N" )
      cWindN := IF( lColor, "W/B" , "W/N"  )
      cWindE := IF( lColor, "N/W" , "N/W"  )
      cErrH  := IF( lColor, "W+/R", "W+/N" )
      cErrN  := IF( lColor, "W/R" , "W/N"  )
      cErrE  := IF( lColor, "N/W" , "N/W"  )

      sDosScrn := SAVESCREEN()
      nDosRow := ROW()
      nDosCol := COL()
      SETCOLOR( "W/N" )
      CLS
      NOSNOW( ( "NOSNOW" $ UPPER( cCmdLine ) ) )
      IF "VGA" $ UPPER( cCmdLine )
         SETMODE(50,80)
      ENDIF
      nMaxRow := MAXROW()
      SETBLINK( .F. )
      SETCOLOR( cWindN + "*" )
      CLS
      SETCOLOR( cNormN )
      @ nMaxRow, 0
      @ nMaxRow, 0 SAY " FT_MENU 2.0  "
      @ nMaxRow,15 SAY "Written by Paul Ferrara [76702,556] For NANFORUM.LIB"
      @ nMaxRow,69 SAY " "+DTOC( DATE() )

      SETCOLOR( cErrH )
      @ nMaxRow-11, 23, nMaxRow-3, 56 BOX "Ŀ "
      @ nMaxRow- 9,23 SAY "Ĵ"
      SETCOLOR( cErrN )
      @ nMaxRow-10,33 SAY "Navigation Keys"
      @ nMaxRow- 8,25 SAY "LeftArrow   RightArrow   Alt-E"
      @ nMaxRow- 7,25 SAY "Home        End          Alt-R"
      @ nMaxRow- 6,25 SAY "Tab         Shift-Tab    Alt-D"
      @ nMaxRow- 5,25 SAY "PgUp        PgDn         Alt-M"
      @ nMaxRow- 4,25 SAY "Enter       ESCape       Alt-Q"
      SETCOLOR( cNormN )

      Ft_Menu( aBar, aOptions, aColors )

      SETCOLOR( "W/N" )
      SETCURSOR( SC_NORMAL )
      SETBLINK( .T. )
      IF "VGA" $ UPPER( cCmdLine )
         SETMODE(25,80)
      ENDIF
      RESTSCREEN(,,,, sDosScrn )
      SETPOS( nDosRow, nDosCol )
      QUIT


   FUNCTION Fubar()
      LOCAL OldColor:= SETCOLOR( "W/N" )
      CLS
      QOUT( "Press Any Key" )
      INKEY(0)
      SETCOLOR( OldColor )
      RETURN .T.
#endif
/* end of demo program */


FUNCTION Ft_Menu( aBar, aOptions, aColors, nTopRow, lShadow )
   LOCAL nTtlWid, nTtlUsed, i, j, nPad, ;
         cScreen, lCancMode, lLooping := .T.

   /* column position for each item on the menu bar */
   LOCAL aBarCol[LEN(aBar)]

   /* inkey code for each item on menu bar */
   LOCAL aBarKeys[ LEN( aBar ) ]

   /* inkey codes for A - Z */
   LOCAL aKeyCodes := { 286, 304, 302, 288, 274, 289, 290, 291, 279, ;
                        292, 293, 294, 306, 305, 280, 281, 272, 275, ;
                        287, 276, 278, 303, 273, 301, 277, 300 }

   /* length of widest array element for for each pulldown menu */
   LOCAL aBarWidth[LEN(aBar)]

   /* starting column for each box */
   LOCAL aBoxLoc[LEN(aBar)]

   /* last selection for each element */
   LOCAL aLastSel[LEN(aBar)]

   /* color memvars */
   LOCAL cBorder  := aColors[1]
   LOCAL cBox     := aColors[2]
   LOCAL cBar     := aColors[3]
   LOCAL cCurrent := aColors[4]
   LOCAL cUnSelec := aColors[5]

   /* push current arrays/memvars onto stack - needed for recursion */
   AADD( aStack, { aChoices, aValidKeys, nHpos, nVpos } )

   nMaxRow := MAXROW()
   nMaxCol := MAXCOL()

   /* row for menu bar */
   nTopRow := IF( nTopRow == NIL, 0, nTopRow )

   AFILL(aLastSel,1)
   aChoices := aOptions

   /* calculate the position of each item on the menu bar */
   nTtlWid := 0
   aBarCol[1] := 0
   nTtlUsed := LEN( aBar[1] ) + 1
   AEVAL( aBar, ;
          {|x,i| aBarcol[i]:= nTtlUsed,nTtlUsed+= (LEN(aBar[i]) +1 )}, ;
          2, LEN(aBar) -1 )

   /* calculates widest element for each pulldown menu */
   AFILL(aBarWidth,1)
   AEVAL( aChoices, { |x,i| _ftWidest( @i, aChoices, @aBarWidth ) } )

   /* box location for each pulldown menu */
   AEVAL( aChoices, { |x,i| _ftLocat( i, aBarCol, aBarWidth, @aBoxLoc, nMaxCol ) } )

   /* valid keys for each pulldown menu */
   AEVAL( aChoices,{|x,i| AADD( aValidkeys,"" ),;
                          _ftValKeys( i,aChoices,@aValidKeys ) } )

   /* display the menu bar */
   SETCOLOR( cBar )
   @ nTopRow, 0
   AEVAL( aBar, { |x,i| Devpos(nTopRow, aBarCol[i]), Devout(aBar[i]) })

   /* store inkey code for each item on menu bar to aBarKeys */
   AEVAL( aBarKeys, {|x,i| aBarKeys[i] := ;
          aKeyCodes[ ASC( UPPER( LTRIM( aBar[i] ) ) ) - 64 ] } )

   /* disable Alt-C and Alt-D so they can be used for ALT-keys */
   lCancMode := SETCANCEL( .f. )
   ALTD( DISABLE )

   cScreen := SAVESCREEN()
   /* which menu and which menu item */
   nHpos := 1; nVpos := 1

   /* main menu loop */
   DO WHILE lLooping
      RESTSCREEN( ,,,, cScreen )
      SETCOLOR( cCurrent )
      @ nTopRow, aBarCol[nHpos] SAY aBar[nHpos]
      IF lShadow == NIL .OR. lShadow
         #ifdef PAULF
            #ifdef UCP
               Shadow( nTopRow+1, aBoxLoc[nHpos], LEN(aChoices[nHpos,1])+nTopRow+2, aBarWidth[nHpos]+3+aBoxLoc[nHpos], 2 )
            #else
               Shadow( nTopRow+1, aBoxLoc[nHpos], LEN(aChoices[nHpos,1])+nTopRow+2, aBarWidth[nHpos]+3+aBoxLoc[nHpos], 1, 8 )
            #endif
         #else
            ft_Shadow( nTopRow+1, aBoxLoc[nHpos], LEN(aChoices[nHpos,1])+nTopRow+2, aBarWidth[nHpos]+3+aBoxLoc[nHpos] )
         #endif
      ENDIF
      SETCOLOR( cBorder )
      @  nTopRow+1, aBoxLoc[nHpos], LEN(aChoices[nHpos,1])+nTopRow+2, aBarWidth[nHpos]+3+aBoxLoc[nHpos] BOX "ͻȺ "
      SETCOLOR( cBox +","+ cCurrent +",,,"+ cUnselec )
      nVpos := ACHOICE( nTopRow+2, aBoxLoc[nHpos]+2, LEN(aChoices[nHpos,1])+nTopRow+2, aBarWidth[nHpos]+1+aBoxLoc[nHpos], aChoices[nHpos,1], aChoices[nHpos,3], "_ftAcUdf", aLastSel[nHpos])
      DO CASE
      CASE LASTKEY() == K_RIGHT .OR. LASTKEY() == K_TAB
         IF( nHpos == LEN( aChoices ), nHpos := 1, nHpos := nHpos + 1 )
      CASE LASTKEY() == K_LEFT .OR. LASTKEY() == K_SH_TAB
         IF( nHpos == 1, nHpos := LEN( aChoices ), nHpos := nHpos - 1 )
      CASE LASTKEY() == K_ESC
         lLooping := !ft_DispMsg( { { "Press [ESC] To Exit",            ;
                                     "Or Any Other Key To Continue" }, ;
                                   { cBox,, cBorder } }, CHR( K_ESC ) )
      CASE LASTKEY() == K_HOME
         nHpos := 1
      CASE LASTKEY() == K_END
         nHpos := LEN( aChoices )
      CASE LASTKEY() == K_ENTER
         aLastSel[nHpos] := nVpos
         IF aChoices[nHpos,2,nVpos] != NIL
            SETCANCEL( lCancMode )
            ALTD( ENABLE )
            lLooping := EVAL( aChoices[nHpos,2,nVpos] )
            ALTD( DISABLE )
            SETCANCEL( .f. )
         ENDIF
      CASE ASCAN( aBarKeys, LASTKEY() ) > 0
         nHpos := ASCAN( aBarKeys, LASTKEY() )
      ENDCASE
   ENDDO
   SETCANCEL( lCancMode )
   ALTD( ENABLE )
   RESTSCREEN( ,,,, cScreen )

   /* pop */
   aChoices   := aStack[ LEN( aStack ), 1 ]
   aValidKeys := aStack[ LEN( aStack ), 2 ]
   nHpos      := aStack[ LEN( aStack ), 3 ]
   nVpos      := aStack[ LEN( aStack ), 4 ]
   ASIZE( aStack, LEN( aStack ) - 1 )

   RETURN NIL


/* ACHOICE() user function */
FUNCTION _ftAcUdf( nMode )
   LOCAL nRtnVal := AC_CONT
   DO CASE
   CASE nMode == AC_HITTOP
      KEYBOARD CHR( K_CTRL_END )
   CASE nMode == AC_HITBOTTOM
      KEYBOARD CHR( K_CTRL_HOME )
   CASE nMode == AC_EXCEPT
      IF UPPER( CHR( LASTKEY() ) ) $ aValidKeys[ nHpos ]
         IF aChoices[ nHpos, 3, AT( UPPER(CHR(LASTKEY())), aValidKeys[ nHpos ] )]
            KEYBOARD CHR( K_ENTER )
            nRtnVal := AC_GOTO
         ENDIF
      ELSE
         nRtnVal := AC_SELECT
      ENDIF
   ENDCASE
   RETURN nRtnVal


STATIC FUNCTION _ftWidest( i, aChoices, aBarWidth )
   AEVAL(aChoices[i,1],{|a,b| aBarWidth[i] := ;
            MAX( aBarWidth[i],LEN(aChoices[i,1,b])) })
   RETURN NIL


STATIC FUNCTION _ftLocat( i, aBarCol, aBarWidth, aBoxLoc, nMaxCol )
   aBoxLoc[i] := IF( aBarCol[i] + aBarWidth[i] + 4 > nMaxCol + 1, ;
                 nMaxCol - 3 - aBarWidth[i], aBarCol[i] )
   RETURN NIL


FUNCTION ft_Fill( aArray, cMenuOption, bBlock, lAvailable )
   AADD( aArray[1], cMenuOption )
   AADD( aArray[2], bBlock )
   AADD( aArray[3], lAvailable )
   RETURN NIL


STATIC FUNCTION _ftValKeys( nNum,aChoices,aValidkeys )
   AEVAL( aChoices[nNum,1], {|x| aValidKeys[nNum] += LEFT( x, 1)} )
   RETURN NIL
