;******************************************************************************
;
;  Function:  FloppyTst
;  Author:    Dave Peckam
;
;  Purpose:   FloppyTst determines the status of the specified floppy
;             disk drive.  FloppyTst 2.0 was designed to remove the
;             need for Tom Rettig's Library and to make FloppyTst less
;             machine-dependent.
;
;             The original version worked under the Autumn '86 release 
;             of Clipper and release 1.1 of Tom Rettig's Library.  In
;             the latest update to the Library, however, Tom Rettig
;             uses the Clipper memory allocation system.  FloppyTst 2.0
;             uses a fixed buffer of 512 bytes.
;             
;             FloppyTst has been tested with:
;
;                 Autumn '86 and Summer '87 releases of Clipper
;                 Release 1.1 of Tom Rettig's Library (TR86 and TR87)
;             
;
;  History:   Dec  9, 1986   FloppyTst 1.0 written in Clipper 
;                            by Charlie Butler with calls to the
;                            ROMBIOS, Allocate and DeAlloc routines
;                            from Tom Rettig's Library
;
;             Feb 12, 1988   Converted FloppyTst to external assembler
;                            function written in MASM 3.0.  Function now
;                            accepts integer instead of character disk
;                            specifiers.
;
;                            Converted Int 13h BIOS calls to 
;                            Int 25h (MS-DOS Absolute Disk Read) and
;                            Int 26h (MS-DOS Absolute Disk Write) calls
;                            for greater MS-DOS compatibility
;
;             Feb 17, 1988   Added support for "Sector not found" 
;                            and "Unknown Media" errors
;
;             Feb 18, 1988   Function now accepts character parameters.
;                            For example, call with "A", "a", "B" or "b".
;
;
;  Date:      February 17, 1988
;  Version:   2.0a for Clipper '87
;  Language:  Microsoft Macro Assembler (MASM) version 3.0
;
;  Syntax:    FloppyTst (<expC>)
;
;             where <expC> is "A", "a", "B" or "b"
;
;  Returns:   A <expN> that will be one of the following:
;
;             0 - Drive loaded and ready to read or write
;             1 - Drive door open or diskette inserted upside-down
;             2 - Diskette is unformatted
;             3 - Diskette has a write-protect tab installed
;             4 - Diskette is of incorrect media type for this floppy drive
;             8 - Diskette has been damaged (Sector not found) or
;                 is not formatted
;            15 - Invalid drive specifier passed in <expN>
;
;             n - Any return code other than those listed here indicates
;                 that an error did occur, though the source of the error
;                 can not be exactly determined
;
;               
;  Notes:     The <expC> passed as a parameter must be one of the following
;             character values: "A", "a", "B" or "b".  Thus, calls of the 
;             form FloppyTst ("A") and FloppyTst ("B") are legal.  If <expC>
;             is not one of these values, the value 15 will be returned as 
;             the function value.  The function may return a misleading 
;             value if drive B: is tested on a machine with only an A: drive.
;             The result will usually correspond to the status of the A: drive.
;
;             The DOS Critical Interrupt handler may, under certain
;             circumstances, report a confusing or ambiguous error.
;             For example, attempting to read or write to a non-DOS
;             diskette or a 1.2Mb diskette formatted for 1.2Mb 
;             capacity may cause FloppyTst to return eight instead
;             of four.  If you understand this, you may provide
;             better error messages for users.  For those who have
;             replaced the MS-DOS Critical Error Handler with their
;             own, you must be sure that your replacement handler 
;             returns the same error codes, or that you modify the 
;             error routines in this assembler module to conform to
;             your return codes.
;
;
;  Algorithm: 
;             1. Verify that valid drive specifier has been passed.
;             2. Read sector zero into buffer using MS-DOS interrupt 25h,
;                Absolute Read.
;             3. Write buffer to sector zero using MS-DOS interrupt 26h,
;                Absolute Write.
;             4. The return value is based upon error codes returned from
;                the MS-DOS Critical Error handler.
;
;
;******************************************************************************

INCLUDE c:\prog\clipper\EXTENDA.MAC

PUBLIC FloppyTst        ; Declare as public.

DGROUP GROUP   DATASG   ; Combine your data segment
                        ; with Clipper's.
DATASG SEGMENT '_DATA'  ; Start of data segment.

drive  db 0             ; drive to be tested (zero=A:  one=B:)
buff   db 512 dup (?)   ; data buffer for one disk sector

DATASG   ENDS           ; End of data segment.
;
;
_PROG segment 'CODE'    ; Start of code segment.
ASSUME   cs:_PROG,ds:DGROUP
;
;
FloppyTst  PROC  FAR    ; Start of process.

   push  bp             ; Preserve return address.
   mov   bp,sp

   push  ds             ; Save registers.
   push  es
   push  si
   push  di

   GET_CHAR 1           ; Get the first character parameter
   push  es             ; save old data segment
   mov   es,dx          ; set DS register to segment of parameter
   mov   bx,ax          ; set BX to offset of parameter
   mov   dl,es:[bx]     ; put parameter in DL
   and   dl,01011111B   ; ensure uppercase ASCII
   sub   dl,('A')       ; convert ASCII to number ('A' = 0)
   pop   es             ; restore old data segment
   mov   drive,dl       ; put number in drive
   
; Is drive specifier valid?

   cmp   dl,0           ; drive = A:?
   je    read           ; yes, continue

   cmp   dl,1           ; no, drive = B:?
   je    read           ; yes, continue

   mov   ax,0Fh         ; no, return code = 15 (invalid drive specifier)
   jmp   exit          

; Read sector zero into buffer

read:
   mov   al,drive       ; drive to test
   mov   cx,1           ; number of sectors
   mov   dx,0           ; beginning sector number
   mov   bx, seg buff   ; address of buffer
   mov   ds,bx
   mov   bx, offset buff
   int   25h            ; request disk read
   jc    error          ; jump if error
   add   sp,2           ; clear stack
      
; Write buffer to sector zero

write:
   mov   al,drive       ; drive to test
   mov   cx,1           ; number of sectors
   mov   dx,0           ; beginning sector number
   mov   bx, seg buff   ; address of buffer
   mov   ds,bx
   mov   bx, offset buff
   int   26h            ; request disk write
   jc    error          ; jump if error
   add   sp,2           ; clear stack
   mov   ax,0           ; return value = disk ready
   jmp   exit
         

error:
   add   sp,2           ; clear stack

; check error code from Critical Error Handler returned in AL

error1:
   cmp   al,2           ; drive not ready?
   jne   error2
   mov   ax,1           ; return code = 1 (drive not ready)
   jmp   exit         

error2:
   cmp   al,0Ch         ; diskette unformatted?
   jne   error3
   mov   ax,2           ; return code = 2 (diskette unformatted)
   jmp   exit         

error3:
   cmp   al,0           ; write-protect error?
   jne   error4
   mov   ax,3           ; return code = 3 (write-protect error)
   jmp   exit
   
error4:
   cmp   al,7h          ; unknown media type?
   jne   error5
   mov   ax,4           ; return code = 3 (unknown media)
   jmp   exit
   
error5:
   mov   ah,0           ; error can not be determined  
                        ; return code = critical error code (whatever it may be)
               
exit:
   pop   di
   pop   si
   pop   es             ; Restore registers.
   pop   ds
   pop   bp

   RET_INT ax           ; Clipper return integer (actual cleaning).
   ret                  ; Actual physical return.
FloppyTst  ENDP         ; End of process.


_prog ENDS
   END
