/***
*   EH.PRG - Jon P. Rognerud - 72220, 1173
*
*   Simple error handling sample for Clipper 5.1, rev. 1.29, USASCII
*   
*   Compile with /n /w /a
*  
*   Rognerud-Wilson Software Research & Development (c) 1991.
*
*   (Portions Copyright Nantucket (c) 1991.)
*/

/***

Hi again!

In my travels thru the "Clipper Computing Community" here in Southern
California, I come across many clients that have several different areas
of concern.  One of them are how to use the new error handling system 
in Clipper.  Although most of them could figure out how to use it,
they are simply too busy developing apps, and they don't want to 
take the extra time to research "new" features...

The below is partly taken from Nantucket \CLIPPER5\SOURCE\SAMPLE disks,
and some of our own.

Keep in mind, this shows a "bare-bones" version of the error system, and 
you should feel free to modify it - maybe you could add a SLICKER user-
interface (colors, shadowing, etc), along with more detail error 
tracking.

To use this module, just link it with your favorite program, and see how
it works.  A quick test program like this might give you an idea:

* BEGIN TEST
  USE JON					// assuming no such file!
  SET DEVICE TO PRINTER		// make sure printer is off-line!
  @ PROW(), PCOL() SAY "Hi!"
* END TEST

Link this program with the eh.obj module, and take a look...

Happy computing from Jon!

*/

// include generic error codes
#include "error.ch"

#define FALSE .F.
#define TRUE  .T.
#define INTTRIM(num)   LTRIM(STR(num))


/***
*	ErrorSys() --> NIL
*
*	This function will be called automatically when an error occurs
*/
FUNCTION ErrorSys()
// the next line installs the error handling stuff...
ERRORBLOCK({|MyError| ErrorInstall(MyError)})
RETURN (NIL)


/***
*	ErrorInstall() --> NIL
*
*	This is the main error handler code
*/
STATIC FUNCTION ErrorInstall(errorObject)
LOCAL cMessage		// build error message
LOCAL aOptions		// array of error "PROMPT" options
LOCAL nChoice		// menu choice

// for network open error, set NETERR() and alert user
IF errorObject:genCode == EG_OPEN .AND. errorObject:osCode == 32
	NETERR(TRUE)
END

// for lock error during APPEND BLANK, set NETERR() and alert user
IF errorObject:genCode == EG_APPENDLOCK .AND. errorObject:canDefault
	NETERR(TRUE)
END

// build error message
cMessage := ""
ErrorMessage(errorObject, @cMessage)		// by reference!

// build options array
aOptions := {" Break ", " Quit "}

IF errorObject:canRetry
	AADD(aOptions, " Retry ")
END

IF errorObject:canDefault
	AADD(aOptions, " Default ")
END

// put up alert box
nChoice := 0
WHILE (nChoice == 0)

	IF Empty(errorObject:osCode)
		nChoice := ALERT(cMessage, aOptions)
	ELSE
		nChoice := ALERT( cMessage + "(DOS Error -> " + ;
			INTTRIM(errorObject:osCode) + ")", aOptions )
	END
END

IF !EMPTY(nChoice)
	IF aOptions[nChoice]     == " Break "
		BREAK(errorObject)
	ELSEIF aOptions[nChoice] == " Retry "
		RETURN (TRUE)
	ELSEIF aOptions[nChoice] == " Default "
		// default for division by zero is zero
		IF errorObject:genCode == EG_ZERODIV
			RETURN (0)
		END
		RETURN (FALSE)
	END
END

// display message and quit
IF !EMPTY(errorObject:osCode)
	cMessage += " (DOS Error " + INTTRIM(errorObject:osCode) + ") "
END

? cMessage
ERRORLEVEL(1)
QUIT

RETURN (FALSE)


/***
*	ErrorMessage()
*
*   Error message build function
*/
STATIC FUNCTION ErrorMessage(errorObject, cMessage)

// start error message
cMessage := IF (errorObject:severity > ES_WARNING, "Error ", "Warning ")

// add subsystem name if available
IF VALTYPE(errorObject:subsystem) == "C"
	cMessage += errorObject:subsystem()
ELSE
	cMessage += "???"
END

// add subsystem's error code if available
IF VALTYPE(errorObject:subCode) == "N"
	cMessage += ("/" + INTTRIM(errorObject:subCode))
ELSE
	cMessage += "/???"
END

// add error description if available
IF VALTYPE(errorObject:description) == "C"
	cMessage += ("  " + errorObject:description)
END

// add either filename or operation
IF !EMPTY(errorObject:filename)
	cMessage += (" " + errorObject:filename)
ELSEIF !Empty(errorObject:operation)
	cMessage += (" " + errorObject:operation)
END

RETURN (NIL)

