****************
****************
**
**  Purpose:  Full-screen, multi-page edit of any record in any data file
**
**   Syntax:  editrec( [ <fldptr>, [ <toprow> [, <leftcol> [, <botrow> 
**            [, <rightcol> [, <frame> [, <title> ] ] ] ] ] ] ] )
**
**   Params:
**
**      Name       T Description
**          
**      fldptr     N # of field to start at in gets
**      toprow     N Top row of edit window
**      leftcol    N Left column of edit window
**      botrow     N Bottom row of edit window
**      rightcol   N Right column of edit window
**      frame      C Frame for box around window
**      title      C Title for box
**
**    Notes:  uses the currently active data file to edit it.
**            EXTEND.LIB must be linked in to use AFIELDS()
**            [PgUp] is the only key to allow the user to get back to the
**            previous screen.  Includes support for memo fields.
**
**  Returns:  .t. or .f.
**
**    Calls:  _fldpic_()  _fldval_()
**
**    Files:  whatever data file is active
**
****************
****************
function editrec
parameter fldptr, toprow, leftcol, botrow, rightcol, frame, title
private pc, rows, numflds, numscrns, i, aliname, fn, topfld, botfld, j, cols,;
        k, editscr, savecurs
store '' to fn, editscr
pc = pcount()
if m->pc < 7 && Parameters will default
  title = "[ Editing " + alias() + " ]"
  if m->pc < 6
    frame = "Ŀ "
    if m->pc < 5
      rightcol  = 79
      if m->pc < 4
        botrow  = 23
        if m->pc < 3
          leftcol = 0
          if m->pc < 2
            toprow  = 1
            if m->pc < 1
              fldptr = 1
            endif
          endif
        endif
      endif
    endif
  endif
endif

****** Set dimensions for actually INSIDE the box
toprow   = m->toprow   + 1
leftcol  = m->leftcol  + 1
botrow   = m->botrow   - 1
rightcol = m->rightcol - 1

****** # of rows in window
rows = m->botrow - m->toprow + 1
****** (# of cols in window) - field display width
cols = m->rightcol - m->leftcol - 13
aliname = alias()
if m->rows < 1 .or. m->cols < 1 .or. empty( m->aliname ) && Come on now!
  return .f.
endif

do while ! rlock()
  @  0, 0 say "Unable to lock record"
  inkey(.5)
  @  0, 0
enddo

****** Determine the number of screens needed for editing the record
numflds = fcount()
numscrns = round( m->numflds / m->rows, 0 )
if m->numscrns * m->rows < m->numflds
  numscrns = m->numscrns + 1
endif

****** Declare array for fields
private _fname[ m->numflds ], _ftype[ m->numflds ], _fwid[ m->numflds ],;
        _fdec[ m->numflds ], _fval[ m->numflds ]

afields( _fname, _ftype, _fwid, _fdec )
****** Assign field equivalents
for i = 1 to m->numflds
  fn = _fname[ m->i ]
  if _ftype[ m->i ] = 'M'
    ****** Do the memo edit from the valid clause!
    _fval[ m->i ] = .f.
  else
    _fval[ m->i ] = &aliname->&fn
  endif
next i

****** Take care of the (+ 1) problem for 0-based screens
toprow  = m->toprow - 1

editscr = savescreen( m->toprow, m->leftcol - 1, m->botrow + 1, ;
                      m->rightcol + 1 )

i = 1
if m->fldptr > 1
  ****** Find the correct screen
  i = int( m->fldptr / m->rows ) + 1
  ****** Move to the correct field - keyboard the down arrow
  keyboard replicate( chr( 24 ), max( 0, m->fldptr % m->rows - 1 ) )
endif

****** savecurs = status( 6 )  && status of cursor - uncomment if you have status()
set cursor on
do while m->i > 0 .and. m->i <= m->numscrns

  ****** Calculate the field offset on this edit window
  topfld = m->rows * ( m->i - 1 )

  ****** Make sure we don't try to display more fields than there are
  if m->topfld + m->rows > m->numflds
    botfld = m->numflds - m->topfld
  else
    botfld = m->rows
  endif

  @ m->toprow, m->leftcol - 1, m->botrow + 1, m->rightcol + 1 box m->frame
  @ m->toprow, m->leftcol say m->title
  @ m->botrow + 1, m->leftcol say "[ Record: " + ltrim( str( recno() ) ) + " ]"

  ****** Setting up the get clauses
  for j = 1 to m->botfld

    k = m->topfld + m->j
    @ m->toprow + m->j, m->leftcol say _fname[ m->k ]
    @ m->toprow + m->j, m->leftcol + 11 get _fval[ m->k ] ;
        picture _fldpic_( _ftype[ m->k ], _fwid[ m->k ], _fdec[ m->k ], m->cols ) ;
        valid _fldval_( m->toprow + 1, m->rows, m->i )

    if _ftype[ m->k ] = 'M' .and. m->cols > 15
      @ m->toprow + m->j, m->leftcol + 13 say "<Y edits>"
    endif

  next j
  read

  ****** Determining which screen to go to from here
  if lastkey() = 27 && [Esc]
    i = 0
  elseif lastkey() = 18 && [PgUp]
    if i > 1
      i = m->i - 1
    endif
  elseif lastkey() = 23 && [^W] or [^End]
    exit
  else
    i = m->i + 1
  endif
enddo

****** set cursor ( m->savecurs )
restscreen( m->toprow, m->leftcol - 1, m->botrow + 1, m->rightcol + 1, ;
            m->editscr )

if m->i = 0 && Escape was pressed
  return .f.
endif

****** Replace values in fields
for i = 1 to m->numflds
  fn = _fname[ m->i ]
  if _ftype[ m->i ] # 'M'
    replace &aliname->&fn with _fval[ m->i ]
  endif
next i

****** Successful edit!
return .t.

*****************
*****************
**
**  Purpose:  Return the field picture used in editrec()
**
**   Syntax:  _fldpic_( <_ftype>, <_fwid>, <_fdec>, <wid> )
**
**  Variable  T Len Description
**      
**  _ftype    C     Type of field
**  _fwid     N     Width of field
**  _fdec     N     Decimal for numeric fields
**  wid       N     Display width of window
**
**  Notes: <wid> only is useful when editing character variables - all others
**         will overlap the window if they don't fit.
**
*****************
*****************
function _fldpic_
parameter _ftype, _fwid, _fdec, wid
if pcount() < 4 && returns a generic field picture
  return '@X'
endif

if m->_ftype = 'N'
  if _fdec > 0
    return replicate( '9', m->_fwid - m->_fdec - 1 ) + '.' + ;
           replicate( '9', m->_fdec )
  endif
  return replicate( '9', m->_fwid )
elseif m->_ftype = 'D'
  return '@D'
elseif m->_ftype $ 'LM'
  return 'Y'
endif
if m->wid < m->_fwid
  return '@S' + ltrim( str( m->wid ) )
endif
return '@X'


*******************
*******************
**
**  Purpose:  provide field validation capability (mainly for memos)
**
**   Syntax:  _fldval_( <toprow>, <rows>, <scrnum> )
**
**   Params:
**      Name       T Len Description
**           
**      toprow     N     Top row of window
**      rows       N     # of rows per edit window
**      scrnum     N     # of current screen
**
**  Notes:  This function is NO GOOD OUTSIDE of editrec().  It assumes that
**          memory variables declared private in editrec() are available to
**          it (because I didn't want a huge parameter list) and because it's
**          set up specifically for that edit window.  Because valid clauses
**          are evaluated on AFTER the READ is invoked, there is no telling
**          what the actual value of 'j' is at that time.  This will figure
**          out what field you're on, and do the validation necessary for that
**          field.  Capiche?
**
**          EXTEND.LIB must be linked because this function used savescreen()
**          and restscreen().
**
**          Be warned that this function is a potential memory grabber because
**          of memoedit() and my making a copy of the memo field before editing
**          it.
**
*******************
*******************
function _fldval_
parameter toprow, rows, scrnum
private fldptr, fn, valscr, tmpmem

****** Nulling out private character variables for maximum memory recovery
store '' to fn, valscr, tmpmem

****** Calculating the field pointer based on the current screen row, top
****** row, and screen number
fldptr = ( row() - m->toprow ) + ( m->rows * ( m->scrnum - 1 ) ) + 1
if _ftype[ m->fldptr ] # 'M'  && Any type OTHER than a memo field
  return .t.
endif

if ! _fval[ m->fldptr ] && .F. was in field value - no edit!
  return .t.
endif
fn = _fname[ m->fldptr ]

****** Copy the memo
tmpmem = &aliname->&fn

****** Save the screen before blanking it out
valscr = savescreen( m->toprow, m->leftcol, m->botrow, m->rightcol )
@ m->toprow, m->leftcol, m->botrow, m->rightcol box "ͻȺ "
@ m->toprow, m->leftcol + 1 say "[Ctrl-W=save,Esc=abort]"
****** edit the memo
tmpmem = memoedit( m->tmpmem, m->toprow + 1, m->leftcol + 1, m->botrow - 1,;
                   m->rightcol - 1, .t., "", 75 )
if lastkey() # 27
  replace &aliname->&fn with m->tmpmem
endif
restscreen( m->toprow, m->leftcol, m->botrow, m->rightcol, m->valscr )

****** set toggle back to .F., so edit isn't ALWAYS called
_fval[ m->fldptr ] = .f.

return .t.
