*****************************************************************************
*                                                                           *
* CONVERSION CODE FOR EBCDIC TO ASCII and vice versa                        *
*                                                                           *
* done '91 by D.BOS                                                         *
*                                                                           *
*****************************************************************************



*---------------------------------------------------------------------------*
* Conversion defines for a single EBCDIC or ASCII character                 *
*---------------------------------------------------------------------------*
#define asc2ebc(ab) if(asc(ab)!=0,chr(ebc[asc(ab)]),chr(0))
#define ebc2asc(eb) if(asc(eb)!=0,chr(ascan(ebc,asc(eb))),chr(0))

*---------------------------------------------------------------------------*
* general defines                                                           *
*---------------------------------------------------------------------------*
#define NUMERIC 0
#define ZONED   1

#define EBCTYPE farr[i,5]
#define FLENGTH farr[i,3]

*---------------------------------------------------------------------------*
* EBCDIC collating sequence                                                 *
*---------------------------------------------------------------------------*
private ebc:={;
  /* 0 */   0,  1,  2,  3, 55, 45, 46, 47, 22,  5, 21, 11, 12, 13, 14, 15,;
  /* 1 */  16, 17, 18, 19, 60, 61, 50, 38, 24, 25, 63, 39, 28, 29, 30, 31,;
  /* 2 */  64, 79,127,123, 91,108, 80,125, 77, 93, 92, 78,107, 96, 75, 97,;
  /* 3 */ 240,241,242,243,244,245,246,247,248,249,122, 94, 76,126,110,111,;
  /* 4 */ 124,193,194,195,196,197,198,199,200,201,209,210,211,212,213,214,;
  /* 5 */ 215,216,217,226,227,228,229,230,231,232,233, 74,224, 90, 95,109,;
  /* 6 */ 129,130,131,132,133,134,135,136,137,138,145,146,147,148,149,150,;
  /* 7 */ 151,152,153,162,163,164,165,166,167,168,169,192,106,208,161,  7}

*---------------------------------------------------------------------------*
* table of decimal positions                                                *
*---------------------------------------------------------------------------*
private dtab:={1,10,100,1000,10000,100000,1000000,10000000,100000000,;
               1000000000,10000000000,100000000000,1000000000000,;
               10000000000000,100000000000000,1000000000000000}


*---------------------------------------------------------------------------*
* DBF2EBC   - DBF file to raw EBCDIC file                                   *
*           - infile   = name of DBF file                                   *
*           - outfile  = name of resulting EBC file                         *
*           - farr     = array with 'extended' record info                  *
*                        1. -> Fieldname c                                  *
*                        2. -> Fieldtype c                                  *
*                        3. -> Fieldlen  n                                  *
*                        4. -> Fielddec  n                                  *
*                        5. -> EBCtype   c (B,P,Z,C)                        *
*                              (B=binary,P=packed,Z=zoned,C=char)           *
*---------------------------------------------------------------------------*
function dbf2ebc(infile,outfile,farr)
local file,i,recsize:=0,fieldbuff:="",osel,recoff:={},fstring:=""
private fmacro:=""
  osel=select()
  if !file(infile)
    @ maxrow(),0 say " <"+alltrim(infile)+"> not available!"
    inkey(0)
    return.f.
  end
  file=fcreate(outfile,0)
  if ferror()!=0
    @ maxrow(),0 say " creation of <"+alltrim(infile)+"> not possible!"
    inkey(0)
    return.f.
  end
  fmacro="infile->(used())"
  if !&fmacro
    use infile ALIAS DB-STRUC NEW
  else
    select infile
  endif
  go top
  while !eof()
    for i=1 to len(farr)
      fmacro="field(i)"
      do case
        case EBCTYPE="Z"
          fbuf=num2zone(&fmacro)
          // stretch to consistent length
          fbuf=replicate(chr(240),FLENGTH-len(fbuf))+fbuf
          fwrite(file,@fbuf,FLENGTH)
        case EBCTYPE="P"
          fbuf=nz2pack(&fmacro)
          fbuf=replicate(chr(240),FLENGTH-len(fbuf))+fbuf
          fwrite(file,@fbuf,FLENGTH)
        case EBCTYPE="B"
          fbuf=l2bin(&fmacro)
          fwrite(file,@fbuf,FLENGTH)
        case EBCTYPE="C"
          fbuf=ascs2ebcs(&fmacro,FLENGTH)
          fwrite(file,@fbuf,FLENGTH)
      end
    next
    skip
  end
  select(osel)
return.t.

*---------------------------------------------------------------------------*
* EBC2DBF   - raw EBCDIC file to DBF file                                   *
*           - infile   = name of EBCDIC file                                *
*           - outfile  = name of resulting DBF file                         *
*           - farr     = array with 'extended' record info                  *
*                        1. -> Fieldname c                                  *
*                        2. -> Fieldtype c                                  *
*                        3. -> Fieldlen  n                                  *
*                        4. -> Fielddec  n                                  *
*                        5. -> EBCtype   c (B,P,Z,C)                        *
*                              (B=binary,P=packed,Z=zoned,C=char)           *
*---------------------------------------------------------------------------*
function ebc2dbf(infile,outfile,farr)
local file,i,recsize:=0,recbuff:="",osel,recoff:={},fstring:=""
private fmacro:=""
  osel=select()
  if !file(infile)
    @ maxrow(),0 say " <"+alltrim(infile)+"> not available!"
    inkey(0)
    return.f.
  end
  file=fopen(infile,0)
  dbcreate(outfile,farr)
  use outfile alias DB-STRUC NEW
  if farr=NIL
    @ maxrow(),0 say " need field description!"
    inkey(0)
    return.f.
  endif
  for i=1 to len(farr)
    recsize+=FLENGTH
    aadd(recoff,(recsize-FLENGTH)+1)
  next
  recbuff=space(recsize)
  while fread(file,@recbuff,recsize)=recsize
    append blank
    for i=1 to len(farr)
      fmacro="DB-STRUC->farr[i,1]"
      fstring=substr(recbuff,recoff[i],FLENGTH)
      do case
        case EBCTYPE="Z"
          replace &fmacro with zone2num(fstring,FLENGTH)
        case EBCTYPE="P"
          replace &fmacro with pack2nz(fstring,FLENGTH,NUMERIC)
        case EBCTYPE="B"
          replace &fmacro with bin2l(fstring)
        case EBCTYPE="C"
          replace &fmacro with ebcs2ascs(fstring,FLENGTH)
      end
    next
  end
  use
  select(osel)
return.t.

*---------------------------------------------------------------------------*
* PACK2NZ   - packed EBCDIC string to numeric or zoned                      *
*           - pstr     = packed EBCDIC string to convert                    *
*           - len      = length of string to convert (numeric)              *
*           - op       = operation -> NUMERIC or ZONED                      *
*                        returns clipper usable numeric                     *
*---------------------------------------------------------------------------*
function pack2nz(pstr,len,op)
local i:=0,hi:=0,lo:=0,res:=0
  for i=1 to len
    hi=(asc(substr(pstr,i,1))-asc(substr(pstr,i,1))%16)/16
    lo=asc(substr(pstr,i,1))%16
    res+=hi*dtab[(len*2)-((i*2)-1)]
    if i!=len;res+=lo*dtab[(len*2)-(i*2)];end
  next
  res*=if(lo=15,1,-1)
  if op=ZONED
    res=num2zone(res)
  endif
return res

*---------------------------------------------------------------------------*
* NZ2PACK   - numeric or zoned to packed EBCDIC string                      *
*           - num      = clipper numeric or zoned ebcstr                    *
*                        returns packed EBCDIC string                       *
*---------------------------------------------------------------------------*
function nz2pack(num)
local i:=0,zone:="",tmp:="",tmp2:="",res:="",sign:=0,len:=0
  if valtype(num)="N"
    zone=num2zone(num)
  else
    zone=num
  end
  // strip zones
  for i=1 to len(zone)
    tmp+=chr(asc(substr(zone,i,1))%16)
  next
  sign=if(asc(substr(zone,len(zone),1))-asc(substr(zone,len(zone),1))%16=240,15,13)
  lastbyte=chr(  ( (asc(substr(zone,len(zone),1))%16)*16)  +sign)
  if len(zone)%2=0;len=len(zone)+1;tmp=chr(0)+tmp;else;len=len(zone);end
  for i=1 to len step 2
    if i=len
      res+=lastbyte
      i++
    else
      b1=asc(substr(tmp,i))*16
      b2=asc(substr(tmp,i+1))
      res+=chr(b1+b2)
    end
  next
return res

*---------------------------------------------------------------------------*
* NUM2ZONE  - numeric to zoned EBCDIC string                                *
*           - num      = clipper numeric                                    *
*                        returns zoned EBCDIC string                        *
*---------------------------------------------------------------------------*
function num2zone(num)
local i:=0,tmp:=0,res:="",zone:=240,negative:=208,tmpnum:=num
  // sign conversion at last, so make sure to use positive number
  if num<0;tmpnum*=-1;end
  for i=16 to 1 step -1
    if tmpnum/dtab[i]>1
      tmp=int(tmpnum/dtab[i])
      tmpnum-=tmp*dtab[i]
      if num>0
        res+=chr(zone+tmp)
      else
        if i=1
          res+=chr(negative+tmp)
        else
          res+=chr(zone+tmp)
        end
      end
    end
  next
return res

*---------------------------------------------------------------------------*
* ZONE2NUM  - zoned EBCDIC string to numeric                                *
*           - estr     = zoned EBCDIC string to convert                     *
*           - len      = length of string to convert (numeric)              *
*                        returns clipper usable numeric                     *
*---------------------------------------------------------------------------*
function zone2num(estr,len)
local i:=0,tmp:=0,res:=0
  for i=1 to len
    tmp=asc(substr(estr,i,1))%16
    tmp*=dtab[(len+1)-i]
    res+=tmp
  next
  // sign calculation
  tmp=asc(substr(estr,len,1))-asc(substr(estr,len,1))%16
  res*=if(tmp=240,1,-1)
return res

*---------------------------------------------------------------------------*
* EBCS2ASCS - EBDCIC string to ASCII string                                 *
*           - string   = EBCDIC string to convert                           *
*           - len      = length of string to convert (numeric)              *
*                        returns converted string                           *
*---------------------------------------------------------------------------*
function ebcs2ascs(string,len)
local i,res:=""
   for i=1 to len
      res+=ebc2asc(substr(string,i,1))
   next
return res

*---------------------------------------------------------------------------*
* ASCS2EBCS - ASCII string to EBCDIC string                                 *
*           - string   = ASCII string to convert                            *
*           - len      = length of string to convert (numeric)              *
*                        returns converted string                           *
*---------------------------------------------------------------------------*
function ascs2ebcs(string,len)
local i,res:=""
   for i=1 to len
      res+=asc2ebc(substr(string,i,1))
   next
return res
