*-- Header ---------------------------------------------------------------
*    Program . DMSJUL.PRG
*    Date .... June 16, 1989
*    Author .. Bob Laszko, Data Management Systems
*    Desc .... Contains CAL2JUL() / JUL2CAL()
*    Notes ... Julian / Calendar conversion routines
*              Written for Clipper S'87
*    Notice .. Copyright 1989, Data Management Systems. All Rights Reserved
*-------------------------------------------------------------------------
*
*-- Ops Notes ------------------------------------------------------------
*
*    The use of SET CENTURY ON is required if you intent to convert dates
*    before / after the twentieth century.
*
*    Since Clipper S'87 only supports dates from 01/01/0100 to 12/31/2999,
*    the practical limits on these functions as written are:
*
*         Calendar dates:     01/01/0100 - 12/31/2999
*         Julian dates:           36,160 -    1095384
*
*-------------------------------------------------------------------------
*
*-- Updates --------------------------------------------------------------
*
*-------------------------------------------------------------------------





*-- Header ---------------------------------------------------------------
*    Function  CAL2JUL()
*    Date .... June 16, 1989
*    Author .. Bob Laszko, Data Management Systems
*    Desc .... Convert a calendar date to a julian numeric date
*    Notes ... Original concept from CAL2JUL.PRG by Luis A Castro
*              Written for Clipper S'87
*    Notice .. Copyright 1989, Data Management Systems. All Rights Reserved
*-------------------------------------------------------------------------
*
*-- Ops Notes ------------------------------------------------------------
*
*    Syntax
*
*         expN = CAL2JUL(cal_date)
*
*    Parameters
*
*         cal_date  D         any valid date from 01/01/0100 - 12/31/2999
*
*    Private
*
*         month     N,2,0     month extracted from cal_date
*         day       N,2,0     day extracted from cal_date
*         year      N,4,0     year extracted from cal_date
*
*    Returns
*
*         jul_date  N,7,0     julian date from cal_date
*
*-------------------------------------------------------------------------
*
*-- Updates --------------------------------------------------------------
*
*-------------------------------------------------------------------------

FUNCTION CAL2JUL

PARAMETERS cal_date

PRIVATE month, day, year, jul_date

* Extract day, month, year from cal_date
day = DAY(cal_date)
month = MONTH(cal_date)
year = YEAR(cal_date)

jul_date = INT(30.57 * month) + INT(365.25 * year - 395.25) + day

* Adjust jul_date if leap year
IF month > 2
     jul_date = jul_date - 1
     IF INT(year / 4) * 4 <> year
          jul_date = jul_date - 1
     ENDIF
ENDIF

RETURN(jul_date)

*------------------------------------------------------------------------------
*    EOF  CAL2JUL()
*------------------------------------------------------------------------------




*-- Header ---------------------------------------------------------------
*    Function  JUL2CAL()
*    Date .... June 16, 1989
*    Author .. Bob Laszko, Data Management Systems
*    Desc .... Convert a Julian date to a calendar date
*    Notes ... Original concept from JUL2CAL.PRG by Luis A Castro
*              Written for Clipper S'87
*    Notice .. Copyright 1989, Data Management Systems. All Rights Reserved
*-------------------------------------------------------------------------
*
*-- Ops Notes ------------------------------------------------------------
*
*    Syntax
*
*         expD = JUL2CAL(jul_date)
*
*    Parameters
*
*         jul_date  N,7,0     any valid julian date from 36,160 - 1,095,384
*
*    Private
*
*         month     N,2,0     month extracted from jul_date
*         day       N,2,0     day extracted from jul_date
*         year      N,4,0     year extracted from jul_date
*         leap_day  N,2,0     used to calculate leap year offsets
*
*    Returns
*
*         cal_date  D         calendar date from jul_date
*
*-------------------------------------------------------------------------
*
*-- Updates --------------------------------------------------------------
*
*-------------------------------------------------------------------------

FUNCTION JUL2CAL

PARAMETERS jul_date

PRIVATE month, day, year, leap_day, cal_date

year = INT(jul_date / 365.26) + 1
day = jul_date + INT(395.25 - 365.25 * year)
leap_day = 1

* Calculate extra day if leap year
IF INT(year / 4) * 4 <> year
     leap_day = 2
ENDIF

* Calculate actual number of days
IF day > (91 - leap_day)
     day = day + leap_day
ENDIF

* Calculate month, day, & year
month = INT(day / 30.57)
day = day - INT(30.57 * month)
IF month > 12
     month = 1
     year = year + 1
ENDIF

cal_date = CTOD(STR(month,2) + '/' + STR(day,2) + '/' + STR(year,4))

RETURN(cal_date)

*------------------------------------------------------------------------------
*    EOF  JUL2CAL()
*------------------------------------------------------------------------------
