*--- Header ---------------------------------------------------------------
*    Procedure DMSOOPS.PRG
*    Version . 1.10
*    Date .... August 18, 1988
*    Author .. Bob Laszko, Data Management Systems
*    Desc .... Displays an error message box in custom or default
*               configuration
*    Notice .. Copyright 1988, Data Management Systems. All Rights Reserved
*--------------------------------------------------------------------------
*
*--- Ops Notes ------------------------------------------------------------
*
*    Requires the following external routines:
*         EXXTEND.OBJ    C routines that return the status of some SET
*                         commands. Written by J. Scott Emerich.
*    Syntax - complete
*         DO DMSOOPS [WITH [title], [frame], [instruction], [location],;
*                    [rest_scrn], [explode, [implode]], [shad_show],;
*                    [shad_char], [shad_side]]
*
*    Syntax - default
*         DO DMSOOPS
*
*    Parameters
*         title          C    title to display on top line of box
*                              default = "OOPS"
*         frame          N    0 = no characters in border
*                             1 = single line box
*                             2 = double line box
*                             3 = double line top/bottom, single line sides
*                             4 = single line top/bottom, double line sides
*                        C    custom frame, include all eight characters as
*                              outlined for @...BOX command
*                              default = 1
*         instruction    C    instructions to display on bottom line of box
*                              default = "Press Any Key to Continue"
*         location       C    UR = upper right corner of screen
*                             UL = upper left corner of screen
*                             LL = lower left corner of screen
*                             LR = lower right corner of screen
*                             C  = center of screen
*                              default = C
*         rest_scrn      L    .T. = restore screen upon RETURN to calling .PRG
*                             .F. = screen not restored
*                              default = .T.
*         explode        L    .T. = exploding box
*                             .F. = no explosion
*                              default = .T.
*         implode        L    .T. = implode screen before restore
*                             .F. = no implosion
*                              default = .T.
*                              must explode box to implode on restore
*         shad_show      L    .T. = shadow
*                             .F. = no shadow
*                              default = .T.
*         shad_char      C    character to use for shadow
*                              default = CHR(177) ""
*         shad_side      C    L = shadow on left side of box
*                             R = shadow on right side of box
*                              default = R
*
*    Public memvars
*         OOPS_MSG[]     C    each line of message to display in box
*         OOPS_RESP[]    C    valid responses to instruction line prompts
*         OOPS_ACTION    C    validated response returned to calling prg
*         M_COOPSFR    * C    color for box frame
*         M_COOPSTIT   * C    color for title
*         M_COOPSTXT   * C    color for messages (text)
*         M_COOPSINS   * C    color for instruction
*         M_COOPSSHD   * C    color for shadow
*         OOPS_SCRN    * C    screen saved prior to calling OOPS.
*                      *      if these memvars are not initialized by the
*                              calling prg, they will become PRIVATE
*
*    Private memvars
*         OOPS_TITLE     C    title parameter
*         OOPS_FRAME     C    frame for box derived from frame parameter
*         OOPS_INS       C    instruction parameter
*         OOPS_SCRN      C    screen saved prior to calling OOPS. PRIVATE if
*                              not initialized by calling .PRG
*         OOPS_LEN       N    length of box
*         OOPS_TOP       N    top row of box
*         OOPS_LEFT      N    left column of box
*         OOPS_BOTT      N    bottom row of box
*         OOPS_RIGHT     N    right column of box
*         EXP_TOP        N    top row of exploding box
*         EXP_LEFT       N    left column of exploding box
*         EXP_BOTT       N    bottom row of exploding box
*         EXP_RIGHT      N    right column of exploding box
*         LIMIT_TOP      N    lowest value allowed for OOPS_TOP
*         LIMIT_LEFT     N    lowest value allowed for OOPS_LEFT
*         LIMIT_BOTT     N    highest value allowed for OOPS_BOTT
*         LIMIT_RIGHT    N    highest value allowed for OOPS_RIGHT
*         MSG[]          C    OOPS_MSG[] used in this routine
*         MSG_NO         N    # of messages (LEN(OOPS_MSG))
*         MSG_LEN        N    length of messages
*         EXP_SCRN[]     C    screens of each step of exploding box
*         EXP_NO         N    # of steps in exploding box
*         T_EXP_NO       N    temp used to find EXP_NO
*         COL_POS        N    column position for @...SAY
*         ROW_POS        N    row position for @...SAY
*         ADJUST         N    adjustment factor for COL_POS
*         CURR_COLOR     C    current SETCOLOR() before calling DMSOOPS
*         CURR_ROW       N    current cursor row before calling DMSOOPS
*         CURR_COL       N    current cursor column before calling DMSOOPS
*         CURR_CURSOR    L    current cursor on/off state before calling DMSOOPS
*         VALID_RESP     N    flag to validate OOPS_RESP[]
*         ACTION         N    INKEY(0) for OOPS_RESP[] validation
*         X              N    FOR...NEXT memvar
*
*    Setup example - custom
*
*         DECLARE OOPS_MSG[2]
*         OOPS_MSG[1] = "Printer is not ready. Make sure"
*         OOPS_MSG[2] = "it is on-line and has paper"
*         DECLARE OOPS_RESP[2]
*         OOPS_RESP[1] = "R"
*         OOPS_RESP[2] = "A"
*         OOPS_ACTION = SPACE(1)
*         DO OOPS WITH "Printer Not Ready", 1, "R = Retry   A = Abort", "UL", .T., .T., .T., .T., "", "R"
*         RELEASE OOPS_MSG, OOPS_RESP
*
*              [ Printer Not Ready ]Ŀ
*                                               
*               Printer is not ready. Make sure 
*               it is on-line and has paper     
*                                               
*              [ R = Retry   A = Abort ]ٱ
*               
*
*    Setup example - default
*         DECLARE OOPS_MSG[2]
*         OOPS_MSG[1] = "This customer has sales"
*         OOPS_MSG[2] = "Cannot delete at this time"
*         DO OOPS
*         RELEASE OOPS_MSG
*
*              [ OOPS ]Ŀ
*                                           
*               This customer has sales     
*               Cannot delete at this time  
*                                           
*              [ Press Any Key to Continue ]
*
*    Misc.
*         All message lengths need not be the same. Widest message
*         is found and spaces added to shorter messages.
*
*--------------------------------------------------------------------------
*
*--- Updates --------------------------------------------------------------
*    09/15/88  Added check for mono systems when setting default colors
*    v.1.10    Added implode parameter & code
*              Added ASCAN() function to validate OOPS_RESP[]
*              Added check for cursor on/off state, restores original state
*               on exit (function GETCURS() from EXXTEND.OBJ)
*              Added save of cursor position, restore on exit
*              Corrected bug in explosion code. Exploding box was sometimes
*               larger than final display box.
*--------------------------------------------------------------------------

** PROCEDURE DMSOOPS          && remove ** to make a procedure

IF PCOUNT() <> 0              && check if parameters passed in command line
     PARAMETERS TITLE, FRAME, INSTRUCTION, LOCATION, REST_SCRN, EXPLODE, IMPLODE, SHAD_SHOW, SHAD_CHAR, SHAD_SIDE
ENDIF

PRIVATE OOPS_TITLE, OOPS_FRAME, OOPS_INS, OOPS_LEN
PRIVATE OOPS_TOP, OOPS_LEFT, OOPS_BOTT, OOPS_RIGHT
PRIVATE MSG_NO, MSG_LEN, ROW_POS, COL_POS, ADJUST
PRIVATE EXP_TOP, EXP_LEFT, EXP_BOTT, EXP_RIGHT
PRIVATE EXP_NO, T_EXP_NO, EXP_SCRN
PRIVATE CURR_COLOR, CURR_ROW, CURR_COL, CURR_CURSOR
PRIVATE X, VALID_RESP, ACTION
PRIVATE LIMIT_TOP, LIMIT_LEFT, LIMIT_BOTT, LIMIT_RIGHT

* Parameter validation & default assignments
IF TYPE("TITLE") = "U"
    TITLE = "OOPS"
ENDIF
IF TYPE("FRAME") = "U"
     FRAME = 1
ENDIF
IF TYPE("INSTRUCTION") = "U"
     INSTRUCTION = "Press Any Key to Continue"
     VALID_RESP = .F.
ENDIF
IF TYPE("LOCATION") = "U"
     LOCATION = "C"
ENDIF
IF TYPE("REST_SCRN") = "U"
     REST_SCRN = .T.
ENDIF
IF TYPE("EXPLODE") = "U"
     EXPLODE = .T.
ENDIF
IF TYPE("IMPLODE") = "U"
     IMPLODE = .T.
ENDIF

IF .NOT. EXPLODE
     IMPLODE = .F.                 && cannot implode if not exploding
ENDIF

IF TYPE("SHAD_SHOW") = "U"
     SHAD_SHOW = .T.
ENDIF
IF TYPE("SHAD_CHAR") = "U"
     SHAD_CHAR = CHR(177)
ENDIF
IF TYPE("SHAD_SIDE") = "U"
     SHAD_SIDE = "R"
ENDIF

* Check other memvars assigned by calling .prg, assign defaults
IF TYPE("OOPS_RESP") = "U"         && no validation required
     VALID_RESP = .F.
ELSE
     IF TYPE("OOPS_RESP") = "A"    && make sure it's an array
          VALID_RESP = .T.
     ELSE
          VALID_RESP = .F.
     ENDIF
ENDIF

IF TYPE("M_COOPSFR") = "U"         && box frame color
     M_COOPSFR = IF(ISCOLOR(), "W+/R", "W+/ ")      && hi white on red or high white on black
ENDIF
IF TYPE("M_COOPSTIT") = "U"        && box title color
     M_COOPSTIT = IF(ISCOLOR(), "BG+/R", " /W")     && hi cyan on red or black on white
ENDIF
IF TYPE("M_COOPSTXT") = "U"        && box text (messages) color
     M_COOPSTXT = IF(ISCOLOR(), "W+/R", "W/ ")      && hi white on red or white or white on black
ENDIF
IF TYPE ("M_COOPSINS") = "U"       && box instructions color
     M_COOPSINS = IF(ISCOLOR(), "GR+/R", " /W")     && hi yellow on red or black on white
ENDIF
IF TYPE ("M_COOPSSHD") = "U"       && shadow color
     M_COOPSSHD = IF(ISCOLOR(), "R/ ", "W/ ")       && red on black or white on black
ENDIF


* Setup for display
CURR_COLOR = SETCOLOR()       && save color setting from calling .prg
CURR_ROW = ROW()              && save current row position from calling .prg
CURR_COL = COL()              && save current column position from calling .prg
CURR_CURSOR = GETCURS()       && save cursor on/off state - routine from GETSTAT.OBJ
SAVE SCREEN TO OOPS_SCRN      && save screen from calling .prg
SET CURSOR OFF

* Initialize private memvars
LIMIT_TOP = 4
LIMIT_LEFT = 1
LIMIT_BOTT = 22
LIMIT_RIGHT = 79
ACTION = 0
X = 0

* Assign private memvars from parameters passed
OOPS_TITLE = SPACE(1) + TITLE + SPACE(1)
OOPS_INS = SPACE(1) + INSTRUCTION + SPACE(1)

MSG_NO = LEN(OOPS_MSG)
DECLARE MSG[MSG_NO]
FOR X = 1 TO MSG_NO
     MSG[X] = OOPS_MSG[X]
NEXT

IF TYPE("FRAME") = "N"     && passed a numeric choice for frame
     DO CASE
          CASE FRAME = 0
               OOPS_FRAME = "        "
          CASE FRAME = 2
               OOPS_FRAME = "ͻȺ"
          CASE FRAME = 3
               OOPS_FRAME = "͸Գ"
          CASE FRAME = 4
               OOPS_FRAME = "ķӺ"
          OTHERWISE
               OOPS_FRAME = "Ŀ"       && FRAME = 1 or not 0,2,3,4
     ENDCASE
ELSE
     OOPS_FRAME = FRAME                      && char string was passed
ENDIF


* Find MSG_LEN
MSG_LEN = LEN(MSG[1])
FOR X = 1 TO MSG_NO           && make sure all messages are same len
     IF LEN(MSG[X]) > MSG_LEN
          MSG_LEN = LEN(MSG[X])
     ENDIF
NEXT

* Make all MSG[] the same length, add spaces to end of each to match
FOR X = 1 TO MSG_NO
     MSG[X] = MSG[X] + SPACE(MSG_LEN - LEN(MSG[X]))
NEXT

* Make sure MSG_LEN >= length of OOPS_TITLE & OOPS_INS
IF MSG_LEN < LEN(OOPS_TITLE)
     MSG_LEN = LEN(OOPS_TITLE)
ENDIF
IF MSG_LEN < LEN(OOPS_INS)
     MSG_LEN = LEN(OOPS_INS)
ENDIF

* Pad both ends of all MSG[] with spaces if MSG_LEN has changed
DO WHILE .T.
     IF LEN(MSG[1]) < MSG_LEN
          FOR X = 1 TO MSG_NO
               MSG[X] = SPACE(1) + MSG[X] + SPACE(1)
          NEXT
     ELSE
          EXIT
     ENDIF
ENDDO
MSG_LEN = LEN(MSG[1])
OOPS_LEN = MSG_LEN + 4   && " " + " "

* Find screen coordinates for oops box
DO CASE
     CASE LOCATION = "UL"
          OOPS_TOP = LIMIT_TOP
          OOPS_LEFT = LIMIT_LEFT
          IF SHAD_SHOW .AND. SHAD_SIDE = "L"
               OOPS_LEFT = OOPS_LEFT + 2
          ENDIF
          OOPS_BOTT = OOPS_TOP + 1 + MSG_NO + 2
          OOPS_RIGHT = OOPS_LEFT + OOPS_LEN - 1

     CASE LOCATION = "UR"
          OOPS_TOP = LIMIT_TOP
          OOPS_LEFT = LIMIT_RIGHT - OOPS_LEN
          IF SHAD_SHOW .AND. SHAD_SIDE = "R"
               OOPS_LEFT = OOPS_LEFT - 2
          ENDIF
          OOPS_BOTT = OOPS_TOP + 1 + MSG_NO + 2
          OOPS_RIGHT = OOPS_LEFT + OOPS_LEN - 1

     CASE LOCATION = "LL"
          OOPS_TOP = LIMIT_BOTT - MSG_NO - 4
          IF SHAD_SHOW
               OOPS_TOP = OOPS_TOP - 1
         ENDIF
          OOPS_LEFT = LIMIT_LEFT
          IF SHAD_SHOW .AND. SHAD_SIDE = "L"
               OOPS_LEFT = OOPS_LEFT + 2
          ENDIF
          OOPS_BOTT = OOPS_TOP + 1 + MSG_NO + 2
          OOPS_RIGHT = OOPS_LEFT + OOPS_LEN - 1

     CASE LOCATION = "LR"
          OOPS_TOP = LIMIT_BOTT - MSG_NO - 4
          IF SHAD_SHOW
               OOPS_TOP = OOPS_TOP - 1
          ENDIF
          OOPS_LEFT = LIMIT_RIGHT - OOPS_LEN
          IF SHAD_SHOW .AND. SHAD_SIDE = "R"
               OOPS_LEFT = OOPS_LEFT - 2
          ENDIF
          OOPS_BOTT = OOPS_TOP + 1 + MSG_NO + 2
          OOPS_RIGHT = OOPS_LEFT + OOPS_LEN - 1

     CASE LOCATION = "C"          && center coord = 12,40
          OOPS_TOP = 12 - INT((MSG_NO + 4) / 2)
          OOPS_LEFT = 40
          IF (OOPS_LEN / 2) <> INT(OOPS_LEN / 2)
               OOPS_LEFT = OOPS_LEFT - (INT(OOPS_LEN / 2) + 1)
          ELSE
               OOPS_LEFT = OOPS_LEFT - (OOPS_LEN / 2)
          ENDIF
          OOPS_BOTT = OOPS_TOP + 1 + MSG_NO + 2
          OOPS_RIGHT = OOPS_LEFT + OOPS_LEN - 1
ENDCASE

* Begin display
* Box
SET COLOR TO (M_COOPSFR)

* Explode
IF EXPLODE
     EXP_TOP = ROUND((OOPS_BOTT - OOPS_TOP) / 2 + (OOPS_TOP - 1),0)
     EXP_LEFT = ROUND((OOPS_RIGHT - OOPS_LEFT) / 2 + (OOPS_LEFT - 1),0)
     EXP_BOTT = ROUND(EXP_TOP + 1,0)
     EXP_RIGHT = ROUND(EXP_LEFT + 1,0)

     * Determine # of steps to explode box (needed for implode, allows explode to occur faster)
     EXP_NO = ROUND((EXP_TOP - OOPS_TOP + 1),0)
     T_EXP_NO = ROUND(((OOPS_RIGHT - EXP_RIGHT + 1) / 3),0)
     EXP_NO = IF(EXP_NO < T_EXP_NO, T_EXP_NO, EXP_NO)

     IF IMPLODE
          DECLARE EXP_SCRN[EXP_NO]
     ENDIF

     FOR X = 1 TO EXP_NO
          @ EXP_TOP, EXP_LEFT, EXP_BOTT, EXP_RIGHT BOX OOPS_FRAME + SPACE(1)

          IF IMPLODE
               EXP_SCRN[X] = SAVESCREEN(OOPS_TOP, OOPS_LEFT, OOPS_BOTT, OOPS_RIGHT)  && save box as explodes for implode
          ENDIF

          IF EXP_TOP > OOPS_TOP
               EXP_TOP = EXP_TOP - 1
          ENDIF
          IF (EXP_LEFT - 3) > OOPS_LEFT
               EXP_LEFT = EXP_LEFT - 3
          ENDIF
          IF EXP_BOTT < OOPS_BOTT
               EXP_BOTT = EXP_BOTT + 1
          ENDIF
          IF (EXP_RIGHT + 3) < OOPS_RIGHT
               EXP_RIGHT = EXP_RIGHT + 3
          ENDIF
     NEXT
ENDIF
@ OOPS_TOP, OOPS_LEFT, OOPS_BOTT, OOPS_RIGHT BOX OOPS_FRAME + SPACE(1)

* Shadow
IF SHAD_SHOW
     SET COLOR TO (M_COOPSSHD)
     IF SHAD_SIDE = "R"
          COL_POS = OOPS_RIGHT + 1
     ELSE
          COL_POS = OOPS_LEFT - 2
     ENDIF
     FOR X = (OOPS_TOP + 1) TO (OOPS_TOP + 1 + MSG_NO + 2)
          @ X,COL_POS SAY SHAD_CHAR + SHAD_CHAR
     NEXT
     IF SHAD_SIDE = "R"
          COL_POS = OOPS_LEFT + 1
     ELSE
          COL_POS = OOPS_LEFT - 2
     ENDIF

     @ X,COL_POS SAY REPLICATE(SHAD_CHAR,(OOPS_LEN + 1))
ENDIF

* Title
ADJUST = (OOPS_LEN - LEN(OOPS_TITLE))
ADJUST = INT(ADJUST / 2)

SET COLOR TO (M_COOPSFR)
@ OOPS_TOP,(OOPS_LEFT + ADJUST - 1) SAY "["
SET COLOR TO (M_COOPSTIT)
@ OOPS_TOP,(OOPS_LEFT + ADJUST) SAY OOPS_TITLE
SET COLOR TO (M_COOPSFR)
@ OOPS_TOP,(OOPS_LEFT + ADJUST + LEN(OOPS_TITLE)) SAY  "]"

* Messages
SET COLOR TO (M_COOPSTXT)
@ (OOPS_TOP + 1),(OOPS_LEFT + 1) SAY SPACE(MSG_LEN + 2)
ROW_POS = OOPS_TOP + 2
FOR X = 1 TO MSG_NO
     @ ROW_POS,(OOPS_LEFT + 1) SAY SPACE(1) + MSG[X] + SPACE(1)
     ROW_POS = ROW_POS + 1
NEXT
@ (OOPS_BOTT - 1),(OOPS_LEFT + 1) SAY SPACE(MSG_LEN + 2)

* Instructions
ADJUST = (OOPS_LEN - LEN(OOPS_INS))
ADJUST = INT(ADJUST / 2)

SET COLOR TO (M_COOPSFR)
@ OOPS_BOTT,(OOPS_LEFT + ADJUST - 1) SAY "["
SET COLOR TO (M_COOPSINS)
@ OOPS_BOTT,(OOPS_LEFT + ADJUST) SAY OOPS_INS
SET COLOR TO (M_COOPSFR)
@ OOPS_BOTT,(OOPS_LEFT + ADJUST + LEN(OOPS_INS)) SAY  "]"


* Get response (ACTION)
IF VALID_RESP
     DO WHILE .T.
          TONE(920,3)
          ACTION = INKEY(0)
          OOPS_ACTION = UPPER(CHR(ACTION))   && alpha/numeric

          IF ASCAN(OOPS_RESP,OOPS_ACTION) <> 0
               EXIT
          ENDIF
     ENDDO
ELSE
     TONE(920,3)
     INKEY(0)
ENDIF


* Implode screen if set
IF IMPLODE .AND. REST_SCRN
     FOR X = EXP_NO TO 1 STEP -1
          RESTSCREEN(OOPS_TOP, OOPS_LEFT, OOPS_BOTT, OOPS_RIGHT, EXP_SCRN[X])
     NEXT
ENDIF


SET COLOR TO (CURR_COLOR)               && restore color setting

IF CURR_CURSOR
     SET CURSOR ON                      && turn cursor on if was on
ENDIF

@ CURR_ROW, CURR_COL SAY SPACE(0)       && restore cursor positions

IF REST_SCRN
     RESTORE SCREEN FROM OOPS_SCRN
ENDIF

RETURN

*--------------------------------------------------------------------------
*    EOP  DMSOOPS.PRG
*--------------------------------------------------------------------------