/*******************************************************************************
*                                                                              *
*  Program: DBSTRUCT.C                                                         *
*                                                                              *
*  Purpose: Access xBase files, derive and list the structure or optionally    *
*           list the commands to generate an xBase STORE or REPLACE operation  *
*           based on parameters supplied by the user in the command line.      *
*                                                                              *
*    Usage: DBSTRUCT[.EXE] infile[.dbf] [S|R|B]                                *
*                                                                              *
*    Parms: infile -                                                           *
*           Infile is the database file. If an extention is not supplied       *
*           then .DBF is assumed.                                              *
*                                                                              *
*           S, R & B switches -                                                *
*           The 'S' switch tells dbstruct to generate STORE commands.          *
*           The 'R' switch tells dbstruct to generate REPLACE commands.        *
*           The 'B' switch tells dbstruct to generate STORE command and        *
*             initialize memvars to blank, zero, .F. or null values.           *
*           If no switches are passed then dbstruct displays the structure     *
*           of the database.                                                   *
*                                                                              *
*    Notes: Redirection to a file is permitted with standard DOS re-           *
*           direction (e.g. DBSTRUCT database.dbf S > filename.ext)            *
*           The above example would generate STORE commands for the            *
*           database database.dbf and redirect to the file filename.ext.       *
*                                                                              *
*   Author: Frank Ray Kinser (CIS:73767,3071)                                  *
*                                                                              *
*  Dialect: Borland C++ 3.0                                                    *
*                                                                              *
*******************************************************************************/

/*******************************
*                              *
*           Includes           *
*                              *
*******************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dir.h>
#include "dbase.h"


/*******************************
*                              *
*     Function Declarations    *
*                              *
*******************************/

void dbstru( char *fname ) ;         /* Displays Structure */

void dbstore( char *fname ) ;        /* Generates STORE commands */

void dbreplace( char *fname ) ;      /* Generates REPLACE commands */

void dbstoreblank( char *fname) ;    /* Generates STORE commands with  *
                                      * fields initialized to blanks,  *
                                      * zeros and .f.                  */

void error_exit( int msgno ) ;       /* Exit to DOS w/error message */


/*******************************
*                              *
*       Public Varaiables      *
*                              *
*******************************/

HEADREC  header ;
FLDDES   field ;
FILE     *infile ;

unsigned int version, memo, nfields, n, total = 1 ;


/*******************************
*                              *
*             Main             *
*                              *
*******************************/

void main( int argc, char *argv[] )
{

	 /* local varaibles */

	 char mode[2];
    char drive[MAXDRIVE], dir[MAXDIR] ;
    char fname[MAXFILE],  ext[MAXEXT], fullname[MAXPATH] ;
	 char *dbalias;

   if( argc < 2 )      /* need at least a database name */
     error_exit(0) ;

	 /* Get filename */
    strncpy(fullname, argv[1], 79) ;
    strncpy(mode, argv[2], 1) ;

   /* Qualify the filename and extract the alias */
   fnsplit( fullname, drive, dir, fname, ext ) ;
   dbalias = fname ;
   if ( strlen(ext) == 0 )
   {
     strcpy( ext, ".dbf" ) ;                        /* give it an .ext */
     fnmerge(fullname, drive, dir, fname, ext ) ;   /* remerge filename */
     fnsplit( fullname, drive, dir, fname, ext ) ;  /* resplit filename */
   }

   /* Open as binary file and read header */
   if(( infile = fopen( fullname, "rb" )) == 0 )
     error_exit(1) ;                                 /* Couldn't open file */
   else
     fread( &header, 32, 1, infile ) ;               /* Read header */

   /* Break down ID byte */
   version = header.id & 0x07 ;               /* bits 0-2 */
   memo    = ( header.id >> 6 ) & 0x03 ;      /* bits 6-7 */
   nfields = ( header.headerlen - 33 ) / 32 ; /* Number of fields */

   /* Stru, Store, Replace or Blank based on mode requested by user */

   switch ( *mode )
   {
     case 's' :
     case 'S' :
       dbstore( dbalias ) ;       break ;
     case 'r' :
     case 'R' :
       dbreplace( dbalias ) ;     break ;
	  case 'b' :
	  case 'B' :
       dbstoreblank( dbalias ) ;  break ;
     default :
       dbstru( dbalias ) ;
   }

   fclose(infile);

}


/*******************************
*                              *
*    User Defined Functions    *
*                              *
*******************************/



/***************
*              *
*  ERROR_EXIT  *
*              *
***************/

void error_exit( int msgno )
{
	char *error_msg[] =
	{
      "\n\aSyntax: DBSTRUCT database.ext [mode]\n"    /* Msg 1: Improper Syntax */
		"Mode = (S)tore, Store (B)lank, (R)eplace\n"
		"or no mode switch to list structure\n"       ,

      "Could not open requested database\a\n"         /* Msg 2: DB does not exist */
	} ;

	printf("%s", error_msg[msgno]) ;
   exit(1) ;
}


/***********
*          *
*  DBSTRU  *
*          *
***********/

void dbstru( char *dbalias )
{
   /* Print contents of header record */
   dbalias = strupr(dbalias) ;
   printf( "\nSTRUCTURE OF: %s", dbalias ) ;
   printf( "\nDate of last update           %u/%u/%u",
    header.mm, header.dd, header.yy ) ;
   printf( "\nHeader Record Length          %u",  header.headerlen ) ;
   printf( "\nNumber of Records             %lu", header.nrecs ) ;


	/* print header */
   printf( "\n\nField  Field Name  Type          Width   Dec" ) ;

   /* Print field information */
   for( n=0; n<nfields; n++ )
   {
       fread( &field, 32, 1, infile ) ;
       total += field.length ;
       printf( "\n%5d  %-11s ", n+1, field.name ) ;

     switch ( field.type )
	  {
              case 'C' : printf( "%-9s", "Character" ) ;  break ;
              case 'D' : printf( "%-9s", "Date" ) ;       break ;
				  case 'N' :
              case 'F' : printf( "%-9s", "Numeric" ) ;    break ;
              case 'L' : printf( "%-9s", "Logical" ) ;    break ;
              case 'M' : printf( "%-9s", "Memo" ) ;       break ;
	  }
          printf( "   %5d", field.length ) ;

       if( field.decimals != 0 )
          printf( "     %3d", field.decimals ) ;

  }

       printf( "\n\nTOTAL ==>                      %5d", total ) ;
}


/************
*           *
*  DBSTORE  *
*           *
************/

void dbstore( char *dbalias )
{
	for( n=0; n<nfields; n++ )
	{
		fread( &field, 32, 1, infile ) ;
		printf( "M->%-11s  = %9s->%-11s\n", field.name, strupr(dbalias), field.name ) ;
	}
}


/**************
*             *
*  DBREPLACE  *
*             *
**************/

void dbreplace( char *dbalias )
{
   for( n=0; n<nfields; n++ )
   {
		 fread( &field, 32, 1, infile );
       printf( "replace %9s->%-11s  with M->%-11s\n", strupr(dbalias), field.name,  field.name ) ;
	}
}


/*****************
*                *
*  DBSTOREBLANK  *
*                *
*****************/

void dbstoreblank( char *dbalias )
{

  dbalias = strupr(dbalias) ;


   /* Print field information */
   for( n=0; n<nfields; n++ )
   {
       fread( &field, 32, 1, infile ) ;
       total += field.length ;
       printf( "\nM->%-11s  = ", field.name ) ;

			 switch ( field.type )
          {
            case 'C' :
               printf( "space(len(%s->%s))", dbalias, field.name) ; break ;

            case 'D' :
               printf( "ctod(\"  /  /  \")" ); break ;

            case 'N' :
            case 'F' :
               printf( "0" ) ; break ;

            case 'L' :
               printf( ".F." ) ; break ;

            case 'M' :
               printf( "\"\"" ) ; break ;
          }

   }

}
