
/* dblook.c */

#include<stdio.h>
#include<string.h>
#include<io.h>
#include<stdlib.h>
#include<ctype.h>

#if defined(MSC)
#include<malloc.h>
#endif

#if defined(TC)
#include<alloc.h>
#endif

#define MAXRECLEN    4000     /* dBase Record Length Limit */
#define MAXFIELD      128     /* fields per record limit */
#define DB4MAXFIELD   255     /* dBASEIV fields/record limit */

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

typedef struct _fld           /* field data structure */
    {
    char name[11];
    char type;
    unsigned char length;
    unsigned char decimal;
    char *value;
    } FLD;

typedef struct _hdr           /* header data structure */
    {
    unsigned version : 3;
    unsigned sqlbits : 3;
    unsigned memobits : 2;
    unsigned int yy, mm, dd;
    unsigned long numrecs;
    unsigned int hdrlen;
    unsigned int reclen;
	} HDR;

FLD field_info[DB4MAXFIELD];
int maxfields = MAXFIELD;
char *version = "dbLook: ";

void main(int argc, char **argv);
void usagemsg(void);
void errexit(char *msg);
int convert(char *buf, FLD *fields, int count);
int dbdata(FILE *fp,HDR *p, FLD *record);

void main(int argc, char **argv)
    {
    char fname[80], *buffer;
    FILE *fp;
    int i, fld_flag, dat_flag, field_count;
    long recno;
    HDR hdr;

    if(argc < 2)              /* check arguments */
        usagemsg();
    else
        strcpy(fname,argv[1]);
	strupr(fname);

    if(!strstr(fname,".DBF")) /* add .DBF extension */
        strcat(fname,".DBF");

    if(access(fname,0) != 0)  /* check for file access */
        errexit("Specified file does not exist.");

    if (argc > 2)             /* Check command line flags */
        {
        fld_flag = dat_flag = FALSE;
        for (i = 2; i < argc; i++)
            if(*argv[i] == '/')
                switch(toupper(argv[i][1]))
                    {
                case 'F':
                    fld_flag = TRUE;
                    break;
                case 'D':
                    dat_flag = TRUE;
                    break;
                default:
                    usagemsg();
                    break;
                    }
        }
    else
        {    
        fld_flag = TRUE;      /* set default flags */
        dat_flag = FALSE;
        }    

    if (!(fp = fopen(fname,"rb")))      /* open data file */
        errexit("Can't open specified file.");

          /* read file structure */
    if ((field_count = dbdata(fp,&hdr,field_info)) <= 0)
        errexit("Can't read structure information from file.");

    if (hdr.version < 2 || hdr.version > 4)  /* check header data */
        {
        printf("ver=%d",hdr.version);
        errexit("dBASE version invalid. File may be corrupted.");
        }
    if (hdr.version >= 4)
        maxfields = DB4MAXFIELD;

    if ((hdr.mm < 1 || hdr.mm > 12) || (hdr.dd < 1 || hdr.dd > 31) ||
            (hdr.yy > 99))
        errexit("Update date invalid. File may be corrupted.");
    if (hdr.reclen > MAXRECLEN)
        errexit("Record Length out of range. File may be corrupted.");

    if (fld_flag)             /* display structure */
        {
        printf("Structure for database: %s\n",fname);
        printf("Number of data records: %7ld\n",hdr.numrecs);
        printf("Date of last update   : %02d/%02d/%02d\n",
                hdr.mm,hdr.dd,hdr.yy);
        printf("Field  Field Name  Type       Width    Dec\n");
        for (i = 0; i < field_count; i++)
            {
            printf("%5d  %-10.10s  ",i+1,field_info[i].name);
            switch (field_info[i].type)
                {
            case 'C':
                printf("Character");
                break;
            case 'D':
                printf("Date     ");
                break;
            case 'L':
                printf("Logical  ");
                break;
            case 'M':
                printf("Memo     ");
                break;
            case 'N':
                printf("Numeric  ");
                break;
            default:
                errexit("Unknown field type");
                break;
                }
            printf("  %5d",field_info[i].length);
            if (field_info[i].type == 'N')
                printf("    %3d",field_info[i].decimal);
            printf("\n");
            }
        printf("** Total **                     %3d\n",hdr.reclen);
        }

    if (dat_flag)             /* if data option specified */
        {
        if (!(buffer = malloc(hdr.reclen)))
            errexit("Not enough memory for record buffer");

        for (i = 0; i < field_count; i++)    /* allocate field space */
            if (!(field_info[i].value = malloc(field_info[i].length+1)))
                errexit("Not enough memory for field conversion");

          /* skip past header area    */
        if (fseek(fp,(long) (hdr.hdrlen),SEEK_SET) != 0)
            errexit("Can't move file pointer to start of data");

        printf("Contents of %s:\n\n",fname);

          /* list record contents */
        for (recno = 0; recno < hdr.numrecs; recno++)
            {
          /* read record */
            if (fread(buffer,hdr.reclen,1,fp) != 1)
                errexit("Can't read data record from file");
          /* convert contents */
            if (convert(buffer,field_info,field_count))
                printf("\n#%d DELETED\n",recno+1);
            else
                printf("\n#%d\n",recno+1);   /* Normal Record */

            for (i = 0; i < field_count; i++)     /* display contents */
                printf("%-10.10s: %s\n",field_info[i].name,
                        field_info[i].value);
            }
        }
    fclose(fp);               /* close file */
    exit(0);
    }

void usagemsg(void)           /* print usage and exit */
    {
    puts("Usage: dblook [options] <filename>[.dbf]");
    puts("\twhere filename a dBASE data file");
    puts("\t/f - show file structure (default)");
    puts("\t/d - show file contents");
    exit(1);
    }

void errexit(char *msg)       /* Display message and exit */
    {
    printf("%s\n",version);
    printf("Error: %s\n",msg);
    exit(255);
    }

          /* convert buffer to fields */
int convert(char *buf, FLD *fields, int count)
    {
    int i, index = 1;

    for (i = 0; i < count; i++)    /* read field values data */
        {
        switch(fields[i].type)
            {
        case 'C':        /* Character */
        case 'N':        /* Numeric */
        case 'L':        /* Logical */

            strncpy(fields[i].value,&buf[index],fields[i].length);
            fields[i].value[fields[i].length] = '\0';
            break;
        case 'D':        /* convert to MM/DD/YY */
            sprintf(fields[i].value,"%2.2s/%2.2s/%2.2s",&buf[index+4],
                    &buf[index+6],&buf[index+2]);
            break;
        case 'M':        /* Memo field */
            strcpy(fields[i].value,"MEMO FIELD");
            break;
        default:
            return(-1);
            }
        index += fields[i].length;
        }
    return (buf[0] == '*') ? 1 : 0;     /* 1 if marked for deletion */
    }

int dbdata(FILE *fp,HDR *p, FLD *record)     /* read header info */
    {
    int fldcount, i;
    unsigned char buffer[32];

    if (fread(buffer, sizeof(buffer), 1, fp) != 1)    /* read file header */
        return(-1);

    p->version = (unsigned char) (buffer[0] & 0x7F);  /* strip off high bit */
    p->yy = buffer[1];
    p->mm = buffer[2];
    p->dd = buffer[3];
    p->numrecs = buffer[4] + (256 * buffer[5]) +
            (4096 * buffer[6]) + (65536 * buffer[7]);
    p->hdrlen = buffer[8] + (256 * buffer[9]);
    p->reclen = buffer[10] + (256 * buffer[11]);

    fldcount = (p->hdrlen - 33)/32;     /* get field structure data */
    
    if (fldcount < 1 || fldcount > maxfields)     /* check field count */
        return(-1);

    for (i = 0; i < fldcount; i++)      /* read fields */
        {
        if (fread(buffer, sizeof(buffer), 1, fp) != 1)
            return(-1);
        sscanf(buffer,"%11s",record[i].name);
        record[i].type = buffer[11];
        record[i].length = buffer[16];
        record[i].decimal = buffer[17];
        }
    rewind(fp);               /* to beginning of file */
    return(fldcount);
    }

/* end of dblook.c */



