/* DBEDIT_R.PRG - Drop-in replacement for Clipper's DBEdit()

   Provides a ability to browse an indexed range of records.

*/

#include "dbedit_r.ch"

STATIC aRange  // Global "RANGE" Setting

PROCEDURE DBEdit( nTr,nLc,nBr,nRc,acFields,cUdf,acPics,acHeads,;
                  acHeadSep,acColSep,acFootSep,acFootings )
   LOCAL tb:= ConfigTBrowse( nTr,nLc,nBr,nRc )
   LOCAL bUdf

   // If these came in as .F., turn them to NIL or whatever
   S87DEFAULT nTr        TO 0
   S87DEFAULT nLc        TO 0
   S87DEFAULT nBr        TO MaxRow()
   S87DEFAULT nRc        TO MaxCol()
   S87DEFAULT acFields   TO NIL
   S87DEFAULT cUdf       TO NIL
   S87DEFAULT acPics     TO NIL
   S87DEFAULT acHeads    TO NIL
   S87DEFAULT acHeadSep  TO NIL
   S87DEFAULT acColSep   TO NIL
   S87DEFAULT acFootSep  TO NIL
   S87DEFAULT acFootings TO NIL

   AddColumns( tb,acFields,acPics,acHeads,acHeadSep,;
                  acColSep,acFootSep,acFootings )

   IF cUdf <> NIL // Turn it into a codeblock
      bUdf:= &( "{| nMode,nCol |" + cUdf + "( nMode,nCol )}" )
   ENDIF

   TBrowseProcess( tb,bUdf )
   RETURN

STATIC PROCEDURE TBrowseProcess( tb,bUdf )
   LOCAL nKey,nMode,bKey
   LOCAL nRet:= DE_CONT
   LOCAL isDone:= .F.
   IF Eof() .AND. Bof()
      IF bUdf == NIL
         isDone:= .F.
      ELSE
         isDone:= Eval( bUdf, DE_EMPTY, tb:colPos ) == DE_ABORT
      ENDIF
   ENDIF
   WHILE !isDone
      WHILE NextKey() == 0
         IF tb:stabilize()
            nRet:= CallUserFunc( tb,bUdf )
            DO CASE
            CASE nRet == DE_ABORT
               isDone:= .T.
               LOOP
            CASE nRet == DE_REFRESH
               LOOP
            ENDCASE
            EXIT
         ENDIF
      END
      nKey:= InKey( 0 )
      DO CASE
      CASE ( bKey:= SetKey( nKey ) ) <> NIL
         Eval( bKey, ProcName( 2 ),ProcLine( 2 ), "" )
      CASE nKey == K_UP;                     tb:up()
      CASE nKey == K_DOWN;                   tb:down()
      CASE nKey == K_LEFT;                   tb:left()
      CASE nKey == K_RIGHT;                  tb:right()
      CASE nKey == K_CTRL_LEFT;              tb:panLeft()
      CASE nKey == K_CTRL_RIGHT;             tb:panRight()
      CASE nKey == K_HOME;                   tb:home()
      CASE nKey == K_END;                    tb:end()
      CASE nKey == K_CTRL_HOME;              tb:panHome()
      CASE nKey == K_CTRL_END;               tb:panEnd()
      CASE nKey == K_PGUP;                   tb:pageUp()
      CASE nKey == K_PGDN;                   tb:pageDown()
      CASE nKey == K_CTRL_PGUP;              tb:goTop()
      CASE nKey == K_CTRL_PGDN;              tb:goBottom()
      CASE nKey == K_ESC .AND. bUdf==NIL;    EXIT
      CASE nKey == K_ENTER.AND. bUdf==NIL;   EXIT
      OTHERWISE
         isDone:= CallUserFunc( tb,bUdf,DE_EXCEPT ) == DE_ABORT
      ENDCASE
   END
   RETURN

STATIC FUNCTION CallUserFunc( tb,bUdf,nMode )
   LOCAL nRet:= DE_CONT
   DO CASE
   CASE nMode <> NIL     // Do nothing; was DE_EXCEPT
   CASE Eof().AND.Bof(); nMode:= DE_EMPTY
   CASE tb:hitTop;       nMode:= DE_HITTOP
   CASE tb:hitBottom;    nMode:= DE_HITBOTTOM
   OTHERWISE;            nMode:= DE_IDLE
   ENDCASE
   IF bUdf <> NIL .AND. ( nRet:= Eval( bUdf, nMode, tb:colPos ) ) == DE_REFRESH
      tb:refreshAll()
   ENDIF
   RETURN nRet  // Could be DE_CONT,DE_ABORT,DE_REFRESH

STATIC PROCEDURE AddColumns(tb,acFields,acPics,acHeads,acHeadSep,;
                               acColSep,acFootSep,acFoots )
   LOCAL aFields
   LOCAL aFldBlocks:= FldBlocks( acFields,@aFields )
   LOCAL aHeads:=     Stuff2Arr( acHeads )
   LOCAL aFoots:=     Stuff2Arr( acFoots )
   LOCAL aPics:=      Stuff2Arr( acPics )
   LOCAL lHeadSeps,lFootSeps,lColSeps
   LOCAL i,c,bFld,cHead

   DEFAULT acHeadSep  TO  ""
   DEFAULT acColSep   TO  "  "
   DEFAULT acFootSep  TO  ""

   // Head,foot,and col seps are TBrowse wide if not array
   IF !( lHeadSeps:= ( ValType( acHeadSep ) == "A" ) )
      tb:headSep:= acHeadSep
   ENDIF
   IF !( lFootSeps:= ( ValType( acFootSep ) == "A" ) )
      tb:footSep:= acFootSep
   ENDIF
   IF !( lColSeps:= ( ValType( acColSep ) == "A" ) )
      tb:colSep:= acColSep
   ENDIF

   // Add all the columns w/appropriate info
   FOR i:= 1 TO Len( aFldBlocks )
      bFld:= aFldBlocks[i]
      IF aPics[i] <> NIL
         bFld:= AddPicture( bFld,aPics[i] )
      ENDIF
      cHead:= IIf( aHeads[i]<>NIL, aHeads[i], aFields[i] )
      c:= TBColumnNew( cHead,bFld )
      IF lHeadSeps;  c:headSep:= acHeadSep[i];  END
      IF lFootSeps;  c:footSep:= acFootSep[i];  END
      IF lColSeps;   c:colSep:=  acColSep[i];   END
      tb:addColumn( c )
   NEXT
   RETURN

// Used by AddColumns
STATIC FUNCTION AddPicture( bFld,aPic )
   RETURN {|| Transform( Eval(bFld),aPic ) }

// Get/Set function for browsing range info.
FUNCTION DBEditRange( aNew )
   LOCAL aTmp:= aRange         // Access Static var
   IF ValType( aNew ) == "A"
      aRange:= aNew
   ENDIF
   RETURN aTmp

// Set the browse range and key value
PROCEDURE DBEditSetRange( acValue,bKey )
   IF !( ValType( acValue ) == "A" )               // Array specifies both upper
      acValue:= { acValue,acValue }                // and lower end of range.
   ENDIF
   aRange:= { acValue,bKey }
   RETURN

// Return array of codeblocks for columns
// given fieldname or array of fieldnames
STATIC FUNCTION FldBlocks( acFields,aNames )
   LOCAL aBlocks,i
   DO CASE
   CASE ValType( acFields ) == "C";    aNames:= { acFields }
   CASE ValType( acFields ) == "A";    aNames:= acFields
   OTHERWISE;                          aNames:= Array( FCount() )
                                       AFields( aNames )
   ENDCASE
   aBlocks:= Array( Len( aNames ) )
   FOR i:= 1 TO Len( aNames )
      IF FieldPos( aNames[ i ] ) > 0  // Is this a field name?
         aBlocks[ i ]:= FieldBlock( aNames[ i ] )
      ELSE // This expression is not a field, must macro compile
         aBlocks[ i ]:= &( "{||"+ aNames[ i ] +"}" )
      ENDIF
   NEXT
   RETURN aBlocks

// Returns an array of stuff given either an array of stuff,
// a single char, or nothing as a parameter
// Designed for headings,footings,and pictures
STATIC FUNCTION Stuff2Arr( acStuffs )
   LOCAL aStuffs
   DO CASE
   CASE ValType( acStuffs ) == "C";    aStuffs:= { acStuffs }
   CASE ValType( acStuffs ) == "A";    aStuffs:= acStuffs
   OTHERWISE;                          aStuffs:= Array( FCount() )
   ENDCASE
   RETURN aStuffs

// Returns a TBrowse object ready to browse
STATIC FUNCTION ConfigTBrowse( nTr,nLc,nBr,nRc )
   LOCAL aRange:= DBEditRange()
   LOCAL uLower,uUpper,bKey
   LOCAL tb
   IF DBEditRange() == NIL
      tb:= TBrowseDB( nTr,nLc,nBr,nRc )
   ELSE
      IF Empty( IndexKey(0) )
         GenerateError( ERR_NO_KEY,"No active index",,.CANDEFAULT. )
      ENDIF
      tb:= TBrowseNew( nTr,nLc,nBr,nRc )
      tb:goTopBlock:=    {||  RangeGoTop( uLower ) }
      tb:goBottomBlock:= {||  RangeGoBott( uLower,uUpper,bKey ) }
      tb:SkipBlock:=     {|n| RangeSkip( uLower,uUpper,bKey,n ) }
      uLower:= aRange[1][1]
      uUpper:= aRange[1][2]
      bKey:=   IIf( aRange[2]==NIL,;                // If not specified
                    &( "{||"+IndexKey(0)+"}"),;     // Use current index key
                    aRange[2] )
   ENDIF
   RETURN tb

STATIC PROCEDURE RangeGoTop( uLower )
   IF Empty( uLower )    // They should work if no scope
      GO TOP
   ELSE
      SEEK uLower
      IF !Found()
         GO LastRec() + 1
      ENDIF
   ENDIF
   RETURN

STATIC PROCEDURE RangeGoBott( uLower,uUpper, bExpr )
   IF Empty( uLower )    // They should work if no scope
      GO BOTTOM
   ELSE
      dbSeek( IncrByOne( uUpper ), .SOFTSEEK. )
      SKIP -1
      IF !( uUpper == Eval( bExpr ) )
         GO LastRec() + 1
      ENDIF
   ENDIF
   RETURN

STATIC FUNCTION RangeSkip( uLower,uUpper,bKeyExpr, nToSkip )
   LOCAL nSkipped:= 0
   LOCAL uTmp
   IF Empty( uLower )
      bKeyExpr:= {|| uLower }
   ENDIF
   IF ( uTmp:= Eval( bKeyExpr ) ) < uLower .OR. uTmp > uUpper
      RangeGoTop( uLower )
   ENDIF
   DO CASE
   CASE nToSkip == 0
      SKIP 0
   CASE nToSkip > 0
      WHILE .T.
         SKIP +1
         DO CASE
         CASE Eof() .OR. Eval( bKeyExpr ) > uUpper
            SKIP -1
            EXIT
         CASE ++nSkipped >= nToSkip
            EXIT
         ENDCASE
      END
   CASE nToSkip < 0
      WHILE .T.
         SKIP -1
         DO CASE
         CASE Bof()
            EXIT
         CASE Eval( bKeyExpr ) < uLower
            SKIP +1
            EXIT
         CASE --nSkipped <= nToSkip
            EXIT
         ENDCASE
      END
   ENDCASE
   IF ( uTmp:= Eval( bKeyExpr ) ) < uLower .OR. uTmp > uUpper
      RangeGoBott( uLower,uUpper,bKeyExpr )
   ENDIF
   RETURN nSkipped

STATIC FUNCTION IncrByOne( uValue )
   LOCAL nLen,cLast,cType:= ValType( uValue )
   LOCAL uRetVal
   DO CASE
   CASE cType == "C"
      cLast:= SubStr( uValue, nLen:= Len( uValue ) )
      uRetVal:= SubStr( uValue, 1, nLen - 1 ) + Chr( Asc( cLast ) + 1 )
   CASE cType $ "ND"
      uRetVal:= uValue++
   ENDCASE
   RETURN uRetVal

STATIC FUNCTION GenerateError( nErr,cMsg,cOp,canDefa,canRetry,canSubs )
   LOCAL e:= ErrorNew()
   DEFAULT cOp      TO "",;
           canDefa  TO .F.,;
           canRetry TO .F.,;
           canSubs  TO .F.
   e:subSystem:=     "DBEDIT_RANGE"
   e:subCode:=       nErr
   e:description:=   cMsg
   e:operation:=     cOp
   e:canDefault:=    canDefa
   e:canRetry:=      canRetry
   e:canSubstitute:= canSubs
   e:severity:=      ES_ERROR
   RETURN Eval( ErrorBlock(), e )





// EOF

