/*
 Date & Time compression routines

 Author: Thom Pantazi
  CISID: 72713,2671

 This is a copyrighted work.

 Copyright Thom Pantazi, (C) 1993  All Rights Reserved.

 You may use this code freely, however, you may not remove this or any
 copyright notice.  If you make changes you may add your name to the copyright
 list.

 Dates and times are stored in a two byte binary character format.  Saving
 six bytes per occurance.

 One caveat exists, dates must start at 01/01/1980 and end 06/06/2159. This
 allows for 178 full years.  If you need to store dates earlier than 01/01/80
 change the manifest constant BASEDATE.

 Syntax:

 To read a date from field: dDate := Bytes2Date(Date)

 To read a time from field: cTime := Bytes2Time(Time)

 To write a date to field: REPLACE Date WITH Date2Bytes(dDate)

 To write a time to field: REPLACE Time WITH Time2Bytes(cTime)

 Field Format:

       DATE Character 2
       TIME Character 2
*/

#DEFINE BASEDATE CTOD("01/01/80")

// Function Ŀ
//         Name: Bytes2Date()          Docs:                                
//  Description: Converts two bytes to a date n days since 01/01/80         
//       Author: Thom Pantazi 72713,2671                                    
// Date created: 10-11-93              Date updated: 10-11-93              
// Time created: 08:41:57pm            Time updated: 08:41:57pm            
//    Copyright: Thom Pantazi (c) 1993                                      
//Ĵ
//    Arguments: cBytes                                                     
// Return Value: BASEDATE + Bytes2Num(cBytes)                               
//     See Also: Date2Bytes                                                 
//
FUNCTION Bytes2Date(cBytes)
RETURN BASEDATE + Bytes2Num(cBytes)

// Function Ŀ
//         Name: Date2Bytes()          Docs:                                
//  Description: Converts a date to a two byte integer of n days since      
//               01/01/80                                                   
//       Author: Thom Pantazi 72713,2671                                    
// Date created: 10-11-93              Date updated: 10-11-93              
// Time created: 08:43:03pm            Time updated: 08:43:03pm            
//    Copyright: Thom Pantazi (c) 1993                                      
//Ĵ
//    Arguments: dDate                                                      
// Return Value: Num2Bytes(BASEDATE - dDate)                                
//     See Also: Bytes2Date                                                 
//
FUNCTION Date2Bytes(dDate)
RETURN Num2Bytes(dDate - BASEDATE)

// Function Ŀ
//         Name: Bytes2Time()          Docs:                                
//  Description: Converts a two byte integer to a time string hh:mm:ss      
//       Author: Thom Pantazi 72713,2671                                    
// Date created: 10-11-93              Date updated: 10-11-93              
// Time created: 08:44:26pm            Time updated: 08:44:26pm            
//    Copyright: Thom Pantazi (c) 1993                                      
//Ĵ
//    Arguments: cBytes                                                     
// Return Value: TimeStr hh:mm:ss                                           
//     See Also:                                                            
//
FUNCTION Bytes2Time(cBytes)
    LOCAL nHrs
    LOCAL nMins
    LOCAL nSecs
    LOCAL nTotSecs := Bytes2Num(cBytes)
    nSecs := (nTotSecs % 3600) % 60
    nTotSecs -= nSecs
    nMins := INT((nTotSecs % 3600) / 60)
    nTotSecs -= nMins
    nHrs  := INT(nTotSecs / 3600)
RETURN ZeroPad(nHrs,2)+":"+ZeroPad(nMins,2)+":"+ZeroPad(nSecs,2)

*  Function Ŀ
*          Name: Time2Bytes()          Docs:                                
*   Description: Converts a Time String to a Two Byte Binary Number         
*              :                                                            
*        Author: Thom Pantazi 72713,2671                                    
*  Date created: 10-21-93              Date updated: 10-21-93              
*  Time created: 10:22:47am            Time updated: 10:22:47am            
*     Copyright: Thom Pantazi (c) 1993                                      
* Ĵ
*     Arguments: cTime "hh:mm:ss"                                           
*  Return Value: Num2Bytes((Hrs@*3600)@+(nMins@*60)@+nSecs)                 
*      See Also:                                                            
* 
FUNCTION Time2Bytes(cTime)
    LOCAL nHrs  := VAL(SUBSTR(cTime,1,2))
    LOCAL nMins := VAL(SUBSTR(cTime,4,2))
    LOCAL nSecs := VAL(SUBSTR(cTime,7,2))
RETURN Num2Bytes((nHrs*3600)+(nMins*60)+nSecs)

// Function Ŀ
//         Name: Bytes2Num()           Docs:                                
//  Description: Converts Two Character Bytes to an Integer                 
*              : Maximum Integer Value of 65535 (256*256)                   
//       Author: Thom Pantazi 72713,2671                                    
// Date created: 10-21-93              Date updated: 10-21-93              
// Time created: 10:13:32am            Time updated: 10:13:32am            
//    Copyright: Thom Pantazi (c) 1993                                      
//Ĵ
//    Arguments: cBytes                                                     
// Return Value: ASC(SUBSTR(cBytes,1,1))@+(ASC(SUBSTR(cBytes,2,1))@*256)    
//     See Also:                                                            
//
FUNCTION Bytes2Num(cBytes)
RETURN (ASC(SUBSTR(cBytes,1,1))+(ASC(SUBSTR(cBytes,2,1))*256))

// Function Ŀ
//         Name: Num2Bytes()           Docs:                                
//  Description: Converts an Integer to Two Character Bytes                 
//       Author: Thom Pantazi 72713,2671                                    
// Date created: 10-21-93              Date updated: 10-21-93              
// Time created: 10:14:24am            Time updated: 10:14:24am            
//    Copyright: Thom Pantazi (c) 1993                                      
//Ĵ
//    Arguments: nNumber                                                    
// Return Value: CHR(nNumber-(INT(nNumber/256)@*256))@+CHR(INT(nNumber/256))
//     See Also:                                                            
//
FUNCTION Num2Bytes(nNumber)
RETURN CHR(nNumber-(INT(nNumber/256)*256))+CHR(INT(nNumber/256))

*  Function Ŀ
*          Name: ZeroPad()             Docs:                                
*   Description: Pads a number with zeros and returns as string             
*        Author: Thom Pantazi 72713,2671                                    
*  Date created: 10-21-93              Date updated: 10-21-93              
*  Time created: 11:04:44am            Time updated: 11:04:44am            
*     Copyright: Thom Pantazi (c) 1993                                      
* Ĵ
*     Arguments: nNum Number to pad                                         
*              : nLen Number of zeros or width of return string             
*  Return Value: PADL( LTRIM( STR( nNum ) ), nLen , "0" )                   
*      See Also:                                                            
* 
FUNCTION ZeroPad( nNum, nLen )
RETURN (PADL( LTRIM( STR( nNum ) ), nLen , "0" ) )

