#include "Clip_Dev.ch"

/*
   This example demonstrates a use for Calculated Instance Variables.  The
   example creates a TEMPERATURE class which has two export instance
   variables Fahrenheit and CELSIUS.  CELSIUS is a calculated instance
   variable which computes its value from the current value of Fahrenheit.

   Notice how CELSIUS can also be set!  Setting the CELSIUS instance
   variable "value" is simply a matter of taking the supplied value and
   converting it to its corresponding Fahrenheit equivalent and then
   setting the Fahrenheit instance variable with this value.  In other
   words you can directly set CELSIUS as the line |< t:Celsius >| := 66.5
   shows, but what really happens is that Fahrenheit is set with the
   equivalent value via the set portion of the get/set calculated code block.

   Notice how the instance variables can never be out of sync.  If you
   update one, the other is automatically updated.  This is because there
   is only one value to update - Fahrenheit!  CELSIUS is an illusion.  It
   only exists at a conceptual level for the user.  Physically there is
   no CELSIUS value!

   This example demonstrates how calculated instance variables can be made
   to simulate completely abstract pieces of "data".  While a Celsius()
   method could have been created with virtually the same code as above,
   this would be using methods to access something that is "conceptually"
   a data item.  By increasing the power of instance variables to access
   "data" we free methods to be used for what they were intended to be
   used - "acting on" the object's instance variables instead of
   providing access to them.  Said another way, a method which merely
   returns a value is conceptually identical to an instance variable
   and should be implemented as such.  All other methods actually
   perform an action or a task and represent the true essence of what a
   method is.  Part of the confusion surrounding OOP is that object
   oriented language implementations fail to adequately make this
   distinction.
*/

FUNCTION Main
   LOCAL t, i

   // Define the TEMPERATURE class.
   DEFINE CLASS Temperature
      EXPORT VARIABLES ;               // Create two instance variables...
	 Fahrenheit := 0, ;            // Only "physical" value required.
	 Celsius          ;            // Celsius is completely derived.
	    CALCULATED BY {|o,c| IIF(c==NIL, ;
				5 / 9 * ( |< o:Fahrenheit >| - 32 ), ;
				|< o:Fahrenheit >| := (9 * c) / 5 + 32 ) }
   ENDCLASS

   t := |< Temperature:New() >|        // Create an instance of Temperature.
   ? "   Fahrenheit     Celsius"       // Display a little table heading.
   ? "   **********************"

   FOR i := -40 TO 100 STEP 10         // Fill in the table with values.
      // Note: Assigning to Fahrenheit automatically "updates" Celsius.
      ? |< t:Fahrenheit >| := i, |< t:Celsius >|
   NEXT i

   ? "   **********************"

   // "Assigning" to Celsius automatically updates Fahrenheit!!!!
   |< t:Celsius >| := 66.5
   ? |< t:Fahrenheit >|, |< t:Celsius >|
RETURN NIL
