#include 'Clip_Dev.ch'

/*
   This TBROWSE class definition is a Clip/++ functional equivalent to
   Clipper's  own built-in class TBrowse.  The design is very simple;
   just "wrap" a UDO called TBrowse around an instance of Clipper's
   TBrowse.  The reason for this is so that other UDOs created with
   Clip/++ can inherit from the TBrowse UDO, since you cannot inherit
   from built-in Clipper objects with Clip/++.

   While a little long, the definition is straightforward.  For every
   instance variable and method documented for Clipper's TBrowse, the
   UDO has an identically-named one.  Notice the one additional instance
   variable called 'Browse'.  This is an import instance variable and
   can only be accessed from within the class.  This variable is itself
   an instance of Clipper's TBrowse.  Rather than completely write my own
   version of TBrowse (undoubtedly a lot of work) I am simply making use
   of all the code and data built into every instance of Clipper's TBrowse.

   Notice how all but one variable is a calculated instance variable.  For
   example, consider the "nTop" instance variable.  It has no actual value
   but instead gets its variable from inside the instance of Clipper's
   TBrowse found in the import variable "Browse".  Implementing the class
   in this way, your users can get and set the nTop instance variable
   normally and never know that no such value exists.  Without the
   calculated instance variable feature, the setting of the nTop instance
   variable would have had to be done with a method which would not have
   been compatible with the way you use nTop in your current TBrowse code.

   The single instance variable 'cargo' need not be calculated since it
   can be implemented directly within the UDO itself.  This is so because
   Clipper's TBrowse objects never internally reference 'cargo', only
   a user's code ever works with this instance variable.

   Notice how 'colCount', 'leftVisible', 'rightVisible' and 'rowCount'
   have all been declared READONLY.  This is because Nantucket's doc-
   umentation states that these instance variables are not assignable.

   Also, note how the instance variables 'colSep' and 'headSep' have a
   trigger associated with them.  This trigger simply calls the Configure()
   method which forces the TBrowse object to re-examine its instance
   variables and re-perform some internal calculations that it normally does
   not do during a stabilize.  I have added these triggers to illustrate
   how they can be used to make the object easier to use.  With Clipper's
   TBrowse objects, if you set one of these two variables you would have
   to follow it with a Configure() method to see the results.  The UDO
   implementation of TBrowse below automates this method invocation using
   triggers.  Just set the instance variables to some new value and the
   result is immediately displayed.

   In addition each instance variable has been typed using the TYPE clause.
   It is always a good idea to type calculated instance variables since they
   are not normally given default values and their types will then default
   to "ANY".

   The first four instance variables are; nTop, nLeft, nBottom, and nRight.
   These are placed first so that the universal constructor method NEW()
   can initialize them (in the same way as TBrowseDB() does) if desired.
   The instance variables have been declared in groups because attempting
   to declare them on a single EXPORT VARIABLES command line will produce
   a line to long for the preprocessor to handle.

   The methods have all been reproduced by simply sending the same
   message to the 'Browse' import instance variable.  Notice the use of
   parenthesis to accomplish this.  Inside the code block of each method
   a message is sent to an actual Clipper object (not a UDO) which is
   why no special bracketing syntax '|<...>|' is required for the method
   invocation.  Note that you still need these brackets to access the
   import instance variable 'Browse' which is of type 'O' - a Clipper
   object.  (i.e. (|<o:Browse>|) --> returns a Clipper object of type
   'O' not 'UDO'.)

   Finally, note that the import instance variable 'Browse' was created
   with a call to TBrowseDB() which means that this TBrowse UDO is currently
   only suitable for browsing databases.

   For an example of this class in action, see the "TBrowse.prg" program.
*/

FUNCTION z_TBrowse

   DEFINE CLASS TBrowse
      EXPORT VARIABLES ;
	 nTop := 0 ;
	    CALCULATED BY {|o,v| IIF(v==NIL, (|<o:Browse>|):nTop, ;
					     (|<o:Browse>|):nTop:=v)}, ;
	 nLeft := 0 ;
	    CALCULATED BY {|o,v| IIF(v==NIL, (|<o:Browse>|):nLeft, ;
					     (|<o:Browse>|):nLeft:=v)}, ;
	 nBottom := MAXROW() ;
	    CALCULATED BY {|o,v| IIF(v==NIL, (|<o:Browse>|):nBottom, ;
					     (|<o:Browse>|):nBottom:=v)}, ;
	 nRight := MAXCOL() ;
	    CALCULATED BY {|o,v| IIF(v==NIL, (|<o:Browse>|):nRight, ;
					     (|<o:Browse>|):nRight:=v)}
      EXPORT VARIABLES ;
	 autoLite      TYPE |'L'| ;
	    CALCULATED BY {|o,v| IIF(v==NIL, (|<o:Browse>|):autoLite, ;
					     (|<o:Browse>|):autoLite := v)}, ;
	 cargo         TYPE |'ANY'|, ;  // Can implement in the object itself
	 colCount      TYPE |'N'| READONLY ;
	    CALCULATED BY {|o| (|<o:Browse>|):colCount}, ;
	 colorSpec     TYPE |'C'| ;
	    CALCULATED BY {|o,v| IIF(v==NIL, (|<o:Browse>|):colorSpec, ;
					     (|<o:Browse>|):colorSpec:=v)}, ;
	 colPos        TYPE |'N'| ;
	    CALCULATED BY {|o,v| IIF(v==NIL, (|<o:Browse>|):colPos, ;
					     (|<o:Browse>|):colPos:=v)}, ;
	 colSep        TYPE |'C'| TRIGGER {|o| |<o:Configure()>|} ;
	    CALCULATED BY {|o,v| IIF(v==NIL, (|<o:Browse>|):colSep, ;
					     (|<o:Browse>|):colSep:=v)}, ;
	 freeze        TYPE |'N'| ;
	    CALCULATED BY {|o,v| IIF(v==NIL, (|<o:Browse>|):freeze, ;
					     (|<o:Browse>|):freeze:=v)}, ;
	 goBottomBlock TYPE |'B'| ;
	    CALCULATED BY {|o,v| IIF(v==NIL, (|<o:Browse>|):goBottomBlock,;
					     (|<o:Browse>|):goBottomBlock:=v)}, ;
	 goTopBlock    TYPE |'B'| ;
	    CALCULATED BY {|o,v| IIF(v==NIL, (|<o:Browse>|):goTopBlock, ;
					     (|<o:Browse>|):goTopBlock:=v)}

      EXPORT VARIABLES  ;
	 headSep      TYPE |'C'| TRIGGER {|o| |<o:Configure()>|} ;
	    CALCULATED BY {|o,v| IIF(v==NIL, (|<o:Browse>|):headSep, ;
					     (|<o:Browse>|):headSep:=v)}, ;
	 leftVisible  TYPE |'N'| READONLY  ;
	    CALCULATED BY {|o| (|<o:Browse>|):leftVisible}, ;
	 rightVisible TYPE |'N'| READONLY  ;
	    CALCULATED BY {|o| (|<o:Browse>|):rightVisible}, ;
	 rowCount     TYPE |'N'| READONLY ;
	    CALCULATED BY {|o| (|<o:Browse>|):rowCount}, ;
	 skipBlock    TYPE |'B'| ;
	    CALCULATED BY {|o,v| IIF(v==NIL, (|<o:Browse>|):skipBlock, ;
					     (|<o:Browse>|):skipBlock:=v)}, ;
	 stable       TYPE |'L'| ;
	    CALCULATED BY {|o,v| IIF(v==NIL, (|<o:Browse>|):stable, ;
					     (|<o:Browse>|):stable:=v)}
      IMPORT VARIABLE ;
	 Browse := TBrowseDB()

      EXPORT METHODS ;
	 Stabilize := { |o|   (|<o:Browse>|):Stabilize() }, ;
	 Up        := { |o|   (|<o:Browse>|):Up() }, ;
	 Down      := { |o|   (|<o:Browse>|):Down() }, ;
	 PageUp    := { |o|   (|<o:Browse>|):PageUp() }, ;
	 PageDown  := { |o|   (|<o:Browse>|):PageDown() }, ;
	 PanHome   := { |o|   (|<o:Browse>|):PanHome() }, ;
	 PanEnd    := { |o|   (|<o:Browse>|):PanEnd() }, ;
	 Home      := { |o|   (|<o:Browse>|):Home() }, ;
	 End       := { |o|   (|<o:Browse>|):End() }, ;
	 Right     := { |o|   (|<o:Browse>|):Right() }, ;
	 Left      := { |o|   (|<o:Browse>|):Left() }, ;
	 PanRight  := { |o|   (|<o:Browse>|):PanRight() }, ;
	 PanLeft   := { |o|   (|<o:Browse>|):PanLeft() }, ;
	 GoTop     := { |o|   (|<o:Browse>|):GoTop() }, ;
	 GoBottom  := { |o|   (|<o:Browse>|):GoBottom() }, ;
	 AddColumn := { |o,p| (|<o:Browse>|):AddColumn(p[1]) }, ;
	 Configure := { |o|   (|<o:Browse>|):Configure() }

      EXPORT METHODS ;
	 ColorRect      := { |o,p| (|<o:Browse>|):ColorRect(p[1],p[2]) }, ;
	 ColWidth       := { |o,p| (|<o:Browse>|):ColWidth(p[1]) }, ;
	 deHilite       := { |o|   (|<o:Browse>|):deHilite() }, ;
	 delColumn      := { |o,p| (|<o:Browse>|):delColumn(p[1]) }, ;
	 getColumn      := { |o,p| (|<o:Browse>|):getColumn(p[1]) }, ;
	 hilite         := { |o|   (|<o:Browse>|):hilite() }, ;
	 insColumn      := { |o,p| (|<o:Browse>|):insColumn(p[1],p[2]) }, ;
	 invalidate     := { |o|   (|<o:Browse>|):invalidate() }, ;
	 refreshAll     := { |o|   (|<o:Browse>|):refreshAll() }, ;
	 setColumn      := { |o,p| (|<o:Browse>|):setColumn(p[1],p[2]) }, ;
	 refreshCurrent := { |o| (|<o:Browse>|):refreshCurrent() }

  ENDCLASS

RETURN NIL
