#include "Clip_Dev.ch"

/*
   Class definition for Odometer.  Demonstrates how to use existing classes
   as building blocks to create higher level classes.  Notice how the class
   ODOMETER is made up of instance variables that are themselves instances
   of another class called DIGIT.  The DIGIT class is defined in the program
   file called 'z_Digit.prg'.

   Also shows how instance variables and methods can be defined for
   internal use only using the IMPORT keyword.  Export instance variables
   that should not be alterable by a user of the object can be declared
   READONLY.

   Finally, notice how the function Increment() referred to in the code
   block for the method 'Increment' is declared STATIC along with the
   supporting function BuildReading().  This enforces encapsulation by
   ensuring that the only way these functions can be called is through
   their associated methods.  Thus, because they are declared STATIC
   they can not be executed by other routines, outside the object, directly.
*/
FUNCTION z_Odometer

   /*
      Will be using class DIGIT in the definition of ODOMETER.
   */
   ANNOUNCE Class:DIGIT FROM LIBRARY SOFTWARE_PERSPECTIVES

   DEFINE Class ODOMETER
      IMPORT VARIABLES Tenths       := |< DIGIT:New() >|, ;
		       Units        := |< DIGIT:New() >|, ;
		       Tens         := |< DIGIT:New() >|
      EXPORT VARIABLE  Reading      := ' 0.0'  READONLY
      EXPORT METHOD    Increment    := { | o | Increment(o) }
      IMPORT METHOD    BuildReading := { | o | BuildReading(o) }
   ENDCLASS

RETURN NIL

/*
   Increment Odometer's Reading by applying the Increment() method to
   each of the Tenths, Units, and Tens Digit objects where necessary.

   Remember that each of Tenths, Units and Tens instance variables are
   DIGIT objects themselves.  Therefore |<o:Tenths>| returns an object
   of type DIGIT.  This means that |< (|<o:Tenths>|):Digit >| returns the
   current value of the Digit instance variable for the object
   |<o:Tenths>|.  The same thing could have been accomplished with the
   slightly longer but perhaps more readable:

   oTenths := |< o:Tenths >|  // Returns an object of type DIGIT and
			      //    assigns it to the variable oTenths
   |< oTenths:Digit >|        // Access the current value of the Digit
			      //    instance variable of the object oTenths

   Remember that the Increment() method being called within this function
   is not THIS Increment() function.  That is, no recursion is occurring
   here.  The Increment() method referred to in this function is that of
   the DIGIT class, as defined in the file 'z_Digit.prg'.  This is an
   excellent example of 'polymorphism' - the idea that different objects
   react to the same message differently.
*/
STATIC FUNCTION Increment( o )
   |< (|<o:Tenths>|):Increment() >|       // Increment |<o:Tenths>|'s Digit.
   IF |< (|<o:Tenths>|):Digit >| = 0      // If Tenths rolled...
      |< (|<o:Units>|):Increment() >|     //    increment |<o:Units>|'s Digit.
      IF |< (|<o:Units>|):Digit >| = 0    // If Units rolled...
	 |< (|<o:Tens>|):Increment() >|   //    increment |<o:Tens>|'s Digit.
      ENDIF
   ENDIF
   |<o:Reading>| := |<o:BuildReading()>|  // Update Reading
RETURN NIL

/*
   BuildReading simply combines the 3 instance variables Tenths, Units,
   and Tens into a string and returns the result.
*/
STATIC FUNCTION BuildReading( o )
   LOCAL ret_val :=  ( 10 * |< (|<o:Tens>|):Digit >|        )  + ;
		     (      |< (|<o:Units>|):Digit >|       ) + ;
		     (      |< (|<o:Tenths>|):Digit >| / 10 )
   // Turn any preceding zeroes into spaces.
   ret_val := STR(ret_val, 4, 1)
RETURN ret_val
