#include "Clip_Dev.ch"
#include "box.ch"

/*
   This file contains the definition and supporting methods for the class
   'Menu'.  This class requires an array of 'MenuOption' objects to create
   a fully recursive menuing system.  While the system could be easily
   extended it serves as a good example of how a well-designed Class can
   be constructed for maximum reusability.
*/
FUNCTION z_Menu
   MEMVAR i                       // Used in the Longest() Method below

   DEFINE CLASS Menu

      EXPORT VARIABLES ;
	 MenuOptions       TYPE |"A"|, ; // An Array of MenuOption objects
	 Top               := 0, ;
	 Left              := 0, ;
	 Color             := SETCOLOR(), ;
	 Horizontal        := .T., ;
	 HorizontalSpacing := 2

      EXPORT METHODS   ;
	 ShowMenu          := { |o| ShowMenu( o ) }

      /*
	 Longest() is a support Method to find the length of the longest
	 MenuOption of a given Menu object.  While ShowMenu() above
	 calls the STATIC function of the same name to perform its task,
	 Longest() is small enough to specify in the code block directly.
	 The first thing it does is initialize the MEMVAR i to 0, then
	 loops through the Labels of each Menu Option in the menu's
	 MenuOption array updating i with the length of the Label if it is
	 greater than i.  Finally, since the code block returns the last
	 value in the expression chain, i is returned containing the result
	 of the calculation.

	 Note that this method is declared IMPORT since it is purely
	 a support function to ShowMenu() below.  It need not have been
	 declared a method at all but is done so to clearly indicate in
	 the Class definition all the supporting functions required.
      */
      IMPORT METHOD ;
	  Longest := { |o| i := 0, ;
			   AEVAL( |<o:MenuOptions>|, ;
				  {|opt| IIF( LEN(|<opt:Label>|) > i, ;
					      i:=LEN(|<opt:Label>|), ;
					      .T. ) } ), ;
			   i }
   ENDCLASS

RETURN NIL



STATIC FUNCTION ShowMenu ( menu )
   LOCAL num_options, option, selection, orgScreen, columns, MenuOptions, ;
	 menuAction, top, left, bottom, right, longestOption, horizspace, ;
	 horizontal, oldcolor

   MenuOptions := |<menu:MenuOptions>|
   num_options := LEN( MenuOptions )
   oldColor    := SETCOLOR( |< menu:Color >| )

   /*
      Because these instance variables are used a lot, copy them to
      local variables.  This speeds things up a bit.
   */
   top        := |<menu:Top>|
   left       := |<menu:Left>|
   horizontal := |<menu:Horizontal>|
   horizspace := |<menu:HorizontalSpacing>|

   /*
      Calculate the 'bottom' and 'right' coordinates of the menu box.
      |<menu:Horizontal>| will only be true the first time this function
      is executed (for the main menu).  From then on all sub-menus are
      displayed vertically.
   */
   IF horizontal
      bottom  := top + 2
      right   := MAXCOL()
      columns := ARRAY( num_options )
   ELSE
      longestOption := |< menu:Longest() >|
      bottom := top + num_options + 1
      right  := left + longestOption + 1
   ENDIF

   /*
      Save the current screen portion that will be affected and draw
      the menu box.
   */
   orgScreen := SAVESCREEN( top, left, bottom, right )
   @ top, left, bottom, right BOX B_SINGLE + CHR(32)

   /*
      Stay in loop until user presses <ESC>.
   */
   selection := 1
   DO WHILE selection <> 0
      @ top + 1, left SAY SPACE(0)
      /*
	 Loop to paint the menu prompts for the menu.  Notice the use of
	 the extended expressions to get the proper MenuOption[] object.
      */
      FOR option := 1 TO num_options
	 IF horizontal
	    columns[option] := COL() + horizspace
	    @ top + 1, columns[option] ;
	       PROMPT |< (MenuOptions[option]):Label >|
	 ELSE
	    @ top + option, left + 1 ;
	       PROMPT Pad( |< (MenuOptions[option]):Label >|, longestOption )
	 ENDIF
      NEXT option

      MENU TO selection                // Get user's menu selection

      IF selection > 0                 // If user did NOT press <ESC>...
	 /*
	    Get the Action of the Menu Option selected.
	 */
	 menuAction := |< (MenuOptions[selection]):Action >|
	 /*
	    If menuAction is a code block - evaluate it - else it must be
	    a 'Menu' object, which indicates a sub-menu.  Therefore,
	    calculate the menu's start co-ords (down and to the right of
	    the currently selected option), and then display it.
	 */
	 IF TVALTYPE( menuAction ) == 'B'
	    EVAL( menuAction )
	 ELSE
	    IF horizontal
	       |< menuAction:Top >|  := top + 2
	       |< menuAction:Left >| := columns[selection]
	    ELSE
	       |< menuAction:Top >|  := selection + top + 1
	       |< menuAction:Left >| := left + 1 + (2 * (longestOption / 3))
	    ENDIF
	    // All Menus except 1st are vertical - must override default .T.
	    |< menuAction:Horizontal >| := .F.
	    |< menuAction:ShowMenu() >|
	 ENDIF
      ENDIF
   ENDDO

   /*
      Put original screen back the way it was.
   */
   RESTSCREEN( top, left, bottom, right, orgScreen )
   SETCOLOR( oldColor )

RETURN NIL
