/*
   This example uses the classes 'Menu' (found in 'z_Menu.prg') and
   'MenuOption' (found in 'z_MenuO.prg') to build a small menu system.
   The Main Menu will display horizontally across the screen while
   all further sub-menus will be displayed vertically.  The system is
   completely automatic in that all you do is specify menu options and
   menu actions (code blocks to be evaluated when the option is selected)
   and it creates the menus with a single message to the main (highest
   level) menu.

   A 'Menu' object is simply a collection of 'MenuOptions'.  A MenuOption
   has a 'Label' and an 'Action'.  The Action can be either a code block
   to evaluate or another 'Menu' object indicating a nested sub-menu. The
   'Save As' sub-menu below demonstrates how to create and insert a sub-
   menu into another sub-menu.  This nesting of menus can be continued
   as deeply as necessary.  You can see how 'MainMenu' contains an
   array of 'MenuOptions' which in turn contains Labels and either an
   Action or more menus.  This is a very good example of using one class
   to define a higher level class.

   While the menus in this example are hard coded in the program, it
   would be possible to drive your menus with these classes from data
   in DBF files.  If this 'data driven' approach were taken the code below
   would be significantly reduced.

   Notice how UDOs can be placed inside of arrays just like other Clipper
   data types.  Also the nesting of UDO bracketing syntax is fully
   supported as the definition of the QUIT menu clearly indicates.
*/

// Compile this example with /n and link in the Clip/++ library

#include "Clip_Usr.ch"

FUNCTION main

   LOCAL FileOptions := ARRAY(5), EditOptions := ARRAY(3), ;
	 MainMenuOptions := ARRAY(3), SaveAsOptions := ARRAY(3), ;
	 File, Edit, Quit, MainMenu, SaveAs

   SET CURSOR OFF

   /*
      Declare the Classes that will be used.
   */
   ANNOUNCE Class:Menu, Class:MenuOption FROM LIBRARY SOFTWARE_PERSPECTIVES

   /*
      Create all Options for SAVE AS Sub menu.
   */
   SaveAsOptions[1] := |< MenuOption:New("ASCII", {|| stub("ASCII")} ) >|
   SaveAsOptions[2] := |< MenuOption:New("Word Perfect",;
	    {|| stub("Word Perfect")} ) >|
   SaveAsOptions[3] := |< MenuOption:New("MS Word",{|| stub("MS Word")} ) >|

   /*
      Create the SAVE AS submenu and give it a different color.
   */
   SaveAs := |< Menu:New( SaveAsOptions ) >|
   |<SaveAs:Color>| := 'W+/BG'

   /*
      Create all Options for FILE sub menu option.
   */
   FileOptions[1] := |< MenuOption:New("Open",{|| stub("Open")} ) >|
   FileOptions[2] := |< MenuOption:New("New",{|| stub("New")} ) >|
   FileOptions[3] := |< MenuOption:New("Save", {|| stub("Save")} )>|
   FileOptions[4] := |< MenuOption:New("Save As", SaveAs) >|
   FileOptions[5] := |< MenuOption:New("Close", {|| stub("Close")} ) >|

   /*
      Create FILE sub menu.
   */
   File := |< Menu:New( FileOptions ) >|

   /*
      Create all options for EDIT sub menu.
   */
   EditOptions[1] := |< MenuOption:New("Copy", {|| stub("Copy")} ) >|
   EditOptions[2] := |< MenuOption:New("Cut", {|| stub("Cut")} ) >|
   EditOptions[3] := |< MenuOption:New("Paste", {|| stub("Paste")} )>|

   /*
      Create EDIT sub menu.
   */
   Edit := |< Menu:New( EditOptions ) >|

   /*
      Create QUIT sub menu in one step since it has only one option.
   */
   Quit := |< Menu:New({ |< MenuOption:New("Quit", {|| quit() } ) >| }) >|

   /*
      Create Main Menu Options.
   */
   MainMenuOptions[1] := |< MenuOption:New("File", File) >|
   MainMenuOptions[2] := |< MenuOption:New("Edit", Edit) >|
   MainMenuOptions[3] := |< MenuOption:New("Quit", Quit) >|

   /*
      Create Main Menu @ row - 0 and column - 0 on the screen
      in a color of Yellow on Blue and with 5 spaces between horizontal
      prompts.
   */
   MainMenu := |< Menu:New( MainMenuOptions, 0, 0, 'GR+/B', .T., 5) >|

   /*
      Activate 'MainMenu'.
   */
   CLEAR SCREEN
   |< MainMenu:ShowMenu() >|

RETURN NIL

FUNCTION stub(x)
   LOCAL tempscrn := SAVESCREEN(20,0,20,79)
   TONE( 100,3 )
   @ 20,10 SAY "The " + UPPER(x) + " module has not yet been constructed."
   INKEY(3)
   RESTSCREEN(20,0,20,79,tempscrn)
RETURN NIL

FUNCTION Quit
   QUIT
RETURN NIL
