/*
   Examples of single Inheritance with Clip/++ over two generations
*/

// Because we use DEFINE CLASS, we need the Developer's header file.
#include "Clip_Dev.ch"

FUNCTION Main
   LOCAL e1, m1

   /*
      Define a class representing a Person.  Notice that we do not assign
      any default values but instead specify a type for each instance
      variable.
   */
   DEFINE CLASS Person
       EXPORT VARIABLES  FirstName TYPE |"C"|,;
			 LastName  TYPE |"C"|,;
			 Age       TYPE |"N"|

       EXPORT METHOD     ListInfo := ;
	  { |o| QOUT( |<o:FirstName>|, ;
		      |<o:LastName>|,  ;
		      |<o:Age>| ) }
   ENDCLASS

   /*
      Define a class representing an Employee of a company.  Since every
      Employee is presumed to be a person, the 'Person' class just created
      is inherited from.  This simply merges the instance variables and
      methods of 'Person' in with those of 'Employee'.  Note that the
      'Employee' variables and methods come first and take precedence over
      any inherited variables and methods.  Thus, the newly defined
      'ListInfo()' method for Employee will replace the 'ListInfo()' method
      for Person.
   */
   DEFINE CLASS Employee INHERIT FROM Person
       EXPORT VARIABLES  Employee_Num TYPE |"C"|

       EXPORT METHOD     ListInfo := ;
	  { |o| QOUT( |<o:Employee_num>|, ;
		      |<o:FirstName>|,    ;
		      |<o:LastName>|,     ;
		      |<o:Age>| ) }
   ENDCLASS

   /*
      Class 'Employee' Inherits all instance variables from Class
      'Person' as displayed by the GetVariables() function.  Note
      how the 'Employee' instance variables come first.
   */
   e1 := |<Employee:NEW("10001-234")>|
   ? "The Instance Variables of the 'Employee' Class are:"
   AEVAL( GetVariables( e1 ), {| x | QOUT( x +" =",|<e1:(x)>| ) } )

   /*
      Define a class representing a Manager of a company.  A Manager is
      both a person and an Employee so we can inherit from 'Employee'
      which itself inherits from 'Person' above.  Note that it is not
      necessary to inherit from both 'Employee' and 'Person' since the
      former already contains all the functionality of the latter.

      The last instance variables and methods to be added to the
      definition of the class are those specifically defined
      for that class (in this case 'Department') so they take ultimate
      precedence in the event of a duplication.  However, Clip/++ places
      these defined variables and methods BEFORE inherited ones in the
      associated variable and methods list.  In this way the New()
      constructor method can still be used to give instance variables
      initial values. This is an example of inheritance over two
      generations.
   */
   DEFINE CLASS Manager INHERIT FROM Employee
       EXPORT VARIABLES Department TYPE |"C"|
   ENDCLASS

   /*
      Use the New() constructor method to initialize ALL instance variables.
      First comes Manager's defined instance variable 'Department', then
      comes Employee's instance variables; 'Employee_Num', 'FirstName',
      'LastName' and 'Age' with the latter three being inherited from
      'Person'.
   */
   m1 := |<Manager:New("Sales","10001-234","Joe","Smith",35) >|

   /*
      Class 'Manager' Inherits all instance variables from Class
      'Employee' as displayed by the GetVariables() function.  Note
      how the 'Manager' instance variables come first.  Also note
      how the 'Employee' instance variables include those inherited
      from the 'Person' class
   */
   ? "The Instance Variables of the 'Manager' Class are:"
   AEVAL( GetVariables( m1 ), {| x | QOUT( x +" =",|<m1:(x)>| ) } )
   ?
   ?
   /*
      The ListInfo() method is the only method belonging to 'Manager'
      and was inherited from 'Employee' taking precedence over the
      similarly named method in 'Person'.
   */
   ? "Manager Information For Dept: " + |<m1:Department>|
   |<m1:ListInfo()>|

RETURN NIL
