/******************************************************************
  CALENDAR.PRG
  A Monthly Calendar function which allows a user to view the
  calendar for an input or current month and select a date.
  Page-Down skips 30 days, Page-Up skips back 30 days.
  Cntr-Page-Down skips ahead 365 days, Cntr-Page-Down skips
  back 365 days.

  By: JP Steffen                      Compuserve: 76525,262
      Leadership Data Services        Telephone:  515-266-0975
      725 Grandview
      Des Moines, Iowa 50316
      
  calendar(dStartDate, nTRow, nLCol, cColorStr) -> dSlctDate

  compile with /n
*******************************************************************/
function calendar(dStartDate, nTRow, nLCol, cColorStr)

#include "Inkey.ch"
#include "Box.ch"
#define MY_HSEP    ""
#define MY_CSEP    "  "
#define MY_COLOR   "N/W, N/BG"
private nTargRow 
private nTargCol

nTRow      := IF(nTRow == NIL, 0, IF(nTRow > 15, 14, nTRow))
nLCol      := IF(nLCol == NIL, 0, IF(nLCol > 30, 28, nLCol))
nBRow      := nTRow + 7
nRCol      := nLCol + 42
dStartDate := IF(dStartDate == NIL, date(), dStartDate)
cColorStr  := IF(cColorStr  == NIL,"B/W,N/BG",cColorStr)
lDone      := .F.
setcolor(cColorStr)
DspCalWin(nTRow, nLCol, nBRow+3, nRCol+2)

WHILE .NOT. lDone
   m1stDay    := firstday(dStartDate)             //  First Day of Mo. (#)
   mLastDate  := lastdate(dStartDate)             //  Last Date of Month
   mWeeksInMo := WeeksInMo(m1stDay,mLastDate)     //  No. of Weeks (rows)
   nTargDay   := day(dStartDate)                  //  Highlite Date 
   aCalArray  := MakCalArr(m1stDay, mLastDate,;   //  Build Calendar Array
                           mWeeksInMo, nTargDay)
   DspCalHead(dStartDate,nTRow+1,nLCol,nRCol)     //  Show Month and Year
   nDaySelct  := CalBrowse(aCalArray, nTRow+2,;   //  Perform Calendar Browse
                      nLCol+1, nBRow+2, nRCol+1)
   nDayMove   := nDaySelct - nTargDay             //  Determine Day Movement
   dStartDate := dStartDate + nDayMove            //  Reset Day of Month
   IF lastkey() == K_RETURN                        
      exit 
   ELSEIF lastkey() == K_PGDN
      dStartDate := movemonth(dStartDate,1)       //  Reset Month Up
   ELSEIF lastkey() == K_PGUP
      dStartDate := movemonth(dStartDate,-1)      //  Reset Month Back
   ELSEIF lastkey() == K_CTRL_PGDN               
      dStartDate := dStartDate + 365              //  Increment Year by 1
   ELSEIF lastkey() == K_CTRL_PGUP
      dStartDate := dStartDate - 365              //  Decrement Year by 1
   ENDIF
END

return dStartDate                                 //  Return Selected Date

/************************************************
   Function DspCalWin
   clear window area and draw box for window
************************************************/
func DspCalWin
parameters Tr,Lc,Br,Rc                      // top row, bot. row, etc
@ Tr,Lc CLEAR TO Br,Rc
@ Tr,Lc,Br,Rc BOX B_DOUBLE_SINGLE
return 

/************************************************
   Function DspCalHead
   create a centered Month and Year String
************************************************/
func DspCalHead
parameters dStartd,nLine,nBeg,nEnd              // Date, Top, Left, Right
nBeg := nBeg + 1
nEnd := nEnd - 1
cStr := upper(trim(cmonth(dStartd)) + " " + ;
        ltrim(str(year(dStartd))))
@ nLine, nBeg CLEAR TO nLine, nEnd
nLineLen := (nEnd-1) - (nBeg+1)
nSpace   := int((nLineLen - len(cStr)+2) / 2)  // Spaces push title to center
@ nLine,nBeg say space(nSpace) + cStr


/************************************************
   Function MakCalArr
   Builds the data structure for the TBrowse in
   CalBrowse.  This is the key to the program &
   can no doubt be done better ie. faster.
************************************************/
function MakCalArr
parameters m1day, mLastd, mWeeks,  mTargd   // mWeeks: number of array rows
mDayOfMo   := 1
private dArray[mWeeks][7]                   
* public nTargRow, nTargCol                  

for r := 1 to mWeeks
    for c := 1 to 7
                                            // store row & col of target day
        if mDayOfMo == mTargd
           nTargRow := r                    // put the browse cursor here
           nTargCol := c
        endif
        if c + (r-1)*7 < m1Day .or. mDayOfMo > mLastD
           dArray[r][c] := "   "
        else
           dArray[r][c] := pad(mDayOfMo,3)    // convert to string w/ len=3
           mDayOfMo = mDayOfMo + 1
        end
    next
next
return dArray


/***************************************************
   pad()
   convert from num., trim, & apply leading space
****************************************************/
   function pad
   parameters In_Num, Out_len
   Num_Len = len(ltrim(str(In_Num))) 
   return space(Out_Len - Num_Len) + ltrim(str(In_Num))


/************************************************
   function movemonth()
   Simply adds or subtracts 30 days from date
   You may want to add more sophistication to
   this to insure new day of month is same as
   current day of month.
   dStartD = Input Date
   nMove   = +1 or -1 (times 30 days)
************************************************/
function movemonth
parameters dStartD, nMove
dStartD := dStartD + (nMove * 30)
return dStartD

/************************************************
   lastdate()
   returns the last date of month for input date
************************************************/
function lastdate
parameters nMth                            // Input Date
nMonthNo := month(nMth)
do case
   case nMonthNo = 1
        return 31
   case nMonthNo = 2
        if day(ctod("02/29/" + ;
           substr(dtoc(nMth),7,2))) <> 0   // Is this leap year?
           return 29
        else
           return 28
        end
   case nMonthNo = 3
        return 31
   case nMonthNo = 4
        return 30
   case nMonthNo = 5
        return 31
   case nMonthNo = 6
        return 30
   case nMonthNo = 7
        return 31
   case nMonthNo = 8
        return 31      
   case nMonthNo = 9
        return 30
   case nMonthNo = 10
        return 31
   case nMonthNo = 11
        return 30
   case nMonthNo = 12
        return 31
endcase

/**************************************************
   firstday()
   returns the day of week for first day of month
***************************************************/
func firstday
parameters nStartD
return dow(nStartD - day(nStartD) + 1)

/**************************************************
   WeeksInMo()
   calculates the number of rows needed for array
**************************************************/
func WeeksInMo
parameters fday,ldate
if ldate == 31                              // 31 day month
   if fday >= 6
      return 6
   else
      return 5
   end
elseif ldate == 30                          // 30 day month
   if fday == 7
      return 6
   else
      return 5
   end
elseif ldate == 29                          // February - leap year
   return 5
elseif ldate == 28                          // February - 28 days
   if fday == 1
      return 4
   else 
      return 5
   end
end

/**************************************************************************
  CalBrowse( <aArray>, <nTop>, <nLeft>, <nBottom>, <nRight> ) --> nDaySelect
  This function adapted from Nantucket Array.prg contains the TBrowse
  implementation
***************************************************************************/
FUNCTION CalBrowse
   PARAMETERS aArray, nT, nL, nB, nR

   LOCAL o			             // TBrowse object
   LOCAL k			             // used in o:SkipBlock
   LOCAL nKey := 0	       // keystroke holder

   PRIVATE n := 1	        // browse row subscript holder
   PRIVATE nACol	         // browse column subscript

   SetCursor( 0 )

   // Create the TBrowse object
   o := TBrowseNew( nT, nL, nB, nR )

   o:headsep   := MY_HSEP
   o:colsep    := MY_CSEP

   // Initialize the TBrowse blocks
   // Note: during browse, the current row subscript is maintained
   // by the blocks in private n
   // LEN(aArray) returns number of rows in array

   o:SkipBlock := { |nSkipVal| SkipFunc( @n, nSkipVal, LEN(aArray)) }
   o:GoTopBlock := { || n := 1 }
   o:GoBottomBlock := { || n := LEN(aArray) }

   // Create TBColumn objects,
   // Initialize data retrieval blocks, and
   // Add to TBrowse object

   FOR nACol = 1 TO LEN( aArray[1] )
    	  o:AddColumn( TBColumnNew(DayHead(nACol), ABlock("aArray[n]", nACol) ) )
   NEXT

   // Postion Cursor to start
   o:ColPos := nTargCol
   o:RowPos := nTargRow

   // Start the event handler loop
   DO WHILE nKey <> K_ESC .AND. nKey <> K_RETURN

      nKey := 0

      // Start the stabilization loop
      DO WHILE .NOT. o:Stabilize()
         nKey := INKEY()
         IF nKey <> 0
            EXIT
         ENDIF
      ENDDO

      IF nKey == 0
         nKey := INKEY(0)
      ENDIF

      // Process the directional keys
      IF o:Stable
         DO CASE
         CASE ( nKey == K_DOWN )
            if n < LEN(aArray) 
               if aArray[ n+1, o:ColPos ] <> "   "
                  o:Down()
               end
            end
         CASE ( nKey == K_UP )
            if n > 1 
               if aArray[ n-1, o:ColPos ] <> "   "
                  o:Up()
               end
            end

         CASE ( nKey == K_RIGHT )
            if o:colPos == 7
               o:down()
               o:home()
            else
               if aArray[ n, o:ColPos + 1] <> "   "
                  o:Right()
               endif
            end
         CASE ( nKey == K_LEFT )
            if o:colPos == 1
               o:up()
               o:end()
            else
               if aArray[ n, o:ColPos - 1] <> "   "
                  o:Left()
               endif
            end

         CASE ( nKey == K_PGDN .or. nKey == K_CTRL_PGDN)
            return val(aArray[ n, o:ColPos ])
         CASE ( nKey == K_PGUP .or. nKey == K_CTRL_PGUP)
            return val(aArray[ n, o:ColPos ])
         CASE ( nKey == K_HOME )
            o:Left()
            o:Up()
         CASE ( nKey == K_END )
            o:Left()
            o:Down()
         ENDCASE
      ENDIF
   ENDDO

   SetCursor( 1 )

   RETURN val(aArray[ n, o:ColPos ])

/*******************************************************************
 SkipFunc
 I don't know about you but I had to dissect the skipblock routine
 in order to understand what it does.
********************************************************************/
static func SkipFunc( n, nSkip_Val, nMaxVal)
local nMove := 0                        // this value will be returned
if nSkip_Val > 0
   do while n + nMove < nMaxVal .and. nMove < nSkip_Val
      nMove++
   enddo
elseif nSkip_Val < 0
   do while n + nMove > 1 .and. nMove > nSkip_Val
      nMove--
   enddo
endif
n := n + nMove
return nMove

/************************************************************************
  ABlock( <cName>, <nSubx> ) -> bABlock

  Given an array name and subscript, return a set-get block for the
  array element indicated.
*************************************************************************/
function ABlock( cName, nSubx )

LOCAL cAXpr

   cAXpr := cName + "[" + LTRIM(STR(nSubx)) + "]"

   RETURN &( "{ |p| IF(PCOUNT()==0, " + cAXpr + "," + cAXpr + ":=p) }" )


/***************************************************
   DayHead
   returns strings to TBColumnNew for column heads
****************************************************/
   function DayHead
   parameters NumDay
   do case
      case NumDay == 1
           return "Sun"
      case NumDay == 2
           return "Mon"
      case NumDay == 3
           return "Tue"
      case NumDay == 4
           return "Wed"
      case NumDay == 5
           return "Thu"
      case NumDay == 6
           return "Fri"
      case NumDay == 7
           return "Sat"
    endcase


