/****************************************************************************

Screen blanking functions for CLIPPER 5.0


compile with Microsoft QuickC 2.0:

   QCL /c /AL /Gs /Oalt /Zl blank.c

link with RTLink for Clipper:

   RTLINK OUTPUT <name> FI <object files>, blank LIB <libs>, llibce


Copyright (c) 1991 KDB Sytems. All rights reserved.


You are free to include these functions in your own programs without
restrictions.  This source code and the compiled version of the source code 
can be freely distributed without charge.  It is strictly  prohibited to
charge for this source code or the the compiled version of the source code.
A small fee, less than $10.00, may be charged to cover the cost of
distribution.  By using this code you assume all responsiblity for any
consequences arising from it's use.

Please report any bugs to:

   Kerry Brown
   KDB Systems
   P.O. Box 445
   Squamish, B.C.
   Canada, V0N 3G0
   
   Compuserve ID: 74545,1262
   
   Bix ID: kerryb

Last Edit: 04/01/91

*****************************************************************************/

#include <dos.h>
#include <string.h>
#include <extend.h>


#define SCREEN_SIZE 4000                  /* 4000 bytes screen size */
#define DEF_GOAL    5400                  /* approx. 5 minutes */
#define MAX_COUNT   65535                 /* maximum unsigned int */
#define TRUE        1
#define FALSE       0


/********** variables, flags, pointers, etc **********/


static unsigned int count = 0;            /* incremented every clock tick */
static unsigned int goal = DEF_GOAL;      /* default goal */
static unsigned char blanked = FALSE;     /* screen already blanked flag */
static unsigned char installed = FALSE;   /* installed flag */
static char far sc_buffer[SCREEN_SIZE];   /* buffer to store old screen */
static char far *screen;                  /* pointer to screen memory */


/********** prototypes for interrupt functions **********/

void (interrupt far *oldtimer)(void);  /* pointer to old timer interrupt */
void (interrupt far *oldkbrd)(void);   /* pointer to old keyboard interrupt */
void interrupt far chk_tick(void);     /* new timer interrupt */
void interrupt far chk_key(void);      /* new keyboard interrupt */


/********** function prototypes **********/

/********** these functions can be called from CLIPPER **********/


CLIPPER sc_blank(void);

/* SYNTAX
       sc_blank([goal])
   ARGUMENTS
       [goal] is a numerical value specifying the number of clock ticks.
       If not specified a default of 5400 is used.  If 0 is specified the screen
       blanker is installed but the screen will not be blanked.  One minute is
       approximately 1080 clock ticks.  If the screen blanker is already
       installed then the timeout value will be reset to [goal]. Maximum value
       is 65,534 minimum is 0.  If [goal] is outside of allowed range [goal]
       will be set to zero.
   RETURNS
       no return value
   DESCRIPTION
       Installs screen blanking function.  Takes over interrupts 09 (keyboard)
       and 1C (timer).  A counter is incremented every clock tick and reset to
       zero every time a key is pressed.  If the counter reaches the number set
       by [goal] the screen is blanked by setting screen memory to 0.  Pressing
       any key will restore the screen.  The key stroke is passed to your
       application.
   WARNING!
       Video memory is directly manipulated.  There is no snow checking for
       older CGA adapters.  It is assumed that video memory is at B000:0000 or
       B800:0000.  The functions will fail miserably if the screen is located
       somewhere else.  A 25 by 80 character screen is assumed.  As with all
       functions which take over interrupts, there may be conflicts with some
       TSR's.  Do not use with graphics modes activated.  If your application
       writes to the screen while the screen is blanked the buffer is not
       updated.  When you press a key the original screen will be restored.  If
       you have a long pause while screen updates may occur use sc_blank(0) to
       temporarily disable screen blanking.
   READ THIS!!!!!!
       If your program exits without uninstalling this function via sc_undo()
       the interrupt vectors will be in an unstable state.  The timer and
       the keyboard interrupts will point to functions that no longer exist.
       You must plan for this by modifying ERRORSYS.PRG or your own error
       handler to include the something like the following:

           IF sc_inst()
               IF !sc_undo()
                   CLOSE ALL
                   // print a warning to the user that the interrupt vectors
                   // are corrupted and possibly reboot the computer after
                   // closing files, etc.
               END
           END

       If something similar to the above is not executed prior to exiting your
       program the computer is almost sure to lock up.



CLIPPER sc_undo(void);

/* SYNTAX
       sc_undo()
   ARGUMENTS
       none
   RETURNS
       .T. if successful, .F. if not
   DESCRIPTION
       Uninstalls the screen blanker.  If interrupts 09 or 1C have been changed
       the screen blanker cannot be uninstalled.
   WARNING!
       This function *MUST* be called before exiting your program. See above. */



CLIPPER sc_inst(void);

/* SYNTAX
       sc_inst()
   ARGUMENTS
       none
   RETURNS
       .T. if sc_blank is installed, .F. if not installed
   DESCRIPTION
       Tests if the screen blanker is installed */


/***************************************************************************/


/********** functions callable from CLIPPER **********/


CLIPPER sc_blank(void)
{
   union REGS r;

   goal = DEF_GOAL;                   /* default goal */
   if (PCOUNT == 1 && ISNUM(1))       /* check for user supplied goal */
   {
       if (_parni(1) < 0 || _parni(1) >= MAX_COUNT)
           goal = 0;                  /* out of range set to zero */
       else
           goal = _parni(1);
   }
   if (!installed)      /* if not installed then install interrupt handlers */
   {
       installed = TRUE;              /* set up basic assumptions */
       count = 0;
       blanked = FALSE;
       r.h.ah = 0x0f;                 /* get video mode */
       int86(0x10, &r, &r);
       if (r.h.al == 7)
       {
           FP_SEG(screen) = 0xB000;   /* monochrome or herc adapter */
           FP_OFF(screen) = 0x0000;
       }
       else
       {
           FP_SEG(screen) = 0xB800;   /* colour adapter */
           FP_OFF(screen) = 0x0000;
       }
       oldtimer = _dos_getvect(0x1c); /* save old interrupt vectors */
       oldkbrd = _dos_getvect(0x09);
       _dos_setvect(0x1c, chk_tick);  /* set new interrupt vectors */
       _dos_setvect(0x09, chk_key);
   }
   _ret();
}



CLIPPER sc_undo(void)
{
   int un_inst_ok;

   if (installed)          /* are we installed? */
   {
       un_inst_ok = TRUE;  /* yes, assume everything's ok */

                           /* do we still own interrupts? */

       if (_dos_getvect(0x1c) == chk_tick && _dos_getvect(0x09) == chk_key)
       {
           _dos_setvect(0x1c, oldtimer);   /* yes, then uninstall */
           _dos_setvect(0x09, oldkbrd);
           installed = FALSE;
       }
       else
           un_inst_ok = FALSE; /* someone else owns interrupts, return false */
   }
   else
       un_inst_ok = FALSE;     /* never was installed, return false */

   _retl(un_inst_ok);
}



CLIPPER sc_inst(void)
{
   _retl(installed);
}



/********** interrupt functions **********/



void interrupt far chk_tick(void)
{
   (*oldtimer)();          /* call original interrupt handler */
   if (blanked == FALSE)   /* is screen blanked? */
   {
       if (++count == MAX_COUNT) /* no, increment counter */
           count = 0;            /* past max. wrap to zero */
       if (count == goal && goal != 0)             /* is the goal reached? */
       {
           memcpy(sc_buffer, screen, SCREEN_SIZE); /* copy screen to buffer */
           memset(screen, 0, SCREEN_SIZE);         /* set screen mem to zeros */
           blanked = TRUE;
       }
   }
}



void interrupt far chk_key(void)
{
   (*oldkbrd)();           /* call original interrupt handler */
   count = 0;              /* reset count */
   if (blanked == TRUE)    /* is screen blanked? */
   {                       /* yes, copy buffer back to screen */
       memcpy(screen, sc_buffer, SCREEN_SIZE);
       blanked = FALSE;
   }
}

