// Windowing Class
// By John D. Van Etten
// Written in CLASSic

#Include "Classic.ch"
#Include "Box.ch"

Begin Class oWin

  Global: // Everybody gets to use these
    Var cBoxType     Type Char            // What type of box?
    Var cBackGround  Type Char            // What type of BackGround?
    Var lCenterH     Type Logical         // Center Horizontal?
    Var lCenterV     Type Logical         // Center Vertical?
    Var nCenterH     Type Numeric         // Center box on which Column?
    Var nCenterV     Type Numeric         // Center box on which Line?
    Var lReSizeH     Type Logical         // Resize the box's width?
    Var lReSizeV     Type Logical         // Resize the box's Height?
    Var cColor       Type Char            // The Color of the text
    Var cBoxColor    Type Char            // the Color of the border
    Var cHeaderColor Type Char            // the Color of the Header
    Var cFooterColor Type Char            // the Color of the Footer
    Var cShadow      Type Char            // Where is the shadow?
    Var lAutoRedraw  Type Logical         // Draw after assigning Text?
    Var lAutoFormat  Type Logical         // Always make everything Fit?
    Var Header       Type Char            // The header
    Var Footer       Type Char            // The footer
    Var cBreaks      Type Char            // Breakes the box to show header/footer
    Var nWidth       Type Numeric         // Set a Width
    Var nHeight      Type Numeric         // Set a Height
    Method Var SetText     Message Text   // Original Text
    Method Var SetWinText  Message WinText// Array of text to put in box
    Method Var ShownText   Message ShownText// Formattd Array of text to put in box
    Method Changed                        // Tell oWin the text has changed!

    Method Init                           // Make New Box
    Method Open                           // Open the Box
    Method Close()                        // Close the Box
    Method ScreenRestore() Message Hide   // Hide the Box
    Method Move                           // Move/Place the box
    Method Move            Message Set    // Move/Place the box
    Method Size                           // Size the box
    Method ReSize                         // Resize the box
    Method ReDraw                         // Redraw the Box
    Method ReDraw          Message Show   // UnHide the Box
    Method Desktop                        // Make the window a desktop
    Method Window()                       // Make the window a window
    Method Line                           // Open the Box
    Method Clear()                        // Clear the Window
    Method Clear()         Message Cls    // Clear the Window
    Method GetRow          Message Row    // Window Row
    Method GetCol          Message Col    // Window Col
    Method SaveColor       Message SetColor  // Defaults to box Color
    Method RestColor                         // Restore the color
    Method ScreenBegin()   Message Begin     // DispBegin()/Cursor off
    Method ScreenEnd()     Message End       // DispEnd()/Restore Cursor Setting

    // You can look, but you can't touch
    Var lOpened READONLY              // Is the Box Open?
    Var nTop    READONLY              // Where is the top of the box?
    Var nLeft   READONLY              // Where is the left side of the box?
    Var nBottom READONLY              // Where is the bottom of the box?
    Var nRight  READONLY              // Where is the right side of the box?

  Class: // No one will need to worry about these
    Var nSetTop                       // Where you wanted the top
    Var nSetLeft                      // Where you wanted the left
    Var nSetBottom                    // Where you wanted the bottom
    Var nSetRight                     // Where you wanted the right
    Var aScreenSave                   // Screen Save
    Var lFormat                       // Is formatting needed
    Var aDesktop                      // Save Window information
    Var cOldColor                     // Save the SetColor
    Var nCursor                       // Save Cursor Setting

    Method ScreenSave()               // Save Screen under box
    Method ScreenRestore()            // Restore Screen
    Method Process                    // Process a Text Line
    Method Shadow                     // Draw a shadow
    Method Format                     // Format box to show text
    Method FormatText()               // Format the text to fit box
    Method AssignText                 // Set Text without redrawing
    Method TextProcess                // Process text array

End Class

Method Init( xText := {}, ;
             cHeader, cHColor, cFooter, cFColor, ;
             cColor := "W+/gb", ;
             cBoxColor, cShadow := "BR", ;
             nWidth, nHeight, ;
             nTop, nLeft, nBottom, nRight )

  // Make sure parameters are OK
  // This is an example of assumption checking
  Check xText     is CHAR, ARRAY
  Check cHeader   is CHAR, NIL
  Check cHColor   is CHAR, NIL
  Check cFooter   is CHAR, NIL
  Check cFColor   is CHAR, NIL
  Check cColor    is CHAR
  Check cBoxColor is CHAR, NIL
  Check cShadow   is CHAR
  Check nWidth    is NUMERIC, NIL
  Check nHeight   is NUMERIC, NIL
  Check nTop      is NUMERIC, NIL
  Check nLeft     is NUMERIC, NIL
  Check nBottom   is NUMERIC, NIL
  Check nRight    is NUMERIC, NIL

  ::cColor       := cColor
  ::cBoxColor    := cBoxColor
  ::cHeaderColor := cHColor
  ::cFooterColor := cFColor
  ::cShadow      := cShadow
  ::Header       := cHeader
  ::Footer       := cFooter

  // Object Defaults
  ::cBoxType     := "B"
  ::cBackGround  := ""
  ::lOpened      := .f.
  ::lAutoRedraw  := .t.
  ::lAutoFormat  := .t.
  ::Text         := xText

  // Place the window
  ::Move( nTop, nLeft, nBottom, nRight, nWidth, nHeight )
Return( Self )


// Open the window.
Method Open( xText )
  ::lOpened := .t.

  if xText != NIL
    ::AssignText( xText ) // They sent use new text!
  endif

  // Size then Open the Window
  ::ReSize()
  ::ReDraw()
Return( Self )


// Close the window.
Method Close()
  // See-Ya
  ::lOpened := .f.
  ::ScreenRestore()
Return( Self )


// Handles the text shown in the window.
// Holds a processed version of the assign value.
Method Var SetWinText( xText )
  If Set // The text has changed
    If ValType( xText ) == "C"
      xText := { xText } // Someone sent a string instead of an array!
    Else
      xText := aClone( xText )
    Endif
    ClassVar := ::TextProcess( xText ) // Pre-Process the array

    if ::lAutoRedraw .and. ::lOpened
      ::ReSize()
      ::ReDraw()
    endif
  Endif

Return( ClassVar )


// Handles the text shown in the window.
// Holds a unchanged copy of the assign value.
Method Var SetText( xText )
  Local aText := {}
  Local nLines
  Local nLine

  if Set // Someone changed the text
    ClassVar  := xText
    ::WinText := xText
  Endif
Return( ClassVar )

Method Var ShownText()
  Local aText := {}
  Local nCounter
  Local nWidth
  Local nHeight

  if full( ::nRight ) .and.  full( ::nLeft ) .and. ;
     full( ::nBottom ) .and. Full( ::nRight ) .and. ;
     ISARRAY( ::WinText )
    nWidth  := ::nRight  - ::nLeft - 3
    nHeight := ::nBottom - ::nTop  - 1
    For nCounter := 1 to nHeight
      if len( ::WinText ) < nCounter
        aadd( aText, space( nWidth ))
      else
        aadd( aText, ::Process( ::WinText[ nCounter ], nWidth ))
      endif
    Next
  endif
Return( aText )

// Sets the size and place of the box.
// All parms. are optional.
// Setting the top and bottom to the same line will make the window
//  center on that line.  Same with left and right.
//
Method Move( nTop, nLeft, nBottom, nRight, nWidth, nHeight )

  // Should We Center?
  ::nSetTop    := nTop
  ::nSetLeft   := nLeft
  ::nSetBottom := nBottom
  ::nSetRight  := nRight
  //::nWidth     := nWidth
  //::nHeight    := nHeight

  ::lCenterV  := ( nTop == nBottom )              // Center Vertical
  ::nCenterV  := iif( ::lCenterV, nTop, NIL )     // Center on which Line

  ::lCenterH  := ( nLeft == nRight )              // Center Horiz.
  ::nCenterV  := iif( ::lCenterH, nLeft, NIL )    // Center on which Col

  ::lAutoFormat := ( nTop == NIL .or. nBottom == NIL )

  ::Size( nWidth, nHeight )
Return( Self )

Method Size( nWidth, nHeight )

  // Should We Center?
  ::nWidth     := nWidth
  ::nHeight    := nHeight

  If nHeight != NIL .and. ::nSetTop != NIL .and. ::nSetBottom != NIL
    ::nSetBottom := NIL
  Endif
  If nWidth != NIL .and. ::nSetLeft != NIL .and. ::nSetRight != NIL
    ::nSetRight := NIL
  Endif

  ::lReSizeV  := ( ::nSetTop  == NIL .or. ::nSetBottom == NIL ) .and. ;
                 ( ::nHeight == NIL )             // Resize the height?
  ::lReSizeH  := ( ::nSetLeft == NIL .or. ::nSetRight  == NIL ) .and. ;
                 ( ::nWidth == NIL )              // Resize the width?

  if ::lOpened
    ::Open() // Redraw the box
  endif
Return( Self )



// Resize the window to fit the text.
Method Format( nTop    := ::nSetTop, ;
               nLeft   := ::nSetLeft, ;
               nBottom := ::nSetBottom, ;
               nRight  := ::nSetRight )
  Local nMaxLen  := 0
  Local aText    := ::WinText
  Local nCenterH := ::nCenterH
  Local nCenterV := ::nCenterV
  Local nWidth   := ::nWidth
  Local nHeight  := ::nHeight
  Local lDoFormat := ::lFormat
  Local aSavePlace := { nTop, nLeft, nBottom, nRight }

  Default nCenterH to MaxCol() / 2
  Default nCenterV to MaxRow() / 2
  ::lFormat := .f.

  // Make Sure their is a place
  Default ::nTop to 1
  Default ::nLeft to 1
  Default ::nBottom to MaxRow() - 1
  Default ::nRight to MaxCol() - 1

  if ::lCenterV
    If nHeight == NIL
      nMaxLen := Len( aText ) + 2

      if nMaxLen + 3 > MaxRow()
        ::lFormat := .t.
      Endif
    Else
      nMaxLen := nHeight + 2
    Endif

    nTop    := max( nCenterV - nMaxLen / 2 - 1, 0 )
    nBottom := min( nTop + nMaxLen + 1, MaxRow() )
  elseif ::lReSizeV
    nMaxLen := Len( aText ) + 2

    Default nTop    to ::nTop
    Default nBottom to ::nBottom

    If ::nSetTop == NIL
      nTop := max( nBottom - nMaxLen - 1, 0 )
    else
      nBottom := min( nTop + nMaxLen + 1, MaxRow() )
    Endif
  Else
    Default nTop to ::nTop
    Default nBottom to ::nBottom
  endif

  if ::lCenterH
    If nWidth == NIL
      aEval( aText, ;
            {| cText | nMaxLen := max( nMaxLen, ::Process( cText ))})

      If Full( ::Header )
        nMaxLen := Max( nMaxLen, Len( ::Header ) + 2 )
      Endif
      If Full( ::Footer )
        nMaxLen := Max( nMaxLen, Len( ::Footer ) + 2 )
      Endif

      if nMaxLen + 3 > MaxCol()
        ::lFormat := .t.
      Endif
    Else
      nMaxLen := nWidth
    Endif

    nLeft  := max( nCenterH - nMaxLen / 2 - 2, 0 )
    nRight := min( nLeft + nMaxLen + 3, MaxCol() )
  elseif ::lReSizeH
    aEval( aText, ;
          {| cText | nMaxLen := max( nMaxLen, ::Process( cText ))})

    If Full( ::Header )
      nMaxLen := Max( nMaxLen, Len( ::Header ) + 2 )
    Endif
    If Full( ::Footer )
      nMaxLen := Max( nMaxLen, Len( ::Footer ) + 2 )
    Endif

    Default nLeft  to ::nLeft
    Default nRight to ::nRight

    If nRight != NIL .and. ::nSetLeft == NIL
      nLeft := max( nRight - nMaxLen - 3, 0 )
    else
      nRight := min( nLeft + nMaxLen + 3, MaxCol() )
    Endif
  else
    Default nLeft to ::nLeft
    Default nRight to ::nRight
  Endif

  // AUTO FORMAT
  if ::lAutoFormat .and. !::lFormat

    Default nTop    to 0
    Default nLeft   to 0
    Default nBottom to nTop
    Default nRight  to nLeft

    if nLeft < 0
      nLeft := 0
      ::lFormat := .t.
    endif
    if nRight > MaxCol()
      nRight := MaxCol()
      ::lFormat := .t.
    endif
    if nTop < 0
      nTop := 0
      ::lFormat := .t.
    endif
    if nBottom > MaxRow()
      nBottom := MaxRow()
      ::lFormat := .t.
    endif

    If !::lFormat
      nMaxLen := 0
      aEval( aText, ;
            {| cText | nMaxLen := max( nMaxLen, ::Process( cText ))})

      If Full( ::Header )
        nMaxLen := Max( nMaxLen, Len( ::Header ) + 2 )
      Endif
      If Full( ::Footer )
        nMaxLen := Max( nMaxLen, Len( ::Footer ) + 2 )
      Endif
      nMaxLen := Min( Max( nMaxLen, ;
                             iif( nWidth == NIL, 0, nWidth )), ;
                      MaxCol() - 1 )

      If nRight < MaxCol()
        if full( nWidth )
          if nRight != nLeft + nWidth + 3
            nRight := Min( nLeft + nWidth + 3, MaxCol())
            ::lFormat := .t.
          endif
        elseif nRight != nLeft + nMaxLen + 3
          nRight := Min( nLeft + nMaxLen + 3, MaxCol())
          ::lFormat := .t.
        endif
      Endif
      If !::lFormat .and. nLeft > 0 .and. nLeft + nMaxLen + 3 > MaxCol()
        if full( nWidth ) .and. nRight != nLeft + nWidth + 3
          nLeft := Max( nRight - nWidth - 3, 0 )
          ::lFormat := .t.
        elseif nBottom < MaxRow()
          nBottom++
          ::lFormat := .t.
        elseif nTop > 0
          nTop--
          ::lFormat := .t.
        Elseif empty( nWidth )
          nLeft := Max( MaxCol() - nMaxLen - 3, 0 )
          ::lFormat := .t.
        endif
      Endif

      If !::lFormat
        nMaxLen := Min( Max( Len( aText ), ;
                             iif( nHeight == NIL, 0, nHeight )), ;
                        MaxRow() - 1 ) + 2
        if nBottom < MaxRow() .and. nTop + nMaxLen + 1 != nBottom
          nBottom := Min( nTop + nMaxLen + 1, MaxRow() )
          ::lFormat := .t.
        endif
        if nTop > 0 .and. nTop + nMaxLen + 1 > MaxRow()
          nTop := max( MaxRow() - nMaxLen - 1, 0 )
          ::lFormat := .t.
        endif
      Endif
    Endif
  Endif

  Default nWidth to Int( nRight ) - Int( nLeft ) - 3
  Default nHeight to Int( nBottom ) - Int( nTop ) - 3

  ::nTop    := int( nTop )
  ::nLeft   := int( nLeft )
  ::nBottom := Int( Min( nTop + nHeight + 3, MaxRow() ))
  ::nRight  := Int( Min( nLeft + nWidth + 3, MaxCol() ))

  If !::lFormat .and. ;
     full( aSavePlace[ 1 ] ) .and. full( aSavePlace[ 2 ] ) .and. ;
     full( aSavePlace[ 3 ] ) .and. full( aSavePlace[ 4 ] )
     ::lFormat := ( ::nTop    != aSavePlace[ 1 ] ) .or. ;
                  ( ::nLeft   != aSavePlace[ 2 ] ) .or. ;
                  ( ::nBottom != aSavePlace[ 3 ] ) .or. ;
                  ( ::nRight  != aSavePlace[ 4 ] )
  Endif
Return( Self )


// Draw the window on the screen
// Use lQuick to only redraw the text.
Method ReDraw( aText, lQuick := .f. )
  Local cColor      := SetColor()
  Local nWidth      := ::nRight  - ::nLeft - 3
  Local nHeight     := ::nBottom - ::nTop  - 1
  Local nTop        := ::nTop
  Local nLeft       := ::nLeft
  Local nBottom     := ::nBottom
  Local nRight      := ::nRight
  Local nRow        := 0
  Local nCol        := nLeft + 2
  Local cHeader
  Local nCenterHead
  Local cBreaks     := ::cBreaks
  Local cHeadColor  := ::cHeaderColor
  Local cFooter
  Local nCenterFoot
  Local cFootColor  := ::cFooterColor
  Local nCounter
  Local nCursor
  Local cBoxType    := ::cBoxType
  Local cBoxColor   := ::cBoxColor

  If aText != NIL
    ::AssignText( aText )
  Endif
  aText := ::WinText

  DispBegin()
  nCursor := SetCursor( 0 )

  if !lQuick .or. !::lOpened

    if !::lOpened
      ::lOpened := .t.
      ::Resize()
    endif

    Default cBoxColor to ::cColor

    If ::aDesktop == NIL
      Default cHeadColor to cBoxColor
      Default cFootColor to cBoxColor
    else
      Default cHeadColor to "w+/rb"
      Default cFootColor to "w+/rb"
    Endif

    ::ScreenRestore()
    ::ScreenSave()

    SetColor( ::cColor )
    @nTop, nLeft Clear to nBottom, nRight
    SetColor( cBoxColor )
    if cBoxType == ""
      cBreaks := ""
      ::Clear()
    elseif Upper( cBoxType ) == "S"
      cBreaks := "[]"
      @nTop, nLeft,nBottom, nRight Box B_SINGLE + ::cBackGround
    elseif Upper( cBoxType ) == "D"
      cBreaks := "[]"
      @nTop, nLeft, nBottom, nRight Box B_DOUBLE + ::cBackGround
    elseif Upper( cBoxType ) == "B" .and. MaxRow() > 24
      cBreaks := ""
      @ nTop, nLeft, nBottom, nRight Box "" + ::cBackGround
    elseif Upper( cBoxType ) == "B"
      cBreaks := ""
      @ nTop, nLeft, nBottom, nRight Box "" + ::cBackGround
    elseif len( cBoxType ) == 1
      Default cBreaks to "[]"
      @ nTop, nLeft, nBottom, nRight ;
        Box Replicate( cBoxType, 8 ) + ::cBackGround
    elseif len( cBoxType ) == 2
      Default cBreaks to "[]"
      @ nTop, nLeft, nBottom, nRight ;
        Box Replicate( left( cBoxType, 1 ), 8 ) + right( cBoxType, 1 )
    else
      Default cBreaks to "[]"
      @nTop, nLeft, nBottom, nRight Box Left( cBoxType + ::cBackGround, 9 )
    Endif

    SetColor( ::cColor )

    If Full( ::Header )
      cHeader     := Left( ::Header, nWidth - 2 )
      if ::aDesktop == NIL
        nCenterHead := Int( nWidth / 2 - len( cHeader ) / 2 )
        @ nTop, nLeft + nCenterHead ;
          say Left( cBreaks, 1 ) Color cBoxColor
        @ nTop, nLeft + nCenterHead + Len( cHeader ) + 3;
          say Right( cBreaks, 1 ) Color cBoxColor
        @ nTop, nLeft + nCenterHead + 1 ;
          say " " + cHeader +" " Color cHeadColor
      Else
        @ nTop, nLeft say padc( cHeader, nRight - nLeft + 1 ) Color cHeadColor
      Endif
    Endif

    If Full( ::Footer )
      cFooter     := Left( ::Footer, nWidth - 2 )
      if ::aDesktop == NIL
        nCenterFoot := Int( nWidth / 2 - len( cFooter ) / 2 )
        @ nBottom, nLeft + nCenterFoot ;
          say Left( cBreaks, 1 ) Color cBoxColor
        @ nBottom, nLeft + nCenterFoot + Len( cFooter ) + 3;
          say Right( cBreaks, 1 ) Color cBoxColor
        @ nBottom, nLeft + nCenterFoot + 1 ;
          say " " + cFooter +" " Color cFootColor
      Else
        @ nBottom, nLeft say padc( cFooter, nRight - nLeft + 1 ) Color cFootColor
      Endif
    Endif
  Else
    SetColor( ::cColor )
    ::Clear()
  Endif

  For nCounter := 1 to Min( nHeight, Len( aText ))
    nRow := nTop + nCounter + 1
    If Full( aText[ nCounter ] )
      @nRow, nCol say RTrim( ::Process( aText[ nCounter ], nWidth ))
    Endif
  Next

  DispEnd()
  SetCursor( nCursor )
  Setcolor( cColor )
Return( Self )

// Save the screen, then draw the shadow
Method ScreenSave()
  Local nTop     := ::nTop
  Local nLeft    := ::nLeft
  Local nBottom  := ::nBottom
  Local nRight   := ::nRight
  Local cShadow  := ::cShadow
  Local nSTop    := nTop
  Local nSLeft   := nLeft
  Local nSBottom := nBottom
  Local nSRight  := nRight

  if Full( cShadow )
    If "T" $ upper( cShadow )
      if nSTop != 0
        nTop--
      endif
      nSTop--
      nSBottom--
    elseIf "B" $ upper( cShadow )
      if nSBottom != MaxRow()
        nBottom++
      endif
      nSTop++
      nSBottom++
    endif
    If "L" $ upper( cShadow )
      if nSLeft != 0
        nLeft--
      endif
      nSLeft--
      nSRight--

      if maxrow() <= 24
        if nLeft != 0
          nLeft--
        endif
        nSLeft--
        nSRight--
      endif
    elseIf "R" $ upper( cShadow )
      if nSRight != MaxCol()
        nRight++
      endif
      nSLeft++
      nSRight++
      if maxrow() <= 24
        if nRight != MaxCol()
          nRight++
        endif
        nSLeft++
        nSRight++
      endif
    endif
  endif

  ::aScreenSave := { nTop, nLeft, nBottom, nRight ,;
               SaveScreen( nTop, nLeft, nBottom, nRight ) }
  if Full( cShadow )
    ::shadow( nSTop, nSLeft, nSBottom, nSRight, 8 )
  endif
Return( Self )

Method ScreenRestore()
  Local aScreenSave := ::aScreenSave

  If Full( aScreenSave )
    RestScreen( aScreenSave[ 1 ], aScreenSave[ 2 ], aScreenSave[ 3 ], ;
                aScreenSave[ 4 ], aScreenSave[ 5 ] )
    ::aScreenSave := {}
  Endif
Return( Self )

Method Shadow( nTop, nLeft, nBottom, nRight, nAttr := 8 )
  Local cColor := SetColor( "n/n")

  @ nTop, nLeft clear to nBottom, nRight
  // RestScreen( nTop, nLeft, nBottom, nRight, ;
  //             ReplAttr( ;
  //             SaveScreen( nTop, nLeft, nBottom, nRight ), nAttr ))

  SetColor( cColor )
Return( Self )

Method ReSize( xText )

  if xText != NIL
    ::AssignText( xText )
  else
    ::AssignText( ::Text )
  Endif

  ::lFormat := .f.
  ::Format()
  Do While ::lAutoFormat .and. ::lFormat
    ::FormatText()
    ::Format( ::nTop, ::nLeft, ::nBottom, ::nRight )
  Enddo
Return( Self )

Method FormatText()
  Local nWidth      := Max( ::nRight - ::nLeft - 3, 1 )
  Local aFormatText := {}
  Local aText       := ::Text
  Local nCount
  Local nLine
  Local nLines
  Local cText
  Local cCmd

  If ISCHAR( aText )
    aText := { aText }
  Endif

  For nCount := 1 to len( aText )
    cText := aText[ nCount ]
    cCmd := left( cText, 3 )
    If < cCmd == "~L,", "~R," > // Continuation lines also need these!
      cText := substr( cText, 4 )
    else
      cCmd := ""
    Endif
    nLines := mlcount(cText, nWidth )
    if nLines == 0
      aadd( aFormatText, "" )
    else
      For nLine := 1 to nLines
        aadd( aFormatText, cCmd + rtrim( memoline(cText, nWidth, nLine,, .t. )))
      Next
    endif
    If Full( ::Header )
      ::Header := Left( ::Header, nWidth - 2 )
    Endif
    If Full( ::Footer )
      ::Footer := Left( ::Footer, nWidth - 2 )
    Endif
  Next

  ::AssignText( aFormatText, .t. )
Return( Self )

Method AssignText( xText, lSetWinText := .f. )
  Local lAutoRedraw := ::lAutoRedraw

  // Turn off AutoRedraw only to Assign Text
  ::lAutoRedraw := .f.

  if lSetWinText
    ::WinText  := xText
  Else
    ::Text := xText
  endif

  ::lAutoRedraw := lAutoRedraw
Return( Self )

Method Process( cText, nWidth )
  Local xRet     := 0
  Local lString  := ( nWidth != NIL )
  Local cHeader  := ""

  If left( upper( cText ), 3 ) == "~L,"
    If lString
      xRet := Left( Substr( cText, 4 ), nWidth )
    else
      xRet := len( cText ) - 3
    Endif
  Elseif left( upper( cText ), 3 ) == "~R,"
    If lString
      xRet := Padl( Substr( cText, 4 ), nWidth )
    else
      xRet := len( cText ) - 3
    Endif
  Elseif Len( cText ) > 1 .and. ;
          left( cText, 1 ) == "~" .and. Right( cText, 1 ) == "~"
    cHeader := "[ " + Substr( left( cText, len( cText ) - 1 ), 2 ) + " ]"
    If lString
      xRet := Padc( cHeader, nWidth, "" )
    else
      xRet := Len( cHeader )
    Endif
  Elseif left( cText, 1 ) == "~"
    If lString
      xRet := Replicate( "", nWidth )
    else
      xRet := 0
    Endif
  Else
    If lString
      cText := Left( cText, nWidth )
      xRet  := Space( int( nWidth / 2 - len( cText ) / 2 )) + cText
    Else
      xRet := Len( cText )
    endif
  Endif
Return ( xRet )

Method Desktop( cBackGround, cColor )

  if ::aDesktop == NIL
    Default cBackGround to ""
    Default cColor to ::cColor

    ::aDesktop := { ::nSetTop, ::nSetLeft, ::nSetBottom, ::nSetRight, ;
                    ::cBoxType, ::cBackGround, ::cColor }

  endif

  ::Move( 0, 0, MaxRow(), MaxCol() )
  ::cBoxType    := ""
  ::cBackGround := cBackGround
  ::cColor      := cColor
Return( Self )

Method Window()

  if ::aDesktop != NIL
    ::Move( ::aDesktop[01], ::aDesktop[02], ::aDesktop[03], ::aDesktop[04] )
    ::cBoxType    := ::aDesktop[05]
    ::cBackGround := ::aDesktop[06]
    ::cColor      := ::aDesktop[07]
    ::aDesktop    := NIL

    ::ReSize()
  endif
Return( Self )

Method Clear()
  Local nHeader := iif( empty( ::Header ), 0, 1 )
  Local nFooter := iif( empty( ::Footer ), 0, 1 )
  Local cColor := SetColor()
  Local nRow
  Local cBack
  Local nLeft   := ::nLeft
  Local nBottom := ::nBottom
  Local nWidth  := ::nRight - ::nLeft + 1


  SetColor( ::cColor )
  if full( ::cBackGround )
    if full( ::aDesktop )
      if len( ::cBackGround ) == 1
        @ ::nTop + nHeader ,nLeft, nBottom - nFooter, ::nRight ;
          Box Replicate( ::cBackGround, 9 )
      else
        cBack := Replicate( ::cBackGround, ;
                            nWidth + Len( ::cBackGround ))
        DispBegin()
        For nRow := ::nTop + nHeader to nBottom - nFooter
          @nRow, nLeft say Substr( cBack, ;
                                   ( nRow % Len( cBack )) + 1 )
        Next
        DispEnd()
      endif
    else
      @ ::nTop + 1, nLeft + 1, nBottom - 1, ::nRight - 1 Box Replicate( ::cBackGround, 9 )
    endif
  else
    @ ::nTop + 1, nLeft + 1 clear to nBottom - 1, ::nRight - 1
  endif
  Setcolor( cColor )
Return (Self)

Method GetRow( nRow := 1 )
Return( ::nTop + nRow + 1 )

Method GetCol( nCol := 1 )
Return( ::nLeft + nCol + 1 )

Method TextProcess( aText )
  Local nCount
  Local nJump
  Local nLoop

  For nCount := 1 to Len( aText )
    if ValType( nJump := aText[ nCount ] ) == "N"
      aText := aSize( aDel( aText, nCount ), Len( aText ) - 1 )
      if nCount < nJump
        aText := aSize( aText, Len( aText ) + ( nJump - nCount ))
        For nLoop := nCount to nJump - 1
          aText := aIns( aText, nCount )
          aText[ nCount ] := ""
        Next
      Endif
    endif
  Next
Return( aText )

Method SaveColor( cColor := ::cColor )
  ::cOldColor := SetColor( cColor )
Return(Self)

Method RestColor( cColor := ::cOldColor )
  if !empty( cColor )
    SetColor( cColor )
  endif
Return(Self)

Method ScreenBegin()
  ::nCursor := SetCursor(0)
  DispBegin()
Return (Self)

Method ScreenEnd()
  DispEnd()
  SetCursor( ::nCursor )
Return (Self)

Method Line( nRow := 1, cChar := "" )
  @ ::nTop + nRow + 1, ::nLeft + 1 say replicate( cChar, ::nRight - ::nLeft - 1 )
Return (Self)

Method Changed()
  ::Text := ::Text
Return( Self )
