/**\
 Program:  chDBF - creates #define statements for fields in your DBFs      
  Author:  Ryan Katri                                                      
    Date:  June 28, 1991                                                   
\**/

#include <stdio.h>
#include <ctype.h>
#include "wild.h"


#define TRUE  1
#define FALSE 0
#define HEADER "chDBF v1.0   Copyright (c) 1991 by Ryan Katri, all rights reserved.\n\n"

void ShowHelp(void);
void new_ext(char filename[], char ext[4]);

main(int argc, char *argv[])
{
  /* Header structure used in .dbf files.  */
  struct header_rec {
    unsigned char version;
    unsigned char year;
    unsigned char month;
    unsigned char day;
    long num_recs;
    unsigned int length;
    unsigned int rec_length;
    unsigned char reserved[20];
  } header;


  /* Field structure in .dbf files */
  struct field_rec {
    char    name[11];                  /*field name*/
    char    type;                      /*field type*/
    unsigned char   *data_ptr;         /*pointer into buffer*/
    unsigned char   *xdata_ptr;        /*remove if large model*/
                                       /*ignore if small model*/
    unsigned char   width;             /*field width*/
    unsigned char   dec;               /*field decimal point*/
    unsigned char   fill[14];          /*not used by us */
  } field;

  FILE *fp, *fout;
  char filepath[80], afiles[40][80], outfile[80];
  char prefix = ' ', globalCH = FALSE, overwrite = FALSE;
  char *parameter;
  int fldcnt, i, j, filecnt = 0;

  printf(HEADER);

  /* if no filename specified, print help */
  if (argc == 1) {
    ShowHelp();
    return;
  }

  /* process all parameters on command line */
  for (i = 1; i < argc; i++) {
    parameter = (char *)strdup(argv[i]);

    /* process options */
    if (*parameter == '/') {
      switch (toupper(*(++parameter))) {
        case 'P': prefix = *(++parameter);         /* define prefix */
                  break;
        case 'O': overwrite = TRUE;                   /* set overwrite on */
                  break;
        case 'F': strcpy(outfile, ++parameter);    /* define header file */
                  if (!strchr(outfile, '.'))
                    new_ext(outfile, ".CH");
                  globalCH = TRUE;
                  break;
      }
    }
    else
    {
      /* it's not an option, so must be a filename */
      strcpy(filepath, parameter);
      /* if no extension on the filename, append 'DBF'  */
      if (!strchr(filepath, '.'))
        new_ext(filepath, ".DBF");

      /* find all files matching filespec, and add to file array */
      while (getfile(filepath, afiles[filecnt])) filecnt++;
    }
  }

  if (globalCH) {
    if ((fout = fopen(outfile, overwrite ? "wt" : "at")) == NULL) globalCH = 0;
    else fprintf(fout, "\n/* %s */\n", strupr(outfile));
  }


  /* PROCESS ALL THE FILES LISTED IN THE ARRAY */
  for (i = 0; i < filecnt; i++) {
    if (!globalCH) {
      strcpy(outfile, afiles[i]);
      new_ext(outfile, ".CH");

      if ((fout = fopen(outfile, overwrite ? "wt" : "at")) == NULL) continue;
    }

    /* open the .dbf file for reading */
    if ((fp = fopen(afiles[i], "rb")) != NULL) {
      printf("%-12s --> ", afiles[i]);
      fread(&header, sizeof(char), sizeof(header), fp);
      if (!(header.version == 3 || header.version == 0x83)) {
        printf("Not a valid .dbf file");
        fclose(fp);
        continue;
      }

      if (!globalCH)
        fprintf(fout, "\n/* %s  (for %s) */\n", strupr(outfile), afiles[i]);
      else
        fprintf(fout, "\n// %s\n", afiles[i]);
      fprintf(fout, "                         // Type Len Dec\n");

      fldcnt = (header.length-1)/32-1;
      for (j = 0; j < fldcnt; j++) {
        fread(&field, sizeof(char), sizeof(field), fp);

        fprintf(fout, "#define %c%-10s %3d  //  %c  %3d  %2d\n", prefix,
                                                               field.name, j+1,
                                                               field.type,
                                                               field.width,
                                                               field.dec);
      }
    }
    fprintf(fout, "\n");
    fclose(fp);
    if (!globalCH) fclose(fout);
    printf("preprocessor info written to %s\n", strupr(outfile));
  }
  fclose(fout);
  if (filecnt == 0) printf("No .DBF files found matching filespec.\n");

  return;
}

void ShowHelp(void)
{
  printf("     CHDBF filespec [filespec...] [options]\n\n");
  printf("Options:\n");
  printf("     /P<prefix>     Prefix for preprocessor field names\n");
  printf("     /F<file>       Header output file\n");
  printf("     /O<file>       Overwrite existing header files (otherwise append)\n\n");
  printf("Examples:\n");
  printf("     CHDBF clients        (process clients.dbf and send output to clients.ch)\n");
  printf("     CHDBF clients /ftest (process clients.dbf and send output to test.ch)\n");
  printf("     CHDBF r* /ftest      (process all r*.dbf files and output to test.ch)\n");
  printf("     CHDBF r*             (process all r*.dbf files and output to r*.ch)\n");
  printf("     CHDBF clients /p_    (process clients.dbf, prefix fieldnames with '_')\n");
  printf("     CHDBF clients prices (process clients.dbf and prices.dbf)\n");
  printf("     CHDBF clients /o /pm (overwrite existing files and use prefix 'm')\n\n");
  return;
}

/*-------------------------------------------------------------------------
   RESULTS: filename is given a new extension
-------------------------------------------------------------------------*/
void new_ext(char filename[], char *ext)
{
  int i;
  for (i = 0; filename[i] && filename[i] != '.'; i++);
  for (; *ext; ext++, i++) filename[i] = *ext;
  filename[i] = '\0';
}

