*****************************************************************************
*  File     : CDDERR.PRG Version 1.03 for Clipper 5.01a
*****************************************************************************
*
*  Purpose  : CDDERR.PRG contains the replacement error handling
*             function V_E_ALT().
*
*             Compile CDDERR.PRG with the following command line:
*
*                CLIPPER CDDERR  /N
*
*  Author...: Vick Perry - Phoenix Systems, Inc.
*             Copyright 1992, 1993 Phoenix Systems, Inc.
*
*  Revisions:
*
*****************************************************************************
#include "cddlib.ch"           // CDDLIB's header


*****************************************************************************
*  Function : V_E_ALT()
*****************************************************************************
*
*  Purpose  : Error handler for CDD functions.  Error messages
*             are displayed and written to the ERROR.TXT text file.
*             The DOS SET USERID=???? command will optionally provide
*             user identification.
*
*  Syntax   : V_E_ALT(<objErr>) --> recover value
*
*  Arguments: <objErr> - The current error object
*
*  Returns  : A recovery value.  The data type for the recovery is 
*             dependent upon the type of error.  V_E_ALT() will
*             terminate the program if no recovery is possible.
*             "Printer not ready" errors are treated as non-recoverable -
*             they should be handled in the printing code.
*
*  Author   : Vick Perry,   Tue  04-23-1991
*
*  Notes    : V_E_ALT() handles all standard Clipper and DOS errors.
*             V_E_ALT() can replace the standard ERRORSYS error handler
*             built into Clipper 5.  To substitute V_E_ALT() for ERRORSYS,
*             link the following code into your application:
*
*                Procedure Errorsys
*                   Errorblock( {|e| V_E_ALT(e)} )
*                Return
*
*             If you modify v_e_alt() insure that any system-wide variables
*             that you use are defined early in v_init(), so that if the
*             failure occurs within v_init() (like out of file handles
*             when opening the DD), no recursive errors will occur.
*             Use the type() function (not valtype()) to detect whether
*             a CDD system exists before using it.
*
*
*  Revisions:
*
*****************************************************************************
function V_E_ALT(e)
   /* local declarations */
   local success := .f.             // flag if error is recoverable
   local r_value := .f.             // value returned to calling function
   local user_id := ""              // contains title and userid
   local s_msg   := ""              // summary message
   local handle, i, j
   local e_block, e_obj             // used by alt error handler
   local is_write := .t.            // flag whether to write to ERROR.TXT
   static ft_err_flag := .f.        // stops recursion of V_E_ALT()

   * check for recursion of error handler
   if ft_err_flag
      set device to screen
      set cursor off
      tone(880,4)
      tempstr := "<<<< Recursive error occurred while in the error handler >>>>"
      @ maxrow(), 0 say padc(tempstr,maxcol()+1)
      inkey(30)

      /* cleanup and quit */
      set cursor on
      set color to
      close all
      quit
   else
      ft_err_flag := .t.  // indicate that V_E_ALT() is now active
   endif

   /* for network open error, set NETERR() and subsystem default */
   if (e:gencode == EG_OPEN) .and. e:osCode == 32 .and. e:candefault
     neterr(.t.)          // indicate network error
     success := .t.       // do not terminate program
     r_value := .f.       // return a failure
   end

   /* for lock error during APPEND BLANK, set NETERR() and subsystem default */
   if ( e:genCode == EG_APPENDLOCK .and. e:canDefault )
     neterr(.t.)
     success := .t.       // do not terminate program
     r_value := .f.       // return a failure
   end

   /* zero divide errors -  return 0 */
   if (e:gencode == EG_ZERODIV) .and. e:cansubstitute
      success := .t.      // do not terminate program
      r_value := 0        // return a zero
   endif

   if !success
      setcolor("W/N,N/W,,,N/W")
      * show message to screen

      /* get the current user's identification from DOS SET USERID=????? */
      do case
         case !empty(getenv("USERID"))
            user_id := upper(getenv("USERID"))
         case !empty(getenv("USER"))
            user_id := upper(getenv("USER"))
         case !empty(getenv("USERNAME"))
            user_id := upper(getenv("USERNAME"))
      endcase

      /* build the summary message */
      /* function name and line number */
      if ! empty(procname(2))
         s_msg += alltrim(procname(2)) +" line "+;
                  alltrim(str(procline(2)))
      endif

      /* description of error */
      if ! empty(e:description())
         s_msg += "  " + alltrim(e:description())
      endif

      /* operation */
      if ! empty(e:operation())
         s_msg += ": " + alltrim(e:operation())
      end

      /* filename */
      if ! empty(e:filename())            // filename
         s_msg += ": " + upper(alltrim(e:filename()))
      end

      /* DOS error code */
      if ( e:genCode == EG_OPEN .or. e:genCode == EG_CREATE )
         s_msg += "  (DOS error = " + alltrim(Str(e:osCode)) + ")"
      end

      /* subsystem name and subsystem code */
      if ! empty(e:subsystem())
         s_msg +=  "  " + alltrim(e:subsystem()) + "/" +;
                  alltrim(str(e:subcode()))
      endif


      * display to screen and write to error.txt
      tone(880,4)
      setpos(maxrow(),0)
      ? "ERROR  ERROR  ERROR  ERROR  ERROR  ERROR  ERROR  ERROR  ERROR"

      * install alternate error handler to trap error.txt file open errors
      * the new error handler simply returns true to all errors
      e_block := errorblock({|e_obj| v_errfile(e_obj) })

      begin sequence

         * open error file for appending
         set alternate to error.txt additive
         set alternate on

      recover using e_objlocal

         * signal that ERROR.TXT is not open
         is_write := .f.

      end

      * restore original error handler
      errorblock(e_block)


      ?
      ? "  Call Stack          CDDLIB Stack"
      ? "Procname     Line  | Type   Name"
      ? "--------------------------------------"

      /* show the call stack and v_stack_  */
      i := 2
      j := len(v_stack_)
      do while .t.
         if empty(procname(i))
            exit
         endif

         * only echo up to 12 lines of the CDDLIB stack to the
         * screen; however, the stack will still echo to the error file
         if i > 14
            set console off
         endif

         ? padr(procname(i),12) + str(procline(i),5) + "  |  " +;
           if(j > 0, left(v_stack_[j],1) + "     " + substr(v_stack_[j],2), "")
         ++i
         --j
      enddo

      * turn the console back on
      set console on

      ?

      * write std error line
      ? "Error   : "+s_msg

      * CDDLIB specific error!
      * if error in middle of a sequence, show sequence name and current line
      if procname(2) = "V_SEQUENCE"
         ? "Sequence: "+upper(v_id)+" - "+v_curline
      endif

      * CDDLIB specific error!
      * if error in middle of a S_LOAD() screen variable init, show info
      if procname(3) = "S_LOAD"
         * NOTE: varnum is declared as private in s_load()
         ? "Screen  : Error initializing variable "+flds_[GINTERNAL,varnum]+ " in screen "+substr(v_stack_[len(v_stack_)],2)
      endif

      * CDDLIB specific error!
      * s_replace() error - usually caused by unknown fieldname or incorrect
      * dbf alias
      if procname(2) = "S_REPLACE"
         * NOTE: tempint is declared as private in s_replace()
         ? "Screen  : Error replacing variable "+tempint+ " in screen "+substr(v_stack_[len(v_stack_)],2)
      endif

      * show user
      if !empty(user_id)
         ? "User    : "+user_id
      endif

      * show date and time
      ? "Date    : "+dtoc(date()) + "  " + time()

      * show system name
      ? "System  : "+if(type("v_sysname")=="C",v_sysname,"??????")

      * end line separator
      ? replicate("-",79)

      * stop writing to error file
      if is_write
         set alternate off
      else
         ? "          <<<<< Cannot open ERROR.TXT.  Please print this screen. >>>>>"
      endif

      ? "Program is terminated.  Press any key..."
      inkey(30)

      /* cleanup and quit */
      set cursor on
      setcolor("W/N,N/W,,,N/W")
      quit
   else
      ft_err_flag := .f.                // user recovered, reset activation flag
   endif
return (r_value)                        // error return values


*****************************************************************************
*  Function : v_errfile()
*****************************************************************************
*
*  Purpose  : Alternate error handler (block) called in case ERROR.TXT
*             cannot be opened for writing.
*
*  Syntax   : v_errfile(obj_err)
*
*  Arguments: None
*
*  Returns  : NIL
*
*  Author   : Vick Perry,   Mon  11-23-1992
*
*  Notes    :
*
*  Revisions:
*
*  Examples :
*
*****************************************************************************
function v_errfile(obj_err)
   break obj_err
return nil


