/*

   Program: DEMO2.PRG
   Purpose: Multi-window demonstration of a Clipper 5.0 application
            written to run under Microsoft Windows (with Clip-4-Win)
   Authors: John Skelton and Greg Lief
   Date:    November 1992 - present (on-going)

   This demo is Copyright (c) 1992-3 by John Skelton and Greg Lief.
   Clip-4-Win is Copyright (c) 1992 by John Skelton.

   IMPORTANT NOTE: This is an abbreviated version of the full demo
   source code which ships with Clip-4-Win.  The following proprietary
   information has been omitted: (a) a customized GET reader; (b) event
   handlers for data entry and browse screens; (c) search/replace logic
   for the edit window; (d) selecting font for edit control and printing
   using same; and (e) other miscellaneous functions.  Once again, all
   of Clipper source code is included with the product.

*/

#define LOGO_FILE  "clip4win.bmp"

external descend  // for index files that might need it

// directory where .WAV files reside
#define WAVE_DIRECTORY  "C:\WINDOWS\"

#define WIN_WANT_CLIPBOARD  // to make Clipboard directives accessible
#define WIN_WANT_EN         // for optional Edit Notification messages
#define WIN_WANT_LBS        // to make Listbox styles accessible

#define WM_SETTEXT 12
#define WM_GETTEXT 13
#define WM_GETTEXTLENGTH 14

#define CR    chr(13)
#define CRLF  chr(13)+chr(10)
#define ID_EDIT   1         // unique ID for edit control

#include "accel.ch"
#include "commdlg.ch"
#include "c4wget.ch"        // customized @..GET directives
#include "dbstruct.ch"
#include "directry.ch"
#include "error.ch"
#include "getexit.ch"
#include "inkey.ch"
#include "windows.ch"
#include "setcaret.ch"
#include "font.ch"

// manifest constants for GetClientRect() and GetDIBRect() arrays
// note that since Top and Left are always 0, they are unused
#define W_RIGHT  3
#define W_BOTTOM 4

// manifest constants for RGB() color combinations
#define C_RED        RGB(255,0,0)
#define C_BLUE       RGB(0,0,255)
#define C_GREEN      RGB(0,255,0)
#define C_MAGENTA    RGB(255,0,255)
#define C_BLACK      RGB(0,0,0)

#define APP_NAME     "Clip-4-Win"

// structure of event queue array (aEvents)
#define E_WINDOW    1
#define E_ACTION    2

static aEvents := {}  // array to hold events/windows info

static hWnd           // handle to main (parent) window
static hMenu          // handle to main menu
static cEditFile      // name of file currently being edited
static cAppName       // name of application (from DEMO.INI)
static cVersion       // version info (from DEMO.INI)
static nLimit         // maximum size of text in edit window
static hEdit          // handle to edit control

/*
   Function: Main()
*/
function main(nTextLimit)
local nEvent
local cPath
local hCurrWnd
local nAccelkey
local AccelKeyList // used as temporary storage for the accel key information

set scoreboard off     // don't even THINK about using it in Windows!!

/*
  The GetPvProfString() function is perfect for retrieving configuration
  information stored in an .INI file (in this case, Here we use it to
  retrieve the name of the application and the version number from
  DEMO.INI.  Note the use of GetModuleFileName() to determine the
  directory from which we are running this application, and thus where
  to look for the .INI file.
*/
cPath    := GetModuleFileName()
cPath    := left(cPath, rat('\', cPath))
cAppName := GetPvProfString("Preferences", "appname", "Clip-4-Win Demo", ;
                            cPath + "demo.ini")
cVersion := GetPvProfString("Preferences", "version", "02/23/93", ;
                            cPath + "demo.ini")
cEditFile := "(Untitled)"
hWnd := WinSetup(APP_NAME, cAppName + " - " + cEditFile, 0, 0, 600, 400, ;
                "fishicon", , WHITE_BRUSH)
HideCaret(hWnd)
ShowLogo(LOGO_FILE)
hMenu := MenuSetup()
ShowCaret(hWnd)
hEdit = CreateWindow("edit",             ;      // window class
                     '',                 ;      // initial buffer
                     WS_CHILD            ;      // child window
                     + WS_VISIBLE        ;      // ... that can be seen
                     + WS_VSCROLL        ;      // ... vert. scroll bar
                     + WS_BORDER         ;      // ... with a border
                     + ES_LEFT           ;      // ... left justified
                     + ES_MULTILINE      ;      // ... allow several lines
                     + ES_AUTOVSCROLL    ;      // ... auto vert. scroll
                     + ES_AUTOHSCROLL    ;      // ... auto horiz. scroll**
                     + WS_HSCROLL,       ;      // ... horiz. scroll bar**
                     0, 0,               ;      // x,y position
                     590, 350,           ;      // width, height
                     hWnd,               ;      // parent window
                     ID_EDIT,            ;      // id for child control to use
                     _GetInstance())            // our own app instance
                                                // **remove for word wrap

// limit maximum size of edit text if passed as parameter
if nTextLimit <> NIL
   nLimit := nTextLimit
   EditLimitText(hEdit, nLimit)
endif
AddHandler(hWnd, {|nEvent| EditEvent(nEvent, hWnd)})
SetFocus(hEdit)

/*
   The C4W_AutoClose() function allows us to make the user confirm their
   decision to quit the app, even if they try to close the window
   with ALT-F4 (or via the System menu).  It will cause the EVENT_CLOSE
   event to be generated, which we can then react to (see below)
*/
C4W_AutoClose(.f.)
SetHandleCount(40)

// initialize our accelerator keys; note that the goofy key
// assignments are intended to emulate other Windows applications
ACCELERATORS
AKEY asc("Z"), 501, VIRTKEY, CONTROL   // Ctrl-Z = Undo
AKEY asc("X"), 502, VIRTKEY, CONTROL   // Ctrl-X = Cut
AKEY asc("C"), 503, VIRTKEY, CONTROL   // Ctrl-C = Copy
AKEY asc("R"), 504, VIRTKEY, CONTROL   // Ctrl-R = Clear
AKEY asc("V"), 505, VIRTKEY, CONTROL   // Ctrl-V = Paste
AKEY VK_F5,    506, VIRTKEY            // F5     = insert Date/Time
AKEY VK_F1,    507, VIRTKEY            // F1     = help!
USE ACCELERATORS

// main event loop
do while .t.
   do while (nEvent := ChkEvent()) == EVENT_NONE
      // "background" processing could go here
   enddo
   HandleEvent(nEvent)
   do case
   case nEvent == EVENT_CLOSE
      // determine if main window is currently active
      hCurrWnd := _LasthWnd()
      if hWnd == hCurrWnd
         DoExit()
      else
         DestroyWindow(hCurrWnd)
      endif
   case nEvent == EVENT_ACCELKEY
      Accelerate()
   case nEvent == EVENT_QUIT
      quit
   endcase
enddo
end accelerators
return nil


/*
   Function: ShowLogo()
   Purpose:  Display opening Clip-4-Win logo then vanish it downwards
*/
static function ShowLogo(cFile)
local hDC
local aDIBRect
local cDIB := ReadDIB(cFile)
local coords_[4]
local hWnd := SelectWindow()
local x
local aClientRect
if cDIB <> NIL
   hDC := GetDC(hWnd)
   aDIBRect := GetDIBRect(cDIB)
   aClientRect := GetClientRect(hWnd)
   coords_[1] := (aClientRect[W_RIGHT] - aDIBRect[W_RIGHT]) / 2
   coords_[2] := (aClientRect[W_BOTTOM] - aDIBRect[W_BOTTOM]) / 2
   coords_[3] := coords_[1] + aDIBRect[W_RIGHT]
   coords_[4] := coords_[2] + aDIBRect[W_BOTTOM]
   ShowDIB(hDC, cDIB, (aClientRect[W_RIGHT] - aDIBRect[W_RIGHT]) / 2, coords_[2])
   ReleaseDC(hWnd, hDC)
   inkey(2)
   // make the logo vanish downwards...
   for x := coords_[2] to coords_[4] step 20
      ScrollWindow(hWnd, 0, 20, , coords_)
   next
endif
return nil


/*
   Function: Accelerate()
   Purpose:  Process an accelerator key
*/
static function accelerate
local nAccelkey := _lastwParam()
do case
   case nAccelkey == 501
      eval(GetMenuBlock(hMenu, "undo"))
   case nAccelkey == 502
      eval(GetMenuBlock(hMenu, "cut"))
   case nAccelkey == 503
      eval(GetMenuBlock(hMenu, "copy"))
   case nAccelkey == 504
      eval(GetMenuBlock(hMenu, "clear"))
   case nAccelkey == 505
      eval(GetMenuBlock(hMenu, "paste"))
   case nAccelkey == 506
      eval(GetMenuBlock(hMenu, "timedate"))
   case nAccelkey == 507
      eval(GetMenuBlock(hMenu, "about"))
endcase
return nil


/*
   Function: DoLogo()
   Purpose:  Create a new window that displays the Clip-4-Win logo
   Seealso:  LogoEvent()
*/
static function DoLogo(cItem)
static hWnd
static nId
if CheckMenuItem(hMenu, cItem) == MF_UNCHECKED
   CheckMenuItem(hMenu, cItem, MF_CHECKED)
   hWnd := WinNew("Logo", 50, 50, 350, 250)
   nId := AddHandler(hWnd, {|nEvent| LogoEvent(nEvent, hWnd, nId, cItem)})
else
   DestroyWindow(hWnd)
   CheckMenuItem(hMenu, cItem, MF_UNCHECKED)
   DelHandler(nId)
endif
return nil


/*
   Function: LogoEvent()
   Purpose:  Redraw logo when necessary
*/
static function LogoEvent(nEvent, hWnd, nId, cItem)
static cDIB
local hDC
local aDIBRect, aClientRect
do case
case nEvent == EVENT_REDRAW
   hDC := GetDC(hWnd)
   if cDIB == NIL
      cDIB := ReadDIB(LOGO_FILE)
   endif
   aDIBRect := GetDIBRect(cDIB)
   aClientRect := GetClientRect(hWnd)
   ShowDIB(hDC, cDIB, (aClientRect[W_RIGHT] - aDIBRect[W_RIGHT]) / 2, ;
                      (aClientRect[W_BOTTOM] - aDIBRect[W_BOTTOM]) / 2)
   ReleaseDC(hWnd, hDC)
case nEvent == EVENT_CLOSE   // clean up, including toggling menu item
   DestroyWindow(hWnd)
   CheckMenuItem(hMenu, cItem, MF_UNCHECKED)
   DelHandler(nId)
endcase
return nil


/*
   Function: Credits()
*/
static function credits
// note: cAppName and cVersion are file-wide statics... assigned
//       above when they were retrieved from our DEMO.INI file
static aDlg
if aDlg == NIL
   aDlg := CreateDialog("About",                                           ;
                       WS_CAPTION + WS_SYSMENU + WS_GROUP + WS_TABSTOP     ;
                       + WS_THICKFRAME + WS_VISIBLE + WS_POPUP,            ;
                       80, 50, 125, 86)
   aDlg := AppendDialog(aDlg, "", DLG_STATIC,                              ;
                        SS_CENTER + WS_CHILD + WS_VISIBLE,                 ;
                        1, 6, 123, 8,                                      ;
                        cAppName)
   aDlg := AppendDialog(aDlg, "", DLG_STATIC,                              ;
                        SS_CENTER + WS_CHILD + WS_VISIBLE,                 ;
                        1, 17, 123, 8,                                     ;
                        "Version " + cVersion)
   aDlg := AppendDialog(aDlg, "", DLG_STATIC,                              ;
                        SS_CENTER + WS_CHILD + WS_VISIBLE,                 ;
                        1, 28, 123, 8,                                     ;
                        "By Greg Lief and John Skelton")
   aDlg := AppendDialog(aDlg, "", DLG_STATIC,                              ;
                        SS_CENTER + WS_CHILD + WS_VISIBLE,                 ;
                        1, 39, 123, 8,                                     ;
                        "Copyright (c) 1993")
   aDlg := AppendDialog(aDlg, "", DLG_STATIC,                              ;
                        SS_CENTER + WS_CHILD + WS_VISIBLE,                 ;
                        1, 50, 123, 8,                                     ;
                        "All Rights Reserved")
   aDlg := AppendDialog(aDlg,"IDCANCEL", DLG_BUTTON,                       ;
                        BS_DEFPUSHBUTTON + WS_CHILD                        ;
                        + WS_VISIBLE + WS_TABSTOP,                         ;
                        64, 64, 40, 14,                                    ;
                        "&Okay")
   aDlg := AppendDialog(aDlg, "", DLG_STATIC,                              ;
                        SS_ICON + WS_CHILD + WS_VISIBLE,                   ;
                        27, 62, 18, 20,                                    ;
                        "fishicon")
endif
ModalDialog(aDlg, _GetInstance(), SelectWindow())
return nil


/*
   Function: DoExit()
   Purpose:  Seek confirmation prior to exit
*/
static procedure DoExit()
local nSel
if EditGetModify(hEdit)
   // Note that MessageBox() returns a value based on the I.D. of
   // the selected item, all of which have corresponding ID* manifest
   // constants in the WINDOWS.CH header file
   nSel := MessageBox(0, "The text in " + cEditFile + " has changed." + ;
                   CR + "Do you want to save the changes?", ;
                   "Notepad", MB_YESNOCANCEL + MB_ICONEXCLAMATION)
   do case
      case nSel == IDYES
         DoSave()
         quit
      case nSel == IDNO
         quit
   endcase
elseif MessageBox(0, "Are you sure you want to exit this demo?", ;
                     "Leaving so soon?", MB_OKCANCEL + MB_ICONQUESTION) == IDOK
   quit
endif
return


/*
   Function: EditEvent()
   Purpose:  Resize edit control if main window is resized.
             Also verify that edit buffer does not exceed
             size limitation (if any was established)
*/
static function EditEvent(nEvent, hWindow)
local nEN
local x
local aClientRect
do case
   case nEvent == EVENT_WINSIZE .and. SelectWindow() == hWnd
      aClientRect := GetClientRect(hWnd)
      MoveWindow(hEdit, aClientRect[1], aClientRect[2], ;
                        aClientRect[3], aClientRect[4], .f.)
      InvalidateRect(hWnd)
   case nEvent == EVENT_SETFOCUS .and. SelectWindow() <> hEdit
      SetFocus(hEdit)
   case nEvent == EVENT_CONTROL
      if _LastwParam() == ID_EDIT     // child id
         // process message sent by the edit control
         nEN = _LastHilParam()   // get the reason for the msg
         if nEN == EN_MAXTEXT .or. nEN == EN_ERRSPACE
            MessageBox( , "No more room!", "Error", MB_ICONHAND + MB_OK)
         elseif nEN == EN_CHANGE
            if EditCanUndo(hEdit)
               EnableMenuItem(hMenu, "undo", MF_ENABLED)
            endif
         endif
      endif
endcase
EnableMenuItem(hMenu, "paste", ;
      if(IsClipBFormatAvailable(CF_TEXT), MF_ENABLED, MF_GRAYED) )
return nil


/*
   Function: DoAudio()
   Purpose:  Play .WAV files.  Also demonstrates use of DLLs.
*/
static function DoAudio()
static waves_ := { "CHORD", "CHIMES", "TADA", "DING" }
local hLib, cSound, hWnd, n
hLib   := LoadLibrary("MMSYSTEM.DLL")
cSound := GetProcAddress(hLib, "SndPlaySound", "Pascal", "Int", "str, int")
// note that Alert() allows you to specify trigger letters by
// preceding them with an ampersand
n := alert("Pick a sound", ;
           { "&Chord", "Chi&mes", "&Tada" } )

// if they escaped out, use sound #4
if n == 0
   n := 4
endif

// second parameter to SndPlaySound() is: 1 == return instantly,
// 0 == wait until finished playing before returning
n := CallDLL(cSound, WAVE_DIRECTORY + waves_[n] + ".WAV", 1)
if n == 0
   MessageBox(, "No audio hardware" + CR + "Or other error", ;
              MB_ICONHAND + MB_OK)
endif
return nil


/*
   Function: DoColor()
   Purpose:  Allow color selection then create new window for color demo
   Seealso:  ColorEvent()
*/
static function DoColor(cItem)
local  nColor
static hWnd
static nId
if CheckMenuItem(hMenu, cItem) == MF_UNCHECKED
   nColor := ChooseColor()
   if nColor >= 0         // else user chose cancel/close or hit Esc
      hWnd := WinNew("Color", 60, 110, 150, 100)
      nId := AddHandler(hWnd, {|nEvent| ColorEvent(nEvent, hWnd, nColor, '
                              nId, cItem)})
      CheckMenuItem(hMenu, cItem, MF_CHECKED)
   endif
else
   DestroyWindow(hWnd)
   CheckMenuItem(hMenu, cItem, MF_UNCHECKED)
   DelHandler(nId)
endif
return nil


/*
   Function: ColorEvent()
   Purpose:  Redraw color demo window when necessary
*/
static function ColorEvent(nEvent, hWnd, nColor, nId, cItem)
local hDC, hBrush
do case
case nEvent == EVENT_REDRAW
   hDC := GetDC(hWnd)
   hBrush := CreateSolidBrush(nColor)
   FillRect(hDC, 20, 20, 100, 50, hBrush)
   DeleteObject(hBrush)
   ReleaseDC(hWnd, hDC)
case nEvent == EVENT_CLOSE   // clean up, including toggling menu item
   DestroyWindow(hWnd)
   CheckMenuItem(hMenu, cItem, MF_UNCHECKED)
   DelHandler(nId)
endcase
return nil


/*
   Function: DaDoRunRun()
   Purpose:  Run an external program
   Note:     Long live Phil Spector!
*/
static function DaDoRunRun
local cFile := GetOpenFileName(, "*.exe;*.com;*.bat", "Run", ;
                               { {"programs", "*.exe;*.com;*.bat"} } )
local hCursor
local hOldcursor
if cFile <> NIL
   hCursor := LoadCursor(, IDC_WAIT)
   hOldcursor := SetCursor(hCursor)
   WinExec(cFile)
   SetCursor(hOldcursor)   // restore previous cursor
endif
return nil


/*
   Function: DoDLL()
   Purpose:  DLL demonstration (using the Rectangle() function)
   Seealso:  DLLEvent()
*/
static function DoDLL()
static	nX := 20, nY := 200
local	hLib, cRectangle, hWnd
hLib := LoadLibrary("GDI.EXE")
cRectangle := GetProcAddress(hLib, "rectangle", "Pascal", "Int", ;
                             "int, int, int, int, int")
hWnd := WinNew("DLL", nX += 40, nY += 60, 200, 100)
AddHandler(hWnd, {|nEvent| DLLEvent(nEvent, hWnd, cRectangle)})
return nil


/*
   Function: DLLEvent()
   Purpose:  Redraw Rectangle() window when necessary
*/
static function DLLEvent(nEvent, hWnd, cRectangle)
local hDC, cText
do case
case nEvent == EVENT_REDRAW
   hDC := GetDC(hWnd)
   cText := "CallDLL(Rectangle, ...) --> " + ;
             nstr(CallDLL(cRectangle, hDC, 10, 30, 100, 50))
   DrawText(hDC, cText, GetClientRect(hWnd))
   ReleaseDC(hWnd, hDC)
case nEvent == EVENT_LCLICK .or. nEvent == EVENT_RCLICK
   InvalidateRect(hWnd)
endcase
return nil


/*
   Function: DoFont()
   Purpose:  Allow font selection then create a window to display font samples
   Seealso:  FontEvent()
*/
static function DoFont(cItem)
static  nX := 100, nY := 50
static aFont := {40, 40, 450, 0, 400, .t., .f., .f., 1, 0, 0, 0, 0, "Arial"}
static hWnd
local nColor := C_RED
static nId
if CheckMenuItem(hMenu, cItem) == MF_UNCHECKED
   aFont := ChooseFont(aFont, , , @nColor)
   if aFont <> NIL         // else user chose cancel/close or hit Esc
      hWnd := WinNew("Font", 140, 110, 300, 200)
      nId := AddHandler(hWnd, {|nEvent| FontEvent(nEvent, hWnd, aFont, nColor, nId, cItem)})
      CheckMenuItem(hMenu, cItem, MF_CHECKED)
   endif
else
   DestroyWindow(hWnd)
   CheckMenuItem(hMenu, cItem, MF_UNCHECKED)
   DelHandler(nId)
endif
return nil


/*
   Function: FontEvent()
   Purpose:  Redraw font sample window when necessary
*/
static function FontEvent(nEvent, hWnd, aFont, nColor, nId, cItem)
local   hDC, hFont, hOldFont, i, j
static msg := "Clip-4-Win"
static  aShow := { {200, 200, 300},  ;     // {x coord, y coord, angle}
                   {0,  350, 1800},  ;     // angle is in 3600 gradiants
                   {20, 0,   2700},  ;     // 0 is horizontal left-to-right
                   {40, 200,  800},  ;     // e.g., 900 points straight up
                   {75, 400, 1350},  ;
                   {100, 20, 0},     ;
                   {100, 300, 450},  ;
                   {125, 425, 450},  ;     // 1800 is horizontal right-to-left
                   {200, 50, 3450},  ;
                   {250, 200, 0},    ;
                   {300, 15, 2100},  ;
                   {300,400, 1080},  ;
                   {375,200, 300},   ;
                   {470,300, 2700},  ;
                   {520,250, 1800},  ;
                   {525,400, 0},     ;
                   {550,100, 3500},  ;
                   {400, 50, 3150} }
do case
case nEvent == EVENT_REDRAW
   hDC := GetDC(hWnd)
   SetTextColor(hDC, nColor)
   j := len(aShow)
   for i := 1 to j
      aFont[LF_Escapement] := aShow[i, 3]
      hFont := CreateFont(aFont)
      hOldFont := SelectObject(hDC, hFont)
      TextOut(hDC, aShow[i, 1], aShow[i, 2], msg)
      // note that SelectObject() returns a handle to the
      // prior object, so you can delete it in one fell swoop
      DeleteObject( SelectObject(hDC, hOldFont) )
   next
   ReleaseDC(hWnd, hDC)
case nEvent == EVENT_CLOSE   // clean up, including toggling menu item
   DestroyWindow(hWnd)
   CheckMenuItem(hMenu, cItem, MF_UNCHECKED)
   DelHandler(nId)
endcase
return nil



/*
   Function: DoPie()
   Purpose:  Create window displaying pie graph
   Seealso:  PieEvent()
*/
static function DoPie(cItem)
static hWnd
static nId
if CheckMenuItem(hMenu, cItem) == MF_UNCHECKED
   hWnd := WinNew("Pie", 440, 110, 100, 100)
   nId := AddHandler(hWnd, {|nEvent| PieEvent(nEvent, hWnd, nId, cItem)})
   CheckMenuItem(hMenu, cItem, MF_CHECKED)
else
   DestroyWindow(hWnd)
   CheckMenuItem(hMenu, cItem, MF_UNCHECKED)
   DelHandler(nId)
endif
return nil


/*
   Function: PieEvent()
   Purpose:  Redraw pie graph when necessary
*/
static function PieEvent(nEvent, hWnd, nId, cItem)
local hDC, aRect, hBrush, hOldbrush
do case
case nEvent == EVENT_REDRAW
   hBrush := CreateSolidBrush(C_RED)
   aRect := GetClientRect(hWnd)
   hDC := GetDC(hWnd)
   hOldbrush := SelectObject(hDC, hBrush)
   pie(hDC, 0, 0, aRect[W_RIGHT], aRect[W_BOTTOM], ;
            aRect[W_RIGHT] / 2, 0, aRect[W_RIGHT], aRect[W_BOTTOM])
   hBrush := CreateSolidBrush(C_MAGENTA)
   // Just as it is important to close each open file handle, you
   // should always delete each object when you are done with it.
   // Note that this can be done with the following syntax, which
   // lets you re-use the same variable name for multiple objects.
   DeleteObject( SelectObject(hDC, hBrush) )

   pie(hDC, 0, 0, aRect[W_RIGHT], aRect[W_BOTTOM], aRect[W_RIGHT], ;
            aRect[W_BOTTOM], aRect[W_RIGHT] * .75, aRect[W_BOTTOM] * .25)
   hBrush := CreateSolidBrush(C_BLUE)
   DeleteObject( SelectObject(hDC, hBrush) )
   pie(hDC, 0, 0, aRect[W_RIGHT], aRect[W_BOTTOM], aRect[W_RIGHT], ;
            aRect[W_BOTTOM] / 2, aRect[W_RIGHT] * .75, aRect[W_BOTTOM] * .25)
   hBrush := CreateSolidBrush(C_GREEN)
   DeleteObject( SelectObject(hDC, hBrush) )
   pie(hDC, 0, 0, aRect[W_RIGHT], aRect[W_BOTTOM], ;
       aRect[W_RIGHT] * .75, aRect[W_BOTTOM] * .25, aRect[W_RIGHT] / 2, 0)
   DeleteObject( SelectObject(hDC, hOldbrush) )
   ReleaseDC(hWnd, hDC)
case nEvent == EVENT_CLOSE   // clean up, including toggling menu item
   DestroyWindow(hWnd)
   CheckMenuItem(hMenu, cItem, MF_UNCHECKED)
   DelHandler(nId)
endcase
return nil


/*
   Function: DoBar()
   Purpose:  Create window to display bar graph
   Seealso:  BarEvent()
*/
static function DoBar(cItem)
static hWnd
static nId
if CheckMenuItem(hMenu, cItem) == MF_UNCHECKED
   CheckMenuItem(hMenu, cItem, MF_CHECKED)
   hWnd := WinNew("Bar Graph", 190, 235, 250, 125)
   nId := AddHandler(hWnd, {|nEvent| BarEvent(nEvent, hWnd, nId, cItem)})
else
   DestroyWindow(hWnd)
   CheckMenuItem(hMenu, cItem, MF_UNCHECKED)
   DelHandler(nId)
endif
return nil


/*
   Function: BarEvent()
   Purpose:  Redraw bar graph when necessary
*/
static function BarEvent(nEvent, hWnd, nId, cItem)
local hDC, aRect, hBrush, hOldbrush, nWidth
do case
case nEvent == EVENT_REDRAW
   hBrush := CreateHatchBrush(HS_BDIAGONAL, C_RED)
   aRect := GetClientRect(hWnd)
   nWidth := aRect[W_RIGHT] / 6
   hDC := GetDC(hWnd)
   hOldbrush := SelectObject(hDC, hBrush)
   TextOut(hDC, nWidth + 10, aRect[W_BOTTOM] * .15 - 20, "U.S.A.")
   rectangle(hDC, nWidth, aRect[W_BOTTOM] * .15, ;
                  nWidth * 2, aRect[W_BOTTOM])
   hBrush := CreateHatchBrush(HS_VERTICAL, C_MAGENTA)
   DeleteObject( SelectObject(hDC, hBrush) )
   TextOut(hDC, nWidth * 2 + 10, aRect[W_BOTTOM] * .4 - 20, "England")
   rectangle(hDC, nWidth * 2, aRect[W_BOTTOM] * .4, ;
                  nWidth * 3, aRect[W_BOTTOM])
   hBrush := CreateHatchBrush(HS_FDIAGONAL, C_BLUE)
   DeleteObject( SelectObject(hDC, hBrush) )
   TextOut(hDC, nWidth * 3 + 10, aRect[W_BOTTOM] * .8 - 20, "Australia")
   rectangle(hDC, nWidth * 3, aRect[W_BOTTOM] * .8, ;
                  nWidth * 4, aRect[W_BOTTOM])
   hBrush := CreateHatchBrush(HS_CROSS, C_GREEN)
   DeleteObject( SelectObject(hDC, hBrush) )
   TextOut(hDC, nWidth * 4 + 10, aRect[W_BOTTOM] * .55 - 20, "Germany")
   rectangle(hDC, nWidth * 4, aRect[W_BOTTOM] * .55, ;
                  nWidth * 5, aRect[W_BOTTOM])
   DeleteObject( SelectObject(hDC, hOldbrush) )
   ReleaseDC(hWnd, hDC)
case nEvent == EVENT_CLOSE   // clean up, including toggling menu item
   DestroyWindow(hWnd)
   CheckMenuItem(hMenu, cItem, MF_UNCHECKED)
   DelHandler(nId)
endcase
return nil


/*
   Function: TestPrint()
   Purpose:  Print sample page with graphics etcetera
*/
static function TestPrint()
local hPrintDC, hIcon, hBrush, hOldBrush, hCursor, hOldCursor
local nBlack := C_BLACK
local i, nWidth, nHeight

// display printer dialog box, so the user can choose the settings
hPrintDC := GetPrintDC()
if empty(hPrintDC)
   // cancelled by the user
   return nil
endif

// print a test page

hCursor := LoadCursor( , IDC_WAIT)
hOldCursor := SetCursor(hCursor)

nWidth := GetDeviceCaps(hPrintDC, HORZRES)
nHeight := GetDeviceCaps(hPrintDC, VERTRES)

StartDoc(hPrintDC, "Test Output")
StartPage(hPrintDC)

TextOut(hPrintDC, 100, 50, "Clip-4-Win Printer Test Page")
Rectangle(hPrintDC, 0, 0, nWidth, nHeight)
MoveTo(hPrintDC, 0, 0)
LineTo(hPrintDC, nWidth, nHeight)
MoveTo(hPrintDC, nWidth, 0)
LineTo(hPrintDC, 0, nHeight)

Arc(hPrintDC,   1000, 1000,   1300, 1200,   1250, 1190,   1100, 1100)

hBrush := CreateHatchBrush(HS_HORIZONTAL, nBlack)
hOldBrush := SelectObject(hPrintDC, hBrush)
Chord(hPrintDC, 1500, 1200,   2000, 1350,   1550, 1340,   1400, 1200)
DeleteObject( SelectObject(hPrintDC, hOldBrush) )

hBrush := CreateHatchBrush(HS_BDIAGONAL, nBlack)
hOldBrush := SelectObject(hPrintDC, hBrush)
Pie(hPrintDC,   100, 1200,   700, 1500,   650, 1490,   120, 1280)
DeleteObject( SelectObject(hPrintDC, hOldBrush) )

hBrush := CreateHatchBrush(HS_FDIAGONAL, nBlack)
hOldBrush := SelectObject(hPrintDC, hBrush)
Polygon(hPrintDC, { {1000, 250}, {1600, 500}, {1800, 100} })
DeleteObject( SelectObject(hPrintDC, hOldBrush) )
PolyLine(hPrintDC, { {300, 700}, {100, 900}, {500, 1000} })
for i := 100 to 500 step 100
   TextOut(hPrintDC, i + 400, i + 100, nstr(i))
next i
EndPage(hPrintDC)
EndDoc(hPrintDC)
DeleteDC(hPrintDC)
SetCursor(hOldCursor)
return nil


/*
   Function: DoTimer()
   Purpose:  Create window to display "ticking clock"
   Seealso:  TimerEvent()
*/
static function DoTimer()
static lTimer := .F.
local hWnd
local nColor
if ! lTimer
   nColor := ChooseColor()
   if nColor == -1  // if user pressed ESC, default to blue
      nColor := C_BLUE
   endif
   hWnd := WinNew("Timer", 400, 150, 135, 75)
   SetTimer(hWnd, 1, 1000)     // every 1000 millisecs (= one second)
   lTimer := .T.
   AddHandler(hWnd, ;
              {|nEvent| TimerEvent(nEvent, hWnd, nColor, @lTimer)})
endif
return nil


/*
   Function: TimerEvent()
   Purpose:  Redraw "ticking clock" once per second
*/
static function TimerEvent(nEvent, hWnd, nColor, lTimer)
local hDC, hOldFont, aRect
static aFont := { -55, -17, 0, 0, FW_SEMIBOLD, .F., .F., .F., 0, 3, 2, 1, 18, ;
                  "Times New Roman" }
static hFont
do case
case nEvent == EVENT_TIMER
   InvalidateRect(hWnd)

case nEvent == EVENT_CLOSE .or. nEvent == EVENT_DESTROY
   KillTimer(hWnd, 1)
   lTimer := .F.
   DestroyWindow(hWnd)

case nEvent == EVENT_WINSIZE
   // adjust font height/width based on new size of this window
   aRect := GetClientRect(hWnd)
   aFont[LF_Height] := - aRect[W_BOTTOM] * .85
   aFont[LF_Width] := - aRect[W_RIGHT] * .11
   hFont := NIL   // force font object to be recreated below...

case nEvent == EVENT_REDRAW
   hDC := GetDC(hWnd)
   SetTextColor(hDC, nColor)
   if hFont == NIL
      hFont := CreateFont(aFont)
   endif
   hOldFont := SelectObject(hDC, hFont)
   // display the system time perfectly centered in the client area
   DrawText(hDC, time(), , DT_CENTER + DT_VCENTER)  
   // note that we are not using DeleteObject() because we are
   // maintaining hFont as a static (otherwise we would delete it)
   SelectObject(hDC, hOldFont)
   ReleaseDC(hWnd, hDC)
endcase
return nil


/*
   Function: NStr()
   Purpose:  Convert number to string and add trailing space
*/
static function nstr(n)
return alltrim(str(n)) + " "


/*
   Function: MenuSetup()
   Purpose:  Create main menu and attach to main window
*/
static function MenuSetup()
local hWnd := SelectWindow()
local hMenu
local hPopupMenu
local hPopup2
if (hMenu := GetMenu(hWnd)) <> NIL
   DestroyMenu(hMenu)
endif

// note the grayed out entries (search for MF_GRAYED)
// also note that ampersand indicates the trigger letter
hMenu := CreateMenu()
hPopupMenu := CreatePopupMenu()
AppendMenu(hMenu, "file", MF_ENABLED + MF_POPUP, "&File", hPopupMenu)
AppendMenu(hPopupMenu, "new", MF_ENABLED + MF_STRING, "&New", {|| DoOpen(.t.)})
AppendMenu(hPopupMenu, "open", MF_ENABLED + MF_STRING, "&Open...", {|| DoOpen(.f.)})
AppendMenu(hPopupMenu, "save", MF_ENABLED + MF_STRING, "&Save", {|| DoSave()})
AppendMenu(hPopupMenu, "saveas", MF_ENABLED + MF_STRING, "Save &As...", {|| DoSave(.t.)})
AppendMenu(hPopupMenu, "print", MF_ENABLED + MF_STRING, "&Print", {|| DoPrint()})
AppendMenu(hPopupMenu, "testprint", MF_ENABLED + MF_STRING, "Print &Test", {|| TestPrint()})
AppendMenu(hPopupMenu, "", MF_SEPARATOR)
AppendMenu(hPopupMenu, "run", MF_ENABLED + MF_STRING, "&Run", {|| DaDoRunRun()})
AppendMenu(hPopupMenu, "", MF_SEPARATOR)
AppendMenu(hPopupMenu, "exit", MF_ENABLED + MF_STRING, "E&xit", {|| DoExit()} )

hPopupMenu := CreatePopupMenu()
AppendMenu(hMenu, "edit", MF_ENABLED + MF_POPUP, "&Edit", hPopupMenu)
AppendMenu(hPopupMenu, "undo", MF_GRAYED + MF_STRING,   "&Undo"+chr(K_TAB)+"Ctrl-Z", {|| DoUndo() })
AppendMenu(hPopupMenu, "", MF_SEPARATOR)
AppendMenu(hPopupMenu, "cut", MF_ENABLED + MF_STRING,   "Cu&t"+chr(K_TAB)+"Ctrl-X", {|| DoCut()})
AppendMenu(hPopupMenu, "copy", MF_ENABLED + MF_STRING,  "&Copy"+chr(K_TAB)+"Ctrl-C", {|| DoCopy()})
AppendMenu(hPopupMenu, "clear", MF_ENABLED + MF_STRING, "Clea&r"+chr(K_TAB)+"Ctrl-R", {|| DoClear()})
AppendMenu(hPopupMenu, "paste", if(IsClipBFormatAvailable(CF_TEXT), MF_ENABLED, MF_GRAYED) + ;
                                + MF_STRING,            "&Paste"+chr(K_TAB)+"Ctrl-V", {|| DoPaste()})
AppendMenu(hPopupMenu, "", MF_SEPARATOR)
AppendMenu(hPopupMenu, "timedate", MF_ENABLED + MF_STRING, "Time/&Date"+chr(K_TAB)+"F5", {|| TimeDate()})
AppendMenu(hPopupMenu, "", MF_SEPARATOR)
AppendMenu(hPopupMenu, "wordwrap", MF_ENABLED + MF_STRING, "&Word Wrap", {|i| WordWrap(i)})
AppendMenu(hPopupMenu, "font", MF_ENABLED + MF_STRING, "&Font...", {|| DoFont(hWnd, hEdit) })

hPopupMenu = CreatePopupMenu()
AppendMenu(hMenu, "search", MF_ENABLED + MF_POPUP, "&Search", hPopupMenu)
AppendMenu(hPopupMenu, "find", MF_ENABLED + MF_STRING, "&Find...", {|| DoFind() })
AppendMenu(hPopupMenu, "replace", MF_ENABLED + MF_STRING, "&Replace...", {|| DoReplace() })

hPopupMenu := CreatePopupMenu()
AppendMenu(hMenu, "misc", MF_ENABLED + MF_POPUP, "&Miscellany", hPopupMenu)
AppendMenu(hPopupMenu, "color", MF_ENABLED + MF_STRING, "&Color", {|c| DoColor(c)})
AppendMenu(hPopupMenu, "dll", MF_ENABLED + MF_STRING, "&DLL", {|| DoDLL()})
AppendMenu(hPopupMenu, "fonttest", MF_ENABLED + MF_STRING, "&Font test", {|c| FontTest(c)})
AppendMenu(hPopupMenu, "logo", MF_ENABLED + MF_STRING, "&Logo", {|c| DoLogo(c)})
AppendMenu(hPopupMenu, "pie chart", MF_ENABLED + MF_STRING, "&Pie Chart", {|c| DoPie(c)})
AppendMenu(hPopupMenu, "bar graph", MF_ENABLED + MF_STRING, "&Bar Graph", {|c| DoBar(c)})
AppendMenu(hPopupMenu, "", MF_SEPARATOR)

// the following chunk of code demonstrates a submenu of a submenu
hPopup2 := CreatePopupMenu()
AppendMenu(hPopupMenu, "morestuf", MF_ENABLED + MF_POPUP, "&More...", hPopup2)
AppendMenu(hPopup2, "audio", MF_ENABLED + MF_STRING, "&Audio", {|| DoAudio()})
AppendMenu(hPopup2, "cursor", MF_ENABLED + MF_STRING, "Cur&sors", {|| DoCursor()})
AppendMenu(hPopup2, "icons", MF_ENABLED + MF_STRING, "&Icons", {|c| DoIcons(c)})
AppendMenu(hPopup2, "timer", MF_ENABLED + MF_STRING, "&Timer", {|| DoTimer()})

hPopupMenu := CreatePopupMenu()
AppendMenu(hMenu, "clipper", MF_ENABLED + MF_POPUP, "&Clipper", hPopupMenu)
AppendMenu(hPopupMenu, "modelessgets", MF_ENABLED + MF_STRING, "&GETs", {|| ModelessGets()})
AppendMenu(hPopupMenu, "valtest", MF_ENABLED + MF_STRING, "&Validation", {|| ValidTest()})
AppendMenu(hPopupMenu, "browse", MF_ENABLED + MF_STRING, "&Browse", {|| DoBrowse()})

hPopupMenu := CreatePopupMenu()
AppendMenu(hMenu, "help", MF_ENABLED + MF_POPUP, "&Help", hPopupMenu)
AppendMenu(hPopupMenu, "about", MF_ENABLED + MF_STRING, "&About"+chr(K_TAB)+"F1", {|| Credits()})
SetMenu(hWnd, hMenu)
return hMenu


/*
   Function: DoCursor()
   Purpose:  Cycle through stock cursors
*/
static function DoCursor()
static cursors_ := { IDC_ARROW, IDC_IBEAM, IDC_WAIT, IDC_CROSS, ;
                     IDC_UPARROW, IDC_SIZE, IDC_ICON, IDC_SIZENWSE, ;
                     IDC_SIZENESW, IDC_SIZEWE, IDC_SIZENS }
local y := len(cursors_)
local x := 0
local hCursor
local key
local hOldcursor := SetCursor()
MessageBox(hWnd, "Press spacebar to cycle through the stock cursors", "Info", ;
                 MB_OK)
do while ++x <= y .and. key != K_ESC
   hCursor := LoadCursor( , cursors_[x])
   if hCursor <> 0
      SetCursor(hCursor)
   endif
   key := inkey(2)
enddo
SetCursor(hOldcursor)
return nil


/*
   Function: DoIcons()
   Purpose:  Create window to display all stock icons
   Seealso:  IconEvent()
*/
static function DoIcons(cItem)
static hWnd
static nId
if CheckMenuItem(hMenu, cItem) == MF_UNCHECKED
   hWnd := WinNew("Icons", 290, 160, 300, 100)
   nId := AddHandler(hWnd, {|nEvent| IconEvent(nEvent, hWnd, nId, cItem)})
   CheckMenuItem(hMenu, cItem, MF_CHECKED)
else
   DestroyWindow(hWnd)
   CheckMenuItem(hMenu, cItem, MF_UNCHECKED)
   DelHandler(nId)
endif
return nil


/*
   Function: IconEvent()
   Purpose:  Redraw icon window when necessary
*/
static function IconEvent(nEvent, hWnd, nId, cItem)
static icons_ := { IDI_EXCLAMATION, IDI_HAND, IDI_ASTERISK, ;
                   IDI_QUESTION, IDI_APPLICATION }
local hDC
local x
local hIcon
local aRect
do case
case nEvent == EVENT_REDRAW
   aRect := GetClientRect(hWnd)
   hDC := GetDC(hWnd)
   if hDC <> 0
      for x := 1 to 5
         if ( hIcon := LoadIcon(, icons_[x]) ) <> 0
            DrawIcon(hDC, aRect[W_RIGHT] * x / 6, ;
                          aRect[W_BOTTOM] * .4, hIcon)
         endif
      next
      ReleaseDC(hWnd, hDC)
   endif
case nEvent == EVENT_CLOSE   // clean up, including toggling menu item
   DestroyWindow(hWnd)
   CheckMenuItem(hMenu, cItem, MF_UNCHECKED)
   DelHandler(nId)
endcase
return nil


/*
   Function: AddHandler()
   Purpose:  Add an event to the event queue
*/
function AddHandler(hWnd, bAction)  // --> nId (for use with DelHandler)
local nId
// scan for an empty slot and re-use it if one is found
if ( nId := ascan(aEvents, { | e | e[E_WINDOW] == NIL } )  ) <> 0
   aEvents[nId][E_WINDOW] := hWnd
   aEvents[nId][E_ACTION] := bAction
else
   aadd(aEvents, { hWnd, bAction } )
   nId := len(aEvents)
endif
return nId


/*
   Function: DelHandler()
   Purpose:  Delete an event from the event queue
*/
static function DelHandler(nId)
aEvents[nId][E_WINDOW] := aEvents[nId][E_ACTION] := NIL
return nil


/*
   Function: HandleEvent()
*/
function HandleEvent(nEvent)
local hWnd := _LasthWnd()
local i := 0
do while (i := ascan(aEvents, { | e | e[E_WINDOW] == hWnd } , ++i)) <> 0
   eval(aEvents[i][E_ACTION], nEvent)
enddo
if nEvent == EVENT_DESTROY
   // clean up, so the event handler needn't bother
   do while (i := ascan(aEvents, { | e | e[E_WINDOW] == hWnd } ) ) <> 0
      DelHandler(i)
   enddo
endif
return nil


/*
   Function: ModelessGets()
   Purpose:  Create window to hold data entry screen
*/
static function ModelessGets
local cFile
local getlist := {}
local x
local f
local nFields
local hWnd
local nOldCaret
local nOldarea := select()
local bOldinsert
local nHand
local nWidth
local aRect
local nY
local hInst
local lContinue := .t.
local aButtons := {}
local nButton
cFile := GetOpenFileName(, "*.dbf", "Select a database")
if cFile <> NIL
   if select(StripPath(StripExt(cFile))) == 0
      use (cFile) new
   else
      dbSelectArea(StripPath(StripExt(cFile)))
   endif
   x := alias()

   // I realize that this is not particularly graceful, but the
   // purpose of this demo is not necessarily to show locking
   // techniques... i.e., puh-leeze feel free to substitute your own!
   if ! rlock()
      MessageBox(, "Cannot edit " + x + " database at this time", ;
                   MB_ICONHAND + MB_OK)
      dbSelectArea(nOldarea)
      return nil
   endif

   hWnd := WinNew("Editing " + left(x, 1) + lower(substr(x, 2)) + ;
                  " Database", 0, 0, 350, 250)

   #define BUTTON_HEIGHT  20
   #define B_NEXT   1
   #define B_PREV   2
   #define B_FIRST  3
   #define B_LAST   4
   #define B_ADD    5

   #define TOTAL_BUTTONS  5

   aRect := GetClientRect(hWnd)
   hInst := _GetInstance()
   nWidth := aRect[W_RIGHT] / TOTAL_BUTTONS
   nY := aRect[W_BOTTOM] - BUTTON_HEIGHT
   aadd(aButtons, CreateWindow("button",         ;       // button class name
                               "Next",           ;       // title text
                               WS_CHILD          ;       // child window
                               + WS_VISIBLE      ;       // ... that can be seen
                               + BS_PUSHBUTTON,  ;       // ... a push button
                               0, nY,            ;       // x,y position
                               nWidth,           ;       // width
                               BUTTON_HEIGHT,    ;       // height
                               hWnd,             ;       // parent window
                               B_NEXT,           ;       // unique child id
                               hInst)  )                 // parent's instance

   aadd(aButtons, CreateWindow("button",         ;       // button class name
                               "Prev",           ;       // title text
                               WS_CHILD          ;       // child window
                               + WS_VISIBLE      ;       // ... that can be seen
                               + BS_PUSHBUTTON,  ;       // ... a push button
                               nWidth, nY,       ;       // x,y position
                               nWidth,           ;       // width
                               BUTTON_HEIGHT,    ;       // height
                               hWnd,             ;       // parent window
                               B_PREV,           ;       // unique child id
                               hInst)   )                // parent's instance

   aadd(aButtons, CreateWindow("button",         ;       // button class name
                               "First",          ;       // title text
                               WS_CHILD          ;       // child window
                               + WS_VISIBLE      ;       // ... that can be seen
                               + BS_PUSHBUTTON,  ;       // ... a push button
                               nWidth*2, nY,     ;       // x,y position
                               nWidth,           ;       // width
                               BUTTON_HEIGHT,    ;       // height
                               hWnd,             ;       // parent window
                               B_FIRST,          ;       // unique child id
                               hInst)   )                // parent's instance

   aadd(aButtons, CreateWindow("button",         ;       // button class name
                               "Last",           ;       // title text
                               WS_CHILD          ;       // child window
                               + WS_VISIBLE      ;       // ... that can be seen
                               + BS_PUSHBUTTON,  ;       // ... a push button
                               nWidth*3, nY,     ;       // x,y position
                               nWidth,           ;       // width
                               BUTTON_HEIGHT,    ;       // height
                               hWnd,             ;       // parent window
                               B_LAST,           ;       // unique child id
                               hInst)   )                // parent's instance

   aadd(aButtons, CreateWindow("button",         ;       // button class name
                               "Add",            ;       // title text
                               WS_CHILD          ;       // child window
                               + WS_VISIBLE      ;       // ... that can be seen
                               + BS_PUSHBUTTON,  ;       // ... a push button
                               nWidth*4, nY,     ;       // x,y position
                               nWidth,           ;       // width
                               BUTTON_HEIGHT,    ;       // height
                               hWnd,             ;       // parent window
                               B_ADD,            ;       // unique child id
                               hInst)   )                // parent's instance

   // create Getlist array and load it with GET objects
   nFields := fcount()
   for x := 1 to nFields
      f := field(x)
      @ x-1, 2 say left(f, 1) + lower(substr(f, 2)) get field f ;
                   in window hWnd buttons aButtons ;
                   picture '@K' saycolor 'n/w' getcolor 'n/bg,+w/b'
   next

   // set INS key to toggle shape of cursor and INS mode
   bOldinsert := setkey( K_INS, {|| SetCaret( ;
                     if(readinsert(! readInsert()), SC_NORMAL, SC_FULL_BLOCK))} )
   // set initial cursor setting based on current mode
   nOldcaret := SetCaret( if(readInsert(), SC_FULL_BLOCK, SC_NORMAL) )

   ShowCaret(hWnd)
   nHand := AddHandler(hWnd, ;
            {|nEvent| ModelessEvent(GetList, nEvent, hWnd, nHand, aButtons)})
endif
return nil


/*
   Function: DoBrowse()
   Purpose:  Create window to display TBrowse
*/
static function DoBrowse
static nX := 10, nY := 10
local b
local hWnd
local x
local f
local c
local e
local nFields
local fields_
local cFile
local cNtxfile
local bOldhandler
cFile := GetOpenFileName(, "*.dbf", "Select a database")
if cFile <> NIL
   if select(StripPath(StripExt(cFile))) == 0
      cNtxfile := GetOpenFileName(, "*.ntx", "Select an index file")
      use (cFile) new shared
      if cNtxfile <> NIL
         bOldhandler := errorblock( { | e | bogusindex(e, bOldhandler) } )
         dbsetindex(cNtxfile)
         // verify the validity of this index key
         begin sequence
            x := eval( &("{ || " + indexkey(0) + "}") )
         recover using e
            MessageBox(, "Invalid index (missing " + e:operation + " "  + ;
                         if(e:genCode == EG_NOVAR, "field", "function") + ;
                         ")" + CR + "Browsing " + StripPath(cFile) +      ;
                         " in natural order", "Bogus Index",              ;
                         MB_ICONHAND + MB_OK)
            dbclearindex()
         end sequence
         errorblock(bOldhandler)
      endif
   else
      dbSelectArea(StripPath(StripExt(cFile)))
      dbGoTop()
   endif
   x := alias()
   hWnd := WinNew("Browsing " + left(x, 1) + lower(substr(x, 2)) + ;
                  " Database", nX += 20, nY += 20, 525, 200, ;
                  WS_OVERLAPPEDWINDOW + WS_VSCROLL + WS_HSCROLL)
   b := TBrowseDB(0, 0, maxrow(), maxcol())
   b:headSep := ""
   b:colSep  := "  "
   b:colorSpec := "w/b, +gr/n, w/r, +w/r"

   nFields := fcount()
   // we are going to store an array of fieldnames in the TBrowse
   // cargo instance variable (to be used when inserting new columns)
   b:cargo := array(nFields)
   fields_ := dbstruct()
   for x := 1 to nFields
      b:cargo[x] := field(x)

      // memos must be treated differently than "regular" fields
      if fields_[x][DBS_TYPE] == "M"
         c := TBColumnNew(b:cargo[x], &("{ || if(empty(" + ;
                          b:cargo[x] + "), '<memo>', '<MEMO>') }"))
         c:cargo := { || ShowMemo(b) }
      else
         c := TBColumnNew(b:cargo[x], fieldblock(b:cargo[x]))
      endif

      // set color block to highlight negative numbers
      if valtype(eval(c:block)) == "N"
         c:colorBlock := { | x | if( x < 0, { 3, 4 }, { 1, 2 } ) }
      endif

      b:AddColumn(c)
   next

   // set range of horizontal scrollbar to match # of columns
   // note that if you later decide to add or delete columns,
   // you should do this again to set the new range
   SetScrollRange(hWnd, SB_HORZ, 1, b:colCount, .f.)

   AddHandler(hWnd, {|nEvent| BrowseEvent(nEvent, hWnd, b)})
   InvalidateRect(hWnd)
endif
return nil

/*
   Function: BogusIndex()
   Purpose:  To catch any errors due to invalid indexes
*/
static function bogusindex(oError, bOldhandler)
if oError:genCode == EG_NOFUNC .or. oError:genCode == EG_NOVAR
   break oError
endif
return eval(bOldhandler, oError)

/*
   Function: EditCell()
*/
static function editcell(b, hWnd)
local c := b:getColumn(b:colPos)
local oldval := eval(c:block)
if rlock()
   ShowCaret(hWnd)
   // this is modal because you're changing a record, which should (in general)
   // be locked -- so you should not spend too long over it!
   // however, it could be modeless with a time-out just by using code like
   // that in ValidTest(), and DoTimer()
   readmodal( { getnew(Row(), Col(), c:block, ;
                c:heading, '@K', b:colorSpec) } )
   dbcommit()
   dbunlock()
   HideCaret(hWnd)
endif
return ( eval(c:block) <> oldval )


/*
   Function: WinNew()
*/
static function WinNew(cTitle, nX, nY, nWidth, nHeight, nStyle)
local hInst := _GetInstance()
local nCmdShow := _GetnCmdShow()
local hWin
if nStyle == NIL
   nStyle := WS_OVERLAPPEDWINDOW
endif
hWin := CreateWindow(APP_NAME,           ;       // window class
                    cTitle,              ;       // caption for title bar
                    nStyle,              ;       // window style
                    nX,                  ;       // x co-ordinate
                    nY,                  ;       // y co-ordinate
                    nWidth,              ;       // width
                    nHeight,             ;       // height
                    hWnd,                ;       // hWnd of parent
                    0,                   ;       // hMenu of menu (none yet)
                    hInst)                       // our own app instance
if hWin == 0
   // probably out of resources
   MessageBox( , "Can't create window", "Error", MB_ICONEXCLAMATION + MB_OK)
   return nil
endif
HideCaret(hWin)
// make sure it's displayed ...
ShowWindow(hWin, nCmdShow)
// ... and up to date
UpdateWindow(hWin)
return hWin


/*
   Function: ValidTest()
   Purpose:  Demonstrate picklist validation
   Seealso:  ValidEvent()
*/
static function ValidTest
local cFruit := space(10)
local hWnd := WinNew("Sample Listbox Validation", 0, 0, 300, 100)
local getlist := {}
local nHand
ShowCaret(hWnd)
@ 2,5 say "Type in a fruit" color 'n/w' ;
      get cFruit color '+w/b,+w/b' valid FruitLook()
nHand := AddHandler(hWnd, ;
         {|nEvent| ValidEvent(GetList, nEvent, hWnd, nHand, cFruit)})
return nil


/*
   Procedure: ValidEvent()
*/
static procedure ValidEvent(GetList, nEvent, hWnd, nHand, cFruit)
local hOldWnd
local lDone := (GetList[1] == NIL)    // only do the read until it finishes
if !lDone
   if nEvent == EVENT_REDRAW
      hOldWnd = SelectWindow(hWnd)
      @ 2,5 say "Type in a fruit" color 'n/w'
      SelectWindow(hOldWnd)
   endif
   if ReadModeless(GetList, nEvent) == NIL
      // not yet done
      return                          // wait for next event
   endif
endif
// if we got here, the read has completed (or been abandoned)

if nEvent == EVENT_DESTROY
   // abandoned
   return
endif
if ! lDone
   // get here exactly once
   // (immediately after ReadModeless() returns non-NIL)
   GetList[1] := NIL               // cancel the GET
   HideCaret(hWnd)
   InvalidateRect(hWnd)            // trigger a re-draw (below)
endif
if nEvent == EVENT_REDRAW
   hOldWnd = SelectWindow(hWnd)
   @ 2,5 say "Fruit selected: " + cFruit color '*r/w'
   SelectWindow(hOldWnd)
endif
return


/*
   Function: FruitLook()
*/
static function FruitLook
static aChoices := {"Apple", "Banana", "Coconut", "Grape", "Grapefuit", ;
                   "Lemon", "Lime", "Orange", "Papaya", "Pineapple",   ;
                   "Raisin", "Raspberry", "Strawberry" }
local g := getactive()
local v := upper( g:varGet() )
static aDlg
if ascan(aChoices, { | f | upper(f) = upper(v) } ) == 0
   if aDlg == NIL
      aDlg := CreateDialog("Fruit List",                                      ;
                          WS_CAPTION + WS_SYSMENU + WS_GROUP + WS_TABSTOP     ;
                          + WS_THICKFRAME + WS_VISIBLE + WS_POPUP,            ;
                          100, 30, 100, 100)
      aDlg := AppendDialog(aDlg, "listbox", DLG_LISTBOX,                      ;
                          LBS_STANDARD + WS_CHILD + WS_VISIBLE,               ;
                          10, 10, 80, 60,                                     ;
                          aChoices)
      aDlg := AppendDialog(aDlg, "ok", DLG_BUTTON,                            ;
                          BS_DEFPUSHBUTTON + WS_TABSTOP + WS_CHILD + WS_VISIBLE,;
                          10, 75, 35, 15,                                     ;
                          "&Ok")
      aDlg := AppendDialog(aDlg, "cancel", DLG_BUTTON,                        ;
                          BS_PUSHBUTTON + WS_TABSTOP + WS_CHILD + WS_VISIBLE, ;
                          55, 75, 35, 15,                                     ;
                          "&Cancel")
   endif
   if ModalDialog(aDlg, _GetInstance(), SelectWindow()) <> 0 .and. ;
                  ! GetDialogResult(aDlg, "cancel")
      g:varPut( padr(GetDialogResult(aDlg, "listbox"), len(g:buffer)) )
   endif
endif
return .t.


/*
   Function: DoOpen()
   Purpose:  Open a new text file for editing
*/
static function DoOpen(lNew)
local nSel
local cTempFile
local lContinue := ! EditGetModify(hEdit)
if ! lContinue
   nSel := MessageBox(0, "The text in " + cEditFile + " has changed." + ;
                   CR + "Do you want to save the changes?", ;
                   "Notepad", MB_YESNOCANCEL + MB_ICONEXCLAMATION)
   if nSel == IDYES
      DoSave()
   endif
   lContinue := ( nSel <> IDCANCEL )
endif
if lContinue
   if lNew
      cEditFile := "(Untitled)"
      EditSetText(hEdit, '')
      SetWindowText(hWnd, cAppName + " - " + cEditFile)
   else
      cTempFile := GetOpenFileName(, "*.txt", "Select a text file for editing")
      if cTempFile <> NIL
         if directory(cTempFile)[1][F_SIZE] < 32762
            cEditFile := cTempFile
            EditSetText(hEdit, memoread(cEditFile) )
            SetWindowText(hWnd, cAppName + " - " + upper(cEditFile))
         else
            MessageBox(, "The " + cTempFile + " file is too large." + CR + ;
                   CR + "Please use another editor to edit the file.", ;
                   "Notepad", MB_OK + MB_ICONEXCLAMATION)
         endif
      endif
   endif
endif
SetFocus(hEdit)
return nil


/*
   Function: DoSave()
   Purpose:  Save current contents of edit buffer to a text file
*/
static function DoSave(lNeedfilename)
local x
local y := ''
local nLines
local cTempfile
local hOldcursor
local hCursor
if lNeedfilename == NIL
   lNeedfilename := (cEditFile == "(Untitled)")
endif
if lNeedfilename
   cTempfile := GetSaveFileName(, "*.txt", "Select file to save to")
   if cTempfile <> NIL
      cEditFile := cTempfile
      SetWindowText(hWnd, cAppName + " - " + upper(cEditFile))
   endif
endif
if cEditFile <> "(Untitled)"
   hCursor := LoadCursor( , IDC_WAIT)
   hOldCursor := SetCursor(hCursor)
   nLines := SendMessage(hEdit, EM_GETLINECOUNT) - 2
   for x := 0 to nLines
      y += EditGetLine(hEdit, x) + CRLF
   next

   /*
      Note: it would also be possible to retrieve the entire contents
      of the edit buffer at one fell swoop like so:

      x := SendMessage(hEdit, WM_GETTEXTLENGTH, 0, 0)
      y := space(x)
      SendMessage(hEdit, WM_GETTEXT, x, @y)

      However, this generally requires that you allocate a lot more
      free memory for the internal heap (i.e., the "heapsize" parameter
      in your .DEF file)
   */

   memowrit(cEditFile, y)
   EditSetModify(hEdit, .f.)   // toggle global "has the file changed?" flag
   SetCursor(hOldCursor)
endif
SetFocus(hEdit)
return nil


/*
   Function: DoPrint()
   Purpose:  Print contents of edit buffer
*/
static function DoPrint
local aPrintDlg := array(PD_LENGTH)   // from COMMDLG.CH
local nCurrLine
local nLines
local hCursor
local hOldcursor
local nLineHeight
local x

// set up initial values for printer dialog
// see COMMDLG.CH for other flags that can be used
aPrintDlg[PD_HWNDOWNER] := SelectWindow()
aPrintDlg[PD_NFLAGS]    := PD_HIDEPRINTTOFILE + PD_RETURNDC
x := PrintDlg(aPrintDlg)
if x == NIL .or. ! x
   // cancelled by the user
   return nil
endif
nLineHeight := GetDeviceCaps(aPrintDlg[PD_HDC], VERTRES) / 66
hCursor := LoadCursor( , IDC_WAIT)
hOldCursor := SetCursor(hCursor)
nLines := SendMessage(hEdit, EM_GETLINECOUNT) - 2
StartDoc(aPrintDlg[PD_HDC], cAppName + upper(cEditFile))
for x := 1 to aPrintDlg[PD_NCOPIES]
   nCurrLine := 0
   StartPage(aPrintDlg[PD_HDC])
   do while nCurrLine <= nLines
      // currently using fixed page length of 60 lines
      TextOut(aPrintDlg[PD_HDC], 0, (nCurrLine % 61) * nLineHeight, EditGetLine(hEdit, nCurrLine) )
      if nCurrLine > 0 .and. nCurrLine % 61 == 0
         EndPage(aPrintDlg[PD_HDC])
         StartPage(aPrintDlg[PD_HDC])
      endif
      nCurrLine++
   enddo
   EndPage(aPrintDlg[PD_HDC])
next
EndDoc(aPrintDlg[PD_HDC])
DeleteDC(aPrintDlg[PD_HDC])
SetCursor(hOldCursor)
return nil


static function DoUndo
if EditCanUndo(hEdit)
   EditUndo(hEdit)
endif
return nil


static function DoCut
SendMessage(hEdit, WM_CUT, 0)
return nil


static function DoCopy
SendMessage(hEdit, WM_COPY, 0)
return nil


static function DoClear
SendMessage(hEdit, WM_CLEAR, 0)
return nil


static function DoPaste
SendMessage(hEdit, WM_PASTE, 0)
return nil


static function TimeDate
EditSetSel(hEdit, -1, -1, .f.)
EditReplaceSel(hEdit, substr(time(), 1, 5) + '  ' + dtoc(date()))
return nil

// end of file DEMO2.PRG
