/*---------------------------------------------------------------------------\
| BLPRF501.prg                        Blinker 1.5x Profiler for Clipper 5.01 |
|                                                                            |
| This is a Clipper 5.01 specific re-write of the profiler included with     |
| Blinker 1.5x (originally written and donated by Frederick W. Stangl,       |
| President of Dynamic Performance Inc., Philadelphia, PA).                  |
|                                                                            |
| This version of the profiler was written by Todd C. MacDonald, Senior      |
| Programmer/Analyst of Key Services Corporation, Albany, NY, and is also    |
| donated to the Blinker community.                                          |
|                                                                            |
| This version takes advantage of 5.01's nested arrays to allow viewing of   |
| the profiling statistics in any user-selectable order.                     |
|                                                                            |
| The setup procedure has been simplified.  To use the profiler, include     |
| the following lines at the beginning of the main Clipper module in your    |
| application (cColorStr is an optional setcolor() type string for the       |
| profiler to use as it's display colors):                                   |
|                                                                            |
|   #ifdef PROFILER                                                          |
|     BlPrfInit( [<cColorStr>] )                                             |
|   #endif                                                                   |
|                                                                            |
| Compile this profiler module as follows:                                   |
|                                                                            |
|   CLIPPER BLPRF501 /N                                                      |
|                                                                            |
| Compile your main module as follows:                                       |
|                                                                            |
|   CLIPPER yourmodule /dPROFILER yourswitches                               |
|                                                                            |
| Link as follows:                                                           |
|                                                                            |
|   BLINKER DEBUG FI yourmodule, BLPRF501                                    |
|                                                                            |
| To invoke the profiler from any wait-state, press [Alt]-[P].               |
\---------------------------------------------------------------------------*/

#include "inkey.ch"
#include "setcurs.ch"

// handles for nested array elements
#define P_FUNC  1
#define P_SIZE  2
#define P_LOAD  3
#define P_CALL  4
#define P_SVC   5
#define P_BAR   6

// containers for statistics
STATIC nMemHi := 0
STATIC nMemLo := 0
STATIC aStats := {}

// profiler display color
STATIC cClrStr


//--------------------------------------------------------------------------//
  FUNCTION BlPrfInit( cColorStr )
//--------------------------------------------------------------------------//

// This function should be called at the beginning of the first module to be
// executed in your application.

// set colors (if nil, current setcolor() at time of display will be used)
cClrStr := cColorStr

// initialize high and low memory containers
nMemHi := memory( 0 )
nMemLo := nMemHi

// [Alt]+[P] pops up profile statistics
set key K_ALT_P to ovl_stat

// turn Blinker's profiling mode on
bliprfmod( .t. )

RETURN nil
//--------------------------------------------------------------------------//


//--------------------------------------------------------------------------//
  FUNCTION BlPrfPrg
//--------------------------------------------------------------------------//

// Profiling function for gathering operating statistics.  This function gets 
// called every time an overlay gets called.

LOCAL nFree := blitotcal()
LOCAL cFunc := padr( blicurnme(), 10 )
LOCAL nPos  := ascan( aStats, { | aFunc | aFunc[ P_FUNC ] == cFunc } )

// if we don't already have stats for the function, add them to array
if nPos = 0

  aadd( aStats, { cFunc, blicursiz(), nil, nil, nil, nil } )
  nPos := len( aStats )

endif

aStats[ nPos, P_LOAD ] := blicurdsk()
aStats[ nPos, P_CALL ] := blicurcal()

// record free memory every 10th call (so memory(0) won't bog us down as much)
if nFree = int( nFree / 10 ) * 10

  nFree := memory( 0 )

  if nFree > nMemHi

    nMemHi := nFree

  elseif nFree < nMemLo

    nMemLo := nFree

  endif

endif

RETURN nil
//--------------------------------------------------------------------------//


//--------------------------------------------------------------------------//
  FUNCTION ovl_stat
//--------------------------------------------------------------------------//

// Display function for viewing statistics via hot-key.

#define WIN_T  0
#define WIN_L  0
#define WIN_B  WIN_T + 16
#define WIN_R  WIN_L + 42

#define WIN_FRAME  chr( 219 ) + chr( 223 ) + chr( 219 ) + chr( 219 ) + ;
  chr( 219 ) + chr( 220 ) + chr( 219 ) + chr( 219 ) + ' '

#define BAR_CHR  chr( 254 )

LOCAL nFree     := memory( 0 )
LOCAL lLastMode := bliprfmod( .f. )
LOCAL nLastRow  := row()
LOCAL nLastCol  := col()
LOCAL cLastScr1 := savescreen( WIN_T, WIN_L, WIN_B, WIN_R )
LOCAL cLastClr  := setcolor( cClrStr )
LOCAL nLastCrs  := setcursor( SC_NONE )
LOCAL nStats    := len( aStats )
LOCAL acDisp    := {}
LOCAL lExit     := .f.

LOCAL nMaxCalls, nKey, cLastScr2, lContinue
LOCAL lLastCan, bAltF, bAltS, bAltV, bAltL, bAltC

// record high/low memory
if nFree > nMemHi

  nMemHi := nFree

elseif nFree < nMemLo

  nMemLo := nFree

endif

// show overall statistics
@ WIN_T, WIN_L, WIN_B, WIN_R box WIN_FRAME
@ WIN_T +  1, WIN_L + 2 say '       Runtime Overlay Analysis'
@ WIN_T +  2, WIN_L + 2 say replicate( chr( 196 ), 39 )
@ WIN_T +  3, WIN_L + 2 say 'Highest Free Pool Memory:       ' + ;
  transform( nMemHi * 1024, '999,999' )
@ WIN_T +  4, WIN_L + 2 say 'Current Free Pool Memory:       ' + ;
  transform( nFree * 1024, '999,999' )
@ WIN_T +  5, WIN_L + 2 say 'Lowest Free Pool Memory:        ' + ;
  transform( nMemLo * 1024,'999,999' )
@ WIN_T +  6, WIN_L + 2 say 'Blinker Overlay Pool OpSize:    ' + ;
  transform( bliovlops(), '999,999' )
@ WIN_T +  7, WIN_L + 2 say 'Current Overlay Pool Size:      ' + ;
  transform( bliovlsiz(), '999,999' )
@ WIN_T +  8, WIN_L + 2 say 'Total Func Size Since Startup:  ' + ;
  transform( blitotsiz(), '999,999' )
@ WIN_T +  9, WIN_L + 2 say 'Functions Currently Loaded      ' + ;
  transform( blitotlod(), '999,999' )
@ WIN_T + 10, WIN_L + 2 say 'Functions Currently Active:     ' + ;
  transform( blitotact(), '999,999' )
@ WIN_T + 11, WIN_L + 2 say 'Total Calls Since Startup:      ' + ;
  transform( blitotcal(), '999,999' )
@ WIN_T + 12, WIN_L + 2 say 'Total Disk Loads Since Startup: ' + ;
  transform( blitotdsk(), '999,999' )
@ WIN_T + 13, WIN_L + 2 say '% Serviced from Overlay Pool:   ' + ;
  transform( if( blitotcal() = 0, 0, 100 * ;
  ( 1 - blitotdsk() / blitotcal() ) ), ' 999.9%' )
@ WIN_T + 15, WIN_L + 2 say ;
  padc( 'Calculating percentages...', WIN_R - WIN_L - 3 )

// descending sort by number of calls
asort( aStats,,, { | x, y | ;
  str( x[ P_CALL ] ) + x[ P_FUNC ] > str( y[ P_CALL ] ) + y[ P_FUNC ] } )

// get max number of calls from first element
nMaxCalls := aStats[ 1, P_CALL ]

// calculate service percentages and make graph bars
aeval( aStats, { | aFunc | ;
  aFunc[ P_SVC ] := iif( aFunc[ P_CALL ] = 0, 0, ;
    100 * ( 1 - aFunc[ P_LOAD ] / aFunc[ P_CALL ] ) ), ;
  aFunc[ P_BAR ] := replicate( BAR_CHR, ;
    int( 1 + 40 * aFunc[ P_CALL ] / nMaxCalls ) ) } )

@ WIN_T + 15, WIN_L + 2 say '[Enter]-Detail               [Esc]-Exit'

do while !lExit

  // wait for keypress
  nKey := inkey( 0 )

  if nKey = K_ENTER

    // use ACHOICE to display a scrolling window of statistics
    cLastScr2 := savescreen( 0, 0, maxrow(), maxcol() )

    cls

    @ 00, 00 say ' Function   Size  Svc %  Loads  Calls  ' + ;
      '          Calls ' + chr( 246 ) + ' Maximum Calls          '

    @ 01, 00 say replicate( chr( 196 ), 10 ) + ' ' + ;
      replicate( chr( 196 ), 6 ) + ' ' + replicate( chr( 196 ), 5 ) + ' ' + ;
      replicate( chr( 196 ), 6 ) + ' ' + replicate( chr( 196 ), 7 ) + ' ' + ;
      replicate( chr( 196 ), 41 )

    @ maxrow(), 0 say padc( 'Please wait...', 80 )

    lContinue := .t.

    while lContinue

      // build array of strings for achoice display
      acDisp := {}
      aeval( aStats, { | aFunc | aadd( acDisp, ;
        aFunc[ P_FUNC ] + ;
        transform( aFunc[ P_SIZE ], ' 99,999' ) + ;
        transform( aFunc[ P_SVC ],  ' 999.9' ) + ;
        transform( aFunc[ P_LOAD ], ' 99,999' ) + ;
        transform( aFunc[ P_CALL ], ' 999,999' ) + ' ' + ;
        aFunc[ P_BAR ] ) } )

      // set up hot-keys
      lLastCan := setcancel( .f. )
      bAltF    := setkey( K_ALT_F, { || SortByFunc() } )
      bAltS    := setkey( K_ALT_S, { || SortBySize() } )
      bAltV    := setkey( K_ALT_V, { || SortBySvc()  } )
      bAltL    := setkey( K_ALT_L, { || SortByLoad() } )
      bAltC    := setkey( K_ALT_C, { || SortByCall() } )

      // use ACHOICE to display a scrolling window of statistics
      @ maxrow(), 0 say padc( ' [Alt] + [F]unction, [S]ize, ser[V]ice, ' + ;
        '[L]oads, [C]alls-Order       [Esc]-Exit', 80 )
      achoice( 2, 0, maxrow() - 2, 79, acDisp )

      // restore hot-keys to previous settings
      setkey( K_ALT_F, bAltF )
      setkey( K_ALT_S, bAltS )
      setkey( K_ALT_V, bAltV )
      setkey( K_ALT_L, bAltL )
      setkey( K_ALT_C, bAltC )
      setcancel( lLastCan )

      if lastkey() = K_ESC

        lContinue := .f.

      endif

    end

    restscreen( 0, 0, maxrow(), maxcol(), cLastScr2 )

  elseif nKey = K_ESC

    lExit := .t.

  endif

end

// restore screen
setpos( nLastRow, nLastCol )
restscreen( WIN_T, WIN_L, WIN_B, WIN_R, cLastScr1 )
setcolor( cLastClr )
setcursor( nLastCrs )

// restore profiling mode
bliprfmod( lLastMode )

RETURN nil
//--------------------------------------------------------------------------//


//--------------------------------------------------------------------------//
  STATIC FUNCTION SortByFunc
//--------------------------------------------------------------------------//

@ maxrow(), 0 say padc( 'Sorting by Function...', 80 )

asort( aStats,,, { | x, y | x[ P_FUNC ] < y[ P_FUNC ] } )

// force exit from achoice
keyboard chr( K_ENTER )

RETURN nil
//--------------------------------------------------------------------------//


//--------------------------------------------------------------------------//
  STATIC FUNCTION SortBySize
//--------------------------------------------------------------------------//

@ maxrow(), 0 say padc( 'Sorting by Size...', 80 )

asort( aStats,,, { | x, y | ;
  str( x[ P_SIZE ] ) + x[ P_FUNC ] > str( y[ P_SIZE ] ) + y[ P_FUNC ] } )

// force exit from achoice
keyboard chr( K_ENTER )

RETURN nil
//--------------------------------------------------------------------------//


//--------------------------------------------------------------------------//
  STATIC FUNCTION SortBySVC
//--------------------------------------------------------------------------//

@ maxrow(), 0 say padc( 'Sorting by Service Percentage...', 80 )

asort( aStats,,, { | x, y | ;
  str( x[ P_SVC ] ) + x[ P_FUNC ] > str( y[ P_SVC ] ) + y[ P_FUNC ] } )

// force exit from achoice
keyboard chr( K_ENTER )

RETURN nil
//--------------------------------------------------------------------------//


//--------------------------------------------------------------------------//
  STATIC FUNCTION SortByLoad
//--------------------------------------------------------------------------//

@ maxrow(), 0 say padc( 'Sorting by Loads...', 80 )

asort( aStats,,, { | x, y | ;
  str( x[ P_LOAD ] ) + x[ P_FUNC ] > str( y[ P_LOAD ] ) + y[ P_FUNC ] } )

// force exit from achoice
keyboard chr( K_ENTER )

RETURN nil
//--------------------------------------------------------------------------//


//--------------------------------------------------------------------------//
  STATIC FUNCTION SortByCall
//--------------------------------------------------------------------------//

@ maxrow(), 0 say padc( 'Sorting by Calls...', 80 )

asort( aStats,,, { | x, y | ;
  str( x[ P_CALL ] ) + x[ P_FUNC ] > str( y[ P_CALL ] ) + y[ P_FUNC ] } )

// force exit from achoice
keyboard chr( K_ENTER )

RETURN nil
//--------------------------------------------------------------------------//
