/****************************************************************************
*   BLOCK.PRG - sample game written in Clipper                              *
*   Written in September 1991 By Skip Key CIS 73130,2102                    *
*   Compiled with Clipper 5.01  Uses the Nanforum Toolkit                   *
*   This is an original work and is released into the public domain         *
*   If you improve it or fix any bugs, I would like to know, however.       *
*****************************************************************************/

/****************************************************************************
*   This game was written a few months ago when a friend of mine was putting*
*   down Clipper with a comment like, "Yeah, you can write database programs*
*   with it, but can you write games?"  So, just to show him I sat down and *
*   wrote this simple little game in about two hours.  It uses quite a few  *
*   of Clipper 5 features, and in fact would have taken much longer to write*
*   in Clipper Summer 87.  I hope that you have as much fun playing it as   *
*   I did writing it.                                                       *
*                                                                           *
*   Skip Key 73130,2102                                                     *
*****************************************************************************/

#include "inkey.ch"
#include "error.ch"
#define NUMSHAPES 7

static shapetable, curshape, currow, curcol, orientation, board, brow, bcol
static gameover:=.f., score:=0, level:=1, rows:=0, timeconst:=0

function blocks(up, left)
    local cScreen, lCont:=.t., n, nKey, oldnum, oldcolor:=setcolor('w/n'), oldcursor:=setcursor(0)
    if up==nil
        up:=0
    end
    if left==nil
        left:=0
    end
    brow:=up
    bcol:=left
    cScreen:=SaveScreen()
    oldnum:=ft_numlock(.t.) /*From Nanforum toolkit, could be left out*/
    SetUpScreen(up, left)
    GetTime()               /*sets static timeconst*/
    while lCont
        if gameover
            lCont:=.f.
        else
            DisplayStats()
            MoveShape(curshape)
            nKey:=0
            for n:= 1 to timeconst
                nKey:=inkey()
                if curshape#nil .and. nKey#0
                    n:=ProcessKey(nKey, n)  &&Changed here
                end
            next
            if nKey==K_ESC
                lCont:=.f.
            end
        end
    end
    inkey(0)
    ft_numlock(oldnum)
    RestScreen(0, 0, maxrow(), maxcol(), cScreen)
    setcolor(oldcolor)
    setcursor(oldcursor)
return nil


/****************************************************************************
*   FUNCTION    SetupScreen(up, left)                                       *             *
*   PARAMETERS  up - row of upper left hand corner                          *
*               left - column of upper left hand corner                     *
*   RETURNS     nil                                                         *
*   PURPOSE     This function draws the board and sets up the shape tables  *
*   NOTES       The shapetable is a nested array that contains the positions*
*               of the blocks in their four rotations.  The innermost array *
*               is an array of eight numbers.  These eight numbers are      *
*               really four coordinates that are relative to the center of  *
*               the piece.  So, for example, the very first one             *
*               {0,-1, 0,0, 0,1, 0,2}   translates to:                      *
*                                                                           *
*                                   X   (0, -1)                             *
*                                   X   (0, 0)  CENTER                      *
*                                   X   (0, 1)                              *
*                                   X   (0, 2)                              *
*                                                                           *
*               those arrays are grouped by fours into the next higher      *
*               array.  Each of the four is for the four different          *
*               orientations.  The outermost array is an array of each of   *
*               the different shapes.  So shapetable[1] would be the first  *
*               shape, shapetable[1][1] would be the first orientation of   *
*               the first piece, and shapetable[1][1][1] would be the       *
*               relative row position of the first block in the first       *
*               orientation of the first piece.                             *
*                                                                           *
*               The board is a nested array of characters that will contain *
*               an 'X' if there is a block there and a space if not.        *
*****************************************************************************/

static function SetupScreen(up, left)
    local m, n
    @up, left, up+20, left+42 BOX "ͻȺ " COLOR "R/N"
    @up+1, left+1,up+18, left+12 BOX " Ⱥ "
    board:={}
    for n:=1 to 16
        aadd(board, {})
        for m:=1 to 10
            aadd(board[n], ' ')
        next
    next
    @ up+5, left+25 say "SCORE"
    @ up+8, left+25 say "ROWS"
    @ up+11, left+25 say "LEVEL"
    shapetable:={{{0,-1, 0,0, 0,1, 0,2},;
                {-2,0, -1,0, 0,0, 1,0},;
                {0,-1, 0,0, 0,1, 0,2},;
                {-2,0, -1,0, 0,0, 1,0}},;
                {{-1,0, -1,1, 0,0, 0,1},;
                {-1,0, -1,1, 0,0, 0,1},;
                {-1,0, -1,1, 0,0, 0,1},;
                {-1,0, -1,1, 0,0, 0,1}},;
                {{0,-1, 0,0, 0,1, 1,0},;
                {-1,0, 0,0, 0,1, 1,0},;
                {-1,0, 0,-1, 0,0, 0,1},;
                {-1,0, 0,-1, 0,0, 1,0}},;
                {{-1,-1, -1,0, 0,0, 0,1},;
                {-1,1, 0,0, 0,1, 1,0},;
                {-1,-1, -1,0, 0,0, 0,1},;
                {-1,1, 0,0, 0,1, 1,0}},;
                {{0,0, 0,1, 1,-1, 1,0},;
                {-1,0, 0,0, 0,1, 1,1},;
                {0,0, 0,1, 1,-1, 1,0},;
                {-1,0, 0,0, 0,1, 1,1}},;
                {{-1,-1, 0,-1, 0,0, 0,1},;
                {-1,0, 0,0, 1,-1, 1,0},;
                {0,-1, 0,0, 0,1, 1,1},;
                {-1,0, -1,1, 0,0, 1,0}},;
                {{-1,1, 0,-1, 0,0, 0,1},;
                {-1,-1, -1,0, 0,0, 1,0},;
                {0,-1, 0,0, 0,1, 1,-1},;
                {-1,0, 0,0, 1,0, 1,1}}}
return nil

/****************************************************************************
*   FUNCTION    GetTime()                                                   *
*   PARAMETERS  none                                                        *
*   RETURNS     nil                                                         *
*   This function sets timeconst, a static variable, based on the number of *
*   loops that occur in a second.                                           *
*****************************************************************************/

static function GetTime()
local starttime
starttime:=seconds()
while seconds()-starttime<1 /*could change this to change speed of game*/
    timeconst++
end
return nil

/****************************************************************************
*   FUNCTION    MoveShape()                                                 *
*   PARAMETERS  none                                                        *
*   RETURNS     nil                                                         *
*   This function determines whether or not a shape can descend a row, and  *
*   if so, causes it to do so.  It also initiates new pieces.               *
*****************************************************************************/

static function MoveShape()
    if curshape==nil
        curshape:=int(ft_rand1(NUMSHAPES))+1
        currow:=1
        curcol:=5
        orientation:=1
    end
    DispBegin()
    DrawShape(.f.)
    if NotBlocked()
        currow++
        DrawShape(.t.)
    else
        FallRow()
        DrawShape(.t.)
        RemoveLines()
    end
    if curshape==nil
        curshape:=int(ft_rand1(NUMSHAPES))+1
        currow:=1
        curcol:=5
        orientation:=1
        drawshape(.t.)
    end
    DispEnd()
return nil

/****************************************************************************
*   FUNCTION    ProcessKey(nKey, n)                                         *
*   PARAMETERS  nKey - value of key pressed                                 *
*               n - current loop position against timeconst                 *
*   RETURNS     now value to set loop to                                    *
*               This function processes the keys that the game accepts.     *
*               note that an escape will exit the game at any time          *
*****************************************************************************/

static function ProcessKey(nKey, n)
    do case
        case nKey==K_ESC
            n:=timeconst
        case nKey==52   &&numeric 4
            if NotBlocked(nKey)
                DrawShape(.f.)
                curcol--
                DrawShape(.t.)
            end
        case nKey==54   &&numeric 6
            if NotBlocked(nKey)
                DrawShape(.f.)
                curcol++
                DrawShape(.t.)
            end
        case nKey==53   &&numeric 5
            if NotBlocked(nKey)
                DrawShape(.f.)
                orientation:=iif(++orientation>4, 1, orientation)
                DrawShape(.t.)
            end
        case nKey==50
            DrawShape(.f.)
            FallRow()
            DrawShape(.t.)
            RemoveLines()
    end
return n

/****************************************************************************
*   FUNCTION    DrawShape(visible)                                          *
*   PARAMETERS  visible - logical indicating whether to draw or erase       *
*   RETURNS     nil                                                         *
*               This function either draws the shape at the current         *
*               row and column or erases it depending on visible            *
*****************************************************************************/

static function DrawShape(visible)
    local oldcolor, row:=brow+currow+1, col:=bcol+curcol+1, fillchar, n
    iif(visible, fillchar:='X', fillchar:=' ')
    oldcolor:=iif(visible, setcolor('0/'+str(curshape)), setcolor())
    for n:=1 to 8
        DevPos(shapetable[curshape][orientation][n]+row, shapetable[curshape][orientation][++n]+col)
        DevOut(fillchar)
    next
    setcolor(oldcolor)
return nil


/****************************************************************************
*   FUNCTION    NotBlocked(direction)                                       *
*   PARAMETERS  direction - inkey value of arrow pressed, or down if nil    *
*   RETURNS     whether or not the shape can move in that direction         *
*               This function tests whether or not the shape would overlap  *
*               any current blocks if it were moved in this direction.      *
*               The only interesting thing about this function is that      *
*               instead of testing for array out of bounds, I use a begin   *
*               sequence ... end sequence construct and install my own error*
*               handler and assume that any array out of bounds errors in   *
*               this section are caused by attempting to move the piece off *
*               of the screen.                                              *
*****************************************************************************/

static function NotBlocked(direction)
    local retval:=.t., testrow:=currow, testcol:=curcol, testorient:=orientation, n, trow, tcol, bOldErr, ObjErr
    bOldErr:=ErrorBlock({|errObj| ArrErr(errObj)}) /*install my error handler*/
    do case
        case direction==nil
            testrow++
        case direction==52
            testcol--
        case direction==53
            testorient:=iif(++testorient>4, 1, testorient)
        case direction==54
            testcol++
    end
    for n:=1 to 8
        trow:=testrow   /*if any of the array accesses here are out of bounds*/
        tcol:=testcol   /*my error handler will be invoked which will break
                          to the recover*/
        trow+=shapetable[curshape][testorient][n]
        tcol+=shapetable[curshape][testorient][++n]
        begin sequence
            if board[trow][tcol]=='X'
                retval:=.f.
            end
        recover using objErr
            if objErr:genCode==EG_BOUND   /*if out of bounds return false*/
                retval:=.f.
            else
                eval(bOldErr, objErr)     /*else chain to other error handler*/
            end
        end sequence
    next
    errorblock(bOldErr)     /*reintall old handler*/
return retval

/*this function simply breaks so that the begin sequence recover end sequence
    construct can be used*/

static function ArrErr(objErr)
    break objErr
return nil

static function FallRow()
    local n, bOldErr, objErr, scorerow:=0
    while NotBlocked()
        currow++
        scorerow++
    end
    score+=scorerow*level
    bOldErr:=errorblock({|errObj| ArrErr(errObj)})
    begin sequence
    for n:=1 to 8
        board[shapetable[curshape][orientation][n]+currow][shapetable[curshape][orientation][++n]+curcol]:='X'
    next
    recover using objErr
    if objErr:genCode==EG_BOUND
        gameover:=.t.
    end
    end sequence
    errorblock(bOldErr)
return nil

static function RemoveLines()
    local n, m
    for n:=1 to 16
        if (ascan(board[n], ' ')==0)
            score+=25
            rows++
            scroll(brow+1, bcol+2, brow+n+1, bcol+11, -1)
            adel(board,n)
            ains(board, 1)
            board[1]:={}
            for m:=1 to 10
                aadd(board[1], ' ')
            next
        end
    next
    curshape:=nil
return nil

static function DisplayStats()
    static deltatime
    if deltatime==nil
        deltatime:=int(timeconst/10)
    end
    if int(rows/10)>(level-1)
        level++
        timeconst-=deltatime
    end
    @ brow+5, bcol+31 say score
    @ brow+8, bcol+31 say rows
    @ brow+11, bcol+31 say level
return nil
