/*
    The source code contained within this file is protected under the
    laws of the United States of America and by International Treaty.
    Unless otherwise noted, the source contained herein is:

    Copyright (c)1990-1994 BecknerVision Inc - No Rights Reserved

    Written by John Wm Beckner
    BecknerVision Inc

        ** THIS FILE IS PUBLIC DOMAIN **

        ** Use of this code is at your own risk, absolutely no guarantees
        ** are made to its usefulness  - you must determine for yourself if
        ** this code is of any use to you.  If you do not agree with this, do
        ** not use this code. **
*/

#include "beckner.inc"
#include "inkey.ch"
#include "setcurs.ch"

////////////////
////////////////
//
// Purpose:
//    Database file browser
//
// Syntax:
//    fBrowse([<nTR>], [<nTC>], [<nBR>], [<nBC>], [<aColumn>], [<nFreeze>],
//            [<cF2Expr>], [<bSeek>], [<cColor>], [<lDisplay>],
//            [<lSuppress>], [<cSubAlias>]) -> NIL
//
// Formal Arguments: (12)
//    Name        Description
//    ___________ ____________
//    nTR         Top row coordinate [2]
//    nTC         Left column coordinate [2]
//    nBR         Bottom row coordinate [MaxRow()-2]
//    nBC         Right column coordinate [MaxCol()-2]
//    aColumn     Column definitions [all current work area fields]
//    nFreeze     Columns to freeze [1, see description]
//    cF2Expr     User-defined expression [none]
//    bSeek       Code block defining SEEK expression [none]
//    cColor      Color table [see description
//    lDisplay    .y. for display-only table [.n.]
//    lSuppress   .y. to suppress record number column [.n.]
//    cSubAlias   for future expansion, not used [NIL]
//
// Returns:
//    NIL
//
// Examples:
//    #include "beckner.inc"
//    FUNCTION TestIt()
//       CreateTest()
//       fBrowse()
//       CLOSE Test
//    ENDFUNCTION
//
//    #include "alias.ch"
//    STATIC FUNCTION CreateTest()
//       fCreateDbf("Test/Name/C/32/Balance/N/12/2/Date/D/Over21/L/Notes/M")
//       USE Test NEW EXCLUSIVE
//       ADDRECORD ALIAS Test
//       Test->Name    := "Abbott, Jake"
//       Test->Balance := 0.00
//       Test->Date    := CtoD("9/26/94")
//       Test->Over21  := .y.
//       Test->Notes   := ""
//       ADDRECORD ALIAS Test
//       Test->Name    := "Beckner, John Wm"
//       Test->Balance := 15000.00
//       Test->Date    := CtoD("9/26/94")
//       Test->Over21  := .y.
//       Test->Notes   := "These are notes!"
//       ADDRECORD ALIAS Test
//       Test->Name    := "Beckner, Elizabeth Anne"
//       Test->Balance := 0.58
//       Test->Date    := CtoD("12/18/96")
//       Test->Over21  := .n.
//       Test->Notes   := ""
//       ADDRECORD ALIAS Test
//       Test->Name    := "Beckner, Joseph Alan"
//       Test->Balance := 1.68
//       Test->Date    := CtoD("6/10/98")
//       Test->Over21  := .n.
//       Test->Notes   := "These are too!"
//       ADDRECORD ALIAS Test
//       Test->Name    := "Zumundi, Beavis"
//       Test->Balance := 123.45
//       Test->Date    := CtoD("")
//       Test->Over21  := .y.
//       Test->Notes   := ""
//       GO TOP
//    ENDFUNCTION
//
// Files:
//    (current work area)
//
// Description:
//    Fully interactive databse file browser.  The first 4 parameters are the
//    screen coordinates and default to 2, 2, MaxRow()-2 and MaxCol()-2,
//    respectively.  aColumnInfo defaults to all field information.  To
//    create a display-only column, use the code block {|| FieldGet(<nPos>)},
//    where <nPos> is the field's ordinal position.  To create an editable
//    column, use the code block returned from this function:
//
//          FieldWBlock(<cField>, <cAlias>)
//
//    where <cField> is the field name and <cAlias> is the work area name.
//    <nFreeze> defaults to 1 if not <lSuppress>, or 0 otherwise.  If you
//    specify <nFreeze>, and the record number column is to be displayed
//    (ie NOT <lSuppress>), then nFreeze is incremented to include the
//    record number column.  <cF2Expr> is any code block which you want to
//    execute if you press the <F2> key.  Any existing <F2> code block is
//    saved and restored by this function.  <bSeek> is a code block which
//    should return a valid SEEK string.  This SEEK is performed anytime
//    <alt-S> is pressed.  <cColor> is a table of color pairs and defaults
//    to "N/W, N/BG, B/W, B/BG, B/W, B/BG, R/W, B/R, N/W, W/N, N/W, W/N, N/W,
//    W/N, N+/W, W*/N".  This includes 8 color pairs or 16 individual colors.
//    If <lDisplay>, then the columns are display-only, otherwise they may be
//    edited.  This only applies if <aColumn> is not passed, and defaults to
//    .false..  <lSuppress> causes the record number column to be suppressed.
//    <cSubAlias> is not currently used.
//
// See Also:
//    BROWSE
//    fDataEdit()
//
// Category:
//    File Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

FUNCTION fBrowse(nTopRow, nTopCol, nBottomRow, nBottomCol, aColumnInfo,;
      nFreeze, cF2Expr, bSeek, cColorTable, lDisplayOnly, lSuppressRec,;
      cSubAlias)
   LOCAL oTable, nCtr, oCol, uCheck, nCursorShape, nKey, cCurColors, nAdder
   LOCAL nFrzWidths := 0, oPasteCol, lAltC := SetCancel(.n.), oTempCol, aRC
   LOCAL GetList := {}, aTemp := {}, nTemp, nOption
   vCursSave()
   vSave()
   cCurColors := SetColor()
   DEFAULT lSuppressRec TO .n.,;
         lDisplayOnly TO .n.,;
         cColorTable TO iif(IsColor(),;
         "N/W, N/BG, B/W, B/BG, B/W, B/BG, R/W, B/R",;
         "N/W, W/N, N/W, W/N, N/W, W/N, N+/W, W*/N"),;
         nTopRow TO 2,;
         nTopCol TO 2,;
         nBottomRow TO MaxRow()-2,;
         nBottomCol TO MaxCol()-2
   nAdder := iif(lSuppressRec, 0, 1)
   IF aColumnInfo=NIL
      aColumnInfo := {}
      FOR nCtr := 1 to fCount()
         aAdd(aColumnInfo, {StrTran(FieldName(nCtr), "_", " "), NIL})
         aColumnInfo[nCtr, 1] := Upper(Left(aColumnInfo[nCtr, 1], 1))+;
               Lower(SubStr(aColumnInfo[nCtr, 1], 2))
         aColumnInfo[nCtr, 2] := iif(lDisplayOnly,;
               {||FieldGet(nCtr)},;
               FieldWBlock(FieldName(nCtr), Select()))
      NEXT
   ENDIF
   oTable := tBrowseDB(nTopRow, nTopCol, nBottomRow, nBottomCol)
   IF !lSuppressRec
      oTable:addColumn(tbColumnNew("  Rec #", {||Transform(RecNo(), "999,999")}))
   ENDIF
   WHILE (nTemp := Len(aColumnInfo))>125
      nOption := Alert("Too many columns to browse ("+sMake(nTemp)+")",;
            {"Delete columns", "Abort"})
      IF nOption!=1
         RETURN NIL
      ENDIF
      vSave(0, 0, 24, 14)
      @ 0, 0 CLEAR TO 24, 14
      @ 0, 0 TO 24, 14 DOUBLE
      aEval(a1From2(aMarkList(1, 1, 23, 13, a1From2(aColumnInfo, 1)), 1),;
            {|nElement| aColumnInfo[nElement] := NIL})
      vRestore()
      aPack(aColumnInfo)
   ENDWHILE
   FOR nCtr := 1 to Len(aColumnInfo)
      oTable:addColumn(tbColumnNew(aColumnInfo[nCtr, 1],;
            aColumnInfo[nCtr, 2]))
   NEXT
   oTable:freeze     := iif(nFreeze=NIL, nAdder, nFreeze+nAdder)
   oTable:Cargo      := .n.
   oTable:SkipBlock  := {|nRecs|SkipTo(nRecs, oTable)}
   oTable:headSep    := Chr(205)+Chr(209)+Chr(205)
   oTable:colSep     :=  " "+Chr(179)+" "
   oTable:colorSpec  := cColorTable
   IF !lSuppressRec
      oCol              := oTable:getColumn(1)
      oCol:colorBlock   := {||iif(Deleted(), {7, 8}, {5, 6})}
   ENDIF
   FOR nCtr := iif(lSuppressRec, 1, 2) to oTable:colCount
      oCol     := oTable:getColumn(nCtr)
      uCheck   := Eval(oCol:block)
      iif(ValType(uCheck)="C", oCol:width := Max(Min(50, Len(uCheck)),;
            Len(oCol:heading)),)
      IF ValType(uCheck)!="N"
         oCol:defColor := {3, 4}
      ELSE
         oCol:colorBlock := {|nValue|iif(nValue<0, {7, 8}, {5, 6})}
         oCol:defColor := {7, 8}
      ENDIF
   NEXT
   vShadow(nTopRow, nTopCol, nBottomRow, nBottomCol)
   nCursorShape := SetCursor(SC_NONE)
   WHILE LOOPING
      iif(oTable:colPos=1 .and. !lSuppressRec, oTable:colPos := 2, NIL)
      nKey := 0
      WHILE nKey=0 .and. !oTable:Stable
         oTable:stabilize()
         nKey := Inkey()
      ENDWHILE
      IF oTable:stable
         IF oTable:hitBottom .and. !oTable:cargo .and. !lDisplayOnly
            oTable:cargo   := .y.
            nKey           := K_DOWN
         ELSE
            IF oTable:hitTop .or. oTable:hitBottom
               Tone(125, 0)
            ENDIF
            oTable:refreshCurrent()
            WHILE !oTable:stabilize()
            ENDWHILE
            nKey := Inkey(0)
         ENDIF
      ENDIF
      DO CASE
      CASE nKey == K_ESC
         EXIT
      CASE nKey == K_F2
         iif(cF2Expr!=NIL, (&(cF2Expr)), NIL)
         oTable:reconfigure()
      CASE nKey == K_ALT_U
         ForceStable(oTable)
         RECALL
         oTable:refreshCurrent()
      CASE nKey == K_DEL
         ForceStable(oTable)
         fLockRec()
         DELETE
         UNLOCK
         oTable:refreshCurrent()
/*
         oTable:refreshAll()
         oTable:invalidate()
         oTable:configure()
         ForceStable(oTable)
         oTable:configure()
         ForceStable(oTable)
*/
      CASE nKey == K_ALT_C
         oPasteCol := oTable:delColumn(oTable:colPos)
         oTable:invalidate()
         oTable:refreshAll()
      CASE nKey == K_ALT_I
         iif(oPasteCol=NIL, pBeep(),;
               oTable:insColumn(oTable:colPos, oPasteCol))
         oTable:invalidate()
         oTable:refreshAll()
      CASE nKey == K_ALT_G
         aSize(aTemp, oTable:colCount)
         FOR nCtr := 1 to oTable:colCount
            aTemp[nCtr] := oTable:getColumn(nCtr):heading
         NEXT
         vSave(3, 49, 23, 79)
         @ 3, 49 CLEAR TO 23, 79
         @ 3, 49 TO 23, 79 DOUBLE
         iif((nTemp := aChoice(4, 50, 22, 78, aTemp))>0,;
               oTable:colPos := nTemp,)
         vRestore()
         oTable:configure()
      CASE nKey == K_ALT_L
         aSize(aTemp, oTable:colCount)
         FOR nCtr := 1 to oTable:colCount
            aTemp[nCtr] := oTable:getColumn(nCtr):heading
         NEXT
         vSave(MaxRow(), 0, MaxRow(), MaxCol())
         @ MaxRow(), 0
         @ MaxRow(), 0 SAY "MARK COLUMNS TO DELETE  "+;
               "<space>=toggle mark  <+/->=set/reset all  <esc>=done"
         vSave(3, 49, 23, 79)
         @ 3, 49 TO 23, 79 DOUBLE
         aTemp := aMarkList(4, 50, 22, 78, aTemp)
         vRestore()
         vRestore()
         FOR nCtr := Len(aTemp) TO 1 STEP -1
            oTable:delColumn(aTemp[nCtr, 1])
         NEXT
         oTable:invalidate()
         oTable:refreshAll()
      CASE nKey == K_ALT_M
         oTempCol := oTable:getColumn(oTable:colPos)
         vSave(8, 56)
         iif(oTempCol:width=NIL, oTempCol:width := 10,)
         iif(oTempCol:colSep=NIL, oTempCol:colSep := oTable:colSep,)
         iif(oTempCol:headSep=NIL, oTempCol:headSep := oTable:headSep,)
         iif(oTempCol:footSep=NIL, oTempCol:footSep := Space(3),)
         iif(oTempCol:heading=NIL, oTempCol:heading := Space(30),)
         iif(oTempCol:footing=NIL, oTempCol:footing := Space(30),)
         sSetLength(oTempCol:colSep, 3)
         sSetLength(oTempCol:heading, 30)
         sSetLength(oTempCol:headSep, 3)
         sSetLength(oTempCol:footing, 30)
         sSetLength(oTempCol:footSep, 3)
         aRC := vWindow(8, 56, .y., "Column/Table Edit")
         @ aRC[1],  aRC[2] SAY "Column separator ......." GET oTempCol:colSep
         @ Row()+1, aRC[2] SAY "Column width ..........." GET oTempCol:width;
               PICTURE "999"
         @ Row()+1, aRC[2] SAY "Column heading .........";
               GET oTempCol:heading PICTURE "@K"
         @ Row()+1, aRC[2] SAY "Column heading separator" GET oTempCol:headSep
         @ Row()+1, aRC[2] SAY "Column footing .........";
               GET oTempCol:footing PICTURE "@K"
         @ Row()+1, aRC[2] SAY "Column footing separator" GET oTempCol:footSep
         @ Row()+1, aRC[2] SAY "Table freeze column ...." GET oTable:freeze;
               PICTURE "999"
         @ Row()+1, aRC[2] SAY "Table coordinates ......" GET oTable:nTop;
               PICTURE "999"
         @ Row(), Col()+1 GET oTable:nLeft   PICTURE "999"
         @ Row(), Col()+1 GET oTable:nBottom PICTURE "999"
         @ Row(), Col()+1 GET oTable:nRight  PICTURE "999"
         SetCursor(nCursorShape)
         READ
         SetCursor(SC_NONE)
         oTempCol:colSep := iif(Empty(oTempCol:colSep), NIL,;
               Trim(oTempCol:colSep))
         oTempCol:headSep := iif(Empty(oTempCol:headSep), NIL,;
               Trim(oTempCol:headSep))
         oTempCol:footSep := iif(Empty(oTempCol:footSep), NIL,;
               Trim(oTempCol:footSep))
         oTempCol:heading := iif(Empty(oTempCol:heading), NIL,;
               Trim(oTempCol:heading))
         oTempCol:footing := iif(Empty(oTempCol:footing), NIL,;
               Trim(oTempCol:footing))
         vRestore()
         oTable:configure()
         oTable:invalidate()
         oTable:refreshAll()
      CASE nKey == K_ALT_S
         IF bSeek!=NIL
            SEEK Eval(bSeek)
            oTable:refreshAll()
         ENDIF
      CASE nKey == K_DOWN
         oTable:down()
      CASE nKey == K_PGDN
         oTable:pageDown()
      CASE nKey == K_CTRL_PGDN
         oTable:goBottom()
         oTable:cargo := .n.
      CASE nKey == K_UP
         oTable:up()
         IF oTable:cargo
            oTable:cargo := .n.
            oTable:refreshAll()
         ENDIF
      CASE nKey == K_PGUP
         oTable:pageUp()
         IF oTable:cargo
            oTable:cargo := .n.
            oTable:refreshAll()
         ENDIF
      CASE nKey == K_CTRL_PGUP
         oTable:goTop()
         oTable:cargo := .n.
      CASE nKey == K_RIGHT
         oTable:right()
      CASE nKey == K_LEFT
         oTable:left()
      CASE nKey == K_HOME
         oTable:home()
      CASE nKey == K_END
         oTable:end()
      CASE nKey == K_CTRL_LEFT
         oTable:panLeft()
      CASE nKey == K_CTRL_RIGHT
         oTable:panRight()
      CASE nKey == K_CTRL_HOME
         oTable:panHome()
      CASE nKey == K_CTRL_END
         oTable:panEnd()
      CASE nKey == K_RETURN
         IF lDisplayOnly
            EXIT
         ELSE
            DoGet(oTable)
         ENDIF
      OTHERWISE
         IF ValType(SetKey(nKey))="B"
            Eval(SetKey(nKey))
         ELSEIF !lDisplayOnly
            KEYBOARD CHR(nKey)
            DoGet(oTable)
         ENDIF
      ENDCASE
   ENDWHILE
   vRestore()
   vCursRest()
   SetColor(cCurColors)
   SetCursor(nCursorShape)
   SetCancel(lAltC)
ENDFUNCTION

////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Description:
//    Internal function.
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC FUNCTION DoGet(oBrowse)
   LOCAL bIns, lScore, lExit
   LOCAL col, get, nKey
   LOCAL lAppend, xOldKey, xNewKey
   ForceStable(oBrowse)
   lAppend := oBrowse:cargo
   oBrowse:cargo := .n. /*xyzzy*/
   IF lAppend .and. RecNo()==LastRec()+1
      APPEND BLANK
   ENDIF
   xOldKey := iif(!Empty(IndexKey()), &(IndexKey()),)
   lScore  := Set(_SET_SCOREBOARD, .F.)
   lExit   := Set(_SET_EXIT, .T.)
   bIns    := SetKey(K_INS)
   SetKey(K_INS, {|| InsToggle()})
   SetCursor(iif(ReadInsert(), SC_INSERT, SC_NORMAL))
   col := oBrowse:getColumn(oBrowse:colPos)
   get := GetNew(Row(), Col(), col:block, col:heading,, oBrowse:colorSpec)
   fLockRec()
   IF Type(get:name)="M"
      vSave()
      CLS
      vCenter(80, 0, 0, "BecknerVision Memo Field Editor")
      get:varPut(MemoEdit(get:varGet()))
      vRestore()
   ELSE
      IF get:type$"CM" .and.;
               Len(get:varGet())>oBrowse:colWidth(oBrowse:colPos)
         get:picture := "@S"+sMake(oBrowse:colWidth(oBrowse:colPos))
      ENDIF
      get:reader  := {|oGet| vGet(oGet)}
      get:cargo   := {}
      ReadModal({get})
   ENDIF
   UNLOCK
   SetCursor(0)
   Set(_SET_SCOREBOARD, lScore)
   Set(_SET_EXIT, lExit)
   SetKey(K_INS, bIns)
   xNewKey := iif(!Empty(IndexKey()), &(IndexKey()),)
   IF .NOT. (xNewKey == xOldKey) .or. (lAppend .AND. xNewKey != NIL)
      oBrowse:refreshAll()
      ForceStable(oBrowse)
      WHILE &(IndexKey())>xNewKey .and. !oBrowse:hitTop()
         oBrowse:up()
         ForceStable(oBrowse)
      ENDWHILE
   ENDIF
   oBrowse:cargo := .n.
   nKey          := LASTKEY()
   IF nKey=K_UP .or. nKey=K_DOWN .or. nKey=K_PGUP .or. nKey=K_PGDN
      KEYBOARD(Chr(nKey))
   ENDIF
ENDFUNCTION

#define ENDPROC RETURN

////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Description:
//    Internal function.
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC PROCEDURE ForceStable(oBrowse)
   WHILE !oBrowse:stabilize()
   ENDWHILE
ENDPROC

////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Description:
//    Internal function.
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC PROCEDURE InsToggle()
   IF ReadInsert()
      ReadInsert(.n.)
      SetCursor(SC_NORMAL)
   ELSE
      ReadInsert(.y.)
      SetCursor(SC_INSERT)
   ENDIF
ENDPROC

////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Description:
//    Internal function.
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC FUNCTION SkipTo(nRecs, oTable)
   LOCAL nCount := 0
   IF nRecs>0 .and. RecNo()!=LastRec()+1
      WHILE nCount<nRecs
         SKIP
         IF EOF()
            IF oTable:cargo
               nCount++
            ELSE
               SKIP -1
            ENDIF
            EXIT
         ENDIF
         nCount++
      ENDWHILE
   ELSEIF nRecs<0
      WHILE nCount>nRecs
         SKIP -1
         IF BOF()
            EXIT
         ENDIF
         nCount--
      ENDWHILE
   ENDIF
   RETURN nCount
ENDFUNCTION

////////////////
////////////////
//
// Purpose:
//    Internal function
//
// Description:
//    Internal function.
//
// Category:
//    Internal Function
//
// Revisions:
//    01/26/94 Added comment blocks
//
////////////////
////////////////

STATIC FUNCTION vShadow(nTopRow, nTopCol, nBottomRow, nBottomCol)
   LOCAL cCurrentColor
   cCurrentColor := SetColor("N/N")
   @ nTopRow+1, nTopCol+1 CLEAR TO nBottomRow+1, nBottomCol+1
   SetColor("W/W")
   @ nTopRow, nTopCol CLEAR TO nBottomRow, nBottomCol
   SetColor(cCurrentColor)
ENDFUNCTION


/*
 1.02 05.15.93 Corrected problem with stability of <del>
*/
