/*
    ZIPDIR2.PRG
    Author: Eric J. Givler
    Written: 09-09-1993
    Purpose: Display Contents of .ZIP file.

(* Structure of a local file header *)
   ZIP_Local_Header_Type =
      RECORD
         Signature           : LONGINT  (* Header signature        *);  1-4
         Version             : WORD     (* Vers. needed to extract *);  5-6
         BitFlag             : WORD     (* General flags           *);  7-8
         CompressionMethod   : WORD     (* Compression type used   *);  9-10
         FileTime            : WORD     (* File creation time      *); 11-12
         FileDate            : WORD     (* File creation date      *); 13-14
         CRC32               : LONGINT  (* 32-bit CRC of file      *); 15-18
         CompressedSize      : LONGINT  (* Compressed size of file *); 19-22
         UnCompressedSize    : LONGINT  (* Original size of file   *); 23-26
         FileNameLength      : WORD     (* Length of file name     *); 27-28
         ExtraFieldLength    : WORD     (* Length of extra stuff   *); 29-30
      END;

(* Structure of the central *)
(* directory record         *)
   ZIP_Central_Header_Type =
      RECORD
          Signature           : LONGINT (* Header signature        *); 01-04
          VersionMadeBy       : WORD    (* System id/program vers. *); 05-06
          VersionNeeded       : WORD    (* Vers. needed to extract *); 07-08
          BitFlag             : WORD    (* General flags           *); 09-10
          CompressionMethod   : WORD    (* Compression type used   *); 11-12
          FileTime            : WORD    (* File creation time      *); 13-14
          FileDate            : WORD    (* File creation date      *); 15-16
          CRC32               : LONGINT (* 32-bit CRC of file      *); 17-20
          CompressedSize      : LONGINT (* Compressed size of file *); 21-24
          UnCompressedSize    : LONGINT (* Original size of file   *); 25-28
          FileNameLength      : WORD    (* Length of file name     *); 29-30
          ExtraFieldLength    : WORD    (* Length of extra stuff   *); 31-32
          CommentFieldLength  : WORD    (* Length of comments      *); 33-34
          DiskStartNumber     : WORD    (* Disk # file starts on   *); 35-36
          InternalAttributes  : WORD    (* Text/non-text flags     *); 37-38
          ExternalAttributes  : LONGINT (* File system attributes  *); 39-42
          LocalHeaderOffset   : LONGINT (* Where local hdr starts  *); 43-46
      END;
*/

#define ZIP_CENTRAL_HEAD_SIG  33639248   // $02014B50
#define ZIP_LOCAL_HEAD_SIG    67324752   // $04034B50
#define ZIP_END_CENTRAL_SIG  101010256   // $06054B50
#define CENTRAL_HEADER_SIZE   46
#define LOCAL_HEADER_SIZE     30

#define END_OF_FILE            6
#define CENTRAL_DIR_FOUND      5
#define READ_ERROR             4         // Read Error or NOT a .ZIP file!

#include "fileio.ch"

#ifdef TEST
FUNCTION Main(cName)
LOCAL a_, i, nTime

    IF ! ( cName == NIL )
         //nTime := seconds()
         //FOR i := 1 TO 50
            a_ := Zipdir( cName )
         //NEXT i
         //Alert( STR( Seconds() - nTime, 10, 2) + ' for 50 times!')

         aeval( a_, { |a2_| qqout(padr(a2_[1],15)),;
                       qqout(tran(a2_[2],'999999')+'  '),;
                       qqout(tran(a2_[3],'999999')+'  '),;
                       qqout(dtoc(a2_[4])+'  '),;
                       qqout(a2_[5]),qout('') } )
    ELSE
         ? 'ZIPDIR2 <filename>'
    ENDIF
RETURN NIL
#endif

*----------------------------------------------------------------------------*
FUNCTION ZipDir2( cFile )
LOCAL cZipEntry := space( CENTRAL_HEADER_SIZE ), ;
      nZipPos   := 0, nBytes, cFileName, nHandle, a_ := {}, nResult
LOCAL nHeadSig, nNameLen

    IF (nHandle := fopen( cFile, FO_READ )) != F_ERROR

        BEGIN SEQUENCE
        IF ( nResult := FindCentralDir( nHandle, @nZipPos ) ) != ;
            CENTRAL_DIR_FOUND
            BREAK
        ENDIF

        // Position in file set at Central Directory Header.
        FSEEK( nHandle, nZipPos )

        WHILE .T.

            // If wrong size read and not at end, Exit immediately
            nBytes   := FREAD( nHandle, @cZIPEntry, CENTRAL_HEADER_SIZE )
            nHeadSig := bin2l( substr(cZipEntry,1,4) )
            IF (nBytes < CENTRAL_HEADER_SIZE)
                IF nHeadSig == ZIP_END_CENTRAL_SIG
                    EXIT
                ELSE
                    BREAK
                ENDIF
            ENDIF

            // Check for a legitimate header type
            DO CASE
            CASE nHeadSig == ZIP_CENTRAL_HEAD_SIG

                 // File name length - read it in!
                 nNameLen := bin2w( substr(cZipEntry,29,2) )
                 cFileName:= space( nNameLen )
                 nBytes   := FREAD( nHandle, @cFileName, nNameLen )
                 IF ( nBytes <> nNameLen )
                      BREAK
                 ENDIF

                 // Store what we want!
                 AADD( a_, { cFileName, ;
                             BIN2L( substr(cZipEntry,21,4) ), ;
                             BIN2L( substr(cZipEntry,25,4) ), ;
                             DOS_DATE( bin2w(substr(cZipEntry,15,2)) ), ;
                             DOS_TIME( bin2w(substr(cZipEntry,13,2)) ), ;
                           } )

                 // Position to next header
                 /*
                 nZIPPos += bin2w( substr(cZipEntry,31,2) ) + ; // ExtraFieldLength
                            bin2w( substr(cZipEntry,33,2) ) + ; // CommentFieldLength
                            nNameLen                        + ; // FileNameLength
                            CENTRAL_HEADER_SIZE
                 FSEEK( nHandle, nZipPos )
                 */
                 fseek( nHandle, bin2w( substr(cZipEntry,31,2) ) + ;
                                 bin2w( substr(cZipEntry,33,2) ), ;
                                 FS_RELATIVE )

            CASE nHeadSig == ZIP_END_CENTRAL_SIG
                EXIT

            // Anything else is bogus!
            OTHERWISE
                BREAK

            ENDCASE

        ENDDO

        RECOVER
            a_ := {}
        END SEQUENCE

        fclose( nHandle )

    ENDIF

RETURN (a_)

*-----------------------------[ FindCentralDir() ]--------------------------*
STATIC FUNCTION FindCentralDir( nHandle, nPos )
LOCAL cZipLocal := space( LOCAL_HEADER_SIZE )
LOCAL nHeadSig, nResult := 0, nBytes
LOCAL cZipEnd := SPACE(22)

    // Check for a ZIP with NO end comment (ends with an info. block)
    FSEEK( nHandle, -22, FS_END )
    IF ( nBytes := FREAD(nHandle, @cZipEnd,22) ) == 22
        IF bin2l( substr(cZipEnd,1,4) ) == ZIP_END_CENTRAL_SIG
            nPos    := bin2l( substr(cZipEnd,17,4) )
            nResult := CENTRAL_DIR_FOUND
        ENDIF
    ENDIF

    // If we didn't find the Central Dir from the end, look from the
    // beginning through the local headers.  We could easily build the
    // array of files from the local headers.  An example of this is in
    // the Lief,Booth,Yellick book. 
    IF nResult == 0

        fseek(nHandle, 0)
        WHILE .T.

            IF (nBytes := fread( nHandle, @cZipLocal, LOCAL_HEADER_SIZE )) ;
                != LOCAL_HEADER_SIZE
                nResult := READ_ERROR
                EXIT
            ENDIF

            // Check for a legitimate header type.
            nHeadSig := bin2l( substr(cZipLocal,1,4) )
            do case
            case nHeadSig == ZIP_LOCAL_HEAD_SIG

                // nPos := FileNameLength + ExtraFieldLength +
                //         CompressedSize + LOCAL_HEADER_SIZE
                nPos += bin2w( substr(cZipLocal,27,2) ) + ;
                        bin2w( substr(cZipLocal,29,2) ) + ;
                        bin2l( substr(cZipLocal,19,4) ) + ;
                        LOCAL_HEADER_SIZE
                // fseek( nHandle, nPos )
                fseek( nHandle, bin2w( substr(cZipLocal,27,2)) + ;
                                bin2w( substr(cZipLocal,29,2)) + ;
                                bin2l( substr(cZipLocal,19,4)), ;
                                FS_RELATIVE )

            case nHeadSig == ZIP_CENTRAL_HEAD_SIG
                nResult := CENTRAL_DIR_FOUND
                EXIT

            case nHeadSig == ZIP_END_CENTRAL_SIG
                nResult := END_OF_FILE
                EXIT

            otherwise
                nResult := READ_ERROR
                EXIT

            endcase

        ENDDO // while .t.

    ENDIF

RETURN (nResult)
