/*
    FILEVIEW.PRG
    AUTHOR:  Eric J. Givler
    WRITTEN: 06/05/92
    MODS:    06/26/92 - Documented support for ARJ files.  Some 'mite' fixes.
                        Added PAK support by modifying ARCDIR.PRG to not
                        choke on 2nd char after chr(26) being > 12.
             06/28/92 - Support for Sorting [F8] by Name, Extension or Date.
             03/13/93 - Fileviewing!
                        LowLevel byte determination of compressed file type.
                        Creates TEMP directory under current.  DOES NOT REMOVE.
             06/26/93 - #ifdef BLINKER for viewing inside archives.
    J.Wright 01/21/94 - Use Overlay() to swap instead of BLINKER (if defined).
                        Remove TEMP directory after viewing file.
                        Disable LHA messages with /n2 parameter (not >NUL).
                        Commented out Col_Split() calls - not found in PRGs.
                        IsTextFile() and Text_Brows() were not found either.
    J.Wright 01/31/94 - Added functions back in after Eric gave me new code.
                        Changed TEMP directory to Z# to avoid possible errors.
                        Modified program to shell using Clipper RUN command
                        or convert to Blinker/Overlay function if specified.
                        Added option to sort on file size when [F8] pressed.
                        Moved FREADLN() into this PRG to avoid conflicts
                        with the generic Clipper FREADLN() function.

    SYNTAX:  FileView( { <file1.ext> [, file2.ext] } )
    PURPOSE: View .ARC, .ARJ, .HYP, .LZH, .PAK, .ZIP, .ZOO.  Will
             support more later.
    NOTES:   Pass files to view INSIDE AN ARRAY.
    Calls:   FReadln(), IsDir(), MkDir(), ChDir(), Text_Browse()
             Col_Split(), IsTextFile(), SwpRunCmd()
*/

#include "box.ch"
#include "fileio.ch"
#include "inkey.ch"

#define F_NAME   1
#translate ext( <f> ) => ;
           if( '.' $ <f>, substr(<f>, at('.', <f>) + 1), '')

#ifdef BLINKER
* Convert Clipper RUN command to BLINKER swap command
#translate CLRUNDOS( <f> ) => SWPRUNCMD( <f>, 0, "", "" )
#endif

#ifdef OVERCL
* Convert Clipper RUN command to OVERLAY swap command
#translate CLRUNDOS( <f> ) => OVERLAY( <f>, 0, "", "" )
#endif

STATIC unCompress_ := { { 'ZIP', 'pkunzip -e -o [COMPFILE] Z# [FILENAME]'  } ,;
                        { 'ARJ', 'ARJ E -y [COMPFILE] Z# [FILENAME]'  } ,;
                        { 'LZH', 'LHA E -n2 -m [COMPFILE] Z#\ [FILENAME]' } ,;
                        { 'ARC', 'PKXARC -re [COMPFILE] Z# [FILENAME]'   } }

*--------------------------------------------------[ FileView() ]------------*
FUNCTION FILEVIEW( aFiles_ )
LOCAL ncursor  := setcursor(), ;
      cCol     := setcolor(), ;
      c_       := { "W+/B,GR+/R", "W+/RB,W+/BG" }, ;
      elems    := len( aFiles_ ), ;
      contents_:= {}, ;
      cTypes_  := {}, ;                       // File Types
      n        := 1, ;                        // FOR..NEXT counter.
      nHandle, cType, i, nkey
LOCAL b1, b2, cb, nCb, index1, index2
LOCAL scrn, longest, files_
LOCAL cConfig, cBuffer

    if elems > 2 .or. VALTYPE(aFiles_) != "A"
        RETURN NIL
    endif

    FOR i := 1 to elems
        files_ := {}
        if aFiles_[n] != NIL

            cType := ArcType( aFiles_[n] )
            DO CASE
            CASE cType == 'PAK' .OR. cType == 'ARC'
                files_ := ArcDir( aFiles_[n] )
            CASE cType == 'HYP'
                files_ := HypDir( aFiles_[n] )
            CASE cType == 'ARJ'
                files_ := ArjDir( aFiles_[n] )
            CASE cType == 'LZH'
                files_ := LzhDir( aFiles_[n] )
            CASE cType == 'ZIP'
                files_ := ZipDir( aFiles_[n] )
            CASE cType == 'ZOO'
                files_ := ZooDir( aFiles_[n] )
            CASE cType == 'DWC'
                files_ := DwcDir( aFiles_[n] )
            ENDCASE

        endif

        if ! empty( files_ )
            aadd( cTypes_, cType )
            aadd( contents_, files_ )
            n++
        else
            adel( aFiles_, n )
            asize( aFiles_, 1)
        endif
    NEXT i


    if empty( contents_ )
        ALERT("Could not view the filenames that were passed")
        RETURN NIL
    endif

    // Read in viewer configuration information, else use what's
    // defined above.
    if !EMPTY( cConfig := getenv('FILEVIEW') )
        if (nhandle := fopen( cConfig, FO_READ )) != F_ERROR
            UnCompress_ := {}
            cbuffer := space(80)
            do while freadln( nhandle, @cbuffer, 80 )
                aadd( UnCompress_, { left(cbuffer,at(' ',cBuffer)-1), ;
                                     substr(cbuffer,at(' ',cbuffer)+1) } )
            enddo
            fclose(nhandle)
        endif
    endif

    scrn  := savescreen( 0, 0, maxrow(), maxcol() )
    elems := len( contents_ )
    //---------------------[ Set up first browse ]--------------------------
    index1 := 1
    nCb := 1
    IF( Col_Split()[1] = Col_Split(c_[1])[1], ;
           setcolor('w+/g,n/bg'), setcolor(c_[1]) )
    setcolor(c_[1])
    b1 := tbrowsenew( 8, 1, 23, if(elems==1, 78,38) )
    dispbox(b1:nTop-1,b1:nLeft-1,b1:nBottom+1,b1:nRight+1, ;
            B_DOUBLE + " ")
    @ b1:nTop-1,b1:nLeft+1 say left(aFiles_[1],35)
    b1:headsep := ""
    b1:colsep  := "  "
    // b1:footsep := ""
    // b1:colorspec := m->c_normcol
    b1:gotopblock    := { || index1 := 1 }
    b1:gobottomblock := { || index1 := len( contents_[1] ) }
    b1:skipblock := { |n| ArraySkip( len( contents_[1] ), @index1, n ) }
    b1:addcolumn( tbcolumnnew( "Filename",   { || contents_[1][index1][1] } ))
    b1:addcolumn( tbcolumnnew( "Compressed", { || contents_[1][index1][2] } ))
    b1:addcolumn( tbcolumnnew( "UnCompress", { || contents_[1][index1][3] } ))
    b1:addcolumn( tbcolumnnew( "FileDate",   { || contents_[1][index1][4] } ))

    // The following 3 lines allow for the embedded ARJ path too.
    longest := 0
    aeval( contents_[1], { |a| longest := max( len(a[1]), longest ) } )
    b1:getcolumn(1):width := longest
    b1:getcolumn(2):width := 10
    b1:getcolumn(3):width := 10
    b1:getcolumn(4):width :=  9

    if elems > 1
        index2 := 1
        setcolor(c_[2])
        b2 := tbrowsenew(8, 41, 23, 78)
        dispbox(b2:nTop-1,b2:nLeft-1,b2:nBottom+1,b2:nRight+1, ;
            B_SINGLE + " ")
        @ b2:nTop-1,b2:nLeft+1 say left(aFiles_[2],35)
        b2:headsep := ""
        b2:colsep  := "  "
        // b2:footsep := ""
        // b2:colorspec := "W+/RB,W+/BG"
        b2:gotopblock    := { || index2 := 1 }
        b2:gobottomblock := { || index2 := len( contents_[2] ) }
        b2:skipblock := { |n| ArraySkip( len( contents_[2] ), @index2, n ) }
        b2:addcolumn( tbcolumnnew( "Filename",   { || contents_[2][index2][1] } ))
        b2:addcolumn( tbcolumnnew( "Compressed", { || contents_[2][index2][2] } ))
        b2:addcolumn( tbcolumnnew( "UnCompress", { || contents_[2][index2][3] } ))
        b2:addcolumn( tbcolumnnew( "FileDate",   { || contents_[2][index2][4] } ))
        longest := 0
        aeval( contents_[2], { |a| longest := max( len(a[1]), longest ) } )
        b2:getcolumn(1):width := longest
        b2:getcolumn(2):width := 10
        b2:getcolumn(3):width := 10
        b2:getcolumn(4):width :=  9
    endif

    if elems > 1
        cb := b2
        do while ! cb:stabilize()
        enddo
        cb:dehilite()
    endif
    cb := b1

    DO WHILE .T.

        DO WHILE ! CB:STABILIZE()
        ENDDO

        nKey := inkey(0)
        IF StdMeth(nKey, cb)
            // The previous function handles standard Tbrowse keystrokes.
            loop
        ELSE
            do case
                case nKey == K_ENTER
                    ViewFile( aFiles_[nCb], ;
                              cTypes_[nCb], ;
                              contents_[nCb][if(nCb=1,index1,index2)][1] )

                case nKey == K_TAB
                    if elems > 1
                        dispbox(cb:nTop-1,cb:nLeft-1,cb:nBottom+1,cb:nRight+1,;
                           B_SINGLE, c_[nCb])
                        @ cb:nTop-1,cb:nLeft+1 say left(aFiles_[nCb],35) color c_[nCb]
                        nCb := if(nCb == 1, 2, 1)
                        cb:dehilite()
                        cb  := if(nCb == 1, b1, b2)
                        cb:hilite()
                        dispbox(cb:nTop-1,cb:nLeft-1,cb:nBottom+1,cb:nRight+1,;
                           B_DOUBLE, c_[nCb])
                        @ cb:nTop-1,cb:nLeft+1 say left(aFiles_[nCb],35) color c_[nCb]
                    endif

                case nKey == K_F8
                    // Sort the array in current window
                    nKey := ALERT("Select a different sort order.", ;
                                  {"Filename", "Extension", "Date", "Size" })
                    do case
                        case nKey == 1
                            //---------------[ FileName ]-------------------
                            asort( contents_[nCb],,, ;
                               { |a,b| a[F_NAME] < b[F_NAME] } )
                        case nKey == 2
                            //---------------[ Extension ]------------------
                            asort( contents_[nCb],,, ;
                               { |a,b| if( ext(a[F_NAME]) == ext(b[F_NAME]), ;
                                          a[F_NAME] < b[F_NAME], ;
                                          ext(a[F_NAME]) < ext(b[F_NAME]) ) } )
                        case nKey == 3
                            //----------------[ Date ]----------------------
                            asort( contents_[nCb],,, ;
                               { |a,b| if( a[4] == b[4], ;
                                           a[F_NAME] < b[F_NAME], ;
                                           a[4]      < b[4]) } )
                        case nKey == 4
                            //---------------[ Compressed ]-----------------
                            asort( contents_[nCb],,, ;
                               { |a,b| a[2] < b[2] } )
                    endcase
                    cb:refreshall()

                case nKey == K_ESC
                    exit
            endcase
        ENDIF
    ENDDO

    setcursor(ncursor)
    restscreen( ,,,, scrn )
    setcolor(cCol)

RETURN NIL

// View compressed file internal to other (based on extension here!)
STATIC FUNCTION ViewFile( cMainFile, cFileType, cInternal )
LOCAL Istemp := .F.                // Whether temp directory created.
LOCAL cCommand, n, cCurDir, nBkSlsh

     IF ( n := ascan( UnCompress_, { |a_| a_[1] == cFileType } ) ) != 0

          cCurDir := '\' + CURDIR()

          if !ISDIR( 'Z#' )
             Istemp := MKDIR( 'Z#' )
          else
             Istemp := .T.
          endif

          IF Istemp

               // Build extraction command with file and dump to Z# <dir>
               cCommand := UnCompress_[ n ][ 2 ]
               cCommand := strtran( cCommand, '[COMPFILE]', cMainFile )
               cCommand := strtran( cCommand, '[FILENAME]', cInternal )
               cCommand += ' >NUL'

               // Check to make sure this is not a file in a nested directory
               // If there is a /, remove the path info from the filename.
               cInternal := IF( ( nBkSlsh := RAT( '/', cInternal) ) <> 0, ;
                                STUFF( cInternal, 1, nBkSlsh, ''), cInternal )

               IF CLRUNDOS( cCommand )
                    CHDIR( 'Z#' )
                    IF FILE( cInternal ) .AND. IsTextFile( cInternal )
                         Text_Browse( cInternal )
                    ELSE
                         TONE( 100, 1 )
                    ENDIF
                    FERASE( cInternal )
               ENDIF

               CHDIR( cCurDir )
               RMDIR( 'Z#' )

          ENDIF
     ENDIF

RETURN NIL

#ifndef BLINKER
 #ifndef OVERLAY
    STATIC FUNCTION CLRUNDOS( cFileName )
    * 01/31/94 - Try to extract the file without Blinker or Overlay.
    *            Created a function so #translate could be used easily.
       RUN (cFileName)
    RETURN .T.
  #endif
#endif
