;[]---------------------[ VECTOR STRUCTURE DEFINITION ]----------------------[]
;| The following is the defintion of a Vector Structure.. For Every Interrupt |
;| trapped, an instance of this structure is declared.. The data contained in |
;| the structure enable saving/setting/resetting interrupt vectors in a       |
;| relative easy method..                                                     |
;[]--------------------------------------------------------------------------[]
VectorStruc     STRUC
        OldVectOf dw ?                          ; Offset Of Old Handler
        OldVectSg dw ?                          ; Segment Of Old Handler
        NewVectOf dw ?                          ; Offset Of New Handler
        VectNum   db ?                          ; Interrupt Number
        VectFlag  db ?                          ; General Purpose Flag
VectorStruc     ENDS                            ; End of Vector Structure



;[]---------------[ MACRO TO DEFINE VECTOR STRUCTURES ] ---------------------[]
;| This MACRO instantiates the proper structures for a group of Interrupts    |
;| that will be hooked and unhooked at the same time.  A GroupName is         |
;| specified for a particular group of Vectors and the subsequent MACROS      |
;| handling the various processes are only passed the GroupName...            |
;[]--------------------------------------------------------------------------[]
@DefVect MACRO   VectGrp,i0,i1,i2,i3,i4,i5,i6,i7,i8,i9,iA
IFNB    <iA>
        @Error   <Too many arguments in call to DefVect>
ENDIF
        ;- A label is inserted before the definitions of all the
        ;- variables for the ISRs
        &VectGrp&Tbl    label  WORD
       .RADIX  16
IRP     Int_Num,<i0,i1,i2,i3,i4,i5,i6,i7,i8,i9,iA>

    IFB    <Int_Num>
       .RADIX  10
    ELSE
        Old&&Int_Num&&Handler   label  DWORD
        Vect&&Int_Num   VectorStruc <,,OFFSET New&&Int_Num&&Handler,&&Int_Num&&h,00>
    ENDIF
ENDM
        ;- A count is generated for the number of Vectors actually lumped
        ;- in the current group of Interrupt Vectors
        &VectGrp&Count  EQU (($ - OFFSET &VectGrp&Tbl) / (TYPE VectorStruc))
ENDM



;[]----------------------[ MACRO TO GRAB VECTORS ]---------------------------[]
;| This MACRO will save the Interrupt Vector addresses of the Interrupts in   |
;| the specified GroupName (see @DefVect for GroupName)...  It uses a loop    |
;| minimizes the overhead, specially when several Interrupts make up a Group. |
;| ENTRY:       DS: Points to DATA Segment                                    |
;| RETURN:      NONE                                                          |
;[]--------------------------------------------------------------------------[]
@GetVect MACRO  VectGrp
         LOCAL  __$$__
IFNDEF  &VectGrp&Tbl
@Error  <@DefVect not called for Group of Vectors: &VectGrp&>
ENDIF
        @PUSH   ax, bx, cx, si, es
        mov     cx, &VectGrp&Count
        mov     si, OFFSET &VectGrp&Tbl
        mov     ah, 35h
__$$__: mov     al, [si].VectorStruc.VectNum
        int     21h
        mov     [si].VectorStruc.OldVectOf, bx
        mov     [si].OldVectSg, es
        add     si, TYPE VectorStruc
        loop    __$$__
        @POP    ax, bx, cx, si, es
ENDM


;[]-----------------------[ MACRO TO INSTALL VECTORS ]-----------------------[]
;| This MACRO installs our Service Routines for the particular Interrupts     |
;| making up the specified group...                                           |
;| It is assumed the for each interrupt number (xx) there's will be a label   |
;| or procedure named NewxxHandler...                                         |
;| ENTRY:       DS: Points to CODE SEGMENT of NewHandlers                     |
;| RETURN:      NONE                                                          |
;[]--------------------------------------------------------------------------[]
@SetVect MACRO  VectGrp
         LOCAL  __$$__
IFNDEF  &VectGrp&Tbl
@Error  <@DefVect not called for Group of Vectors: &VectGrp&>
ENDIF
        @PUSH   ax, cx, dx, si
        mov     ah, 25h
        mov     cx, &VectGrp&Count
        mov     si, OFFSET &VectGrp&Tbl
__$$__: mov     al, [si].VectorStruc.VectNum
        mov     dx, [si].VectorStruc.NewVectOf
        int     21h
        add     si, TYPE VectorStruc
        loop    __$$__
        @POP    ax, cx, dx, si
ENDM


;[]------------------- [ VERIFY OWNERSHIP OF VECTORS ]-----------------------[]
;| This Macro verifies that the Interrupt Vector Table still contains the     |
;| addresses of the Interrupts we've installed with @SetVec                   |
;| ENTRY:       DX: Code Segment Of our Interrupt Handlers                    |
;| RETURN:      CF: Somebody Hooked up on of our Vectors                      |
;|              AL: Interrupt Number Hooked by someone else                   |
;|       ELSE  NCF: No Carry -> We're still owner of Vectors                  |
;[]--------------------------------------------------------------------------[]
@ChkVect MACRO  VectGrp
         LOCAL  __$$1_, __$$2_, __$$3_
IFNDEF  &VectGrp&Tbl
@Error  <@DefVect not called for Group of Vectors: &VectGrp&>
ENDIF
        @PUSH   bx, cx, si, es
        mov     cx, &VectGrp&Count
        mov     si, OFFSET &VectGrp&Tbl
        mov     ah, 35h
__$$1_: mov     al, cs:[si].VectorStruc.VectNum
        int     21h
        cmp     bx, cs:[si].VectorStruc.NewVectOf
        jne     __$$2_
        mov     bx, es
        cmp     dx, bx
        jne     __$$2_
        add     si, TYPE VectorStruc
        loop    __$$1_
        clc
        jmp     short __$$3_
__$$2_: stc
__$$3_: @POP    bx, cx, si, es
ENDM


;[]------------------- [ RESTORE VECTORS HOOKED...   ]-----------------------[]
;| This Macro restores the Vectors hooked with @SetVect with the addresses    |
;| of the prior Handlers...                                                   |
;| ENTRY:       NONE                                                          |
;| RETURN:      NONE                                                          |
;[]--------------------------------------------------------------------------[]
@RstVect MACRO  VectGrp
         LOCAL  __$$__
IFNDEF  &VectGrp&Tbl
@Error  <@DefVect not called for Group of Vectors: &VectGrp&>
ENDIF
        @PUSH   ax, cx, dx, si, ds
        mov     ah, 25h
        mov     cx, &VectGrp&Count
        mov     si, OFFSET &VectGrp&Tbl
__$$__: mov     al, cs:[si].VectorStruc.VectNum
        mov     dx, cs:[si].VectorStruc.OldVectOf
        mov     ds, cs:[si].VectorStruc.OldVectSg
        int     21h
        add     si, TYPE VectorStruc
        loop    __$$__
        @POP    ax, cx, dx, si, ds
ENDM


