/*
 *                 Author:  Christopher G. Phillips
 *              Copyright (C) 1994 All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appear in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * The author makes no representations about the suitability of this
 * software for any purpose.  This software is provided ``as is''
 * without express or implied warranty.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "dis.h"

#ifdef DEBUG
int	debug = 0;
#endif

long	curoffset;	/* current offset into file */
short	flags;		/* flags for current instruction */
addr_t	required[3];	/* instructions that must be valid for current
			   instruction to also be valid */
int	pcrelative = 0;

extern char	instbuf[];
extern size_t	leninstbuf;

struct inst	*insts;		/* instruction database */
addr_t		maxoffset;

#define NREQD(o)	(insts[o].flags & 3)
#define JUMP(o)		(insts[o].flags & (ISBRA | ISRTS | ISJMP | ISJSR))
#define FETCH(o,s)	(/* (o) >= 0 && */ (o) <= maxoffset && \
			  ((s) == -1 && insts[o].size \
			  || insts[o].size == (s)))
#define DELETE(o)	free(insts[o].required); \
			  insts[o].required = NULL; \
			  insts[o].size = 0
#define SETLABEL(o)	if ((o) >= initialpc && (o) <= initialpc + maxoffset) \
				insts[(o) - initialpc].flags |= ISLABEL

static short	longestinstsize;
static long	gfsize = 0;

static void
jumpfree(void)
{
	while (gfsize--)
		free(insts[gfsize].required);
	free(insts);
	insts = NULL;
	gfsize = 0;
	longjmp(jmp, 1);
}

static int
findstring(size_t stored, const unsigned char *p)
{
	int	i;
	int	inarow;

	for (inarow = i = 0; i < stored; i++, p++)
		if (isprint(*p)) {
			if (++inarow >= minlen)
				return i - inarow + 1;
		} else
			inarow = 0;

	return -1;
}

static void
dcflush(size_t todo, const unsigned char *p)
{
	char	format[BUFSIZ];
	int	i;
	addr_t	value;
	size_t	n;
	size_t	j;
	size_t	length;
	int	lflags = ops2f(1);
	int	first = 1;

	while (todo) {
		lflags &= ~(size2f(BYTE) | size2f(WORD));
		length = 2 * sizeof(word_t);
		if (todo < length)
			length = todo;
		if (first) {
			if ((ppc & 1) && length != todo)
				length = 1;
			first = 0;
		}
		if ((length % sizeof(word_t)) == 0) {
			lflags |= size2f(WORD);
			for (n = 0, i = 0; i < length / 2; i++) {
				for (value = 0, j = 0; j < sizeof(word_t); j++)
					value += p[i * sizeof(word_t) + j]
					  << (CHAR_BIT
					  * (sizeof(word_t) - 1 - j));
					value = signextend(value, 16);
				if (i)
					n += sprintf(format + n, ",");
				n += sprintf(format + n, "#%d", value);
			}
		} else {
			lflags |= size2f(BYTE);
			for (n = 0, i = 0; i < length; i++) {
				if (i)
					n += sprintf(format + n, ",");
				n += sprintf(format + n, "#%u", p[i]);
			}
		}
		leninstbuf = 0;
		for (i = 0; i < length; i++)
			leninstbuf += sprintf(instbuf + leninstbuf, "%02x",
			  p[i]);
		pc += length;
		instprint(lflags, "DC", format);
		todo -= length;
		p += length;
	}
}

static size_t
ascflush(size_t stored, const unsigned char *p)
{
	char	format[BUFSIZ];
	size_t	length;
	size_t	i;
	size_t	n;

	for (length = 0; length < stored; length++)
		if (!isprint(p[length]))
			break;

	format[0] = '\'';
	n = 1;
	leninstbuf = 0;
	for (i = 0; i < length; i++) {
		leninstbuf += sprintf(instbuf + leninstbuf, "%02x", p[i]);
		n += sprintf(format + n, "%c", p[i]);
	}
	format[n++] = '\'';
	format[n++] = '\0';
	pc += length;
	instprint(ops2f(1) | size2f(BYTE), "DC", format);

	return length;
}

static void
flush(size_t stored, const unsigned char *consts)
{
	size_t	length;
	int	spos = -2;

	while (stored) {
		spos = findstring(stored, consts);
		if (spos) {
			length = (spos < 0) ? stored : spos;
			dcflush(length, consts);
			stored -= length;
			consts += length;
		}
		if (spos >= 0) {
			length = ascflush(stored, consts);
			stored -= length;
			consts += length;
		}
	}
}

static int
validinst(void)
{
	word_t	inst;

	valid = 0;

	if (nextword(&inst) == 0) {
		switch (inst >> 12) {
		case 0:
			bit_movep_immediate(inst);
			break;
		case 1:
			movebyte(inst);
			break;
		case 2:
			movelong(inst);
			break;
		case 3:
			moveword(inst);
			break;
		case 4:
			misc(inst);
			break;
		case 5:
			addq_subq_scc_dbcc_trapcc(inst);
			break;
		case 6:
			valid = bcc_bsr(inst);
			break;
		case 7:
			moveq(inst);
			break;
		case 8:
			or_div_sbcd(inst);
			break;
		case 9:
			sub_subx(inst);
			break;
		case 10:
			aline(inst);
			break;
		case 11:
			cmp_eor(inst);
			break;
		case 12:
			and_mul_abcd_exg(inst);
			break;
		case 13:
			add_addx(inst);
			break;
		case 14:
			shift_rotate_bitfield(inst);
			break;
		case 15:
			if (chip >= MC68020) {
				coprocessor(inst);
				if (!valid)
					fline(inst);
			} else
				unimplemented(inst);
			break;
		}
	}

	return valid;
}

static void
dcpass(void)
{
	addr_t	offset;

	pass = DCLABELSPASS;

	for (curoffset = offset = 0; offset <= maxoffset; )
		if (insts[offset].size) {
			if (curoffset != offset
			  && fseek(infp, curoffset = offset, SEEK_SET) == -1) {
				perror("fseek");
				jumpfree();
			}
			flags = 0;
			pc = ppc = offset + initialpc;
			leninstbuf = 0;
			validinst();
			offset += insts[offset].size;
		} else
			offset += sizeof(word_t);
}

static void
printall(void)
{
	addr_t		offset;
	unsigned char	consts[BUFSIZ];
	size_t		stored = 0;

	if (fseek(infp, 0, SEEK_SET) == -1) {
		perror("fseek");
		jumpfree();
	}
	pc = ppc = initialpc;
	leninstbuf = 0;
	pass = DEBUGPASS;
	for (curoffset = offset = 0; offset <= maxoffset;
	  offset += sizeof(word_t)) {
		/*
		 * Determine if there might be a valid instruction
		 * which has the bytes at ``offset'' as operands.
		 */
		if (insts[offset].size == 0) {
			int	i = 0;
			size_t	size;

			for (size = sizeof(word_t); size <= longestinstsize
			  && size <= offset; size += sizeof(word_t))
				if (size < insts[offset - size].size) {
					i = 1;
					break;
				}
			if (i)
				continue;
		}

		if (curoffset != offset
		  && fseek(infp, curoffset = offset, SEEK_SET) == -1) {
			perror("fseek");
			jumpfree();
		}
		flags = 0;
		if (insts[offset].size) {
			if (stored) {
				flush(stored, consts);
				stored = 0;
			}

			pc = ppc = initialpc + offset;
			leninstbuf = 0;
			validinst();
		} else {
			size_t	i;

			if (stored == 0)
				pc = ppc = initialpc + offset;
			for (i = 0; i < sizeof(word_t); i++)
				if (fread(consts + stored, 1, 1, infp) == 1)
					if (++stored >= sizeof consts) {
						flush(stored, consts);
						stored = 0;
					}

		}
	}

	if (stored) {
		flush(stored, consts);
		stored = 0;
	}

	fflush(NULL);
}

void
pass1(void)
{
	addr_t	offset;

	pass = FIRSTPASS;
	for (curoffset = offset = 0; offset <= maxoffset;
	  offset += sizeof(word_t)) {
		if (curoffset != offset
		  && fseek(infp, curoffset = offset, SEEK_SET) == -1) {
			perror("fseek");
			jumpfree();
		}
		flags = 0;
		pc = ppc = offset + initialpc;
		leninstbuf = 0;
		if (validinst()) {
			/*
			 * It's a potential instruction.
			 */
			insts[offset].size = curoffset - offset;
			insts[offset].flags = flags;
			if (flags & 3) {
				addr_t	*reqd;

				if ((reqd = malloc((flags & 3) * sizeof(*reqd)))
				  == NULL) {
					perror("malloc");
					jumpfree();
				}
				switch (flags & 3) {
				case 3:	reqd[2] = required[2];
				case 2:	reqd[1] = required[1];
				case 1:	reqd[0] = required[0];
				}
				insts[offset].required = reqd;
			} else
				insts[offset].required = NULL;
#ifdef DEBUG
			if (debug >= 3) {
				int	i;

				fprintf(stderr,
				  "Writing offset = %lx, size = %d, flags = %d",
				  (long)offset, insts[offset].size,
				  insts[offset].flags);
				for (i = 0; i < (insts[offset].flags & 3); i++)
					fprintf(stderr, ", reqd = %lx",
					  (long)insts[offset].required[i]);
				fprintf(stderr, "\n");
			}
#endif
		}
	}
}

void
findbadrefs(void)
{
	long	offset;		/* must be signed */
	long	offset2;	/* must be signed */
	int	i;
	size_t	size;
	int	changes;

#ifdef DEBUG
	if (debug >= 2) {
		printall();
		fprintf(stderr, "\n\n\n");
	}
#endif

	/*
	 * Instructions that don't set PC
	 * must be followed by a valid instruction.
	 */
	do {
		changes = 0;
		for (offset = maxoffset + longestinstsize; offset >= 0;
		  offset -= sizeof(word_t)) {
			/*
			 * Back up to a possible instruction.
			 * We do this to jump over a large data section.
			 */
			for (offset2 = offset; offset2 >= 0
			  && insts[offset2].size == 0;
			  offset2 -= sizeof(word_t))
				;
			if (offset2 < 0)
				break;
			if (offset2 + longestinstsize < offset)
				offset = offset2 + longestinstsize;
			if (FETCH(offset, -1))
				continue;

			/*
			 * We've found an invalid instruction.
			 * See if any instructions advance PC here based on
			 * the size of the instruction and its operands.
			 */
			for (size = sizeof(word_t); size <= longestinstsize
			  && size <= offset; size += sizeof(word_t))
				if (FETCH(offset - size, size)
				  && !JUMP(offset - size)) {
#ifdef DEBUG
if (debug >= 2)
fprintf(stderr,
  "1. Deleting offset %lx, size %d, flags = %d\n", (long)(offset - size),
  size, insts[offset - size].flags);
#endif
					DELETE(offset - size);
					changes++;
				}
		}

		/*
		 * See if any instructions require
		 * an invalid instruction to be valid.
		 */
		for (offset2 = 0; offset2 <= maxoffset;
		  offset2 += sizeof(word_t)) {
			if (insts[offset2].size == 0)
				continue;
			for (i = 0; i < NREQD(offset2); i++)
				if (insts[offset2].required[i] >= initialpc
				  && insts[offset2].required[i] <= initialpc
				  + maxoffset
				  && !FETCH(insts[offset2].required[i]
				  - initialpc, -1)) {
#ifdef DEBUG
					if (debug >= 2)
fprintf(stderr, "2. Deleting offset %lx, size %d because %lx is not valid\n",
(long)offset2, insts[offset2].size, (long)insts[offset2].required[i]);
#endif
					DELETE(offset2);
					changes++;
					break;
				}
		}

#ifdef DEBUG
		if (debug >= 2) {
			static int	try = 1;

			fprintf(stderr,
			  "TRY %d ###############################\n", try);
			printall();
			fprintf(stderr, "\n\n\n");
			try++;
		}
#endif
	} while (changes);
}

struct queue {
	addr_t		address;
	struct queue	*next;
};

static struct queue	head = { 0, NULL };

static void
writeq(addr_t address)
{
	struct queue	*tail;
	struct queue	*newq;

	if ((newq = malloc(sizeof(*newq))) == NULL) {
		perror("malloc");
		jumpfree();
	}
	newq->address = address;

#ifdef DEBUG
	if (debug >= 2)
		fprintf(stderr, "Wrote offset = %lx\n", (long)address);
#endif

	newq->next = NULL;

	for (tail = &head; tail->next; tail = tail->next)
		;
	tail->next = newq;
}

static struct queue *
readq(void)
{
	struct queue	*result;

	if (head.next) {
		result = head.next;
		head.next = head.next->next;

#ifdef DEBUG
		if (debug >= 2)
			fprintf(stderr, "Read offset = %lx\n",
			  (long)result->address);
#endif
	} else
		result = NULL;

	return result;
}

static void
makegood(addr_t offset)
{
	size_t		size;
	struct queue	*qptr = NULL;

#ifdef DEBUG
	if (debug >= 2)
		fprintf(stderr, "makegood(%lx)\n", (long)offset);
#endif

	if (insts[offset].flags & ISGOOD)
		return;

	do {
		if (qptr) {
			offset = qptr->address;
			free(qptr);
		}

#ifdef DEBUG
		if (debug >= 3)
			fprintf(stderr, "Going with offset = %lx\n",
			  (long)offset);
#endif

		while (1) {

#ifdef DEBUG
			if (debug >= 3)
				fprintf(stderr, "Offset = %lx\n", (long)offset);
#endif

			if (insts[offset].size == 0 ||
			  insts[offset].flags & ISGOOD)
				break;

			for (size = sizeof(word_t); size < longestinstsize
			  && size <= offset; size += sizeof(word_t))
				if (insts[offset - size].size > size) {
#ifdef DEBUG
					if (debug >= 2)
fprintf(stderr, "3. Deleting offset %lx, size %d, flags = %d because of %lx\n",
  (long)offset, insts[offset].size, insts[offset].flags,
  (long)(offset - size));
#endif
					DELETE(offset - size);
					break;
				}
			for (size = sizeof(word_t); size < insts[offset].size
			  && size <= offset; size += sizeof(word_t))
				if (insts[offset + size].size) {
#ifdef DEBUG
					if (debug >= 2)
fprintf(stderr, "4. Deleting offset %lx, size %d, flags = %d because of %lx\n",
  (long)(offset + size), insts[offset + size].size,
  insts[offset + size].flags, (long)offset);
#endif
					DELETE(offset + size);
				}

			insts[offset].flags |= ISGOOD;
			if ((insts[offset].flags & ISBRA)
			  || ((insts[offset].flags & ISJMP) &&
			  insts[offset].flags & 3)) {
				if (insts[offset].required[0] >= initialpc
				  && insts[offset].required[0] <= initialpc
				  + maxoffset)
					offset = insts[offset].required[0]
					  - initialpc;
				else
					break;
			} else if ((insts[offset].flags
			  & (ISBSR | ISJSR | ISBRA | ISBRcc | ISDBcc | ISJMP))
			  && (insts[offset].flags & 3)
			  && insts[offset].required[0] >= initialpc
			  && insts[offset].required[0] <= initialpc
			  + maxoffset) {
				writeq(insts[offset].required[0] - initialpc);
				offset += insts[offset].size;
			} else if (insts[offset].flags & (ISRTS | ISJMP))
				break;
			else
				offset += insts[offset].size;
		}
	} while (qptr = readq());
}

long
overlaps(void)
{
	long	num = 0;
	long	offset;
	size_t	size;

	for (offset = 0; offset <= maxoffset; offset += sizeof(word_t)) {
		if (insts[offset].size == 0)
			continue;

		for (size = sizeof(word_t); size < longestinstsize
		  && size <= offset; size += sizeof(word_t))
			if (insts[offset - size].size > size)
				num++;
	}

	return num;
}

static long
nextgood(addr_t offset)
{
	long	next;

	for (next = offset + sizeof(word_t); next <= maxoffset
	  && (insts[next].size == 0 || (insts[next].flags & ISGOOD) == 0);
	  next += sizeof(word_t))
		;

	return (next > maxoffset) ? -1 : next;
}

static int
reach(addr_t prevonly, addr_t target)
{
	addr_t	offset;

	for (offset = prevonly; insts[offset].size && offset < target;
	  offset += insts[offset].size)
		;

	return offset == target;
}

static void
fixoverlaps(void)
{
	addr_t	offset;
	addr_t	good = nextgood(0);

	for (offset = 0; offset <= maxoffset; ) {
		if (insts[offset].size == 0) {
			offset += sizeof(word_t);
			continue;
		}
		if (insts[offset].flags & ISGOOD) {
			offset += insts[offset].size;
			continue;
		}

		if (/*good >= 0 &&*/ good <= offset)
			good = nextgood(offset);

		if (reach(offset, good))
			makegood(offset);
		else
			offset += sizeof(word_t);
	}
}

struct data {
	addr_t	start;
	size_t	size;
};

void
disassemble(void)
{
	addr_t		offset;
	long		fsize;
	int		changes;
	short		nlabels;
	int		i;
	unsigned char	consts[BUFSIZ];
	size_t		stored = 0;

	switch (chip) {
	case MC68000:
	case MC68010:
		longestinstsize = 10;
		break;
	case MC68020:
	case MC68030:
	default:
		longestinstsize = 20;
		break;
	}

	if (fseek(infp, 0, SEEK_END) == -1) {
		perror("fseek");
		jumpfree();
	}
	if ((gfsize = fsize = ftell(infp)) == -1) {
		perror("ftell");
		jumpfree();
	} else if (fsize < sizeof(word_t))
		jumpfree();
	if (fseek(infp, 0, SEEK_SET) == -1) {
		perror("fseek");
		jumpfree();
	}
	maxoffset = (fsize - (sizeof(word_t) - 1)) & ~(sizeof(word_t) - 1);

	/*
	 * Malloc and initialize instruction structures.
	 */
	if ((insts = malloc(fsize * sizeof(*insts))) == NULL) {
		perror("malloc");
		jumpfree();
	}
	memset(insts, '\0', fsize * sizeof(*insts));
	while (fsize--)
		insts[fsize].required = NULL;

	/*
	 * Pass 1:
	 *
	 * Determine where the code is and where the data is.
	 */
	pass1();

	if (onepass == INCONSISTENT) {
		printall();
		jumpfree();
	}

	/*
	 * Process offsets specified by the user as being data.
	 */
	if (nfile) {
		FILE	*nfp;

		if (nfp = fopen(nfile, "r")) {
			char		nbuf[80];
			unsigned long	ul;
			char		*cp;

			while (fgets(nbuf, sizeof nbuf, nfp)) {
				ul = strtoul(nbuf, &cp, 0);
				if (cp != nbuf) {
					if (ul >= initialpc && ul <= initialpc
					  + maxoffset) {
						DELETE(ul - initialpc);
					} else
						fprintf(stderr,
						  "File %s: bad pc: %s\n",
						  nfile, nbuf);
				} else
					fprintf(stderr,
					  "File %s: bad pc: %s\n", nfile, nbuf);
			}
			(void)fclose(nfp);
		} else
			perror(nfile);
	}

	/*
	 * Process offsets specified by the user as being instructions.
	 */
	if (ifile) {
		FILE	*ifp;

		if (ifp = fopen(ifile, "r")) {
			char		ibuf[80];
			unsigned long	ul;
			char		*cp;

			while (fgets(ibuf, sizeof ibuf, ifp)) {
				ul = strtoul(ibuf, &cp, 0);
				if (cp != ibuf) {
					if (ul >= initialpc && ul <= initialpc
					  + maxoffset)
						makegood(ul - initialpc);
					else
						fprintf(stderr,
						  "File %s: bad pc: %s\n",
						  ifile, ibuf);
				} else
					fprintf(stderr,
					  "File %s: bad pc: %s\n", ifile, ibuf);
			}
			(void)fclose(ifp);
		} else
			perror(ifile);
	}

	/*
	 * Instructions that don't set PC
	 * must be followed by a valid instruction.
	 *
	 * Instructions must reference valid instructions.
	 */
	findbadrefs();

	if (onepass == CONSISTENT) {
		printall();
		jumpfree();
	}

	for (offset = 0; offset <= maxoffset; offset += sizeof(word_t))
		if (insts[offset].size
		  && (insts[offset].flags & (ISBSR | ISJSR))
		  && (insts[offset].flags & 3)
		  && insts[offset].required[0] >= initialpc
		  && insts[offset].required[0] <= initialpc + maxoffset
		  && (insts[insts[offset].required[0] - initialpc].flags
		  & ISLINK)) {
#ifdef DEBUG
			if (debug >= 3)
				fprintf(stderr,
				  "Number of overlaps = %ld\n", overlaps());
#endif
			makegood(insts[offset].required[0] - initialpc);
		}

	findbadrefs();

#ifdef DEBUG
	if (debug)
		fprintf(stderr,
		  "Number of overlaps after LINKs with labels = %ld\n",
		  overlaps());
#endif

	for (offset = 0; offset <= maxoffset; offset += sizeof(word_t))
		if (insts[offset].size
		  && (insts[offset].flags & ISGOOD) == 0
		  && (insts[offset].flags & ISLINK)) {
#ifdef DEBUG
			if (debug >= 3)
				fprintf(stderr, "Number of overlaps = %ld\n",
				  overlaps());
#endif
			makegood(offset);
		}

	findbadrefs();

#ifdef DEBUG
	if (debug)
		fprintf(stderr, "Number of overlaps after all LINKs = %ld\n",
		  overlaps());
#endif

	do {
		changes = 0;
		for (offset = 0; offset <= maxoffset; offset += sizeof(word_t))
			if (insts[offset].size
			  && (insts[offset].flags & ISGOOD) == 0
			  && (insts[offset].flags & (ISBSR | ISJSR | ISBRA
			  | ISBRcc | ISDBcc | ISJMP))
			  && (insts[offset].flags & 3)
			  && insts[offset].required[0] >= initialpc
			  && insts[offset].required[0] <= initialpc
			  + maxoffset
			  && (insts[insts[offset].required[0] - initialpc].flags
			  & ISGOOD)) {
#ifdef DEBUG
				if (debug >= 3)
					fprintf(stderr,
					  "Number of overlaps = %ld\n",
					  overlaps());
#endif
				makegood(offset);
				changes++;
			}
	} while (changes);

	findbadrefs();

#ifdef DEBUG
	if (debug)
		fprintf(stderr,
"Number of overlaps after all LINKs and jumps to good addresses = %ld\n",
	  overlaps());
#endif

	do {
		changes = 0;
		for (offset = 0; offset <= maxoffset; offset += sizeof(word_t))
			if (insts[offset].size
			  && (insts[offset].flags & ISGOOD) == 0
			  && (insts[offset].flags & (ISBSR | ISJSR | ISBRA
			  | ISBRcc | ISDBcc | ISJMP | ISRTS))) {
#ifdef DEBUG
				if (debug >= 3)
					fprintf(stderr,
					  "Number of overlaps = %ld\n",
					  overlaps());
#endif
				makegood(offset);
				changes++;
			}
	} while (changes);

	findbadrefs();

#ifdef DEBUG
	if (debug)
		fprintf(stderr,
		  "Number of overlaps after all LINKs and branches = %ld\n",
		  overlaps());
#endif

	findbadrefs();
	fixoverlaps();

#ifdef DEBUG
	if (debug)
		fprintf(stderr, "Number of overlaps = %ld (should be 0)\n",
		  overlaps());
#endif

	findbadrefs();

	/*
	 * Get the labels.
	 */
	dcpass();
	for (offset = 0; offset <= maxoffset; ) {
		if (insts[offset].size) {
			for (i = 0; i < NREQD(offset); i++)
				SETLABEL(insts[offset].required[i]);
			offset += insts[offset].size;
		} else
			offset += sizeof(word_t);
	}
	for (nlabels = 0, offset = 0; offset <= maxoffset; offset ++)
		if (insts[offset].flags & ISLABEL) {
#ifdef DEBUG
			if (debug >= 2)
				fprintf(stderr, "Label %d at %lx\n",
				  nlabels + 1, (long)offset);
#endif
			insts[offset].labelnum = ++nlabels;
		}

	/*
	 * Last pass: Print!
	 */
	if (fseek(infp, 0, SEEK_SET) == -1) {
		perror("fseek");
		jumpfree();
	}
	pc = ppc = initialpc;
	leninstbuf = 0;
	pass = LASTPASS;
	for (stored = 0, curoffset = offset = 0; offset <= maxoffset; )  {
		if (curoffset != offset
		  && fseek(infp, curoffset = offset, SEEK_SET) == -1) {
			perror("fseek");
			jumpfree();
		}
		flags = 0;
		if (insts[offset].size) {
			if (stored) {
				flush(stored, consts);
				stored = 0;
			}

			pc = ppc = initialpc + offset;
			leninstbuf = 0;
			validinst();
			offset = curoffset;
		} else {
			if (stored == 0)
				pc = ppc = initialpc + offset;
			if (insts[offset].flags & ISLABEL) {
				flush(stored, consts);
				stored = 0;
			}
			if (fread(consts + stored, 1, 1, infp) == 1)
				if (++stored >= sizeof consts) {
					flush(stored, consts);
					stored = 0;
				}

			offset++;
		}
	}
	if (stored) {
		flush(stored, consts);
		stored = 0;
	}

	jumpfree();
	/* NOTREACHED */
}
