;------------------------------------*\
; FLUSH -     Bruneau Babet - 1991    |
;------------------------------------*/

.MODEL  TINY, C
INCLUDE MACRO.INC
INCLUDE VECTOR.INC

;[]--------------------------[ MCB STRUCTURES ]------------------------------[]
;| DOS preceedes every allocated Memory Block with a Memory Control Block     |
;| Each MCBs starts with a byte ('M' or 'Z') followed by the Process ID (PSP) |
;| of the owner.  The number of paragraphs follows.  The rest of the fields   |
;| are not used (except that in DOS 4.x+ name of the owner is stored at Offset|
;| 08h in an ASCIIZ string)                                                   |
;[]--------------------------------------------------------------------------[]
MCBStruct       STRUC                           ; MCB structure
        TypeOfMCB       db      ?               ; Block type
        OwnerOfMCB      dw      ?               ; Block Owner
        SizeOfMCB       dw      ?               ; Block Size
MCBStruct       ENDS                            ; End of MCB structure

_MIDLMCB        EQU     4DH                     ; Sig. of Valid MCB Block - 'M'
_LASTMCB        EQU     5AH                     ; Sig. of Last MCB Block  - 'Z'
_FREEMCB        EQU     00H                     ; Sig. of a Free Block


;[]-------------------[ RETURN NEXT MCB SEGMENT IN AX  ]---------------------[]
;|  This MACRO assumes ES to be pointing at the Segment of an MCB and returns |
;|  the segment of the next MCB in AX..                                       |
;|  ENTRY:     ES:  MCB Segment                                               |
;|  RETURN:    AX:  Next MCB Segment                                          |
;[]--------------------------------------------------------------------------[]
@NextMCB        MACRO
        mov     ax, es
        add     ax, word ptr es:[MCBStruct.SizeOfMCB]
        inc     ax
        ENDM


;[]--------------------[ CONVERT AL TO LOWER CASE ]--------------------------[]
;|  This MACRO converts the character in AL to lower case...                  |
;|  ENTRY:     AL: Byte to be converted                                       |
;|  RETURN:    AL: Lower Case (if 'A' <= AL <= 'Z')                           |
;[]--------------------------------------------------------------------------[]
@ToLwr  MACRO
        LOCAL   __$0__
        cmp     al, 'A'
        jb      __$0__
        cmp     al, 'Z'
        ja      __$0__
        or      al, 00100000B
__$0__:
        ENDM

;[]-------------------[ CLEAR OUT KEYBOARD BUFFER ]--------------------------[]
;|  This MACRO clears the keyboard buffer by repeated retrieving all keys     |
;|  current waiting in the keyboard buffer...                                 |
;|  ENTRY:      NONE                                                          |
;|  RETURN:     NONE  (NOTE: AX is destroyed !)                               |
;[]--------------------------------------------------------------------------[]
@ClrKbdBuff     MACRO
        LOCAL   __$0__, __$1__
__$0__: mov     ah, 01
        int     16H
        jz      __$1__
        mov     ah, 00
        int     16H
        jmp     __$0__
__$1__:
        ENDM


;[]-----------------[ MACRO TO SAVE INTERRUPT VECTOR TABLE ]-----------------[]
;|  This macro is specific to FLUSH and saves a copy of the interrupt vector  |
;|  table into IntBuffer...                                                   |
;[]--------------------------------------------------------------------------[]
@SaveIVT        MACRO
        push    ds
        pop     es
        push    ds
        sub     si, si
        mov     ds, si
        mov     di, OFFSET IntBuffer
        mov     cx, (_FLUSH_INTSAVED SHL 1)
        cld
        rep     movsw
        pop     ds
        ENDM


.CODE
        ORG     2CH                     ; Offset 2CH within PSP
EnvSeg  label   word                    ; Segment Of Environment Block
        ORG     80H                     ; Offset 80H within PSP
CmdLen  label   byte                    ; Length of Command Line Argument
        ORG     81H                     ; Offset 81H within PSP
CmdArg  label   byte                    ; Command Line Argument


.STARTUP                                ; Startup Macro
        jmp     InitTSR                 ; Jmp to Initialize TSR

_STKLEN         EQU     200             ; Length of Stack of TSR
_KEYSTATUS      EQU     008H            ; Status of Shift/Alt/Ctrl Key (HotKey)
_TSR_SCANCODE   EQU     001H            ; Scan Code of the HotKey

_TSR_DORMANT    EQU     000H            ; Flag to signal TSR's dormant
_TSR_REQUESTED  EQU     001H            ; Flag to signal TSR's about to pop up
_TSR_ACTIVATED  EQU     002H            ; Flag to signal TSR's active

_2Fh_PRESENT    EQU     000H            ; 2F API: Present Request
_2Fh_UNHOOK     EQU     001H            ; 2F API: Unhook Vectors Request
_2Fh_FLUSH      EQU     002H            ; 2F API: Flush Request


TsrStack        dw      (_STKLEN SHR 1) dup (00)        ; TSR's Stack
TsrStackEnd     label   word                            ; End of Stack Label
TsrMltplxID     db      0BBH                            ; TSR's Multiplex ID
TsrScanCode     db     _TSR_SCANCODE                    ; Hotkey's ScanCode
TsrStateFlg     db     _TSR_DORMANT                     ; Hotkey's Status Flag
TsrPSP          dw      00                              ; The TSR's PSP
TsrSaveDS       dw      00                              ; To Save Data Segment
TsrSaveSS       dw      00                              ; To Save Stack
TsrSaveSP       dw      00                              ; To Save Stack Ptr
@DefVect        MainVectors,09,13,2F                    ; Vectors Trapped



;[]--------------------[ KEYBOARD HANDLER INT 09 ]---------------------------[]
;| The following INT 09 (Keyboard) Handler looks for the Hot Key when the TSR |
;| is Dormant;  When the Hot Key is detected, the TSR's Entry routine is      |
;| called after making sure that no DISK I/O (INT 13H Flag) is taking place.  |
;[]--------------------------------------------------------------------------[]
New09Handler    PROC    FAR
        sti                                     ; Set interrupt Flag
        cmp     cs:TsrStateFlg, _TSR_DORMANT    ; Look for HotKey if Dormant !
        jne     @@L01                           ; If not Dormant, simply chain
        push    ax                              ; Save AX register
        in      al, 60H                         ; Get Scan code from Keyboard
        cmp     al, cs:TsrScanCode              ; Is it the Hot Key ScanCode ?
        jne     @@L00                           ; No: Jmp to chain
        mov     ah, 02                          ; Function 02: Get Key Status
        int     16H                             ; Call Keyboard BIOS
        test    al, _KEYSTATUS                  ; Compare to Hotkey Status
        je      @@L00                           ; If not equal, just chain..
        inc     cs:TsrStateFlg                  ; Signal that TSR's Requested
@@L00:  pop     ax                              ; Restore AX register
@@L01:  pushf                                   ; Push Flags on Stack
        cli                                     ; Clear Interrupt Flag
        call    cs:Old09Handler                 ; Chain to Old Handler
@@L02:  cmp     cs:TsrStateFlg, _TSR_REQUESTED  ; Have we been requested
        jne     @@L03                           ; Nope...
        cmp     cs:Vect13.VectFlag, 00          ; Any Disk I/O ?
        jne     @@L00                           ; Yes.... skip TSR !
        call    TsrEntry                        ; Active the TSR
        dec     cs:TsrStateFlg                  ; Change Flag to _TSR_DORMANT
@@L03:  iret                                    ; End of Interrupt
New09Handler    ENDP                            ; End of Keyboard Handler



;[]--------------------[ DISK I/O HANDLER - INT 13 ]-------------------------[]
;|  The purpose of the INT 13H handler is to prevent any TSR popup while      |
;|  some type of DISK I/O is taking place.  Since our TSR is launched         |
;|  from a Hardware Interrupt (KeyBoard) this allows for safe PopUps..        |
;[]--------------------------------------------------------------------------[]
New13Handler    PROC    FAR                     ; DISK I/O Int. Serv. Rout.
        inc     cs:Vect13.VectFlag              ; Flag we're in INT 13H
        pushf                                   ; Push the Flags
        call    cs:Old13Handler                 ; Call the Old Handler
        dec     cs:Vect13.VectFlag              ; Clear Flag
        retf    2                               ; Ret. without disturbing Flgs
New13Handler    ENDP                            ; End of DISK I/O Handler




;[]--------------------[ MULTIPLEX INTERFACE - 2F  ]-------------------------[]
;|  The 2F handler handles presence requests along with the request to release|
;|  all trapped vectors.                                                      |
;|  ENTRY:  AH: Our_ID and AL: _2Fh_PRESENT                                   |
;|  RETURN: AL: 0FFh                                                          |
;|                                                                            |
;|  ENTRY:  AH: Our_ID and AL: _2Fh_UNHOOK                                    |
;|  RETURN: BX  == 0000H -> Unable to Unhook (AL = Offending Interrupt #)     |
;|          BX  != 0000H -> Vectors Unhooked  BX = PSP of TSR                 |
;[]--------------------------------------------------------------------------[]
New2FHandler    PROC    FAR
        cmp     ah, cs:TsrMltplxID              ; Verify if our ID
        je      @@L00                           ; Yes... process request
        jmp     cs:Old2FHandler                 ; Else.. chain to Old Handler
@@L00:  cmp     al, _2Fh_PRESENT                ; Look for Presence Chk Request
        jne     @@L02                           ; Nope...
        mov     al, 0FFH                        ; Signal' we're present
@@L02:  cmp     al, _2Fh_UNHOOK                 ; UnHook Request ?
        jne     @@L03                           ; Nope.. just return
        push    cs                              ; Make DX point to the
        pop     dx                              ;      Code Segment
        xor     bx, bx                          ; Clear BX (Assume Error)

      ;[ DX MUST POINT TO TSR'S CODE SEGMENT !! ]
        @ChkVect MainVectors                    ; Verify if we're still owner
        jc      @@L04                           ; Nope... error: BX=0000H

      ;[ DX MUST POINT TO TSR'S CODE SEGMENT !! ]
        @RstVect MainVectors                    ; Reset the Vectors
        mov     bx, dx                          ; Return our PSP in BX
        iret                                    ; Return
@@L03:  cmp     al, _2Fh_FLUSH                  ; A request to Flush ?
        jne     @@L04                           ; Nope..
        push    cs                              ; Push Code Segment Value
        pop     ds                              ; Make DS == CS
        jmp     FlushFrom2FH                    ; And... go for it   !
@@L04:  iret                                    ; Return
New2FHandler    ENDP




;[]-----------------------[ DATA VARIABLES OF TSR  ]-------------------------[]
;|  Various variables and messages used by TSR...                             |
;[]--------------------------------------------------------------------------[]
_FLUSH_INTSAVED EQU  0FFH                               ; # of Vectors to Save
IntBuffer       dd      (_FLUSH_INTSAVED) dup (00)      ; Buffer to save Vects
_Title_         db  'FLUSH - TSR to save/restore the vector table - '
                db  ' Chamonix and Chocolat - 1991',10,13,0
QuerryMsg       db  'FLUSH: Do you really want to Flush (Y/N) ?:',0
VectMsg         db  '<< FLUSH: Vector Table has been restored.. >>',10,13,0
BadMCBMsg       db  '<< FLUSH: Corrupted Entry in MCB Chain.... >>',10,13,0
MemMsg          db  '<< FLUSH: Memory Blocks have been freed... >>',10,13,0
FirstMCB        dw  00                                  ; Seg. of Fisrt MCB
VideoMode       db  -1                                  ; Assume not saving




;[]-----------------[ RESTORE THE INTERRUPT VECTOR TABLE ]-------------------[]
;|   The Function restores the copy of the Interrupt Vector table saved and   |
;|   displays a message after the restoration...                              |
;|   ENTRY:     ASSUMES DS -> CODE SEG OF TSR                                 |
;[]--------------------------------------------------------------------------[]
RestoreIVT      PROC
        @PUSH   bx, si, di, es                          ; Save Registers
        mov     si, OFFSET IntBuffer                    ; DS:SI -> IntBuffer
        xor     di, di                                  ; Clear DI
        mov     es, di                                  ; ES:DI -> 0000:0000
        cld                                             ; Clear Direction Flag
        mov     cx, (_FLUSH_INTSAVED SHL 1)             ; Load # of Int * 2
        rep     movsw                                   ; Restore Vectors
        mov     si, OFFSET VectMsg                      ; DS:SI -> Vect Message
        mov     bl, 10                                  ; Load Attribute
        call    PutStr                                  ; Display Message
        @POP    bx, si, di, es                          ; Restore Registers
        ret                                             ; Return to Caller
RestoreIVT      ENDP



;[]------------------------[ FREE MEMORY BLOCKS  ]---------------------------[]
;|   This routine walks down the MCB chain and frees Memory blocks which are  |
;|   beyond the TSR's PSP...                                                  |
;|   ENTRY:     ASSUMES DS -> CODE SEGMENT OF TSR                             |
;[]--------------------------------------------------------------------------[]
FreeMemory      PROC
        @PUSH   ax, es, bx, si                          ; Save Registers
        mov     ax, FirstMCB                            ; Load First MCB Seg.
        mov     es, ax                                  ; ES = First MCB Seg.
ChkBlk: mov     ax, es:[MCBStruct.OwnerOfMCB]           ; Load Owner in AX
        cmp     ax, _FREEMCB                            ; Is it a Free Block
        je      CkLast                                  ; See if last Block
        cmp     ax, TsrPSP                              ; Higher than TSR ?
        jbe     CkLast                                  ; No, See if last !
FreeBk: push    es                                      ; Save MCB Segment
        mov     ax, es                                  ; Load MCB in ax
        inc     ax                                      ; AX = Seg of Block
        mov     es, ax                                  ; ES = Seg of Block
        mov     ah, 49h                                 ; Free Blk Function
        int     21h                                     ; Call DOS
        pop     es                                      ; Restore MCB
CkLast: cmp     es:[MCBStruct.TypeOfMCB], _LASTMCB      ; Is it the Last One ?
        je      Done01                                  ; Yes.. we're done   !
        cmp     es:[MCBStruct.TypeOfMCB], _MIDLMCB      ; Should be Middle MCB
        jne     BadMCB                                  ; If not, bad MCB !
        cmp     es:[MCBStruct.OwnerOfMCB], 0A000h       ; Owned by UMB prog  ?
        jae     Done01                                  ; Yes.. we're done   !
        @NextMCB                                        ; Get Next MCB
        mov     es, ax                                  ; Next MCB in AX
        jmp     ChkBlk                                  ; Check Next Block
BadMCB: mov     si, OFFSET BadMCBMsg                    ; Load BadMCBMsg
        mov     bl, 12                                  ; Load Attribute
        call    PutStr                                  ; Display Messsage
        jmp     Done02                                  ; Get out of here !
Done01: mov     si, OFFSET MemMsg                       ; Load Memory Message
        mov     bl, 10                                  ; Load Attribute
        call    PutStr                                  ; Display Message
Done02: @POP    ax, es, bx, si                          ; Restore registers
        ret                                             ; Return to Caller
FreeMemory      ENDP



;[]------------------[ TSR's ENTRY POINT - MAIN OF TSR ]---------------------[]
;|  The following is the *Entry Point* to the TSR... Registers are saved, a   |
;|  new stack installed and off we go...                                      |
;[]--------------------------------------------------------------------------[]
TsrEntry        PROC
        inc     cs:TsrStateFlg                          ; Flag we're Active
        mov     cs:TsrSaveSS, ss                        ; Save Stack Segment
        mov     cs:TsrSaveSP, sp                        ; Save Stack Pointer
        push    cs                                      ; Make Stack equal
        pop     ss                                      ;      to Code Segment
        mov     sp, OFFSET TsrStackEnd                  ; Point SP to end Stack
        @PUSH   ax, bx, cx, dx, es, si, di, ds, bp      ; Save Registers
        push    cs                                      ; Push Code Segment
        pop     ds                                      ; Make DS=CS
        assume  ds:_TEXT                                ; Set up assumption
        mov     ah, 0Fh                                 ; Get Video Info (page)
        int     10h                                     ; Call BIOS
        ;- BH contains the current video page           ;
        mov     ah, 03h                                 ; Read cursor loc/shape
        int     10h                                     ; Call BIOS
        ;- DX contains the current cursor location      ;
        push    dx                                      ; Save on stack *(1)
        xor     dx, dx                                  ; Make dx =(0, 0)
        mov     ah, 02h                                 ; Set Cursor Location
        int     10h                                     ; Call BIOS
        mov     si, OFFSET QuerryMsg                    ; Load Msg -> DS:SI
        mov     bl, 0Eh                                 ; Attribute <<
        call    PutStr                                  ; Display Message
        @ClrKbdBuff                                     ; Clear Keyboard Buffer
L0000:  mov     ah, 00h                                 ; Function 0 - Get Key
        int     16h                                     ; Get a Key
        @ToLwr                                          ; Convert to Lower Case
L0001:  cmp     al, 'y'                                 ; To Flush or not to..
        je      L0003                                   ; Positive answer !
L0002:  cmp     al, 'n'                                 ; To Flush or not to..
        je      L0005                                   ; Do not Flush !!!!
        mov     ax, 0E07H                               ; Teletype/Bell
        int     10h                                     ; Beep via BIOS
        jmp     L0000                                   ; Loop again for (Y/N)
L0003:  cmp     VideoMode, -1                           ; Restore Mode ?
        je      L0004                                   ; Nope...
        mov     al, VideoMode                           ; Load Video Mode
        mov     ah, 00                                  ; Set Mode Function
        int     10h                                     ; Restore Mode...
L0004:  mov     ah, 02h                                 ; Set Cursor Location
        int     10h                                     ; Goto (0, 0)
        mov     si, OFFSET _Title_                      ; SI-> Title Message
        mov     bl, 15                                  ; Load Attribute
        call    PutStr                                  ; Display Message.

FlushFrom2FH    EQU   $                                 ; Label For 2FH Entry
        call    RestoreIVT                              ; Restore IVT
        call    FreeMemory                              ; Restore Memory Blocks
        push    cs                                      ; Put CS on stack
        pop     bx                                      ; Get PSP in BX
        mov     ah, 50h                                 ; Seg PID function
        int     21h                                     ; Set PID via DIOS
        mov     cs:TsrStateFlg, _TSR_DORMANT            ; Reset TSR's Flag
        mov     dx, OFFSET Transient                    ; DX -> End of Res.
        int     27h                                     ; Stay Resident

       ;[WE'LL GET HERE ONLY IF USER CHOSE NOT TO FLUSH ]
L0005:  pop     dx                                      ; Restore Cursor *(1)
        mov     ah, 02                                  ; Set Cursor Loc. Func
        int     10h                                     ; Call BIOS
        @POP    ax, bx, cx, dx, es, si, di, ds, bp      ; Restore registers
        assume  ds: NOTHING                             ; Clear assumption
        mov     ss, cs:TsrSaveSS                        ; Restore SS register
        mov     sp, cs:TsrSaveSP                        ; Restore SP register
        dec     cs:TsrStateFlg                          ; Flag we're done
        ret                                             ; Return to caller
TsrEntry        ENDP                                    ; End of TSREntryPoint




;[]---------------------[ DISPLAY STRING USING BIOS ]------------------------[]
;| This function displays string using BIOS...                                |
;| Upon ENTRY the following is assumed:                                       |
;| ENTRY        BL: Attribute                                                 |
;|           DS:SI: Points to ASCIIZ string                                   |
;| RETURN:      NONE                                                          |
;|              SI: Points to the End of the String !                         |
;[]--------------------------------------------------------------------------[]
PutStr  PROC
        @PUSH   ax, bx, cx                      ; Save Registers
        mov     ah, 0FH                         ; Get Video Info (Current page)
        int     10H                             ; Call BIOS
        mov     ah, 0EH                         ; Use Teletype
        mov     cx, 1                           ; Load Count
@@L00:  lodsb                                   ; Load Character in AL
        or      al, al                          ; At end of string ?
        jz      @@L02                           ; Exit if Null (EOS)
        cmp     al, 0DH                         ; Carriage return ?
        je      @@L01                           ; Use Teletype !
        cmp     al, 0AH                         ; Line Feed ?
        je      @@L01                           ; Use Teletype !
        cmp     al, 07H                         ; Bell ?
        je      @@L01                           ; Use Teletype !
        cmp     al, 08H                         ; BS   ?
        je      @@L01                           ; Use Teletype !
        push    ax                              ; Save AX register
        mov     ax, 0920H                       ; Put Space character w/Attr
        int     10H                             ; Call BIOS
        pop     ax                              ; Restore AX
@@L01:  int     10H                             ; Call BIOS using Teletype
        jmp     @@L00                           ; Loop for next character
@@L02:  @POP    ax, bx, cx                      ; Restore Registers
        ret                                     ; Return to Caller
PutStr  ENDP


_TSR_SIZE       EQU     ($  -   OFFSET @Startup + 100H)
_TSR_SIZE_PARA  EQU     ((_TSR_SIZE + 15) SHR 4)




;[]--------------------[ END OF RESIDENT PORTION OF CODE ]-------------------[]
;|  The following portion is the Transient portion of the TSR... It is        |
;| discarded after the TSR's initialization...                                |
;[]--------------------------------------------------------------------------[]
Transient:




;[]--------------------[  TSR Messages/Warnings/Errors  ]--------------------[]
;| The following is a list of generic messages displayed by the TSR when      |
;| initializing, unloading etc.. etc..                                        |
;[]--------------------------------------------------------------------------[]
Options      db 'Usage:  FLUSH /Option  [ or -Option ]',10,13
             db '        Options:   /?     Quick info regarding FLUSH',10,13
             db '                   /F     Restore the IVT saved and free Memory',10,13
             db '                   /U     Unload the resident copy of FLUSH',10,13
             db '                   /V     Save and Restore the Video Mode',10,13,00
Installed    db '<< FLUSH: FLUSH has now been made resident.>>',10,13,0
Already      db '<< FLUSH: Copy of TSR is already resident. >>',10,13,0
Removed      db '<< FLUSH: FLUSH has been uninstalled...    >>',10,13,0
NoneFound    db '<< FLUSH: No copy of FLUSH in memory to uninstall >>',10,13,0
CannotFlush  db '<< FLUSH: No copy of FLUSH is currently resident  >>',10,13,0
CannotRem    db '<< FLUSH: Unable to uninstall the resident copy.. >>',10,13
             db '<<        Unable to unchain Interrupt Vector      >>',10,13,0
UnLoadFlag   db     00                                  ; Flag to Unload TSR !
FlushFlag    db     00                                  ; Flag to Flush...
InfoFlag     db     00                                  ; Flag to display Info

Info    label    byte
db '                   << FLUSH: Description of Utility >>',10,13
db '  FLUSH is a TSR that saves a copy of the Interrupt Vector Table prior to   ',10,13
db '  going resident.   Once resident the TSR can be activated either via a     ',10,13
db '  Hot Key or by running a second copy of itself.   The TSR will restore     ',10,13
db '  the saved copy of the Interrupt Vector Table, walk down the MCB chain     ',10,13
db '  and free every allocated memory block (including programs) with a segment ',10,13
db "  value greater than that of the TSR's Segment.   The utility can be useful ",10,13
db '  when writing/testing TSRs. It can complement the ones that do not offer   ',10,13
db '  an Unload Option and also helps in *surviving* crashes.                   ',10,13,0



;[]------------------[ TERMINATE PID IN BX REGISTER ]------------------------[]
;|  This function is passed a *VALID* PSP in segment BX and terminates the    |
;|  process...  Termination is done via DOS, therefore ensuring that all      |
;|  all necessary cleanups are done regarding MCBs and SFTs...                |
;|  ENTRY:      BX: PSP of Program                                            |
;|  RETURN:     CF: Error Terminating Process                                 |
;|             !CF: Process Terminated                                        |
;[]--------------------------------------------------------------------------[]
Terminate       PROC
        @PUSH   ax, bx, es                              ; Save Registers
        mov     TsrSaveDS, ds                           ; Save DS
        mov     TsrSaveSS, ss                           ; Save SS
        mov     TsrSaveSP, sp                           ; Save SP
        mov     es, bx                                  ; ES: =  PSP of TSR
        mov     es:[16H], cs                            ; Load Parent Segment
        mov     word ptr es:[0AH], OFFSET @@L00         ; Set Terminate Add Off
        mov     word ptr es:[0CH], cs                   ; Set Terminate Add Seg
        mov     ah, 50H                                 ; Function 50H: SET PID
        int     21H                                     ; Call DOS
        mov     ax, 4C00H                               ; Terminate the TSR
        int     21H                                     ; Call DOS
        stc                                             ; Set Carry Flag
        jmp     @@L01                                   ; Restore Seg. Regs
@@L00:  clc                                             ; Clear Carry Flag
@@L01:  mov     ds, cs:TsrSaveDS                        ; Restore DS register
        mov     ss, cs:TsrSaveSS                        ; Restore SS register
        mov     sp, cs:TsrSaveSP                        ; Restore SP register
        @POP    ax, bx, es                              ; Restore Registers
        ret                                             ; Return
Terminate       ENDP




;[]--------------------[  UNLOAD A RESIDENT COPY OF TSR  ]-------------------[]
;|  This section of the TSR attempts to unload a resident copy of the TSR by  |
;|  using Multiplex Calls                                                     |
;|  ENTRY:      NONE                                                          |
;|  RETURN:     CF: Unable to Unload (BX=00 -> Vectors Hooked )               |
;|             !CF: Copy Unloaded...                                          |
;[]--------------------------------------------------------------------------[]
UnLoad  PROC
        @PUSH   ax, bx                          ; Save Registers
        mov     ah, TsrMltplxID                 ; Load our ID
        mov     al, _2Fh_UNHOOK                 ; Unhook Vectors Request
        int     2FH                             ; Call Resident Copy
        or      bx, bx                          ; Successful ?
        jnz     @@L00                           ; Yes.. Free Memory/Terminate
        stc                                     ; Set Carry Flag
        jmp     @@L01                           ; Jmp to return
@@L00:  call    Terminate                       ; Terminate Resident Copy
@@L01:  @POP    ax, bx                          ; Restore registers
        ret                                     ; Return
UnLoad  ENDP                                    ; End of Unloading Routine




;[]-----------------------[ RETRIEVE OPTIONS ]-------------------------------[]
;|   MACRO to retrieve next Option from the String pointed to by DS:SI...     |
;|   ENTRY:   SI point to string containing Options                           |
;|   RETURN:  CF -> No Options found                                          |
;|           !CF -> AL = Options and SI -> String following String            |
;[]--------------------------------------------------------------------------[]
GetOpt  PROC
        cld                                     ; Clear Direction Flag
@@L00:  lodsb                                   ; Load Next Char
        cmp     al, 20H                         ; Is it a Space Character ?
        je      @@L00                           ; Skip 'em
        cmp     al, '/'                         ; Switch character ?
        je      @@L02                           ; Get the Option char then !
        cmp     al, '-'                         ; Switch character ?
        je      @@L02                           ; Get the Option char then !
        cmp     al, 0DH                         ; Carriage Return  ?
        jne     @@L00                           ; If CR, end routine !
@@L01:  stc                                     ; Set Carry Flag
        ret                                     ; Return to caller
@@L02:  lodsb                                   ; Load Option
        cmp     al, 0DH                         ; Carriage Return ?
        je      @@L01                           ; If CR, end routine !
        clc                                     ; Clear Carry Flag
        ret                                     ; Return to caller
GetOpt  ENDP




;[]--------------------[ TSR's INITIALIZATION ROUTINE ]----------------------[]
;|   The following initializes the TSR's variables - verifies the various     |
;|   command line options and (if all's OK) terminates and stays resident !   |
;[]--------------------------------------------------------------------------[]
InitTSR:
        mov     ax, es                          ; Load PSP in ax
        mov     TsrPSP, ax                      ; Save PSP in Variable
        push    ax                              ; Save PSP on Stack  <------+
L00000: mov     es, EnvSeg                      ; Load Env. Seg from AX     |
        mov     ah, 49H                         ; Function 49H: Free Memory |
        int     21H                             ; Call DOS                  |
        pop     es                              ; Restore ES -> PSP  <------+
        mov     al, CmdLen                      ; Load Command Line Length
        or      al, al                          ; Zero or not ?
        jz      L00005                          ; Zero i.e. No arguments
        mov     si, OFFSET CmdArg               ; DS:SI -> Argument String
L00001: call    GetOpt                          ; Get Option from CmdLine
        jc      L00005                          ; No more option ?, move on !
        @ToLwr                                  ; Convert to Lower Case
        cmp     al, 'u'                         ; Unload Option ?
        jne     L00002                          ; Yes...
        inc     UnLoadFlag                      ; Set up Unload Flag
L00002: cmp     al, 'f'                         ; Flush  Option ?
        jne     L00003                          ; Skip if not..
        inc     FlushFlag                       ; Increment Flush Flag
L00003: cmp     al, '?'                         ; Info Request Option ?
        jne     L00004                          ; Nope...
        inc     InfoFlag                        ; Increment Info Flag
L00004: cmp     al, 'v'                         ; Save Video Mode Request ?
        jne     L00001                          ; Nope...
        mov     ah, 0Fh                         ; Get Mode Functio (INT 10H)
        int     10h                             ; Call Video BIOS
        mov     VideoMode, al                   ; Save Current Video Mode
        jmp     L00001                          ; Get Next argument
L00005: mov     si, OFFSET _Title_              ; Point to Title Message
        mov     bl, 15                          ; Load Attribute
        call    PutStr                          ; Display Message.
        cmp     InfoFlag, 00                    ; Info Request
        je      L00006                          ; Nope...
        mov     si, OFFSET Info                 ; DS:SI -> Info String
        mov     bl, 03                          ; Load Attribute
        jmp     L00010                          ; Display Message and Exit
L00006: mov     ah, TsrMltplxID                 ; Load our ID
        mov     al, _2Fh_PRESENT                ; Presence request in AL
        int     2FH                             ; Querry Multiplex
        cmp     al, 0FFH                        ; Is present ?
        jne     L00012                          ; Nope...
        mov     si, OFFSET Already              ; Already in Memory Message
        mov     bl, 12                          ; Load Attribute
        call    PutStr                          ; Display Message
        cmp     FlushFlag, 00                   ; Flush Request   ?
        je      L00007                          ; Nope... skip flush
        mov     ah, TsrMltplxID                 ; Load our ID
        mov     al, _2Fh_FLUSH                  ; Load Flush Request
        int     2FH                             ; Wipe 'em, including us too !
L00007: cmp     UnLoadFlag, 00                  ; Unload Requested ?
        ja      L00008                          ; Yes.  Jmp to exit
        mov     si, OFFSET Options              ; DS:SI -> Options
        mov     bl, 03                          ; Load Attribute
        call    PutStr                          ; Display Message
        jmp     L00011                          ; Exit...
L00008: call    UnLoad                          ; Unload Resident Copy
        mov     si, OFFSET CannotRem            ; Assume trouble unloading
        jc      L00009                          ; If trouble, show msg !
        mov     si, OFFSET Removed              ; Load Removed Message
L00009: mov     bl, 11                          ; Load Attribute
L00010: call    PutStr                          ; Display Message
L00011: mov     ax, 4C00H                       ; Terminate function
        int     21H                             ; Call DOS..
L00012: cmp     UnLoadFlag, 00                  ; Unload Request ?
        je      L00013                          ; Nope...
        mov     si, OFFSET NoneFound            ; Load 'No TSR installed' msg
        jmp     L00009                          ; Display Msg and Exit
L00013: cmp     FlushFlag, 00                   ; Flush Request
        je      L00014                          ; Nope...
        mov     si, OFFSET CannotFlush          ; Cannot Flush Message
        jmp     L00009
L00014: mov     ah, 52H                         ; List of List - Function
        int     21H                             ; Call MS DOS
        sub     bx, 2                           ; Make ES:BX -> First MCB
        mov     ax, word ptr es:[bx]            ; Load FirstMCB segment in AX
        mov     FirstMCB, ax                    ; Save FirstMCB Segment

        @GetVect MainVectors                    ; Save Old Interrupt Vectors
        @SetVect MainVectors                    ; Install our ISRs
        @SaveIVT                                ; Save Interrupt Vector Table
        mov     si, OFFSET Installed            ; DS:SI   -> Installed Msg.
        mov     bl, 10                          ; Load Attribute
        call    PutStr                          ; Display Message.
        mov     si, OFFSET Options              ; DS:SI   -> Options Msg
        mov     bl, 03                          ; Load Attribute
        call    PutStr                          ; Display Options
        mov     dx, OFFSET Transient            ; Point to end of Resident
        int     27H                             ; Stay Resident
END     @Startup
