;-----------------------------------------------------------------------------;
;      MODULE NAME:   EMM25_A.ASM                                             ;
;                                                                             ;
;    FUNCTION NAME:   copy_matching_struct_entries                            ;
;                                                                             ;
;      DESCRIPTION:   This function copies entries from an array of           ;
;                     structures (mappable_region_struct[]) which have a      ;
;                     member (mappable_region_seg) which falls between an     ;
;                     inclusive lower & upper bound.  Source structure        ;
;                     entries which qualify are copied into a destination     ;
;                     array of structures.  A count of the number of entries  ;
;                     copied is returned.                                     ;
;                                                                             ;
;                     This function provides a means of extracting either     ;
;                     the conventional or expanded memory mappable regions    ;
;                     ONLY from the total mappable_region_struct structure.   ;
;                                                                             ;
;           PASSED:   DS:SI = ptr_source                                      ;
;                             Pointer to the source mappable_region_struct    ;
;                             which contains qualified & unqualified members. ;
;                                                                             ;
;                     ES:DI = ptr_dest                                        ;
;                             Pointer to the dest mappable_region_struct      ;
;                             into which only qualified members will be       ;
;                             copied.                                         ;
;                                                                             ;
;                     AX    = begin_seg                                       ;
;                             Beginning segment of qualifying members.        ;
;                                                                             ;
;                     BX    = end_seg                                         ;
;                             Ending segment of qualifying members.           ;
;                                                                             ;
;                     CX    = source_count                                    ;
;                             Total number of members in the                  ;
;                             source mappable_region_struct structure.        ;
;                                                                             ;
;         RETURNED:   CX    = dest_count                                      ;
;                             Number of entries copied into the destination   ;
;                             structure. An entry is copied from the source   ;
;                             into the destination structure if:              ;
;                                                                             ;
;                             (begin_seg <= mappable_region_seg <= end_seg)   ;
;                                                                             ;
;    REGS MODIFIED:   SI   DI   CX                                            ;
;                                                                             ;
;   USE CONVENTION:   LDS    SI, ptr_source_mappable_region_struct            ;
;                     LES    DI, ptr_dest_mappable_region_struct              ;
;                     MOV    AX, FIRST_EXP_SEG                                ;
;                     MOV    BX, LAST_EXP_SEG                                 ;
;                     MOV    CX, source_mappable_region_struct_count          ;
;                     CALL   copy_matching_struct_entries                     ;
;                     MOV    dest_mappable_region_struct_count, DX            ;
;-----------------------------------------------------------------------------;
.XLIST
PAGE	60,132

IFDEF SMALL
   .MODEL SMALL, C
ENDIF
IFDEF MEDIUM
   .MODEL MEDIUM, C
ENDIF
IFDEF LARGE
   .MODEL LARGE, C
ENDIF
IFDEF COMPACT
   .MODEL COMPACT, C
ENDIF
IFDEF HUGE
   .MODEL HUGE, C
ENDIF

INCLUDE emmlib.equ
INCLUDE emmlib.str
INCLUDE emmlib.mac
.LIST
.CODE

copy_matching_struct_entries	PROC

	;---------------------------------------------------------------------;
	;   do;                                                               ;
	;   .   move direction is FORWARD;                                    ;
	;---------------------------------------------------------------------;
	CLD

	;---------------------------------------------------------------------;
	;   .   dest_count = 0;                                               ;
	;   .   if (source_count <= MAX_MAPPABLE_REGIONS) do;                 ;
	;---------------------------------------------------------------------;
	XOR		CH, CH
	CMP		CL, MAX_MAPPABLE_REGIONS
	JA		exit_copy_matching_struct_entries

	;---------------------------------------------------------------------;
	;   .   .   do while (source_count <> 0);                             ;
	;   .   .   .   if     ((mappable_region_seg => begin_seg)            ;
	;   .   .   .   .   and (mappable_region_seg <= end_seg)) do;         ;
	;---------------------------------------------------------------------;
	JCXZ		exit_copy_matching_struct_entries
begin_copy_matching_struct_entries:
	CMP		[SI].mappable_region_seg, AX
	JB		end_copy_matching_struct_entries
	CMP		[SI].mappable_region_seg, BX
	JA		end_copy_matching_struct_entries

	;---------------------------------------------------------------------;
	;   .   .   .   .   copy source struc entry to dest struct;           ;
	;   .   .   .   .   ptr_dest_struct = ptr_dest_struct                 ;
	;   .   .   .   .                     + dest struct size;             ;
	;---------------------------------------------------------------------;
	MOVSW
	MOVSW
	SUB		SI, SIZE mappable_region_struct

	;---------------------------------------------------------------------;
	;   .   .   .   .   dest_count = dest_count + 1;                      ;
	;   .   .   .   endif  ((mappable_region_seg => begin_seg)            ;
	;   .   .   .       and (mappable_region_seg <= end_seg));            ;
	;---------------------------------------------------------------------;
	INC		CH

end_copy_matching_struct_entries:
	;---------------------------------------------------------------------;
	;   .   .   .   ptr_source = ptr_source + source struct size;         ;
	;   .   .   .   source_count = source_count - 1;                      ;
	;   .   .   endwhile (source_count <> 0);                             ;
	;   .   endif (source_count <= MAX_MAPPABLE_REGIONS);                 ;
	;---------------------------------------------------------------------;
	ADD		SI, SIZE mappable_region_struct
	DEC		CL
	JNZ		begin_copy_matching_struct_entries

exit_copy_matching_struct_entries:
	;---------------------------------------------------------------------;
	;   .   return (dest_count);                                          ;
	;   end;                                                              ;
	;---------------------------------------------------------------------;
	MOVE		CH:CL, 0:CH
	RET

copy_matching_struct_entries	ENDP

PAGE
;-----------------------------------------------------------------------------;
;      MODULE NAME:   EMM25_A.ASM                                             ;
;                                                                             ;
;    FUNCTION NAME:   copy_mappable_regions_struct                            ;
;                                                                             ;
;      DESCRIPTION:   This function copies entries from an array of           ;
;                     structures (mappable_region_struct[]) which have a      ;
;                     member (mappable_region_seg) which falls between an     ;
;                     inclusive lower & upper bound.  Source structure        ;
;                     entries which qualify are copied into a destination     ;
;                     array of structures.  A count of the number of entries  ;
;                     copied is returned.                                     ;
;                                                                             ;
;                     This function provides a means of extracting either     ;
;                     the conventional or expanded memory mappable regions    ;
;                     ONLY from the total mappable_region_struct structure.   ;
;                                                                             ;
;           PASSED:   AX    = begin_seg                                       ;
;                             Beginning segment of qualifying members.        ;
;                                                                             ;
;                     BX    = end_seg                                         ;
;                             Ending segment of qualifying members.           ;
;                                                                             ;
;                     DS:SI = ptr_struct                                      ;
;                             Pointer to the dest mappable_region_struct      ;
;                             into which only qualified members will be       ;
;                             copied.  If the pointer is NULL, the structure  ;
;                             is not copied & discarded.                      ;
;                                                                             ;
;         RETURNED:   DX    = struct_count                                    ;
;                             Number of entries copied into the destination   ;
;                             structure. An entry is copied from the source   ;
;                             into the destination structure if:              ;
;                                                                             ;
;                             (begin_seg <= mappable_region_seg <= end_seg)   ;
;                                                                             ;
;    REGS MODIFIED:   AX   BX   CX   DX   SI   DI   DS   ES                   ;
;                                                                             ;
;   USE CONVENTION:   LDS    SI, ptr_dest_mappable_region_struct              ;
;                     MOV    AX, FIRST_EXP_SEG                                ;
;                     MOV    BX, LAST_EXP_SEG                                 ;
;                     CALL   copy_mappable_region_struct                      ;
;                     MOV    dest_mappable_region_struct_count, DX            ;
;-----------------------------------------------------------------------------;

copy_mappable_regions_struct	PROC
				LOCAL	begin_seg:WORD,                       \
					end_seg:WORD

	;---------------------------------------------------------------------;
	;   do;                                                               ;
	;   .   move direction is FORWARD;                                    ;
	;   .   save passed variables on local stack;                         ;
	;   .   # mappable region struct entries = 0;                         ;
	;---------------------------------------------------------------------;
	CLD
	MOVE		begin_seg, AX
	MOVE		end_seg,   BX 
	MOVE		DX, 0

	;---------------------------------------------------------------------;
	;   .   get the size of the mappable region struct from EMM;          ;
	;---------------------------------------------------------------------;
	MOVE		AX, get_mappable_array_entries_fcn
	INT     	EMM_int

	;---------------------------------------------------------------------;
	;   .   if (EMM status  = passed) do;                                 ;
	;   .   .   if (# mappable region struct entries <> 0) do;            ;
	;   .   .   .   if (# mappable region struct entries > MAX valid #) do;
	;   .   .   .   .   EMM status = s/w error;                           ;
	;   .   .   .   else do;                                              ;
	;---------------------------------------------------------------------;
	TEST		AH, AH
	JNZ		end_get_regions
	JCXZ		end_get_regions
	MOVE		AH, software_err_stat
	CMP		CX, MAX_MAPPABLE_REGIONS
	JA		end_get_regions

	;---------------------------------------------------------------------;
	;   .   .   .   .   allocate mappable region struct on stack;         ;
	;---------------------------------------------------------------------;
	MULT		CX, <SIZE mappable_region_struct>
	SUB		SP, CX
	MOVE		BX, CX

	;---------------------------------------------------------------------;
	;   .   .   .   .   get the mappable region struct from EMM;          ;
	;---------------------------------------------------------------------;
	MOVE		AX, get_mappable_phys_addr_array_fcn
	MOVE		ES:DI, SS:SP
	INT     	EMM_int

	;---------------------------------------------------------------------;
	;   .   .   .   .   if (EMM status = passed) do;                      ;
	;---------------------------------------------------------------------;
	TEST		AH, AH
	JNZ		continue_get_regions

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   if (ptr_dest_struct = 0)                      ;
	;   .   .   .   .   .   .   ptr_dest_struct = ptr_source_struct;      ;
	;---------------------------------------------------------------------;
	MOVE		ES:DI, DS:SI
	MOVE		DS:SI, SS:SP
	MOVE		AX, ES
	OR		AX, DI
	JNZ		begin_copy_regions
	MOVE		ES:DI, SS:SP

begin_copy_regions:
	;---------------------------------------------------------------------;
	;   .   .   .   .   .   do while (ptr_source_struct < end of struct); ;
	;   .   .   .   .   .   .   if ((mappable region seg                  ;
	;   .   .   .   .   .   .        >= start of specified region)        ;
	;   .   .   .   .   .   .   and (mappable region seg                  ;
	;   .   .   .   .   .   .        <= end of specified region)) do;     ;
	;---------------------------------------------------------------------;
	MOVE		AX, [SI].mappable_region_seg
	CMP		AX, begin_seg
	JB		end_copy_regions
	CMP		AX, end_seg
	JA		end_copy_regions

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   .   .   copy source structure entry to        ;
	;   .   .   .   .   .   .   .   destination structure;                ;
	;   .   .   .   .   .   .   .   ptr_dest_struct = ptr_dest_struct     ;
	;   .   .   .   .   .   .   .                     + dest struct size; ;
	;---------------------------------------------------------------------;
	MOVSW
	MOVSW
	SUB		SI, SIZE mappable_region_struct

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   .   .   # mappable region struct entries =    ;
	;   .   .   .   .   .   .   .   # mappable region struct entries + 1; ;
	;---------------------------------------------------------------------;
	INC		DX

end_copy_regions:
	;---------------------------------------------------------------------;
	;   .   .   .   .   .   .   endif ((mappable region seg               ;
	;   .   .   .   .   .   .           >= start of specified region)     ;
	;   .   .   .   .   .   .      and (mappable region seg               ;
	;   .   .   .   .   .   .           <= end of specified region));     ;
	;   .   .   .   .   .   .   ptr_source_struct = ptr_source_struct     ;
	;   .   .   .   .   .   .                       + source struct size; ;
	;   .   .   .   .   .   end while (ptr_source_struct < end of struct);;
	;   .   .   .   .   .   EMM status = passed;                          ;
	;---------------------------------------------------------------------;
	ADD		SI, SIZE mappable_region_struct
	LOOP		begin_copy_regions
	MOVE		AX, fcn_passed_stat

continue_get_regions:
	;---------------------------------------------------------------------;
	;   .   .   .   .   endif (EMM status = passed);                      ;
	;   .   .   .   .   deallocate mappable region struct on stack;       ;
	;---------------------------------------------------------------------;
	ADD		SP, BX

end_get_regions:
	;---------------------------------------------------------------------;
	;   .   .   .   endif (# mappable region struct entries<=MAX valid #);;
	;   .   .   endif (# mappable region struct entries <> 0);            ;
	;   .   endif (EMM status  = passed);                                 ;
	;   .   return (AH = EMM status, DX = mappable region count);         ;
	;   end;                                                              ;
	;---------------------------------------------------------------------;
	RET		

copy_mappable_regions_struct	ENDP

PAGE
;-----------------------------------------------------------------------------;
;      MODULE NAME:   EMM25_A.ASM                                             ;
;                                                                             ;
;    FUNCTION NAME:   get_mappable_conv_regions                               ;
;                                                                             ;
;      DESCRIPTION:   This function returns an array containing the segment   ;
;                     address and physical page number for each mappable      ;
;                     physical page within the conventional memory range      ;
;                     in a system.  The contents of this array provide a      ;
;                     cross reference between physical page numbers and the   ;
;                     actual segment addresses for each mappable page of      ;
;                     conventional memory in the system.  The array is sorted ;
;                     in ascending segment order.  This does not mean that    ;
;                     the physical page numbers associated with the segment   ;
;                     addresses are also in ascending order.                  ;
;                                                                             ;
;           PASSED:   &mcr_count:                                             ;
;                        is a far pointer to a count of the number of entries ;
;                        in the mcr array of structures that will be returned ;
;                        by the function.                                     ;
;                                                                             ;
;                     mcr:                                                    ;
;                        is a far pointer to an application-supplied          ;
;                        array of structures that will be initialized by the  ;
;                        function.                                            ;
;                                                                             ;
;         RETURNED:   status:                                                 ;
;                        is the status EMM returns from the call.  All other  ;
;                        returned results are valid only if the status        ;
;                        returned is zero.  Otherwise they are undefined.     ;
;                                                                             ;
;                     mcr_count:                                              ;
;                        is a count of the number of entries in the mcr       ;
;                        array of structures returned by the function.        ;
;                                                                             ;
;                     mcr:                                                    ;
;                        is an application supplied structure that contains   ;
;                        two elements initialized by the function.  The       ;
;                        structure members are described here:                ;
;                                                                             ;
;                        mcr.mappable_region_seg:                             ;
;                           is the segment address of the corresponding       ;
;                           physical page.  The array entries are sorted in   ;
;                           ascending segment address order.                  ;
;                                                                             ;
;                        mcr.phys_page:                                       ;
;                           is the physical page number of the corresponding  ;
;                           segment address.  The physical page numbers are   ;
;                           not in ascending order.                           ;
;                                                                             ;
; C USE CONVENTION:   unsigned int           status;                          ;
;                     unsigned int           mcr_count;                       ;
;                     MAPPABLE_REGION_STRUCT mcr [MAX_MAPPABLE_REGIONS];      ;
;                                                                             ;
;                     status = get_mappable_conv_regions (&mcr_count,         ;
;                                                         mcr);               ;
;-----------------------------------------------------------------------------;

get_mappable_conv_regions	PROC                                          \
				USES DS SI DI, 	                              \
				ptr_mappable_conv_reg_count:FAR PTR WORD,         \
				ptr_mappable_conv_reg_struct:FAR PTR BYTE

	;---------------------------------------------------------------------;
	;   do;                                                               ;
	;   .   get the size of the mappable region struct from EMM;          ;
	;---------------------------------------------------------------------;
	MOVE		AX, get_mappable_array_entries_fcn
	INT     	EMM_int

	;---------------------------------------------------------------------;
	;   .   if (EMM status = passed) do;                                  ;
	;---------------------------------------------------------------------;
	TEST		AH, AH
	JNZ		end_get_conv_regions

	;---------------------------------------------------------------------;
	;   .   .   if (# mappable region struct entries > MAX valid #) do;   ;
	;   .   .   .   EMM status = s/w error;                               ;
	;   .   .   else do;                                                  ;
	;---------------------------------------------------------------------;
	CMP		CX, MAX_MAPPABLE_REGIONS
	JBE		begin_get_conv_regions
	MOVE		AH, software_err_stat
	JMP		end_get_conv_regions

begin_get_conv_regions:
	;---------------------------------------------------------------------;
	;   .   .   .   if (# mappable region struct entries <> 0) do;        ;
	;---------------------------------------------------------------------;
	JCXZ		end_get_conv_regions

	;---------------------------------------------------------------------;
	;   .   .   .   .   allocate mappable region struct on stack;         ;
	;---------------------------------------------------------------------;
	MOVE		DX, CX
	MULT		DX, <SIZE mappable_region_struct>
	SUB		SP, DX

	;---------------------------------------------------------------------;
	;   .   .   .   .   get the mappable region struct from EMM;          ;
	;---------------------------------------------------------------------;
	MOVE		AX, get_mappable_phys_addr_array_fcn
	MOVE		ES:DI, SS:SP
	INT     	EMM_int

	;---------------------------------------------------------------------;
	;   .   .   .   .   if (EMM status = passed) do;                      ;
	;---------------------------------------------------------------------;
	TEST		AH, AH
	JNZ		continue_get_conv_regions

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   pass mappable conv region struct back         ;
	;                       to caller;                                    ;
	;---------------------------------------------------------------------;
	MOVE		DS:SI, SS:SP
	MOVE		ES:DI, ptr_mappable_conv_reg_struct
	PUSH		AX
	MOVE		AX:BX, FIRST_CONV_SEG:LAST_CONV_SEG
	CALL		copy_matching_struct_entries
	POP		AX

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   pass the # mappable conv regions back         ;
	;                       to caller;                                    ;
	;---------------------------------------------------------------------;
	MOVE		ES:BX, ptr_mappable_conv_reg_count
	MOVE		ES:[BX], CX

continue_get_conv_regions:
	;---------------------------------------------------------------------;
	;   .   .   .   .   end if (EMM status = passed);                     ;
	;   .   .   .   .   deallocate mappable region struct on stack;       ;
	;---------------------------------------------------------------------;
	ADD		SP, DX

end_get_conv_regions:
	;---------------------------------------------------------------------;
	;   .   .   .   end if (# mappable region struct entries <> 0);       ;
	;   .   .   end if (# mappable region array entries <= MAX valid #);  ;
	;   .   end if (EMM status  = passed);                                ;
	;   .   return (EMM status);                                          ;
	;   end;                                                              ;
	;---------------------------------------------------------------------;
	RET_EMM_STAT	AH

get_mappable_conv_regions	ENDP

PAGE
;-----------------------------------------------------------------------------;
;      MODULE NAME:   EMM25_A.ASM                                             ;
;                                                                             ;
;    FUNCTION NAME:   get_mappable_exp_regions                                ;
;                                                                             ;
;      DESCRIPTION:   This function returns an array containing the segment   ;
;                     address and physical page number for each mappable      ;
;                     physical page within the expanded memory range          ;
;                     in a system.  The contents of this array provide a      ;
;                     cross reference between physical page numbers and the   ;
;                     actual segment addresses for each mappable page of      ;
;                     expanded memory in the system.  The array is sorted     ;
;                     in ascending segment order.  This does not mean that    ;
;                     the physical page numbers associated with the segment   ;
;                     addresses are also in ascending order.                  ;
;                                                                             ;
;           PASSED:   &mer_count:                                             ;
;                        is a far pointer to a count of the number of entries ;
;                        in the mer array of structures that will be returned ;
;                        by the function.                                     ;
;                                                                             ;
;                     mer:                                                    ;
;                        is a far pointer to an application-supplied          ;
;                        array of structures that will be initialized by the  ;
;                        function.                                            ;
;                                                                             ;
;         RETURNED:   status:                                                 ;
;                        is the status EMM returns from the call.  All other  ;
;                        returned results are valid only if the status        ;
;                        returned is zero.  Otherwise they are undefined.     ;
;                                                                             ;
;                     mer_count:                                              ;
;                        is a count of the number of entries in the mer       ;
;                        array of structures returned by the function.        ;
;                                                                             ;
;                     mer:                                                    ;
;                        is an application supplied structure that contains   ;
;                        two elements initialized by the function.  The       ;
;                        structure members are described here:                ;
;                                                                             ;
;                        mer.mappable_region_seg:                             ;
;                           is the segment address of the corresponding       ;
;                           physical page.  The array entries are sorted in   ;
;                           ascending segment address order.                  ;
;                                                                             ;
;                        mer.phys_page:                                       ;
;                           is the physical page number of the corresponding  ;
;                           segment address.  The physical page numbers are   ;
;                           not in ascending order.                           ;
;                                                                             ;
; C USE CONVENTION:   unsigned int           status;                          ;
;                     unsigned int           mer_count;                       ;
;                     MAPPABLE_REGION_STRUCT mer [MAX_MAPPABLE_REGIONS];      ;
;                                                                             ;
;                     status = get_mappable_exp_regions (&mer_count,          ;
;                                                        mer);                ;
;-----------------------------------------------------------------------------;

get_mappable_exp_regions	PROC                                          \
				USES DS SI DI,                                \
				ptr_mappable_exp_region_count:FAR PTR WORD,       \
				ptr_mappable_exp_region_struct:FAR PTR BYTE

	;---------------------------------------------------------------------;
	;   do;                                                               ;
	;   .   get the size of the mappable region struct from EMM;          ;
	;---------------------------------------------------------------------;
	MOVE		AX, get_mappable_array_entries_fcn
	INT     	EMM_int

	;---------------------------------------------------------------------;
	;   .   if (EMM status = passed) do;                                  ;
	;---------------------------------------------------------------------;
	TEST		AH, AH
	JNZ		end_get_exp_regions

	;---------------------------------------------------------------------;
	;   .   .   if (# mappable region struct entries > MAX valid #) do;   ;
	;   .   .   .   EMM status = s/w error;                               ;
	;   .   .   else do;                                                  ;
	;---------------------------------------------------------------------;
	CMP		CX, MAX_MAPPABLE_REGIONS
	JBE		begin_get_exp_regions
	MOVE		AH, software_err_stat
	JMP		end_get_exp_regions

begin_get_exp_regions:
	;---------------------------------------------------------------------;
	;   .   .   .   if (# mappable region struct entries <> 0) do;        ;
	;---------------------------------------------------------------------;
	JCXZ		end_get_exp_regions

	;---------------------------------------------------------------------;
	;   .   .   .   .   allocate mappable region struct on stack;         ;
	;---------------------------------------------------------------------;
	MOVE		DX, CX
	MULT		DX, <SIZE mappable_region_struct>
	SUB		SP, DX

	;---------------------------------------------------------------------;
	;   .   .   .   .   get the mappable region struct from EMM;          ;
	;---------------------------------------------------------------------;
	MOVE		AX, get_mappable_phys_addr_array_fcn
	MOVE		ES:DI, SS:SP
	INT     	EMM_int

	;---------------------------------------------------------------------;
	;   .   .   .   .   if (EMM status = passed) do;                      ;
	;---------------------------------------------------------------------;
	TEST		AH, AH
	JNZ		continue_get_exp_regions

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   pass mappable exp region struct back          ;
	;                       to caller;                                    ;
	;---------------------------------------------------------------------;
	MOVE		DS:SI, SS:SP
	MOVE		ES:DI, ptr_mappable_exp_region_struct
	PUSH		AX
	MOVE		AX:BX, FIRST_EXP_SEG:LAST_EXP_SEG
	CALL		copy_matching_struct_entries
	POP		AX

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   pass the # mappable exp regions back          ;
	;                       to caller;                                    ;
	;---------------------------------------------------------------------;
	MOVE		ES:BX, ptr_mappable_exp_region_count
	MOVE		ES:[BX], CX

continue_get_exp_regions:
	;---------------------------------------------------------------------;
	;   .   .   .   .   end if (EMM status = passed);                     ;
	;   .   .   .   .   deallocate mappable region struct on stack;       ;
	;---------------------------------------------------------------------;
	ADD		SP, DX

end_get_exp_regions:
	;---------------------------------------------------------------------;
	;   .   .   .   end if (# mappable region struct entries <> 0);       ;
	;   .   .   end if (# mappable region array entries <= MAX valid #);  ;
	;   .   end if (EMM status  = passed);                                ;
	;   .   return (EMM status);                                          ;
	;   end;                                                              ;
	;---------------------------------------------------------------------;
	RET_EMM_STAT	AH

get_mappable_exp_regions	ENDP

PAGE
;-----------------------------------------------------------------------------;
;      MODULE NAME:   EMM25_A.ASM                                             ;
;                                                                             ;
;    FUNCTION NAME:   get_mappable_conv_region_count                          ;
;                                                                             ;
;      DESCRIPTION:   This function gets the number of entries which will be  ;
;                     required for the array of structures the                ;
;                     get_mappable_conv_regions function returns.  Note that  ;
;                     this function does not actually return the array but    ;
;                     merely a count of the entries in the array returned     ;
;                     by the get_mappable_conv_regions function.  This        ;
;                     number also represents the total number of mappable     ;
;                     conventional memory pages in the system.                ;
;                                                                             ;
;           PASSED:   &mcr_count:                                             ;
;                        is a far pointer to a count of the number of entries ;
;                        in the array of structures that are returned by the  ;
;                        get_mappable_conv_regions function.                  ;
;                                                                             ;
;         RETURNED:   status:                                                 ;
;                        is the status EMM returns from the call.  All other  ;
;                        returned results are valid only if the status        ;
;                        returned is zero.  Otherwise they are undefined.     ;
;                                                                             ;
;                     mcr_count:                                              ;
;                        is a count of the number of entries in the mcr       ;
;                        array of structures returned by the                  ;
;                        get_mappable_conv_regions function.                  ;
;                                                                             ;
; C USE CONVENTION:   unsigned int status;                                    ;
;                     unsigned int mcr_count;                                 ;
;                                                                             ;
;                     status = get_mappable_conv_region_count (&mcr_count);   ;
;-----------------------------------------------------------------------------;

get_mappable_conv_region_count	PROC                                          \
				USES DS SI DI,                                \
				ptr_mappable_conv_region_count:FAR PTR WORD

	;---------------------------------------------------------------------;
	;   do;                                                               ;
	;   .   get the size of the mappable region struct from EMM;          ;
	;---------------------------------------------------------------------;
	MOVE		AX, get_mappable_array_entries_fcn
	INT     	EMM_int

	;---------------------------------------------------------------------;
	;   .   if (EMM status = passed) do;                                  ;
	;---------------------------------------------------------------------;
	TEST		AH, AH
	JNZ		end_get_conv_regions_count

	;---------------------------------------------------------------------;
	;   .   .   if (# mappable region struct entries > MAX valid #) do;   ;
	;   .   .   .   EMM status = s/w error;                               ;
	;   .   .   else do;                                                  ;
	;---------------------------------------------------------------------;
	CMP		CX, MAX_MAPPABLE_REGIONS
	JBE		begin_get_conv_regions_count
	MOVE		AH, software_err_stat
	JMP		end_get_conv_regions_count

begin_get_conv_regions_count:
	;---------------------------------------------------------------------;
	;   .   .   .   if (# mappable region struct entries <> 0) do;        ;
	;---------------------------------------------------------------------;
	JCXZ		end_get_conv_regions_count

	;---------------------------------------------------------------------;
	;   .   .   .   .   allocate mappable region struct on stack;         ;
	;---------------------------------------------------------------------;
	MOVE		DX, CX
	MULT		DX, <SIZE mappable_region_struct>
	SUB		SP, DX

	;---------------------------------------------------------------------;
	;   .   .   .   .   get the mappable region struct from EMM;          ;
	;---------------------------------------------------------------------;
	MOVE		AX, get_mappable_phys_addr_array_fcn
	MOVE		ES:DI, SS:SP
	INT     	EMM_int

	;---------------------------------------------------------------------;
	;   .   .   .   .   if (EMM status = passed) do;                      ;
	;---------------------------------------------------------------------;
	TEST		AH, AH
	JNZ		continue_get_conv_regions_count

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   count the # of mappable conv regions;         ;
	;---------------------------------------------------------------------;
	MOVE		DS:SI, SS:SP
	MOVE		ES:DI, SS:SP
	PUSH		AX
	MOVE		AX:BX, FIRST_CONV_SEG:LAST_CONV_SEG
	CALL		copy_matching_struct_entries
	POP		AX

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   pass the # mappable conv regions back         ;
	;                       to caller;                                    ;
	;---------------------------------------------------------------------;
	MOVE		ES:BX, ptr_mappable_conv_region_count
	MOVE		ES:[BX], CX

continue_get_conv_regions_count:
	;---------------------------------------------------------------------;
	;   .   .   .   .   end if (EMM status = passed);                     ;
	;   .   .   .   .   deallocate mappable region struct on stack;       ;
	;---------------------------------------------------------------------;
	ADD		SP, DX

end_get_conv_regions_count:
	;---------------------------------------------------------------------;
	;   .   .   .   end if (# mappable region struct entries <> 0);       ;
	;   .   .   end if (# mappable region array entries <= MAX valid #);  ;
	;   .   end if (EMM status  = passed);                                ;
	;   .   return (EMM status);                                          ;
	;   end;                                                              ;
	;---------------------------------------------------------------------;
	RET_EMM_STAT	AH

get_mappable_conv_region_count	ENDP

PAGE
;-----------------------------------------------------------------------------;
;      MODULE NAME:   EMM25_A.ASM                                             ;
;                                                                             ;
;    FUNCTION NAME:   get_mappable_exp_region_count                           ;
;                                                                             ;
;      DESCRIPTION:   This function gets the number of entries which will be  ;
;                     required for the array of structures the                ;
;                     get_mappable_exp_regions function returns.  Note that   ;
;                     this function does not actually return the array but    ;
;                     merely a count of the entries in the array returned     ;
;                     by the get_mappable_exp_regions function.  This number  ;
;                     also represents the total number of mappable expanded   ;
;                     memory pages in the system.                             ;
;                                                                             ;
;           PASSED:   &mer_count:                                             ;
;                        is a far pointer to a count of the number of entries ;
;                        in the array of structures that are returned by the  ;
;                        get_mappable_exp_regions function.                   ;
;                                                                             ;
;         RETURNED:   status:                                                 ;
;                        is the status EMM returns from the call.  All other  ;
;                        returned results are valid only if the status        ;
;                        returned is zero.  Otherwise they are undefined.     ;
;                                                                             ;
;                     mer_count:                                              ;
;                        is a count of the number of entries in the mer       ;
;                        array of structures returned by the                  ;
;                        get_mappable_exp_regions function.                   ;
;                                                                             ;
; C USE CONVENTION:   unsigned int status;                                    ;
;                     unsigned int mer_count;                                 ;
;                                                                             ;
;                     status = get_mappable_exp_region_count (&mer_count);    ;
;-----------------------------------------------------------------------------;

get_mappable_exp_region_count	PROC                                          \
				USES DS SI DI,                                \
				ptr_mappable_exp_region_count:FAR PTR WORD

	;---------------------------------------------------------------------;
	;   do;                                                               ;
	;   .   get the size of the mappable region struct from EMM;          ;
	;---------------------------------------------------------------------;
	MOVE		AX, get_mappable_array_entries_fcn
	INT     	EMM_int

	;---------------------------------------------------------------------;
	;   .   if (EMM status = passed) do;                                  ;
	;---------------------------------------------------------------------;
	TEST		AH, AH
	JNZ		end_get_exp_regions_count

	;---------------------------------------------------------------------;
	;   .   .   if (# mappable region struct entries > MAX valid #) do;   ;
	;   .   .   .   EMM status = s/w error;                               ;
	;   .   .   else do;                                                  ;
	;---------------------------------------------------------------------;
	CMP		CX, MAX_MAPPABLE_REGIONS
	JBE		begin_get_exp_regions_count
	MOVE		AH, software_err_stat
	JMP		end_get_exp_regions_count

begin_get_exp_regions_count:
	;---------------------------------------------------------------------;
	;   .   .   .   if (# mappable region struct entries <> 0) do;        ;
	;---------------------------------------------------------------------;
	JCXZ		end_get_exp_regions_count

	;---------------------------------------------------------------------;
	;   .   .   .   .   allocate mappable region struct on stack;         ;
	;---------------------------------------------------------------------;
	MOVE		DX, CX
	MULT		DX, <SIZE mappable_region_struct>
	SUB		SP, DX

	;---------------------------------------------------------------------;
	;   .   .   .   .   get the mappable region struct from EMM;          ;
	;---------------------------------------------------------------------;
	MOVE		AX, get_mappable_phys_addr_array_fcn
	MOVE		ES:DI, SS:SP
	INT     	EMM_int

	;---------------------------------------------------------------------;
	;   .   .   .   .   if (EMM status = passed) do;                      ;
	;---------------------------------------------------------------------;
	TEST		AH, AH
	JNZ		continue_get_exp_regions_count

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   count the # of mappable exp regions;          ;
	;---------------------------------------------------------------------;
	MOVE		DS:SI, SS:SP
	MOVE		ES:DI, SS:SP
	PUSH		AX
	MOVE		AX:BX, FIRST_EXP_SEG:LAST_EXP_SEG
	CALL		copy_matching_struct_entries
	POP		AX

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   pass the # mappable exp regions back          ;
	;                       to caller;                                    ;
	;---------------------------------------------------------------------;
	MOVE		ES:BX, ptr_mappable_exp_region_count
	MOVE		ES:[BX], CX

continue_get_exp_regions_count:
	;---------------------------------------------------------------------;
	;   .   .   .   .   end if (EMM status = passed);                     ;
	;   .   .   .   .   deallocate mappable region struct on stack;       ;
	;---------------------------------------------------------------------;
	ADD		SP, DX

end_get_exp_regions_count:
	;---------------------------------------------------------------------;
	;   .   .   .   end if (# mappable region struct entries <> 0);       ;
	;   .   .   end if (# mappable region array entries <= MAX valid #);  ;
	;   .   end if (EMM status  = passed);                                ;
	;   .   return (EMM status);                                          ;
	;   end;                                                              ;
	;---------------------------------------------------------------------;
	RET_EMM_STAT	AH

get_mappable_exp_region_count	ENDP

PAGE
;-----------------------------------------------------------------------------;
;      MODULE NAME:   EMM25_A.ASM                                             ;
;                                                                             ;
;    FUNCTION NAME:   get_page_frame_count                                    ;
;                                                                             ;
;      DESCRIPTION:   This function returns the number of contiguous pages    ;
;                     that make up the EMM page frame.                        ;
;                                                                             ;
;           PASSED:   &pf_count:                                              ;
;                        is a far pointer to a count of the contiguous number ;
;                        of entries which make up the page frame in the array ;
;                        of structures that are returned by the               ;
;                        get_mappable_exp_regions function.  This number does ;
;                        not represent the total number of mappable expanded  ;
;                        regions, just the contiguous pages that make up the  ;
;                        page frame.                                          ;
;                                                                             ;
;         RETURNED:   status:                                                 ;
;                        is the status EMM returns from the call.  All other  ;
;                        returned results are valid only if the status        ;
;                        returned is zero.  Otherwise they are undefined.     ;
;                                                                             ;
;                     pf_count:                                               ;
;                        is a count of the number of entries that represent   ;
;                        the page frame in the mer array of structures        ;
;                        returned by the get_mappable_exp_regions function.   ;
;                                                                             ;
; C USE CONVENTION:   unsigned int status;                                    ;
;                     unsigned int pf_count;                                  ;
;                                                                             ;
;                     status = get_page_frame_count (&pf_count);              ;
;-----------------------------------------------------------------------------;

get_page_frame_count         	PROC                                          \
				USES DS SI DI,                                \
				ptr_page_frame_count:FAR PTR WORD

	;---------------------------------------------------------------------;
	;   do;                                                               ;
	;   .   get the size of the mappable region struct from EMM;          ;
	;---------------------------------------------------------------------;
	MOVE		AX, get_mappable_array_entries_fcn
	INT     	EMM_int

	;---------------------------------------------------------------------;
	;   .   if (EMM status = passed) do;                                  ;
	;---------------------------------------------------------------------;
	TEST		AH, AH
	JNZ		end_get_page_frame_count

	;---------------------------------------------------------------------;
	;   .   .   if (# mappable region struct entries > MAX valid #) do;   ;
	;   .   .   .   EMM status = s/w error;                               ;
	;   .   .   else do;                                                  ;
	;---------------------------------------------------------------------;
	CMP		CX, MAX_MAPPABLE_REGIONS
	JBE		begin_get_page_frame_count
	MOVE		AH, software_err_stat
	JMP		end_get_page_frame_count

begin_get_page_frame_count:
	;---------------------------------------------------------------------;
	;   .   .   .   if (# mappable region struct entries <> 0) do;        ;
	;---------------------------------------------------------------------;
	JCXZ		end_get_page_frame_count

	;---------------------------------------------------------------------;
	;   .   .   .   .   allocate mappable region struct on stack;         ;
	;---------------------------------------------------------------------;
	MOVE		DX, CX
	MULT		DX, <SIZE mappable_region_struct>
	SUB		SP, DX

	;---------------------------------------------------------------------;
	;   .   .   .   .   get the mappable region struct from EMM;          ;
	;---------------------------------------------------------------------;
	MOVE		AX, get_mappable_phys_addr_array_fcn
	MOVE		ES:DI, SS:SP
	INT     	EMM_int

	;---------------------------------------------------------------------;
	;   .   .   .   .   if (EMM status = passed) do;                      ;
	;---------------------------------------------------------------------;
	TEST		AH, AH
	JNZ		continue_get_page_frame_count

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   count the # of pages in the page frame;       ;
	;---------------------------------------------------------------------;
	MOVE		DS:SI, SS:SP
	PUSH		AX
        PUSH            DX
        
	;---------------------------------------------------------------------;
	;   .   .   .   .   .   get page frame segment from EMM;              ;
	;---------------------------------------------------------------------;
	MOVE		AH, get_page_frame_fcn
	INT     	EMM_int

        ;---------------------------------------------------------------------;
        ;   .   .   .   .   .   if (EMM status = passed) do;                  ;
        ;---------------------------------------------------------------------;
        TEST            AH, AH
        JNZ             continue_get_page_frame_count        

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   Save the last valid index;                    ;
	;---------------------------------------------------------------------;
        MOVE            DI, CX
	MULT		DI, <SIZE mappable_region_struct>
        ADD             DI, SI
        XOR             CX, CX

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   See if current segment is the page frame seg; ;
	;---------------------------------------------------------------------;
check_for_page_frame_seg:
        MOVE            AX, [SI]
        ADD             SI, 4
        CMP             AX, BX
        JNE             check_for_page_frame_seg

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   We have found the page frame;                 ;
	;---------------------------------------------------------------------;
check_next_seg_in_page_frame:
        INC             CX

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   have we exhausted the array?;                 ;
	;---------------------------------------------------------------------;
        CMP             SI, DI
        JAE             return_page_frame_count

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   See if next segment is contiguous;            ;
	;---------------------------------------------------------------------;
        ADD             AX, 0400H
        MOVE            DX, AX
        MOVE            AX, [SI]
        ADD             SI, 4
        CMP             AX, DX
        JE              check_next_seg_in_page_frame

return_page_frame_count:
        POP             DX
	POP		AX

	;---------------------------------------------------------------------;
	;   .   .   .   .   .   pass the page frame count back to caller;     ;
	;---------------------------------------------------------------------;
	MOVE		ES:BX, ptr_page_frame_count
	MOVE		ES:[BX], CX

continue_get_page_frame_count:
	;---------------------------------------------------------------------;
	;   .   .   .   .   end if (EMM status = passed);                     ;
	;   .   .   .   .   deallocate mappable region struct on stack;       ;
	;---------------------------------------------------------------------;
	ADD		SP, DX

end_get_page_frame_count:
	;---------------------------------------------------------------------;
	;   .   .   .   end if (# mappable region struct entries <> 0);       ;
	;   .   .   end if (# mappable region array entries <= MAX valid #);  ;
	;   .   end if (EMM status  = passed);                                ;
	;   .   return (EMM status);                                          ;
	;   end;                                                              ;
	;---------------------------------------------------------------------;
	RET_EMM_STAT	AH

get_page_frame_count	ENDP


PAGE
;-----------------------------------------------------------------------------;
;                STACK SPACE REQUIRED BY THE INTERFACE LIBRARY                ;
;-----------------------------------------------------------------------------;

.STACK (2 * MAX_MAPPABLE_REGIONS * (SIZE mappable_region_struct))

END
