/******************************************************************************

COMRESET.C

A program to reset UART hardware on multiple user-specified COM ports.

NOTE -- THIS IS A PC SPECIFIC PROGRAM AND MUST BE MODIFIED
        (NOT JUST RELINKED) TO FUNCTION ON OTHER SYSTEMS!

Program copyright 1991, Greg Smith & FBN Productions

Version 1.0 of 3 December 1990
    new program

Version 1.1 of 23 January 1991
    Added method to override default port assignments through environment
        strings of the form "COM4=736,3" which indicate the port address
        and interrupt of the specified COM ports.

Version 1.11 of 12 April 1991
    Format of COMx environment strings changed from decimal to hex.

Version 1.12 of 5 June 1991
    Checks and prints a message if a null command line is entered;
    Returns different errorlevels depending on different sources of error.

Special version of 6 June 1991
    Modified for use in COMRESET source package.  No changes to actual 
    code, but compiler directives modified so they do not depend on the 
    arrangement of directories on the FBN Productions development 
    machine, and so program can be compiled without the MSC 6.0 
    Programmer's WorkBench.

******************************************************************************/

/********************************
  external functions & variables
 ********************************/

#include "fbnlib.h"
#include <process.h>
#include <stdio.h>
#include <stdlib.h>

/**********************************
  modem port and interrupt numbers
 **********************************/

/* port and interrupt numbers for standard COM ports */
unsigned modem_port[] = {0x03f8, 0x02f8, 0x3e8, 0x2e8};
unsigned modem_interrupt[] = {4, 3, 4, 3};

/*************************
  main executable section
 *************************/

/*---------------------------------------------------------------------------*
 * int reset_port(int portnum, unsigned base_port, unsigned intrpt);         *
 *---------------------------------------------------------------------------*/

int reset_port(int portnum, unsigned base_port, unsigned intrpt)
{

/*
Resets all registers and interrupts associated with port number <portnum>
located at port address <base_port> on IRQ <intrpt>.

Returns FALSE if the port was reset correctly or TRUE if the port could not
be opened (probably indicating that the port was specified on the command line
but did not exist, but it could mean there is a hardware problem).
*/

    char display[100];
    int inkey;
    unsigned port;

    if (! mpopen(base_port, intrpt, 400, 400))
        {
        printf("No serial port at COM%d.\n\a", portnum);
        return TRUE;
        }

    printf("Resetting port COM%d.\n", portnum);
    mdbaud(300);

    while (mdist())
        inkey = mdin();

    mdclr();
    mdoclr();
    mpclose(FALSE);

    /* disable all interrupts */
    outp(base_port + 1, 0x00);

    /* pulse OUT1, OUT2 high, then low to reset -- this is all that's really
       required to rest for Windows, but the rest helps with other port
       resetting chores as well */
    outp(base_port + 4, 0x0c);
    outp(base_port + 4, 0x00);

    /* load 300 baud in the divisor latch */
    outp(base_port + 3, 0x83);
    outp(base_port, 0x80);
    outp(base_port + 1, 0x01);
    outp(base_port + 3, 0x03);

    return FALSE;
}

/*---------------------------------------------------------------------------*
 * void main(int argc, char *argv[]);                                        *
 *---------------------------------------------------------------------------*/

void main(int argc, char *argv[])
{
    char *envstring, port_name[5], display[100];
    int which, loop, noport, reset[4];

    /* <noport> is used to track whether one or more specified ports did not
       exist */
    noport = FALSE;

    printf("\n");

    /* initialize array of ports to reset */
    for (loop = 0; loop < 4; loop++)
        {
        reset[loop] = FALSE;

        /* search environment for strings of the form "COM1=2f8,3" which can
           be used to override the default modem port and interrupt settings */
        sprintf(port_name, "COM%d", loop + 1);

        if ((envstring = getenv(port_name)) != NULL)
            {
            if (sscanf(envstring, "%x, %x", &modem_port[loop],
                &modem_interrupt[loop]) != 2)
                {
                sprintf(display, "Can't read %s environment variable.",
                    port_name);
                printf("%s\n\a", display);
                exit(1);
                }
            }
        }

    /* exit if nothing specified on command line */
    if (argc < 2)
        {
        printf("Enter numbers of COM ports to reset on command line.\n"
               "Example: COMRESET 1 2        (resets COM1 and COM2)\n\a");
        exit(2);
        }

    /* look at all command line arguments */
    for (loop = 1; loop < argc; loop++)
        {
        which = atoi(argv[loop]);

        if (which < 1 || which > 4)
            {
            printf("Valid command line arguments are: 1 2 3 4\n\a");
            exit(3);
            }

        reset[which - 1] = TRUE;
        }

    for (which = 0; which < 4; which++)
        {
        if (reset[which])
            {
            noport |= reset_port(which + 1, modem_port[which],
                modem_interrupt[which]);
            }
        }

    exit(noport ? 4 : 0);
}

/* end of program */
