;
;A customized boot routine, which will load an application program
;  from tracks 1 through HIGHEST_TRACK and start it running.
;
;The segment to which the boot routine will be relocated so that it will
;  be out of the way of the program to be loaded.
moved_segment	segment at 0020h
	org	0
moved_routine	label	far
moved_segment	ends
;
;The segment in which the application program will be loaded
program_segment	segment at 0040h
	org	100h
start_point	label	far	;Since it is a COM program, the application
				; program will start at offset 100 (hex)
program_segment	ends
;
highest_track	equ	1	;The highest-numbered track to be read in
;
cseg	segment
	assume	cs:cseg
;
;While this is a COM program, it is designed to be independent of its
;  location in memory, and so it need not start at offset 100 (hex).
;
start	proc	near
;
;Set the stack to 0020:0200, at the end of the unused interrupt vector area
;
	cli		;Turn off interrupts so that we can't get caught
			; with the stack segment set the new way and the
			; stack pointer set the old way
	mov	ax,0020h
	mov	ss,ax
	mov	sp,0200h
	sti
;
;Move the rest of the boot routine to the unused vector area at 0020:0000
;
	mov	ax,moved_segment
	mov	es,ax	;The extra segment will be the destination segment
	mov	ax,cs
	mov	ds,ax	;The code segment will be the source, since we're
			; relocating the program code
	call	self	;Since we can't be sure where the routine is loaded,
			; we'll execute a call to the next instruction to get
			; the instruction pointer pushed on the top of the
			; stack
self:
	pop	si	;Get the instruction pointer back off the stack
	add	si,((offset c1)-(offset self))	;Adjust SI, which will be the
			; source for the move, to point the the start of the
			; working part of the boot routine
	sub	di,di	;The destination for the move will be 0020:0000
	mov	cx,256	;Number of bytes to move
	rep movsb	;Move the rest of the routine to 0020:0000
	jmp	moved_routine	;Execute a far jump to the start of the
			; moved portion of the boot routine
;
;By the time the program reaches this point, the following code will have
;  been moved to 0020:0000.
;
c1:
	sub	ax,ax	;0 is the number of the diskette reset function
	int	13h	;Call BIOS to reset the disk 
	mov	ax,program_segment
	mov	es,ax
	mov	ds,ax	;Point the data and extra segments to point to
			; the segment in which the program will go
	mov	bx,offset start_point	;point to the location at which
			; the program is to be loaded.  COM programs must
			; start at CS:100 (hex)
	mov	ch,1	;Initial track #
lp1:
;
;Get the current track (4K bytes).  For 9 sectors/track, you will need
;  to set a new disk parameter block-see Appendix A of the Tech Ref manual.
;Try to read the track-retry 10 times if the read attempt
;  is not successful, and if still unsuccessful, enter an infinite loop.
;
	mov	bp,10	;# times to retry
disk_retry_loop:
	push	bp
	sub	dx,dx	;Head 0, drive 0
	mov	cl,1	;Start at sector 1
	mov	ah,2	;BIOS read function #
	mov	al,8	;We want to read 8 sectors
	int	13h	;Invoke BIOS to read this track
	pop	bp
;
;Check for error
;
	jnc	track_is_in
;
;If there is an error, reset disk and retry
;
	push	bp
	sub	ah,ah
	int	13h	;reset disk
	pop	bp
	dec	bp
	jnz	disk_retry_loop
;
;If we've already retried many times, enter an infinite loop
;
error_loop:
	jmp	error_loop
;
;We've got this track
;
track_is_in:
	add	bx,1000h	;Point to next load point (4K farther on)
	inc	ch		;Point to the next track
	cmp	ch,highest_track	;See if we've read all the tracks in
	jbe	lp1		;If not, read the next track
;
;Set the stack for the application program.  Stack goes at the far end of
;  the program segment.  Turn off interrupts so we can't get caught between
;  setting SS and SP.
;
	cli
	mov	ax,program_segment
	mov	ss,ax
	mov	sp,0FC00h	;Set this value to 0000h if you have more
				; than 64K of memory, to leave as much room
				; as possible free for the program.  A lower
				; value is required if you have less than
				; 64K so that the stack will fit in memory
	sti
;
;The program is all loaded in.  Jump to the start of it.
;
	jmp	start_point
start	endp
cseg	ends
	end	start	;Program is to begin at label START
