
/*  ADJTIME program to adjust DOS time.  Allows adjustment forward or
	    backward, specified in seconds, hours, or timezones.
	    (Executing the program without arguments will cause
	    display of a help screen.)

	    Examples:
		ADJTIME +30    advances DOS time by 30 seconds.
		ADJTIME -53    moves DOS time back by 53 seconds.
		ADJTIME +2h    advances DOS time by 2 hours.
		ADJTIME -3h    sets DOS time back by 3 hours.
		ADJTIME p e    indicates that current time is Pacific,
				 and that desired time is Eastern.  The
				 program will set the DOS time to the
				 Eastern time that is equivalent to the
				 current corresponding Pacific time.
		ADJTIME g p    indicates that the current DOS time is set
				 to Greenwich Mean Time, and that it should
				 be reset to give Pacific time.

     ADJTIME will take Daylight Savings Time into account and will handle
	     the case where the time adjustment causes a change in the
	     day, month, or year.  The normal DOS limitation on the date
	     is maintained (valid dates are only those from 1980 through
	     2099).

     Time zones are built in for the U.S. (see ZONE array, below), but
	     users may add one or more zones through command line
	     arguments.  Such arguments should precede the one-letter
	     current-zone and desired-zone arguments.  The first letter
	     of the zone must not duplicate the first letter of any of
	     the other zones.  The argument should begin with "tz=", and
	     the zone should follow the format of the built-in zones,
	     which is a 3-letter standard-time code, a number indicating
	     how many hours to add to GMT to get this zone's time (can be
	     negative), and if Daylight Savings Time can be in effect in
	     this zone, a 3 letter abbreviation for that.
		     Example:
			  ADJTIME tz=sst2sdt e s    (establishes a zone
			     called "s" which is 2 hours ahead of GMT,
			     and then converts the current Eastern time
			     to the correct time for that "s" zone.)

    This program was written and compiled using Borland's Turbo-C,
    Small model, and is hereby placed in the Public Domain.
*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>

char zone[24][10] ={"GMT0",        /*   Greenwich              */
		    "AST4ADT",     /*   Atlantic               */
		    "EST5EDT",     /*   Eastern                */
		    "CST6CDT",     /*   Central                */
		    "MST7MDT",     /*   Mountain               */
		    "PST8PDT",     /*   Pacific                */
		    "YST9YDT",     /*   Yukon                  */
		    "HST10HDT",    /*   Hawaiian/Alaskan       */
		    "BST11BDT"};   /*   Bering                 */

char env_var[13] = "TZ=";

main(int argc, char **argv)
{

 int x, argnum, zonecount, zonefrom, zoneto;
 long lx;
 time_t now;

 printf("\n");

      /*  get number of active zones in ZONE table */
 for (zonecount = 8; *zone[zonecount] != NULL; zonecount++);

 if ( argc < 2 )
   {
    print_usage();
    return(1);
   }

 for ( x = 1; x < argc; x++)      /*  set all command line       */
    strupr ( argv[x] );           /*    arguments to upper case  */

 /*******************************************************************
 *   Handle adding or subtracting the specified number of seconds   *
 *      or hours to the DOS time                                    *
 *******************************************************************/

 if ( *argv[1] == '+' || *argv[1] == '-' )
   {
    lx = atoi(argv[1]);
    if (argv[1][strlen(argv[1])-1] == 'H') /* If hours specified, multiply */
       lx *= 3600;                         /*      given number by 3600    */
    tzset();                               /* Set global time variables    */
    time (&now);                           /* Get current DOS time         */
    now += lx;                             /* Adjust as requested          */
    stime (&now);                          /* Reset DOS time               */
    return(0);                             /*    And end happily           */
   }

 if ( argc < 3 )
   {
    printf("****  NOT ENOUGH ARGUMENTS  ****\n");
    print_usage();
    return(2);
   }

 /***********************************************************************
 *   Add any timezone (TZ=SSSnDDD) parameters to ZONE table             *
 ***********************************************************************/

 for (argnum=1; argnum < argc && strnicmp(argv[argnum],"TZ=",3) == 0;
						     zonecount++,argnum++)
   {
    for (x = 0; x < zonecount; x++)       /*  First be sure that first  */
      {                                   /*     letter of new zone is  */
       if (*zone[x] == argv[argnum][3])   /*     unique, then....       */
	 {
	  printf("****  NON-UNIQUE FIRST LETTER ON TZ= PARAMETER.  ****\n");
	  print_usage();
	  return(6);
	 }
      }
     strcpy(zone[zonecount],&argv[argnum][3]);  /* Add new zone to table */
    }

 if ( argc - argnum != 2)
   {
    printf("****  ZONE ARGUMENT COUNT NOT 2.  ****\n");
    print_usage();
    return(3);
   }

 /************************************************************************
 *  Validate "from" and "to" zone parameters and set 'zonefrom' and      *
 *  'zoneto' variables to point to the proper entries in the ZONE table. *             *
 ************************************************************************/

 for ( zonefrom = 0; *argv[argnum] != *zone[zonefrom]
				 && zonefrom != zonecount; zonefrom++);
 if (zonefrom == zonecount)
   {
    printf("****  'CURRENT' ZONE IS INVALID.  ****\n");
    print_usage();
    return(4);
   }

 for ( zoneto = 0,argnum++; *argv[argnum] != *zone[zoneto]
				 && zoneto != zonecount; zoneto++);
 if (zoneto == zonecount)
   {
    printf("****  'TARGET' ZONE IS INVALID.  ****\n");
    print_usage();
    return(5);
   }

 /********************************************************************
 *         Change DOS time according to supplied zones               *
 ********************************************************************/

 strcat (env_var, zone[zonefrom]); /*  Set up environment string  */
 putenv (env_var);                 /*  Put it in the environment  */
 tzset();                          /*  Set global variables       */
 time(&now);                       /*  Get the current time       */
 strcpy (env_var, "TZ=");          /*  Reset env_var to reflect   */
 strcat (env_var, zone[zoneto]);   /*    the target time zone     */
 putenv (env_var);                 /*  Set TZ= to target zone     */
 tzset();                          /*  Set globals again          */
 stime(&now);                      /*  Set to the new time        */
 return(0);                        /*     and split               */
}

 /***********************************************************************
 *    Function to print instruction screen if no arguments given, or    *
 *        if error is discovered during parameter processing            *
 ***********************************************************************/

print_usage()
{
 printf("USAGE: ADJTIME  +nnnn        adds nnnn seconds to DOS time.\n");
 printf("       ADJTIME  -nnnn        subtracts seconds from DOS time.\n");
 printf("        (Adding the letter 'h' after the number will specify that\n");
 printf("         hours (rather than seconds) will be added/subtracted.)\n");
 printf("       ADJTIME  <Zone-1>  <Zone-2>\n");
 printf("        (Each zone indicator is a single letter (see below).\n");
 printf("         Zone-1 is the 'current' time zone, on which the current\n");
 printf("         DOS time is based.  Zone-2 is the 'target' time zone.\n");
 printf("         The current DOS time will be changed so the new time\n");
 printf("         reflects the correct time for the target zone.\n");
 printf("  The following time zones are automatically supported:\n");
 printf("         A - Atlantic           P - Pacific\n");
 printf("         E - Eastern            Y - Yukon\n");
 printf("         C - Central            H - Hawaiian/Alaskan\n");
 printf("         M - Mountain           B - Bering\n");
 printf("                                G - Greenwich Mean Time\n");
 printf("  Additional zone(s) may be added by preceding the Zone-1 and\n");
 printf("         Zone-2 arguments with parameters such as TZ=FST9FDT and\n");
 printf("         TZ=JST-2JDT (hours from GMT, preceded (followed) by 3-char\n");
 printf("         abbreviation for standard (and daylight time if allowed)).\n");
 printf("         First letters of zone parameters must be unique, and are\n");
 printf("         used to specify zones in the command line (zone-1,zone2).");
}