/**********************************************************************
 *
 *   NAME
 *      strs.cpp
 *
 *   DESCRIPTION
 *      Holds string routines that pretty much match their basic
 *      equivalents.
 *
 *   HISTORY
 *      08-26-94 JL Created.
 *
 *      Copyright (c) 1994 Graceland Research.
 *      All Rights Reserved.
 *********************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "strs.h"

#define TRUE	1
#define FALSE 	0

/*
 * right -- Returns the rightmost number of characters of a string
 *
 */

char *
right(char *szDest, char *szBuf, int nLen)
{
	char szTemp[MAX_CHARS];
	int nCurLen = 0;
	int nNumChars = 0;
	int jdx = 0;

	nCurLen = strlen(szBuf);

	if ( nLen > nCurLen ) {
		nNumChars = nCurLen;
	} else {
		nNumChars = nLen;
	}

	for (szBuf += nCurLen - nNumChars; *szBuf; szBuf++) {
		szTemp[jdx++] = *szBuf;
	}
	szTemp[jdx] = '\0';

	strcpy(szDest, szTemp);

	return szDest;
}


/*
 * mid -- Returns the middlemost number of characters
 *        nStart is 0 based
 *
 */

char *
mid(char *szDest, char *szBuf, int nStart, int nLen)
{
	char szTemp[MAX_CHARS];
	int nCurLen = 0;
	int jdx = 0;

	nCurLen = strlen(szBuf);

    if ( nStart >= nCurLen ) {
		*szDest = '\0';
		return szDest;
	}

	for (szBuf += nStart ; *szBuf && jdx < nLen; szBuf++) {
		szTemp[jdx++] = *szBuf;
	}
	szTemp[jdx] = '\0';

	strcpy(szDest, szTemp);

	return szDest;
}


/*
 * left -- Returns the leftmost number of characters
 *
 */

char *
left(char *szDest, char *szBuf, int nLen)
{
	char szTemp[MAX_CHARS];
	char *spEndPoint;
	int nCurLen = 0;
	int jdx = 0;

	nCurLen = strlen(szBuf);

	if ( nLen > nCurLen ) {
		spEndPoint = szBuf + nCurLen;
	} else {
		spEndPoint = szBuf + nLen;
	}

	for ( ; szBuf < spEndPoint; szBuf++) {
		szTemp[jdx++] = *szBuf;
	}
	szTemp[jdx] = '\0';

	strcpy(szDest, szTemp);

	return szDest;
}



/*
 * cleartabs -- Replaces tabs in the given line with spaces
 *
 */

char *
cleartabs(char *szBuf, const int nMaxLen)
{
	char *szOutput;
	int	idx = 0;
	int jdx = 0;

	/* allocate memory for string */
	szOutput = new char[nMaxLen];

	/* replace the tabs with spaces */
	for (idx = 0; !szBuf[idx] || idx < nMaxLen - 1 ; idx++) {

		if ( szBuf[idx] == '\t' ) {
			do {
				szOutput[jdx++] = ' ';
			} while (jdx % TAB_WIDTH != 0);

		} else {
			szOutput[jdx++] = szBuf[idx];
		}
	}
	szOutput[jdx] = '\0';

	strcpy(szBuf, szOutput);

	delete szOutput;

	return szBuf;
}


/*
 * trim -- Trims the spaces off of both sides of the given string
 *
 */

char *
trim(char *buf)
{
	char *spBuf;
	char szOutput[MAX_CHARS];
	int idx = 0;
	int nDirty = FALSE;
	int nLastChar = 0;

	for (spBuf = buf; *spBuf; spBuf++) {

		if ( nDirty || (!nDirty && *spBuf != ' ') ) {
			nDirty = TRUE;
			szOutput[idx++] = *spBuf;

			if ( *spBuf != ' ' ) {
				nLastChar = idx;
			}
		}
	}

	szOutput[nLastChar] = '\0';

	strcpy(buf, szOutput);

	return buf;

}


/*
 * rtrim -- Remove trailing white space from a string
 *
 */

char *
rtrim(char *buf)
{
	char    *cp;

	if (buf == NULL) {
		return (char *) NULL;
	}

	for (cp = str_last(buf); cp != buf; cp--) {

		if ( !isspace( *cp )) {
			cp++;                /* Move back to space or terminator */
			break;
		}
	}

	/* cp is where we want to terminate string */

	*cp = '\0';

	return buf;
}


/*
 * ltrim -- Remove leading white space from a string
 *
 */

char *
ltrim(char *cp)
{
	while (*cp) {
        if ( !isspace( *cp )) {
			break;
        }

		cp++;
	}

	return cp;
}


/*
 * str_end -- Return a pointer to the \0 at the end of the string
 *
 */

char *
str_end(char *buf)
{
	if (buf == (char *) NULL) {
		return (char *) NULL;
	}

	return buf + strlen(buf);
}


/*
 * str_last -- Return a pointer to the last character of the string
 *
 */

char *
str_last(char *buf)
{
	if (buf == (char *) NULL) {
		return (char *) NULL;

	} else if (*buf == '\0') {
		return buf;

	} else {
		return buf + strlen(buf) - 1;
	}
}


/*
 * str_center_leading -- return number of space required to center str in
 * width
 *
 */
int
str_center_leading(char *str, int width)
{
	int     str_len;

	str_len = strlen(str);

	if (str_len >= width) {
		return 0;
	} else {
		return (width - str_len) / 2;
	}
}


/*
 * str_right_leading -- return no. of spaces needed to right justify str
 * in width
 *
 */
int
str_right_leading(char *str, int width)
{
	int     str_len;

	str_len = strlen(str);

	if (str_len >= width) {
		return 0;
	} else {
		return width - str_len;
	}
}

/*
 * str_pad -- Pad string with ch until it's wide enough
 *
 */

char *
str_pad(char *str, char ch, int width)
{
	int     len;
	int     togo;

	len = strlen(str);
	togo = width - len;

	if (togo > 0) {
		memset(str + len, ch, togo);
		str[width] = '\0';
	}
	return str;
}


/*
 * extract_file -- Given a full path, extracts just the filename
 *
 */

char *
extract_file(char *szDest, char *szFile)
{
    char    *spLast;
	int		nLen = 0;

    nLen = strlen(szFile);

	// look for the last occurring slash in the name
	// it comes in like "c:\bin\debug.exe", and we want
	// to make it say "debug"
    for ( spLast = szFile + nLen; spLast > szFile; spLast-- ) {

		if ( *spLast == '\\' ) {
			spLast++;
			break;
		}
	}

    mid(szDest, szFile, spLast - szFile, szFile + strlen(szFile) - spLast - 4);

    strlwr(szDest);

    return szDest;

}

