Option Explicit

 ' Description:
 '  This module contains constants, types, and
 '  procedures that deal with files, paths, directories
 '  and other disk related information.
  
 ' Constants:

  ' maximum length of directory name
  Const MAX_DIRECTORY_LENGTH = 145

 ' Types:

  ' paths structure
  Type zzPATH
    nCount                       As Integer         ' number of directories in path
    sDir(1 To 30)                As String * 128    ' directories in path
  End Type

Function zzDirExists (ByVal sDirectory$) As Integer

 ' Description:
 '  See if directory exists

 ' Parameters:
 '  sDirectory            directory to test

 ' Variables:
  Dim sChkDirectory       As String
  Dim sChkDrive           As String
  Dim sCurDirectory       As String
  Dim sCurDrive           As String

  ' handle errors
  On Error Resume Next

  ' blanks not allowed
  If sDirectory = gsEMPTY Then
    zzDirExists = False
    Exit Function
  End If
  
  ' get current directory and drive
  sCurDirectory = UCase$(zzPathFormat(CurDir$))
  Err = 0
  If Mid$(sCurDirectory, 2, 1) = ":" Then
    sCurDrive = Left$(sCurDirectory, 1)
  Else
    sCurDrive = gsEMPTY
  End If
  sCurDirectory = Left$(sCurDirectory, Len(sCurDirectory) - 1)
  If Right$(sCurDirectory, 1) = ":" Then
    sCurDirectory = sCurDirectory & "\"
  End If
  
  ' get directory and drive to check
  sChkDirectory = UCase$(zzPathFormat(sDirectory))
  If Mid$(sChkDirectory, 2, 1) = ":" Then
    sChkDrive = Left$(sChkDirectory, 1)
  Else
    sChkDrive = sCurDrive
  End If
  sChkDirectory = Left$(sChkDirectory, Len(sChkDirectory) - 1)
  If Right$(sChkDirectory, 1) = ":" Then
    sChkDirectory = sChkDirectory & "\"
  End If
  
  ' if different drive
  If sChkDrive <> sCurDrive Then

    ' change to that drive
    On Error Resume Next
    ChDrive sChkDrive
    
    ' if error occurred
    If Err <> 0 Then
      zzDirExists = False
    
    ' if no error occurred
    Else

      ' try to change to path in question
      Err = 0
      ChDir sChkDirectory
      If Err = 0 Then
	zzDirExists = True
      Else
	zzDirExists = False
      End If

      ' change back to current drive
      ChDrive sCurDrive

      ' change back to current directory
      ChDir sCurDirectory

    End If

  ' if same drive
  Else

    ' try to change to path in question
    On Error Resume Next
    ChDir sChkDirectory

    ' if no error then path exists else it does not
    If Err = 0 Then
      zzDirExists = True
    Else
      zzDirExists = False
    End If

    ' change back to current directory
    ChDir sCurDirectory

  End If

End Function

Function zzDirWin () As String

 ' Description:
 '  Returns windows directory

 ' Variables:
  Dim nDirAndPathLength   As Integer   ' length of directory and path
  Dim sDirAndPathName     As String    ' directory and path

  ' make field big enough
  sDirAndPathName = String$(MAX_DIRECTORY_LENGTH, 0)

  ' get windows directory
  nDirAndPathLength = GetWindowsDirectory(sDirAndPathName, MAX_DIRECTORY_LENGTH)

  ' trim excess and add "\"
  zzDirWin = LCase$(zzPathFormat(Left$(sDirAndPathName, nDirAndPathLength)))
  
End Function

Function zzDirWinSys () As String

 ' Description:
 '  Returns windows system directory
 
 ' Variables:
  Dim nDirAndPathLength   As Integer   ' length of directory and path
  Dim sDirAndPathName     As String    ' directory and path

  ' make field big enough
  sDirAndPathName = String$(MAX_DIRECTORY_LENGTH, 0)

  ' get windows directory
  nDirAndPathLength = GetSystemDirectory(sDirAndPathName, MAX_DIRECTORY_LENGTH)

  ' trim excess and add "\"
  zzDirWinSys = LCase$(zzPathFormat(Left$(sDirAndPathName, nDirAndPathLength)))
  
End Function

Function zzDiskFreeSpace (ByVal sDrive$) As Long

 ' Description:
 '  Get the disk space free for specific drive

 ' Parameters:
 '  sDrive              drive to get free space for
  
 ' Variables:
  Dim sPriorDrive As String

  ' handle errors
  On Error Resume Next

  ' save current drive
  sPriorDrive = CurDir

  ' make drive we want info on current
  ChDrive sDrive

  ' if no error
  If Err = 0 Then

    ' return free space
    zzDiskFreeSpace = DiskSpaceFree()
  
    ' make prior drive current
    ChDrive sPriorDrive

  ' if error
  Else
    
    ' no disk space available
    zzDiskFreeSpace = 0

  End If

End Function

Function zzFileExists (ByVal sFile$) As Integer

 ' Description:
 '  Returns TRUE if file exists

 ' Parameters:
 '  sFile           file being checked

  On Error Resume Next
  zzFileExists = GetAttr(sFile) <> 16
  
End Function

Sub zzFileParse (ByVal sPath$, sDir$, sFile$)
 
 ' Description:
 ' Splits the path and file name from one field

 ' Parameters:
 '  sPath                 combined file and directory specification
 '  sDir                        directory extracted
 '  sFile                       file extracted
 
 ' Variables:
  Dim nSlashLocation As Integer ' location of last slash
  
  ' find last "\"
  nSlashLocation = zzFileParseLastSlash(sPath)

  ' if no slash use current dir
  If nSlashLocation = 0 Then
    sDir = gsEMPTY
    sFile = sPath
  Else
    sDir = Left$(sPath, nSlashLocation)
    sFile = Mid$(sPath, nSlashLocation + 1)
  End If

End Sub

Function zzFileParseLastSlash (ByVal sSearchMe$) As Integer
 
 ' Description:
 '  Returns last location of "\" character in path

 ' Parameters:
 '  sSearchMe            string to search
  
 ' Variables:
  Dim nPos1 As Integer   ' found in position 1
  Dim nPos2 As Integer   ' found in position 2

  ' get first location
  nPos1 = InStr(sSearchMe, "\")
  nPos2 = nPos1

  ' loop until no more locations found
  Do While nPos2 > 0
    nPos2 = InStr(nPos1 + 1, sSearchMe, "\")
    If nPos2 > 0 Then nPos1 = nPos2
  Loop

  ' return value to caller
  zzFileParseLastSlash = nPos1

End Function

Function zzFileRHSA (ByVal sFile$) As String

 ' Description:
 '  Returns "rhsa" as shown in file manager

 ' Parameters:
 '  sFile            file to get info on

 ' Variables:
  Dim nAttr As Integer
  Dim sRHSA As String

  ' get attributes
  On Error Resume Next
  nAttr = GetAttr(sFile)
  On Error GoTo 0

  ' test attributes
  If nAttr And ATTR_READONLY Then sRHSA = "r"
  If nAttr And ATTR_HIDDEN Then sRHSA = sRHSA & "h"
  If nAttr And ATTR_SYSTEM Then sRHSA = sRHSA & "s"
  If nAttr And ATTR_ARCHIVE Then sRHSA = sRHSA & "a"

  ' return to caller
  zzFileRHSA = sRHSA

End Function

Function zzFilesDelete (ByVal sFileSpec$) As Integer
 
 ' Description:
 '  Deletes one or more files.

 ' Parameters:
 '  sFileSpec        file specification (may include wild cards)

  ' delete files
  On Error Resume Next
  Kill sFileSpec
  zzFilesDelete = (Err = 0)

End Function

Sub zzPathClear (ByVal sPath$)

 ' Description:
 '  Deletes all the files from a path including subdirectories.
 '  It will not delete the first subdirectory that is referenced.
 '  For example if "C:\DELETEME" is passed all the files and
 '  subdirectories within "C:\DELETEME" will be removed but
 '  "C:\DELETEME" will remain.

 '  !!!BE VERY CAREFUL WHEN USING THIS ROUTINE!!!
 
 ' Parameters:
 '  sPath                        path to clear

 ' Variables:
  Dim nFileAttr    As Integer    ' file attribute
  Dim sTmp         As String     ' work string
  
  ' turn on error handling
  On Error Resume Next

  ' sanity check
  If sPath = gsEMPTY Then Exit Sub
  If LCase$(sPath) = "c:" Then Exit Sub

  ' get first file
  sTmp = Dir$(sPath & "\*.*", ATTR_NORMAL Or ATTR_READONLY Or ATTR_HIDDEN Or ATTR_SYSTEM Or ATTR_DIRECTORY)

  ' loop while file to process
  Do While sTmp <> gsEMPTY
      
    ' if not root or subdirectory
    If Left$(sTmp, 1) <> "." Then
    
      ' get attributes of file
      nFileAttr = GetAttr(sPath & "\" & sTmp)

      ' if directory retrieved
      If nFileAttr And ATTR_DIRECTORY Then

	' recursive call to routine to delete files
	' for the new subdirectory
	Call zzPathClear(sPath & "\" & sTmp)
	
	' set all directory attributes off
	SetAttr sPath & "\" & sTmp, ATTR_NORMAL

	' delete subdirectory
	RmDir sPath & "\" & sTmp
	
	' reset from start
	sTmp = Dir$(sPath & "\*.*", ATTR_NORMAL Or ATTR_READONLY Or ATTR_HIDDEN Or ATTR_SYSTEM Or ATTR_DIRECTORY)
	
	' if nothing then done
	If sTmp = gsEMPTY Then Exit Do
	
      ' if file retrieved
      Else
      
	' set all file attributes off
	SetAttr sPath & "\" & sTmp, ATTR_NORMAL

	' delete the file
	Kill sPath & "\" & sTmp
	
	' clear current entry
	sTmp = gsEMPTY

      End If

    ' clear current entry
    Else
      sTmp = gsEMPTY
    End If

    ' get next file
    If sTmp = gsEMPTY Then sTmp = Dir$

    ' give windows time
    DoEvents

    ' if disk not ready
    If Err = 71 Then Exit Do

  Loop

End Sub

Function zzPathCreate (ByVal sPath$) As Integer
 
 ' Description:
 '  Create new directory path for as many levels
 '  as specified. Path must include drive designation.
 '  Returns TRUE is successful.

 ' Parameters:
 '  sPath                 path to create
 
 ' Variables:
  Dim nBackPos            As Integer
  Dim nForePos            As Integer
  Dim sCurDirectory       As String
  Dim sCurDrive           As String
  Dim sPathToCreate       As String
  
  ' get current directory and drive
  sCurDirectory = UCase$(zzPathFormat(CurDir$))
  If Mid$(sCurDirectory, 2, 1) = ":" Then
    sCurDrive = Left$(sCurDirectory, 1)
  Else
    sCurDrive = gsEMPTY
  End If
  sCurDirectory = Left$(sCurDirectory, Len(sCurDirectory) - 1)

  ' setup error handling
  On Error Resume Next

  ' add slash to end of path if not there already
  sPath = zzPathFormat(sPath)

  ' change to specified drive
  ChDrive sPath
  
  ' if couldn't change then
  If Err <> 0 Then
  
    ' path cannot be created
    zzPathCreate = False
  
  ' if could change then
  Else

    ' change to root
    ChDir "\"

    ' if could not
    If Err <> 0 Then

      ' path cannot be created
      zzPathCreate = False
      
    ' if could
    Else

      ' start after drive specification
      nBackPos = 3
      
      ' find end location of highest directory
      nForePos = InStr(4, sPath, "\")

      ' loop until each directory created
      Do While nForePos <> 0
  
	' pull out next directory
	sPathToCreate = Mid$(sPath, nBackPos + 1, nForePos - nBackPos - 1)

	' try to make directory
	Err = 0
	MkDir sPathToCreate
	
	' if error besides directory already exists
	If Err <> 0 And Err <> 75 Then
	
	  ' path cannot be created
	  zzPathCreate = False
	  Exit Do
    
	' if any other error
	Else
	
	  ' try to change to directory
	  Err = 0
	  ChDir sPathToCreate
	  
	  ' if cannot change to directory
	  If Err <> 0 Then
	  
	    ' path cannot be created
	    zzPathCreate = False
	    Exit Do
	  
	  End If
	
	End If

	' reset starting locations
	' for next directory
	nBackPos = nForePos
	nForePos = InStr(nBackPos + 1, sPath, "\")

      Loop
  
      ' path created OK
      zzPathCreate = True
    
    End If

    ' change back to current drive
    ChDrive sCurDrive

    ' change back to current directory
    ChDir sCurDirectory
  
  End If

End Function

Function zzPathFormat (ByVal sPathToFormat$) As String
 
 ' Description:
 '  Adds slash to end of directory

 ' Parameters:
 '  sPathToFormat         path to format
  
 ' Variables:
  Dim sPath As String

  ' place value into work field
  sPath = sPathToFormat
  
  ' if nothing passed do nothing
  If sPath = gsEMPTY Then
    zzPathFormat = gsEMPTY
    Exit Function
  End If

  ' add ":" if only one character
  If Len(sPath) = 1 Then sPath = sPath & ":"
  
  ' add "\" to end
  If Right$(sPath, 1) <> "\" Then
    zzPathFormat = sPath & "\"
  Else
    zzPathFormat = sPath
  End If

End Function

Sub zzPathIntoArray (tPATH As zzPATH)
 
 ' Description:
 '  Put windows, windows system, and PATH
 '  directories into array in lowercase.

 ' Parameters:
 '  tPath                        ' path structure\

 ' Variables:
  Dim nCE           As Integer   ' current array element
  Dim nCE1          As Integer   ' current array element for lookup
  Dim nCE2          As Integer   ' current array element for lookup
  Dim nStartPos     As Integer   ' location of next ";"
  Dim sDir          As String    ' directory
  Dim sPaths        As String    ' DOS path

  tPATH.sDir(1) = LCase$(zzDirWinSys())
  tPATH.sDir(2) = LCase$(zzDirWin())

  ' if paths found
  sPaths = Environ$("PATH")
  If sPaths <> gsEMPTY Then

    ' make sure terminator at end
    If Right$(sPaths, 1) <> ";" Then
      sPaths = sPaths & ";"
    End If

    ' last used element
    nCE1 = 2
    
    ' do path
    For nCE = 3 To 30
      
      ' get end of next direcotry
      nStartPos = InStr(sPaths, ";")
      If nStartPos = 0 Then Exit For

      ' pull out directory and format properly
      sDir = LCase$(zzPathFormat(Left$(sPaths, nStartPos - 1)))

      ' see if already in list
      For nCE2 = 1 To nCE
	If sDir = RTrim$(tPATH.sDir(nCE2)) Then Exit For
      Next nCE2

      ' if not found then add
      If nCE2 >= nCE Then
	nCE1 = nCE1 + 1
	tPATH.sDir(nCE1) = sDir
      End If
      
      ' remove from string
      sPaths = Mid$(sPaths, nStartPos + 1)

    Next nCE

  End If

  ' return count
  tPATH.nCount = nCE1

End Sub

Sub zzPathInvalidMsg (ByVal sCaption$, ByVal sPath$)
 
 ' Description:
 '  Displays message handling invalid path selection.

 ' Parameters:
 '  sPath           invalid path and file name

 ' Variables:
  Dim nRC      As Integer
  Dim sDrive   As String
  Dim sMessage As String

  ' get drive letter
  sDrive = UCase(Left$(sPath, 1))

  ' if removable drive then
  If sDrive = "A" Or sDrive = "B" Then
    sMessage = "Drive " & sDrive & ": either contains the wrong diskette, contains no diskette at all, or is an invalid drive specification."
  ElseIf sDrive = gsEMPTY Then
    sMessage = "Blank path specification not allowed."
  Else
    sMessage = UCase(sPath) & " is a non-existant or invalid path specification."
  End If

  ' show message box
  MsgBox sMessage, 16, sCaption

End Sub

Function zzPathValid (sPath$) As Integer

 ' Description:
 '  Check for valid path name (not existence of the path).
 '  Returns TRUE or FALSE and the path name properly formatted.

 ' Parameters:
 '  sPath                             path name to check and format

 ' Variables:
  Dim nBackPos     As Integer         ' search location
  Dim nForePos     As Integer         ' search location
  Dim nI           As Integer         ' loop counter
  Dim nLength      As Integer         ' length of string
  Dim nPeriodPos   As Integer         ' period location
  
  Dim sLegalChar   As String          ' characters that can be used in path
  Dim sTmp         As String          ' work field
  Dim sWrkPath     As String          ' work field

  ' assume path name OK
  zzPathValid = True

  ' put into work string
  sWrkPath = Trim(sPath)

  ' turn on error handling
  On Error Resume Next

  ' blanks are not allowed
  If sWrkPath = gsEMPTY Then GoTo ValidPathError

  ' insert default drive if path begins with root backslash
  If Left$(sWrkPath, 1) = "\" Then sWrkPath = "c:" & sWrkPath

  ' wildcard characters or spaces not allowed
  If (InStr(sWrkPath, "*") <> 0) GoTo ValidPathError
  If (InStr(sWrkPath, "?") <> 0) GoTo ValidPathError
  If (InStr(sWrkPath, " ") <> 0) GoTo ValidPathError

  ' check for drive indicator
  If Mid$(sWrkPath, 2, 1) <> ":" Then GoTo ValidPathError

  ' insert root backslash if needed
  If Len(sWrkPath) > 2 Then
    If Right$(Left$(sWrkPath, 3), 1) <> "\" Then
      sWrkPath = Left$(sWrkPath, 2) & "\" & Right$(sWrkPath, Len(sWrkPath) - 2)
    End If
  End If

  ' check drive to install on
  ChDrive (Left$(sWrkPath, 1))
  If Err <> 0 Then GoTo ValidPathError

  ' add final "\"
  If Right$(sWrkPath, 1) <> "\" Then sWrkPath = sWrkPath & "\"

  ' Root dir is a valid dir
  If Len(sWrkPath) = 3 Then
    If Right$(sWrkPath, 2) = ":\" Then GoTo ValidPathOK
  End If

  ' check for repeated slashs
  If InStr(sWrkPath, "\\") <> False Then GoTo ValidPathError

  ' check for illegal directory names
  sLegalChar = "!#$%&'()-0123456789@ABCDEFGHIJKLMNOPQRSTUVWXYZ^_`{}~."
  nBackPos = 3
  nForePos = InStr(4, sWrkPath, "\")
  
  ' loop til no more to directories to check
  Do
  
    ' get work string
    sTmp = Mid$(sWrkPath, nBackPos + 1, nForePos - nBackPos - 1)
  
    ' test for illegal characters
    For nI = 1 To Len(sTmp)
      If InStr(sLegalChar, UCase(Mid$(sTmp, nI, 1))) = 0 Then GoTo ValidPathError
    Next nI

    ' check combinations of periods and lengths
    nPeriodPos = InStr(sTmp, ".")
    nLength = Len(sTmp)
  
    ' if no extension
    If nPeriodPos = 0 Then
    
      ' base too long
      If nLength > 8 Then GoTo ValidPathError
  
    ' if extension used
    Else
    
      ' base too long
      If nPeriodPos > 9 Then GoTo ValidPathError

      ' extension too long
      If nLength > nPeriodPos + 3 Then GoTo ValidPathError

      ' two periods not allowed
      If InStr(nPeriodPos + 1, sTmp, ".") <> 0 Then GoTo ValidPathError
  
    End If

    nBackPos = nForePos
    nForePos = InStr(nBackPos + 1, sWrkPath, "\")

  Loop Until nForePos = 0

' path name was ok valid
ValidPathOK:
  sPath = sWrkPath
  Exit Function

' path name was not valid
ValidPathError:
  zzPathValid = False

End Function

