Option Explicit

 ' Description:
 '  This module contains code to support the
 '  file transfer functions.
 
 ' Constants:
 
  ' File Transfer: return codes
  Global Const gnTF_OK = &H0                                 ' the previous operation was successful
  Global Const gnTF_REQ_ALREADY_ENDED = &H103                ' transfer request is already ended
  Global Const gnTF_UNXLATABLE_DATA = &H300                  ' untranslatable data found in transferred record
  Global Const gnTF_UNXLATABLE_PC_TO_400_DATA = &H302        ' untranslatable data in record to be transferred to AS/400
  Global Const gnTF_AS400_WARNING = &H400                    ' warning detected by the AS/400
  Global Const gnTF_EOF = &H1FFF                             ' end of file
  Global Const gnTF_MAX_REQ_ACTIVE = &H2000                  ' maximum number of transfer requests active
  Global Const gnTF_XFER_LENGTH_INVALID = &H2001             ' transfer request length not valid
  Global Const gnTF_XFER_REQ_CHAR_UNXLATABLE = &H2002        ' transfer request character cannot be translated
  Global Const gnTF_NO_XFER_UNXLAT_NUM_DATA = &H2003         ' record not transferred to the AS/400 because of untranslatable numeric data
  Global Const gnTF_RANGE_ERROR_NUMERIC_DATA = &H2004        ' numeric data does not fit into range specified by digits
  Global Const gnTF_BAD_APP_RECORD_LENGTH = &H2005           ' record length given by application is not correct
  Global Const gnTF_INCORRECT_NULL_MAP_OFFSET = &H2007       ' null map offset provided by application is not correct
  Global Const gnTF_FIELD_LENGTH_OVER_4096 = &H2008          ' total length of record exceed 4096 bytes
  Global Const gnTF_WAIT_REQUEST_STILL_ACTIVE = &H2009       ' a previous request to the DLL has not finished
  Global Const gnTF_XFER_REQ_NOT_OPENED = &H2010             ' transfer request not opened
  Global Const gnTF_CANNOT_RETRIEVE_TEMPLATES = &H2011       ' templates cannot be retrieved now
  Global Const gnTF_CANNOT_GET_REC_PC_TO_400 = &H2012        ' cannot get records on a PC-to-AS/400 request
  Global Const gnTF_CANNOT_SEND_REC_400_TO_PC = &H2013       ' cannot send records on a AS/400-to-PC request
  Global Const gnTF_INCORRECT_RTR_CONV_ADDR_PARM = &H2019    ' incorrect address specified for the router conversation ID
  Global Const gnTF_INCORRECT_FUNCTION_REQUEST = &H2020      ' incorrect function requested
  Global Const gnTF_INCORRECT_BUFFER_LENGTH = &H2021         ' incorrect buffer length
  Global Const gnTF_INCORRECT_BUFFER_ADDRESS = &H2022        ' incorrect buffer address
  Global Const gnTF_INCORRECT_BUFFER_ADDRESS_PARM = &H2023   ' transfer request buffer overlaps a previous transfer request buffer
  Global Const gnTF_INCORRECT_RETURN_ADDRESS_PARM = &H2024   ' incorrect address specified for the buffer address parameter
  Global Const gnTF_INCORRECT_RETURN_LENGTH_PARM = &H2025    ' incorrect address specified for the return length parameter
  Global Const gnTF_INCORRECT_RETURN_LENGTH_PARM2 = &H2026   ' incorrect address specified for the return length parameter
  Global Const gnTF_INCORRECT_SEC_RET_CODE_PARM = &H2027     ' incorrect address specified for the secondary return code parameter
  Global Const gnTF_INCORRECT_SYSTEM_ADDRESS_PARM = &H2028   ' incorrect address specified for the system parameter name
  Global Const gnTF_PCSWIN_COMM_BUFFER_TOO_SMALL = &H2029    ' PCSWIN communications buffer space exhausted
  Global Const gnTF_OPTIONS_NOT_SUPPORTED = &H2030           ' options statement not supported on this system
  Global Const gnTF_OPTIONS_NOT_ALL_SUPPORTED = &H2031       ' options not fully supported on this system
  Global Const gnTF_FUNCTION_AND_STYLE_INVALID = &H2032      ' conversion format not allowed with STYLE(NEW) in effect
  Global Const gnTF_STYLE_KEYWORD_NOT_SUPPORT = &H2033       ' STYLE keyword not supported on this system
  Global Const gnTF_AS400_ERROR_OFFSET_GIVEN = &H3000        ' error detected by AS/400 system; statement offset provided
  Global Const gnTF_AS400_ERROR_NO_OFFSET = &H3100           ' error detected by AS/400 system; no statement offset provided
  Global Const gnTF_ROUTER_NOT_STARTED = &H5042              ' router has not been started
  Global Const gnTF_CONNECT_TO_SYSTEM_FAILED = &H5048        ' connection to system failed
  Global Const gnTF_SYS_NAME_INCORR_OR_INACTIVE = &H5050     ' system name is incorrect or inactive
  Global Const gnTF_SYS_PROGRAM_NOT_FOUND = &H5052           ' system program not found
  Global Const gnTF_SYS_PROGRAM_ENDED_UNEXPECT = &H5054      ' system program ended unexpectedly
  Global Const gnTF_SYS_SECURITY_ERROR = &H5056              ' security error occurred for system
  Global Const gnTF_SYS_NOT_SUPPORTED = &H5058               ' system is not a supported system
  Global Const gnTF_WRONG_PROGRAM_VERSION = &H5060           ' wrong version level
  Global Const gnTF_SYS_CONTACT_ENDED = &H5062               ' contact with system ended
  Global Const gnTF_SYS_CONTACT_INTERRUPTED = &H5064         ' contact with system temporarily interrupted
  Global Const gnTF_SYS_RESOURCE_FAILURE = &H5066            ' resource failure on system
  Global Const gnTF_COV_ENDED_UNEXPECTEDLY = &H5067          ' conversation was ended unexpectedly
  Global Const gnTF_INSUFFICIENT_MEMORY = &H5071             ' not enough memory
  Global Const gnTF_PCSWIN_NOT_FOUND = &H5310                ' PCSWIN not loaded
  Global Const gnTF_INSUFFICIENT_MEMORY2 = &H5311            ' DLL does not support real mode windows
  Global Const gnTF_OPERATING_SYSTEM_ERROR = &H5400          ' operating system error
  Global Const gnTF_UNDEFINED_ERROR = &H9999                 ' undefined error

  ' File Transfer: conversion options
  Global Const gnTF_NO_CONVERSION = &H0                      ' no conversion
  Global Const gnTF_DOS_RANDOM = &H1                         ' DOS
  Global Const gnTF_DOS_RANDOM_EXCEPT = &H2                  ' DOS with exceptions
  Global Const gnTF_DOS_RANDOM2 = &H11                       ' type 2
  Global Const gnTF_DOS_RANDOM2_EXCEPT = &H12                ' type 2 with exceptions
  Global Const gnTF_NO_CONVERSION_NULL = &H80                ' no conversion, handle nulls
  Global Const gnTF_DOS_RANDOM_NULL = &H81                   ' DOS, handle nulls
  Global Const gnTF_DOS_RANDOM_EXCEPT_NULL = &H82            ' DOS with exceptions, handle nulls
  Global Const gnTF_DOS_RANDOM2_NULL = &H91                  ' type 2, handle nulls
  Global Const gnTF_DOS_RANDOM2_EXCEPT_NULL = &H92           ' type 2 with exceptions, handle nulls

  ' File Transfer: transfer options
  Global Const gnTF_SEND_REQUEST = 1                         ' send request
  Global Const gnTF_RETRIEVE_TEMPLATES = 2                   ' retrieve templates
  Global Const gnTF_RETRIEVE_RECORDS = 3                     ' retrieve records
  Global Const gnTF_CLOSE_REQUEST = 4                        ' close request
  Global Const gnTF_END_ALL_REQ_CONVERSATIONS = 5            ' end all conversations
  Global Const gnTF_SEND_RECORDS = 6                         ' send records
  Global Const gnTF_END_ONE_REQ_CONVERSATION = 7             ' end one conversation

  ' File Transfer: AS/400 EBCDIC field types
  Global Const gnTF_HEX_FIELD = &H0                          ' hex
  Global Const gnTF_BINARY_FIELD = &H1                       ' binary
  Global Const gnTF_CHAR_FIELD = &H2                         ' character
  Global Const gnTF_ZONED_FIELD = &H3                        ' zoned
  Global Const gnTF_PACKED_FIELD = &H4                       ' packed
  Global Const gnTF_RSVRD_FIELD = &H5                        ' reserved
  Global Const gnTF_IGCOPEN_FIELD = &H6                      ' IGC open
  Global Const gnTF_IGCONLY_FIELD = &H7                      ' IGC only
  Global Const gnTF_IGCEITHER_FIELD = &H8                    ' IGC either
  Global Const gnTF_TIME_FIELD = &HB                         ' time
  Global Const gnTF_DATE_FIELD = &HC                         ' date
  Global Const gnTF_TIMESTAMP_FIELD = &HD                    ' time stamp
  Global Const gnTF_GRAPHIC_FIELD = &HF                      ' graphic
								    
 ' Variables:
  Global gsTFBuffer       As String                          ' global transfer buffer

 ' Types:

  ' column information
  Type TFColType
    sName                 As String * 10                     ' field name
    nType                 As Integer                         ' type of field
    nDigits               As Integer                         ' number of digits
    nDecPos               As Integer                         ' decimal position
    nLen                  As Integer                         ' buffer length of field
    bNullCap              As Integer                         ' is field Null capable?
    bVarLen               As Integer                         ' is field variable length?
    sText                 As String * 50                     ' field description
  End Type

 ' APIs:
  Declare Function EHNTF_STF% Lib "EHNTFW.DLL" (ByVal hWnd%, ByVal lTFCallBack&, lTFConvID&, ByVal nTFFunction%, ByVal sTFBuffer$, ByVal nTFBufferLen%, ByVal nTFRequestLen%, ByVal sTFSystem$, ByVal nTFConversion%, lTFReturnAddr&, nTFReturnLen%, nTF2ndCode%)
  Declare Function EHNTF_STF_VB% Lib "EHNTFWVB.DLL" (ByVal hWnd%, ByVal lTFCallBack&, lTFConvID&, ByVal nTFFunction%, ByVal sTFBuffer$, ByVal nTFBufferLen%, ByVal nTFRequestLen%, ByVal sTFSystem$, ByVal nTFConversion%, lTFReturnAddr&, nTFReturnLen%, nTF2ndCode%)

Function zzTFClose (ByVal hWnd%, ByVal lTFCallBack&, ByVal lTFConvID&, ByVal sTFSystem$) As Integer
	
 ' Description:
 '  Close a currently open transfer request

 ' Parameters:
 '  hWnd                 windows handle
 '  lTFCallBack          procedural call back address
 '  lTFConvID            conversation ID
 '  sTFSystem            system name

 ' Variables:
  Dim nTFReturnLen       As Integer ' returned length of data
  Dim sTFDataReturned    As String  ' data returned
  
  ' call transfer request wrapper
  zzTFClose = zzTFPerformRequest(hWnd, lTFCallBack, lTFConvID, gnTF_CLOSE_REQUEST, gsEMPTY, sTFSystem, gnTF_NO_CONVERSION, nTFReturnLen, sTFDataReturned)

End Function

Function zzTFEndConversation (ByVal hWnd%, ByVal lTFCallBack&, ByVal lTFConvID&, ByVal sTFSystem$) As Integer

 ' Description:
 '  Ends an open transfer conversation
 
 ' Parameters:
 '  hWnd                 windows handle
 '  lTFCallBack          procedural call back address
 '  lTFConvID            conversation ID
 '  sTFSystem            system name

 ' Variables:
  Dim nTFReturnLen       As Integer ' returned length of data
  Dim sTFDataReturned    As String  ' data returned
 
  ' call transfer request wrapper
  zzTFEndConversation = zzTFPerformRequest(hWnd, lTFCallBack, lTFConvID, gnTF_END_ONE_REQ_CONVERSATION, gsEMPTY, sTFSystem, gnTF_NO_CONVERSION, nTFReturnLen, sTFDataReturned)

End Function

Function zzTFGetColumnsAll (ByVal hWnd%, ByVal sTFSystem$, ByVal sTFLib$, ByVal sTFFile$, atTFCols() As TFColType) As Integer

 ' Description:
 '  Reads all column information (field descriptions) for file
 '  specified into user defined data type atTemplates(). This
 '  wrapper uses the "EXTRACT COLUMNS" request.
 
 ' Parameters:
 '  hWnd                     windows handle
 '  sTFSystem                AS400 system name
 '  sTFLib                   AS400 library
 '  sTFFile                  AS400 file
 '  atTFCols()               User defined type that contains
 '                           properties for each field in the file

 ' Variables:
  Dim lTFCallBack            As Long     ' call back flag
  Dim lTFConvID              As Long     ' transfer conversation
  Dim nRC                    As Integer  ' transfer API return code
  Dim nTFNumFlds             As Integer  ' fields retrieved
  Dim nTFNumTemplates        As Integer  ' number of templates
  Dim sTFColInfo             As String   ' column information
  Dim sTFRequest             As String   ' transfer request text

  ' field properties
  Dim bFldNullCap            As Integer  ' is it Null capable?
  Dim bFldVarLen             As Integer  ' is it variable length?
  Dim nFldLength             As Integer  ' buffer length
  Dim nFldNumDecs            As Integer  ' number of decimal positions
  Dim nFldNumDigits          As Integer  ' number of digits
  Dim nFldType               As Integer  ' type
  Dim sFldName               As String   ' name
  Dim sFldText               As String   ' text

  ' build open transfer request without call back
  sTFRequest = UCase$("EXTRACT COLUMNS " & sTFLib & "/" & sTFFile)
  lTFCallBack = 0
  
  ' open transfer request
  nRC = zzTFOpen(hWnd, lTFCallBack, lTFConvID, sTFRequest, sTFSystem, nTFNumTemplates)
  
  ' if "EXTRACT" worked
  If nRC = gnTF_OK Then

    ' retrieve columns
    Do

      ' give windows chance to run
      DoEvents

      ' get next column
      nRC = zzTFGetRecord(hWnd, lTFCallBack, lTFConvID, sTFSystem, gnTF_DOS_RANDOM, sTFColInfo)

      ' exit on error or EOF
      If nRC <> gnTF_OK Then Exit Do

      ' increment counter
      nTFNumFlds = nTFNumFlds + 1

      ' parse the column information down into individual fields
      Call zzTFParseColumn(sTFColInfo, sFldName, nFldType, nFldNumDigits, nFldNumDecs, nFldLength, bFldNullCap, bFldVarLen, sFldText)
      
      ' move fields into type structure
      atTFCols(nTFNumFlds).sName = sFldName
      atTFCols(nTFNumFlds).nType = nFldType
      atTFCols(nTFNumFlds).nDigits = nFldNumDigits
      atTFCols(nTFNumFlds).nDecPos = nFldNumDecs
      atTFCols(nTFNumFlds).nLen = nFldLength
      atTFCols(nTFNumFlds).bNullCap = bFldNullCap
      atTFCols(nTFNumFlds).bVarLen = bFldVarLen
      atTFCols(nTFNumFlds).sText = sFldText
      
    Loop

    ' close the conversation
    nRC = zzTFEndConversation(hWnd, lTFCallBack, lTFConvID, sTFSystem)
    
  End If

  ' if templates found then return number of templates found
  zzTFGetColumnsAll = nTFNumFlds

End Function

Function zzTFGetRecord (ByVal hWnd%, ByVal lTFCallBack&, ByVal lTFConvID&, ByVal sTFSystem$, ByVal nTFConversion%, sTFRecordReturned$) As Integer
 
 ' Description:
 '  Retrieves next record from transfer request.
 
 ' Parameters:
 ' Input:
 '  hWnd                 windows handle
 '  lTFCallBack          procedural call back address
 '  lTFConvID            ID of transfer session
 '  sTFSystem            system name
 '  nTFConversion        conversion options
 ' Output:
 '  sTFRecordReturned    record returned
  
 ' Variables:
  Dim nTFReturnLen       As Integer ' returned length of data
  Dim sTFDataReturned    As String  ' data returned
  
  ' call transfer request wrapper
  zzTFGetRecord = zzTFPerformRequest(hWnd, lTFCallBack, lTFConvID, gnTF_RETRIEVE_RECORDS, gsEMPTY, sTFSystem, nTFConversion, nTFReturnLen, sTFRecordReturned)

End Function

Function zzTFGetRecordCount (ByVal hWnd%, lTFCallBack&, ByVal sTFSystem$, ByVal sTFLib$, ByVal sTFFile$) As String

 ' Description:
 '  Returns the record count of a file
 '  as a 4-byte binary string. The
 '  Bin4toASCII routine in EHNDTW2.BAS
 '  may be used to convert it into
 '  an ASCII value.
 
 ' Parameters:
 '  hWnd               windows handle
 '  lTFCallBack        call back address
 '  sTFSystem          AS400 system name
 '  sTFLib             AS400 library
 '  sTFFile            AS400 file

 ' Variables:
  Dim lTFConvID              As Long     ' transfer conversation
  Dim nRC                    As Integer  ' transfer API return code
  Dim nTFNumTemplates        As Integer  ' number of fields
  Dim nTFNumFlds             As Integer  ' templates downloaded
  Dim sTFBinRecordCount      As String   ' binary record count
  Dim sTFRequest             As String   ' transfer request text
  
  ' build open transfer request
  sTFRequest = "SELECT COUNT(*) FROM " & sTFLib & "/" & sTFFile
  
  ' open transfer request
  nRC = zzTFOpen(hWnd, lTFCallBack, lTFConvID, sTFRequest, sTFSystem, nTFNumTemplates)
  
  ' if select worked and file has fields then
  If nRC = gnTF_OK And nTFNumTemplates > 0 Then

    ' get first record which will contain record count
    nRC = zzTFGetRecord(hWnd, lTFCallBack, lTFConvID, sTFSystem, gnTF_NO_CONVERSION, sTFBinRecordCount)

    ' close the conversation
    nRC = zzTFEndConversation(hWnd, lTFCallBack, lTFConvID, sTFSystem)
    
  End If

  ' if templates found then return number of templates found
  zzTFGetRecordCount = sTFBinRecordCount

End Function

Function zzTFGetTemplate (ByVal hWnd%, ByVal lTFCallBack&, ByVal lTFConvID&, ByVal sTFSystem$, sTFTemplate$) As Integer
 
 ' Description:
 '  Retrieves next template from open transfer request.

 ' Parameters:
 '  hWnd                 windows handle
 '  lTFCallBack          procedural call back address
 '  lTFConvID            ID of transfer session
 '  sTFSystem            system name
 '  sTFTemplate          template returned

 ' Variables:
  Dim nTFReturnLen       As Integer ' returned length of data
  
  ' call transfer request wrapper
  zzTFGetTemplate = zzTFPerformRequest(hWnd, lTFCallBack, lTFConvID, gnTF_RETRIEVE_TEMPLATES, gsEMPTY, sTFSystem, gnTF_NO_CONVERSION, nTFReturnLen, sTFTemplate)

End Function

Function zzTFGetTemplatesAll (ByVal hWnd%, ByVal sTFSystem$, ByVal sTFLib$, ByVal sTFFile$, atTFCols() As TFColType) As Integer

 ' Description:
 '  Reads all templates (field descriptions) for file
 '  specified into user defined data type atTemplates().
 '  Function will return number of field templates or
 '  zero if routine not successful.
 
 ' Parameters:
 '  hWnd                     windows handle
 '  sTFSystem                AS400 system name
 '  sTFLib                   AS400 library
 '  sTFFile                  AS400 file
 '  atTFCols()               User defined type that contains
 '                           properties for each field in the file

 ' Variables:
  Dim lTFCallBack            As Long     ' call back flag
  Dim lTFConvID              As Long     ' transfer conversation
  Dim nRC                    As Integer  ' transfer API return code
  Dim nTFNumFlds             As Integer  ' templates downloaded
  Dim nTFNumTemplates        As Integer  ' number of fields
  Dim sTFRequest             As String   ' transfer request text
  Dim sTFTemplate            As String   ' transfer template

  ' field properties
  Dim bFldNullCap            As Integer  ' is it Null capable?
  Dim bFldVarLen             As Integer  ' is it variable length?
  Dim nFldLength             As Integer  ' buffer length
  Dim nFldNumDecs            As Integer  ' number of decimal positions
  Dim nFldNumDigits          As Integer  ' number of digits
  Dim nFldType               As Integer  ' type
  Dim sFldName               As String   ' name

  ' build open transfer request without call back
  sTFRequest = UCase$("SELECT * FROM " & sTFLib & "/" & sTFFile)
  lTFCallBack = 0
  
  ' open "SELECT" request
  nRC = zzTFOpen(hWnd, lTFCallBack, lTFConvID, sTFRequest, sTFSystem, nTFNumTemplates)
  
  ' if select worked and file has fields then
  If nRC = gnTF_OK And nTFNumTemplates > 0 Then

    ' make sure array is big enough to hold all fields
    ReDim atTemplates(1 To nTFNumTemplates)

    ' retrieve templates
    Do

      ' give windows chance to run
      DoEvents

      ' get next template
      nRC = zzTFGetTemplate(hWnd, lTFCallBack, lTFConvID, sTFSystem, sTFTemplate)

      ' exit on error or EOF
      If nRC <> gnTF_OK Then Exit Do

      ' increment counter
      nTFNumFlds = nTFNumFlds + 1

      ' parse the template down into individual fields
      Call zzTFParseTemplate(sTFTemplate, sFldName, nFldType, nFldNumDigits, nFldNumDecs, nFldLength, bFldNullCap, bFldVarLen)
      
      ' move fields into user-defined type
      atTFCols(nTFNumFlds).sName = sFldName
      atTFCols(nTFNumFlds).nType = nFldType
      atTFCols(nTFNumFlds).nDigits = nFldNumDigits
      atTFCols(nTFNumFlds).nDecPos = nFldNumDecs
      atTFCols(nTFNumFlds).nLen = nFldLength
      atTFCols(nTFNumFlds).bNullCap = bFldNullCap
      atTFCols(nTFNumFlds).bVarLen = bFldVarLen
      atTFCols(nTFNumFlds).sText = gsEMPTY
      
    Loop

    ' close the conversation
    nRC = zzTFEndConversation(hWnd, lTFCallBack, lTFConvID, sTFSystem)
    
  End If

  ' if templates found then return number of templates found
  zzTFGetTemplatesAll = nTFNumFlds

End Function

Function zzTFOpen (ByVal hWnd%, ByVal lTFCallBack&, lTFConvID&, ByVal sTFBuffer$, ByVal sTFSystem$, nTFNumTemplates%) As Integer
 
 ' Description:
 '  Opens a transfer request. If successful it will return the
 '  conversation ID "lTFConvID" and the number of templates
 '  (fields) "nTFNumTemplates" that match the specification.

 ' Parameters:
 '  hWnd             windows handle                 Input
 '  lTFCallBack      procedural call back address   Input
 '  lTFConvID        conversation ID returned       Output
 '  sTFBuffer        initial transfer request       Input
 '  sTFSystem        AS400 system name              Input
 '  nTFNumTemplates  number of fields returned      Input
 
 ' Variables:
  Dim nTFReturnLen       As Integer ' returned length of data
  Dim sTFDataReturned    As String  ' data returned
 
  ' call transfer request wrapper
  zzTFOpen = zzTFPerformRequest(hWnd, lTFCallBack, lTFConvID, gnTF_SEND_REQUEST, sTFBuffer, sTFSystem, gnTF_NO_CONVERSION, nTFNumTemplates, sTFDataReturned)
	      
End Function

Sub zzTFParseColumn (sTFColInfo$, sFldName$, nFldType%, nFldNumDigits%, nFldNumDecs%, nFldLength%, bFldNullCap%, bFldVarLen%, sFldText$)
 
 ' Description:
 '  Breaks column information into individual components as shown below.
 
 ' Parameters:
 ' Input:
 '  sTFColInfo             original column information
 ' Output:
 '  sFldName               name of field
 '  nFldType               type of field
 '  nFldNumDigits          number of digits
 '  nFldNumDecs            number of decimals
 '  nFldLength             length of field
 '  bFldNullCap            is field Null-capable?
 '  bFldVarLen             is field variable length?
 '  sFldText               description of field

 ' Variables:
  Dim sFldType             As String

  ' fill with Nulls to prevent string handling errors
  sTFColInfo = Left$(sTFColInfo & String$(74, 0), 74)
  
  ' retrieve field name
  sFldName = RTrim$(Mid$(sTFColInfo, 1, 10))

  ' retrieve field type
  sFldType = RTrim$(Mid$(sTFColInfo, 11, 5))

  ' convert text to numeric representation
  Select Case sFldType
    Case "CHAR", "VCHAR": nFldType = gnTF_CHAR_FIELD
    Case "HEX", "VHEX": nFldType = gnTF_HEX_FIELD
    Case "BIN": nFldType = gnTF_BINARY_FIELD
    Case "PACK": nFldType = gnTF_PACKED_FIELD
    Case "ZONED": nFldType = gnTF_ZONED_FIELD
    Case "FLOAT": nFldType = gnTF_RSVRD_FIELD
    Case "DATE": nFldType = gnTF_DATE_FIELD
    Case "TIME": nFldType = gnTF_TIME_FIELD
    Case "TSTMP": nFldType = gnTF_TIMESTAMP_FIELD
    Case "OPEN", "VOPEN": nFldType = gnTF_IGCOPEN_FIELD
    Case "ONLY", "VONLY": nFldType = gnTF_IGCONLY_FIELD
    Case "EITHER", "VETHR": nFldType = gnTF_GRAPHIC_FIELD
    Case "GRAPH", "VGRPH": nFldType = gnTF_GRAPHIC_FIELD
  End Select

  ' get length of field
  nFldLength = Asc(Mid$(sTFColInfo, 16, 1)) + (Asc(Mid(sTFColInfo, 17, 1)) * 256)
  
  ' set number of digits
  nFldNumDigits = Asc(Mid$(sTFColInfo, 18, 1)) + (Asc(Mid(sTFColInfo, 19, 1)) * 256)

  ' set number decimal positions
  nFldNumDecs = Asc(Mid$(sTFColInfo, 20, 1)) + (Asc(Mid(sTFColInfo, 21, 1)) * 256)

  ' Null capable?
  bFldNullCap = Mid$(sTFColInfo, 72, 3) = "YES"
    
  ' variable length?
  bFldVarLen = Left$(sFldType, 1) = "V"

  ' get field description
  sFldText = RTrim$(Mid$(sTFColInfo, 22, 50))

End Sub

Sub zzTFParseTemplate (sTFTemplate$, sFldName$, nFldType%, nFldNumDigits%, nFldNumDecs%, nFldLength%, bFldNullCap%, bFldVarLen%)
 
 ' Description:
 '  Breaks field templete into individual components as shown below.
 
 ' Parameters:
 '  sTFTemplate            original field template
 '  sFldName               name of field
 '  nFldType               type of field
 '  nFldNumDigits          number of digits
 '  nFldNumDecs            number of decimals
 '  nFldLength             length of field
 '  bFldNullCap            is field Null-capable?
 '  bFldVarLen             is field variable length?

 ' Constants:
  Const bNULL_CAPABLE = 128     ' NULL capable
  Const bVARIABLE_LENGTH = 64   ' variable length
  Const nTEMPLATE_LENGTH = 40   ' length of template

  ' fill with Nulls to prevent string handling errors
  sTFTemplate = Left$(sTFTemplate & String$(nTEMPLATE_LENGTH, 0), nTEMPLATE_LENGTH)
  
  ' retrieve field name
  sFldName = RTrim$(Mid$(sTFTemplate, 4, 30))

  ' retrieve field type
  nFldType = Asc(Left$(sTFTemplate, 1))

  ' first portion of field length
  nFldLength = Asc(Mid$(sTFTemplate, 2, 1))
  
  ' set digits and decimals to zero
  nFldNumDecs = 0
  nFldNumDigits = 0

  ' if hex or char then add second byte
  If nFldType = gnTF_HEX_FIELD Or nFldType = gnTF_CHAR_FIELD Then
    nFldLength = nFldLength + Asc(Mid(sTFTemplate, 3, 1)) * 256

  ' if binary, zoned, or packed then get decimal position and digits
  ElseIf nFldType = gnTF_BINARY_FIELD Or nFldType = gnTF_ZONED_FIELD Or nFldType = gnTF_PACKED_FIELD Then
    nFldNumDecs = Asc(Mid(sTFTemplate, 3, 1))
    nFldNumDigits = Asc(Mid(sTFTemplate, 34, 1))
  End If

  ' Null capable?
  bFldNullCap = (Asc(Mid(sTFTemplate, 35, 1)) And bNULL_CAPABLE) > 0
    
  ' variable length?
  bFldVarLen = (Asc(Mid(sTFTemplate, 35, 1)) And bVARIABLE_LENGTH) > 0

End Sub

Function zzTFPerformRequest (ByVal hWnd%, ByVal lTFCallBack&, lTFConvID&, ByVal nTFFunction%, ByVal sTFRequest$, ByVal sTFSystem$, ByVal nTFConversion%, nTFReturnLen%, sTFDataReturned$) As Integer
 
 ' Description:
 '  Transfer API wrapper - This is a generic routine that
 '  handles submission of all types of transfer requests.
 '  It is called from the other routines in this module.
 
 ' Parameters:
 ' Input:
 '  hWnd                windows handle
 '  lTFCallBack         procedural call back address
 '  lTFConvID           conversation ID passed or returned (if open)
 '  nTFFunction         function being performed "see declarations"
 '  sTFRequest          transfer request text (SELECT, REPLACE, etc.)
 '  sTFSystem           system to transfer with
 '  nTFConversion       data conversion option
 ' Output:
 '  nTFReturnLen        length of data returned
 '  sTFDataReturned     data returned

 ' Constants:
  Const nBUFFER_SIZE = 4096       ' required buffer size
 
 ' Variables:
  Dim nRC             As Integer  ' API return code
  Dim n2ndRC          As Integer  ' transfer secondary return code which is ignored with this wrapper
  Dim lAddressPointer As Long     ' returns string pointer
  Dim lReturnAddress  As Long     ' memory location of data in buffer which is returned from the API call
  Dim lReturnLength   As Long     ' contains the # of bytes to be copied

  ' place request into global buffer making it the correct size
  gsTFBuffer = Left(sTFRequest & String$(nBUFFER_SIZE, 0), nBUFFER_SIZE)
										
  ' call transfer API
  nRC = EHNTF_STF(hWnd, lTFCallBack, lTFConvID, nTFFunction, gsTFBuffer, nBUFFER_SIZE, Len(sTFRequest), "SYSTEM(" & Trim$(sTFSystem) & ")", nTFConversion, lReturnAddress, nTFReturnLen, n2ndRC)
  
  ' if something is returned
  If nTFReturnLen > 0 Then

    ' make string buffer big enough
    sTFDataReturned = Space$(nBUFFER_SIZE)

    ' set the copy length
    lReturnLength = nTFReturnLen

    ' copy data to returned data variable using pointer routine
    Call hMemCpyAny2(ByVal sTFDataReturned, lReturnAddress, lReturnLength)
    
    ' drop the remaining spaces
    sTFDataReturned = Left$(sTFDataReturned, nTFReturnLen)

  Else
  
    ' clear returned data field
    sTFDataReturned = gsEMPTY
  
  End If

  ' return code to caller
  zzTFPerformRequest = nRC
  
End Function

Function zzTFSendRecord (ByVal hWnd%, ByVal lTFCallBack&, ByVal lTFConvID&, ByVal sTFSystem$, ByVal nTFConversion%, ByVal sTFRecordToSend$, ByVal nTFRecordLen%) As Integer
 
 ' Description:
 '  Sends a record to open transfer request

 ' Parameters:
 '  hWnd                windows handle
 '  lTFCallBack         procedural call back address
 '  lTFConvID           ID of transfer session
 '  sTFSystem           system name
 '  nTFConversion       conversion option
 '  sTFRecordToSend     data to be sent
 '  nTFRecordLen        length of data to be sent

 ' Variables:
  Dim nTFReturnLen      As Integer ' returned length of data
  Dim sTFDataReturned   As String  ' data returned
  
  ' pad record to correct length
  sTFRecordToSend = Left$(sTFRecordToSend & Space$(nTFRecordLen), nTFRecordLen)

  ' call transfer request wrapper
  zzTFSendRecord = zzTFPerformRequest(hWnd, lTFCallBack, lTFConvID, gnTF_SEND_RECORDS, sTFRecordToSend, sTFSystem, nTFConversion, nTFReturnLen, sTFDataReturned)

End Function

