Option Explicit

 ' Description:
 '  This module contains constants, types,
 '  and wrappers related to the Data Queue APIs.

 ' Support Modules:
 '  zzc
 '  zzpdtw1

 ' Constants:

  ' Data Queue: return codes
  Global Const gnDQ_SUCCESS = 0                              ' successful operation
  Global Const gnDQ_COMMERROR = 1                            ' communication error
  Global Const gnDQ_EXCEPTERROR = 2                          ' exception error
  Global Const gnDQ_PCERROR = 3                              ' pc memory error

  ' Data Queue: translation options
  Global Const gnDQ_NOXLAT_NOPEEK = 0                        ' translation-OFF  peek-OFF
  Global Const gnDQ_YESXLAT_NOPEEK = 1                       ' translation-ON   peek-OFF
  Global Const gnDQ_NOPEEK = 2                               ' translation-SAME peek-OFF
  Global Const gnDQ_NOXLAT_YESPEEK = 4                       ' translation-OFF  peek-ON
  Global Const gnDQ_YESXLAT_YESPEEK = 5                      ' translation-ON   peek-ON
  Global Const gnDQ_YESPEEK = 6                              ' translation-SAME peek-ON
  Global Const gnDQ_NOXLAT = 8                               ' translation-OFF  peek-SAME
  Global Const gnDQ_YESXLAT = 9                              ' translation-ON   peek-SAME

  ' Data Queue: authorities
  Global Const gnDQ_ALL = 0                                  ' *ALL
  Global Const gnDQ_EXCLUDE = 1                              ' *EXCLUDE
  Global Const gnDQ_CHANGE = 2                               ' *CHANGE
  Global Const gnDQ_USE = 3                                  ' *USE
  Global Const gnDQ_LIBCRTAUT = 4                            ' authority based on library DQ is in

  ' Data Queue: sequence
  Global Const gnDQ_LIFO = 0                                 ' last in, first out
  Global Const gnDQ_FIFO = 1                                 ' first in, first out
 
  ' Data Queue: force and sender ID flag
  Global Const gnDQ_FALSE = 0                                ' no
  Global Const gnDQ_TRUE = 1                                 ' yes
 
 ' Variables:
  Dim nDQrc      As Integer           ' API return code

 ' APIs:
  Declare Function EHNDQ_CancelRequest% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$)
  Declare Function EHNDQ_CancelRequestKeyed% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal sDQKey$)
  Declare Function EHNDQ_Clear% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$)
  Declare Function EHNDQ_Create% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal lDQMaxLen&, ByVal nDQSeq%, ByVal bDQForce%, ByVal nDQAuth%, ByVal bDQSenderID%, ByVal sDQText$)
  Declare Function EHNDQ_CreateKeyed% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal lDQMaxLen&, ByVal bDQForce%, ByVal nDQAuth%, ByVal bDQSenderID%, ByVal sDQText$, ByVal nDQKeyLen%)
  Declare Function EHNDQ_Delete% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$)
  Declare Function EHNDQ_GetCapability% Lib "EHNDQW.DLL" (ByVal hWnd%, nDQFunLvl%)
  Declare Function EHNDQ_GetMessage% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQSystem$, ByVal sDQErrMsg$, nDQErrMsgLen%)
  Declare Function EHNDQ_Put% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal sDQRec$, ByVal lDQRecLen&)
  Declare Function EHNDQ_PutKeyed% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal sDQRec$, ByVal lDQRecLen&, ByVal sDQKey$)
  Declare Function EHNDQ_Query% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, lDQMaxLen&, nDQSeq%, bDQForce%, bDQSenderID%, ByVal sDQText$, nDQKeyLen%)
  Declare Function EHNDQ_Receive% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal lDQWait&, ByVal bDQSenderID%, ByVal sDQRec$, lDQRecLen&, ByVal sDQSender$)
  Declare Function EHNDQ_ReceiveData% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$)
  Declare Function EHNDQ_ReceiveDataKeyed% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal sDQKey$)
  Declare Function EHNDQ_ReceiveKeyed% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal lDQWait&, ByVal sDQKeyFind$, ByVal sDQKeySearch$, ByVal bDQSenderID%, ByVal sDQRec$, lDQRecLen&, ByVal sDQKeyRec$, ByVal sDQSender$)
  Declare Function EHNDQ_ReceiveRequest% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal lDQWait&, ByVal bDQSenderID%, ByVal sDQRec$, lDQRecLen&, ByVal sDQSender$)
  Declare Function EHNDQ_ReceiveRequestKeyed% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal lDQWait&, ByVal sDQKey$, ByVal sDQKeySearch$, ByVal bDQSenderID%, ByVal sDQRec$, lDQRecLen&, ByVal sDQKeyRec$, ByVal sDQSender$)
  Declare Function EHNDQ_Send% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal sDQRec$, ByVal lDQRecLen&)
  Declare Function EHNDQ_SendKeyed% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal sDQRec$, ByVal lDQRecLen&, ByVal sDQKey$)
  Declare Function EHNDQ_SetMode% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQSystem$, ByVal lDQMode&)
  Declare Function EHNDQ_Stop% Lib "EHNDQW.DLL" (ByVal hWnd%, ByVal sDQSystem$)

Function zzDQClear (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, sDQErrMsg$) As Integer

 ' Description:
 '  Clear a data queue

 ' Parameters:
 ' Input:
 '  hWnd                     windows handle
 '  sDQ                      data queue
 '  sDQSystem                AS/400 system name
 ' Output:
 '  sDQErrMsg                error message returned
  
  ' clear the data queue
  nDQrc = EHNDQ_Clear(hWnd, RTrim$(sDQ) & gsCHR_NUL, RTrim$(sDQSystem) & gsCHR_NUL)
  
  ' if any error get message
  If nDQrc <> gnDQ_SUCCESS Then
    sDQErrMsg = zzDQGetErrMsg(hWnd, sDQSystem, nDQrc)
  Else
    sDQErrMsg = gsEMPTY
  End If

  ' return code to caller
  zzDQClear = nDQrc

End Function

Function zzDQConvertNo (ByVal hWnd%, ByVal sDQSystem$, ByVal bDQPeek%, sDQErrMsg$) As Integer

 ' Description:
 '  Set data queue mode to no conversion

 ' Parameters:
 ' Input:
 '  hWnd                     windows handle
 '  sDQSystem                AS/400
 '  bDQPeek                  peek at message
 ' Output:
 '  sDQErrMsg                error message returned
 
 ' Variables:
  Dim nDQMode As Integer     ' mode selected

  ' handle DOS errors
  On Error Resume Next

  ' set up mode option
  nDQMode = gnDQ_NOXLAT
  If bDQPeek Then nDQMode = nDQMode And gnDQ_YESPEEK
 
  ' set the DQ mode to no convert
  nDQrc = EHNDQ_SetMode(hWnd, RTrim$(sDQSystem) & gsCHR_NUL, nDQMode)

  ' DOS error
  If Err <> 0 Then nDQrc = gnDQ_PCERROR
  
  ' if any error get message
  If nDQrc <> gnDQ_SUCCESS Then
    sDQErrMsg = zzDQGetErrMsg(hWnd, sDQSystem, nDQrc)
  Else
    sDQErrMsg = gsEMPTY
  End If
  
  ' return code to caller
  zzDQConvertNo = nDQrc

End Function

Function zzDQConvertYes (ByVal hWnd%, ByVal sDQSystem$, ByVal bDQPeek%, sDQErrMsg$) As Integer

 ' Description:
 '  Set data queue mode to conversion

 ' Parameters:
 ' Input:
 '  hWnd                     windows handle
 '  sDQSystem                AS/400
 '  bDQPeek                  peek at message
 ' Output:
 '  sDQErrMsg                error message returned

 ' Variables:
  Dim nDQMode As Integer     ' mode selected

  ' set up mode option
  nDQMode = gnDQ_YESXLAT
  If bDQPeek Then nDQMode = nDQMode And gnDQ_YESPEEK

  ' set the DQ mode to convert
  nDQrc = EHNDQ_SetMode(hWnd, RTrim$(sDQSystem) & gsCHR_NUL, nDQMode)
  
  ' if any error get message
  If nDQrc <> gnDQ_SUCCESS Then
    sDQErrMsg = zzDQGetErrMsg(hWnd, sDQSystem, nDQrc)
  Else
    sDQErrMsg = gsEMPTY
  End If
  
  ' return code to caller
  zzDQConvertYes = nDQrc

End Function

Function zzDQCreate (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal lDQMaxLen&, ByVal nDQSeq%, ByVal bDQForce%, ByVal nDQAuth%, ByVal bDQSenderID%, ByVal sDQText$, sDQErrMsg$) As Integer
  
 ' Description:
 '  Create a data queue

 ' Parameters:
 ' Input:
 '  hWnd                windows handle
 '  sDQ                 data queue
 '  sDQSystem           AS/400 system name
 '  lDQMaxLen           data queue max length
 '  nDQSeq              data queue sequence
 '  bDQForce            data queue force flag
 '  nDQAuth             data queue authority
 '  bDQSenderID         data queue sender ID flag
 '  sDQText             data queue description
 ' Output:
 '  sDQErrMsg           error message returned
  
  ' create the data queue
  nDQrc = EHNDQ_Create(hWnd, RTrim$(sDQ) & gsCHR_NUL, RTrim$(sDQSystem) & gsCHR_NUL, lDQMaxLen, nDQSeq, bDQForce, nDQAuth, bDQSenderID, sDQText)

  ' if any error get message
  If nDQrc <> gnDQ_SUCCESS Then
    sDQErrMsg = zzDQGetErrMsg(hWnd, sDQSystem, nDQrc)
  Else
    sDQErrMsg = gsEMPTY
  End If

  ' return code to caller
  zzDQCreate = nDQrc

End Function

Function zzDQCreateKeyed (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal lDQMaxLen&, ByVal nDQSeq%, ByVal bDQForce%, ByVal nDQAuth%, ByVal bDQSenderID%, ByVal sDQText$, ByVal nDQKeyLen%, sDQErrMsg$) As Integer
  
 ' Description:
 '  Create a data queue (keyed)

 ' Parameters:
 ' Input:
 '  hWnd                windows handle
 '  sDQ                 data queue
 '  sDQSystem           AS/400 system name
 '  lDQMaxLen           data queue max length
 '  nDQSeq              data queue sequence
 '  bDQForce            data queue force flag
 '  nDQAuth             data queue authority
 '  bDQSenderID         data queue sender ID flag
 '  sDQText             data queue description
 '  nDQKeyLen           data queue key length
 ' Output:
 '  sDQErrMsg           error message returned
  
  ' create the data queue
  nDQrc = EHNDQ_CreateKeyed(hWnd, RTrim$(sDQ) & gsCHR_NUL, RTrim$(sDQSystem) & gsCHR_NUL, lDQMaxLen, bDQForce, nDQAuth, bDQSenderID, sDQText, nDQKeyLen)
  
  ' if any error get message
  If nDQrc <> gnDQ_SUCCESS Then
    sDQErrMsg = zzDQGetErrMsg(hWnd, sDQSystem, nDQrc)
  Else
    sDQErrMsg = gsEMPTY
  End If

  ' return code to caller
  zzDQCreateKeyed = nDQrc

End Function

Function zzDQDelete (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, sDQErrMsg$) As Integer
  
 ' Description:
 '   Delete a data queue
 
 ' Parameters:
 ' Input:
 '  hWnd                     windows handle
 '  sDQ                      data queue
 '  sDQSystem                AS/400 system name
 ' Output:
 '  sDQErrMsg                error message returned
  
  ' delete the data queue
  nDQrc = EHNDQ_Delete(hWnd, RTrim$(sDQ) & gsCHR_NUL, RTrim$(sDQSystem) & gsCHR_NUL)
  
  ' if any error get message
  If nDQrc <> gnDQ_SUCCESS Then
    sDQErrMsg = zzDQGetErrMsg(hWnd, sDQSystem, nDQrc)
  Else
    sDQErrMsg = gsEMPTY
  End If

  ' return code to caller
  zzDQDelete = nDQrc

End Function

Function zzDQGetCapability (ByVal hWnd%, ByVal sDQSystem$, nDQFunLvl%, sDQErrMsg$) As Integer

 ' Description:
 '  Get function level of API
 
 ' Parameters:
 ' Input:
 '  hWnd                     windows handle
 '  sDQSystem                AS/400 system name
 ' Output:
 '  nDQFunLvl                functional level returned
 '  sDQErrMsg                error message returned
  
  ' get function level
  nDQrc = EHNDQ_GetCapability(hWnd, nDQFunLvl)

  ' if any error get message
  If nDQrc <> gnDQ_SUCCESS Then
    sDQErrMsg = zzDQGetErrMsg(hWnd, sDQSystem, nDQrc)
  Else
    sDQErrMsg = gsEMPTY
  End If

  ' return code to caller
  zzDQGetCapability = nDQrc
  
End Function

Function zzDQGetErrMsg (ByVal hWnd%, ByVal sDQSystem$, ByVal nDQErrRC%) As String

 ' Description:
 '  Get error message for data queue function.

 ' Parameters:
 '  hWnd                         windows handle
 '  sDQSystem                    AS/400 name
 '  nDQrc                        return code

 ' Constants:
  Const DEFAULT_MSG_SIZE = 152   ' default DQ message size

 ' Variables:
  Dim nDQErrMsgLen  As Integer   ' error message length
  Dim nDQGetRC      As Integer   ' return code for get message
  Dim sDQErrMsg     As String    ' error message returned

  ' setup string space
  sDQErrMsg = Space$(DEFAULT_MSG_SIZE)

  ' get the error message
  nDQGetRC = EHNDQ_GetMessage(hWnd, RTrim$(sDQSystem) & gsCHR_NUL, sDQErrMsg, nDQErrMsgLen)
  
  ' internal message not found
  If nDQErrMsgLen = 0 Then
    sDQErrMsg = "Data Queue error number x'" & Hex$(nDQErrRC) & "'"
  
  ' return internal message
  Else
    sDQErrMsg = Left$(sDQErrMsg, nDQErrMsgLen)
  End If
  
  ' return message to caller
  zzDQGetErrMsg = sDQErrMsg

End Function

Function zzDQPut (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal sDQRec$, sDQErrMsg$) As Integer
  
 ' Description:
 '  Put record into data queue.
  
 ' Parameters:
 ' Input:
 '  hWnd                     windows handle
 '  sDQ                      data queue
 '  sDQSystem                AS/400 system name
 '  sDQRec                   record to be sent
 ' Output:
 '  sDQErrMsg                error message returned
  
  ' put record into data queue
  nDQrc = EHNDQ_Put(hWnd, RTrim$(sDQ) & gsCHR_NUL, RTrim$(sDQSystem) & gsCHR_NUL, sDQRec, Len(sDQRec))
  
  ' if any error get message
  If nDQrc <> gnDQ_SUCCESS Then
    sDQErrMsg = zzDQGetErrMsg(hWnd, sDQSystem, nDQrc)
  Else
    sDQErrMsg = gsEMPTY
  End If

  ' return code to caller
  zzDQPut = nDQrc

End Function

Function zzDQPutKeyed (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal sDQRec$, ByVal sDQKey$, sDQErrMsg$) As Integer

 ' Description:
 '  Put record into keyed data queue.
 
 ' Parameters:
 ' Input:
 '  hWnd                     windows handle
 '  sDQ                      data queue
 '  sDQSystem                AS/400 system name
 '  sDQRec                   record to be sent
 '  sDQKey                   record key to be sent
 ' Output:
 '  sDQErrMsg                error message returned
  
  ' send to keyed queue
  nDQrc = EHNDQ_PutKeyed(hWnd, RTrim$(sDQ) & gsCHR_NUL, RTrim$(sDQSystem) & gsCHR_NUL, sDQRec, Len(sDQRec), sDQKey)

  ' if any error get message
  If nDQrc <> gnDQ_SUCCESS Then
    sDQErrMsg = zzDQGetErrMsg(hWnd, sDQSystem, nDQrc)
  Else
    sDQErrMsg = gsEMPTY
  End If

  ' return code to caller
  zzDQPutKeyed = nDQrc

End Function

Function zzDQQueryAttributes (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, lDQMaxLen&, nDQSeq%, bDQForce%, bDQSenderID%, sDQText$, nDQKeyLen%, sDQErrMsg$) As Integer

 ' Description:
 '  Query data queue characteristics.

 ' Parameters:
 ' Input:
 '  hWnd                        windows handle
 '  sDQ                         data queue beign queried
 '  sDQSystem                   AS/400 name
 ' Output:
 '  lDQMaxLen                   maximum length
 '  nDQSeq                      arrival sequence
 '  bDQForce                    force flag
 '  bDQSenderId                 sender id
 '  sDQText                     description
 '  nDQKeyLen                   key length
 '  sDQErrMsg                   error message

 ' Constants:
  Const nTEXT_FIELD_SIZE = 50   ' size of text description

  ' make description field big enough
  sDQText = Space$(nTEXT_FIELD_SIZE)

  ' get attributes
  nDQrc = EHNDQ_Query(hWnd, RTrim$(sDQ) & gsCHR_NUL, RTrim$(sDQSystem) & gsCHR_NUL, lDQMaxLen, nDQSeq, bDQForce, bDQSenderID, sDQText, nDQKeyLen)

  ' trim spaces off description
  sDQText = RTrim$(sDQText)
  
  ' if any error get message
  If nDQrc <> gnDQ_SUCCESS Then
    sDQErrMsg = zzDQGetErrMsg(hWnd, sDQSystem, nDQrc)
  Else
    sDQErrMsg = gsEMPTY
  End If
  
  ' return code to caller
  zzDQQueryAttributes = nDQrc
  
End Function

Function zzDQReceive (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal nDQMaxLen%, ByVal sDQKey$, ByVal sDQKeySearch$, ByVal bDQSenderID%, ByVal lDQWait&, sDQMsgRtnd$, sDQKeyRtnd$, sDQSenderRtnd$, nDQLenRtnd%, sDQErrMsg$) As Integer

 ' Description:
 '  Receive a data queue record

 ' Parameters:
 ' Input:
 '  hWnd                             windows handle
 '  sDQ                              data queue
 '  sDQSystem                        AS/400 system name
 '  nDQMaxLen                        maximum length of message
 '  sDQKey                           key to search for
 '  sDQKeySearch                     search parameter ("EQ","GT",etc.)
 '  bDQSenderID                      receive sender ID
 '  lDQWait                          time to wait
 ' Output:
 '  sDQMsgRtnd                       record returned
 '  sDQKeyRtnd                       key returned
 '  sDQSenderRtnd                    sender ID returned
 '  nDQLenRtnd                       length of record returned
 '  sDQErrMsg                        error message

 ' Constants:
  Const nSENDER_ID_SIZE = 44         ' size of sender ID

 ' Variables:
  Dim lDQMsgLen          As Long     ' length of message
  Dim nDQKeyLen          As Integer  ' length of key
  Dim sKeyEBCDIC         As String   ' key value EBCDIC value
  
  ' sender id field must be big enough
  sDQSenderRtnd = Space$(nSENDER_ID_SIZE)
  
  ' get length of key field
  nDQKeyLen = Len(sDQKey)

  ' if keyed data queue
  If nDQKeyLen > 0 Then

    ' make key the right size and convert it to EBCDIC
    sKeyEBCDIC = zzCV_ASCIIToEBCDIC(hWnd, sDQKey)
  
    ' returned key field must be big enough to handle full key
    sDQKeyRtnd = Space$(nDQKeyLen)

  End If
  
  ' field must be big enough to
  ' handle largest possible record
  sDQMsgRtnd = Space$(nDQMaxLen)
  
  ' move maximum size to long
  lDQMsgLen = nDQMaxLen
   
  ' if keyed data queue
  If nDQKeyLen > 0 Then

    ' receive data from a keyed data queue and wait for response
    nDQrc = EHNDQ_ReceiveKeyed(hWnd, RTrim$(sDQ) & gsCHR_NUL, RTrim$(sDQSystem) & gsCHR_NUL, lDQWait, sKeyEBCDIC, sDQKeySearch, bDQSenderID, sDQMsgRtnd, lDQMsgLen, sDQKeyRtnd, sDQSenderRtnd)

  ' if not keyed
  Else
    
    ' receive data from a LIFO,FIFO data queue and wait for response
    nDQrc = EHNDQ_Receive(hWnd, RTrim$(sDQ) & gsCHR_NUL, RTrim$(sDQSystem) & gsCHR_NUL, lDQWait, bDQSenderID, sDQMsgRtnd, lDQMsgLen, sDQSenderRtnd)
    
  End If

  ' if any error
  If nDQrc <> gnDQ_SUCCESS Then

    ' get error message
    sDQErrMsg = zzDQGetErrMsg(hWnd, sDQSystem, nDQrc)
    
    ' no data returned
    nDQLenRtnd = 0
    sDQMsgRtnd = gsEMPTY

  ' if no error
  Else

    ' no error message
    sDQErrMsg = gsEMPTY

    ' data returned
    nDQLenRtnd = lDQMsgLen
    sDQMsgRtnd = Left$(sDQMsgRtnd, nDQLenRtnd)

  End If

  ' return code to caller
  zzDQReceive = nDQrc

End Function

Function zzDQReceiveBlock (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal lDQMaxLen&, ByVal sDQKey$, ByVal sDQKeySearch$, ByVal lDQWait&, ByVal nDQWaitTotal%, nBlkrc%, sDQErrMsg$) As String

 ' Description:
 '  Receives messages from data queue until no more left,
 '  will not return key or sender ID to calling routine
 '  since the purpose of this routine is to return many
 '  records as a single string.

 ' Parameters:
 ' Input:
 '  hWnd               windows handle
 '  sDQ                data queue
 '  sDQSystem          AS/400 system name
 '  lDQMaxLen          maximum message length
 '  sDQKey         key value to search for
 '  sDQKeySearch       search order (ie: "GE","EQ",etc.)
 '  lDQWait            record wait time
 '  nDQWaitTotal       the total wait time is:
 '                       -1 = wait until done
 '                        0 = no wait
 '                       >0 = wait that many seconds.
 ' Output:
 '  nBlkrc             procedural return code
 '  sDQErrMsg          procedural error message

 ' Variables:
  Dim lBlkRequestStart As Long     ' time last request of group of records started
  Dim nDQRecLenRtnd    As Integer  ' single message length
  Dim sBlk             As String   ' all records combined
  Dim sDQMsgRtnd       As String   ' single data record returned
  Dim sDQKeyRtnd       As String   ' single key returned
  Dim sDQSenderRtnd    As String   ' single sender ID
  
  ' record starting time
  lBlkRequestStart = Timer

  ' clear storage area
  sBlk = gsEMPTY

  ' loop to read records
  Do
    
    ' read single record from queue
    nBlkrc = zzDQReceive(hWnd, sDQ, sDQSystem, lDQMaxLen, sDQKey, sDQKeySearch, 0, lDQWait, sDQMsgRtnd, sDQKeyRtnd, sDQSenderRtnd, nDQRecLenRtnd, sDQErrMsg)
    
    ' if any error then exit loop
    If nBlkrc <> gnDQ_SUCCESS Then Exit Do

    ' if no current record and prior records read then exit loop
    If nDQRecLenRtnd = 0 And Len(sBlk) > 0 Then Exit Do

    ' if record found
    If nDQRecLenRtnd > 0 Then
    
      ' add record to others, clear record string
      sBlk = sBlk & sDQMsgRtnd
      sDQMsgRtnd = gsEMPTY

    Else

      ' if being timed then see if too much expired
      If nDQWaitTotal >= 0 Then
        If Timer - lBlkRequestStart > nDQWaitTotal Then Exit Do
      End If

    End If

  Loop
  
  ' return messages as single string to caller
  zzDQReceiveBlock = sBlk

End Function

Function zzDQSend (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal sDQRec$, sDQErrMsg$) As Integer

 ' Description:
 '  Put record into data queue.

 ' Parameters:
 ' Input:
 '  hWnd              windows handle
 '  sDQ               data queue
 '  sDQSystem         AS/400 system name
 '  sDQRec            data to be sent
 ' Output:
 '  sDQErrMsg         error message returned
  
  ' send record to DQ
  nDQrc = EHNDQ_Send(hWnd, RTrim$(sDQ) & gsCHR_NUL, RTrim$(sDQSystem) & gsCHR_NUL, sDQRec, Len(sDQRec))
  
  ' if any error get message
  If nDQrc <> gnDQ_SUCCESS Then
    sDQErrMsg = zzDQGetErrMsg(hWnd, sDQSystem, nDQrc)
  Else
    sDQErrMsg = gsEMPTY
  End If

  ' return code to caller
  zzDQSend = nDQrc

End Function

Function zzDQSendKeyed (ByVal hWnd%, ByVal sDQ$, ByVal sDQSystem$, ByVal sDQRec$, ByVal sDQKey$, sDQErrMsg$) As Integer

 ' Description:
 '  Put record into keyed data queue.
 
 ' Parameters:
 ' Input:
 '  hWnd              windows handle
 '  sDQ               data queue
 '  sDQSystem         AS/400 system name
 '  sDQRec            data to be sent
 '  sDQKey            key to be sent
 ' Output:
 '  sDQErrMsg         error message returned
  
  ' send to keyed queue
  nDQrc = EHNDQ_SendKeyed(hWnd, RTrim$(sDQ) & gsCHR_NUL, RTrim$(sDQSystem) & gsCHR_NUL, sDQRec, Len(sDQRec), sDQKey)

  ' if any error get message
  If nDQrc <> gnDQ_SUCCESS Then
    sDQErrMsg = zzDQGetErrMsg(hWnd, sDQSystem, nDQrc)
  Else
    sDQErrMsg = gsEMPTY
  End If

  ' return code to caller
  zzDQSendKeyed = nDQrc

End Function

Function zzDQStop (ByVal hWnd%, ByVal sDQSystem$) As Integer

 ' Description:
 '  Stop the data queue session
 
 ' Parameters:
 '  hWnd                     windows handle
 '  sDQSystem                AS/400

  On Error Resume Next
  zzDQStop = EHNDQ_Stop(hWnd, sDQSystem)

End Function

