Option Explicit

 ' Description:
 '  This module contains constants and wrappers
 '  related to the shared folder APIs.

 ' Constants:

  ' Shared Folders: return codes
  Global Const gnSF_SUCCESS = &H0                         ' command completed successfully
  Global Const gnSF_INVALID_FUNCTION = &H1                ' invalid function
  Global Const gnSF_FILE_NOT_FOUND = &H2                  ' file not found
  Global Const gnSF_PATH_NOT_FOUND = &H3                  ' path not found
  Global Const gnSF_ACCESS_DENIED = &H5                   ' access denied
  Global Const gnSF_ENVIRONMENT_ERROR = &HA               ' does not support real mode windows
  Global Const gnSF_INVALID_DRIVE = &HF                   ' invalid drive
  Global Const gnSF_NO_MORE_FILES = &H12                  ' no more files
  Global Const gnSF_GENERAL_FAILURE = &H1F                ' general failure
  Global Const gnSF_NET_NOT_SUPPORTED = &H32              ' network request not supported
  Global Const gnSF_NO_ROUTER = &H33                      ' router not found
  Global Const gnSF_SYSTEM_NAME_NOT_FOUND = &H35          ' system name not found
  Global Const gnSF_UNEXPECT_NET_ERROR = &H3B             ' unexpected network error
  Global Const gnSF_NET_ACCESS_DENIED = &H41              ' network access denied
  Global Const gnSF_FOLDER_NAME_NOT_FOUND = &H43          ' folder name not found
  Global Const gnSF_NET_NAME_LIMIT = &H44                 ' name too long
  Global Const gnSF_TEMP_CANNOT_ASSIGN = &H48             ' assign function temporarily suspended
  Global Const gnSF_TOO_MANY_ASSIGNED = &H54              ' maximum number of drives already assigned
  Global Const gnSF_ALREADY_ASSIGNED = &H55               ' drive already assigned
  Global Const gnSF_INVALID_PASSWORD = &H56               ' invalid password
  Global Const gnSF_INVALID_PARAMETER = &H57              ' invalid parameter
  Global Const gnSF_INVALID_HOST_LEVEL = &H5A             ' host system does not support function
  Global Const gnSF_BUFFER_OVERFLOW = &H6F                ' buffer overflow
  Global Const gnSF_INVALID_NAME = &H7B                   ' invalid name
  Global Const gnSF_INVALID_LEVEL = &H7C                  ' DLL is not at a function level to support this API

 ' Variables:
  Dim nSFrc            As Integer                         ' API return code
  Dim nSFDriveASCII    As Integer                         ' ASCII value of drive letter

 ' APIs:
  Declare Function EHNSF_AssignFlrDrive% Lib "EHNSFW.DLL" (ByVal hWnd%, ByVal nSFDrive%, ByVal sSFSystem$, ByVal sSFName$)
  Declare Function EHNSF_GetCapability% Lib "EHNSFW.DLL" (ByVal hWnd%)
  Declare Function EHNSF_GetFlrDesc% Lib "EHNSFW.DLL" (ByVal hWnd%, ByVal nSFDrive%, ByVal sSFName$, ByVal sSFDescription$)
  Declare Function EHNSF_FindAvailDrive% Lib "EHNSFW.DLL" (ByVal hWnd%, ByVal sSFDrive$)
  Declare Function EHNSF_QueryAssignedFlrDrive% Lib "EHNSFW.DLL" (ByVal hWnd%, ByVal nSFDrive%, ByVal sSFSystem$, ByVal sSFName$)
  Declare Function EHNSF_QueryDriveStatus% Lib "EHNSFW.DLL" (ByVal hWnd%, ByVal sSFDriveStatusArray$)
  Declare Function EHNSF_ReleaseFlrDrive% Lib "EHNSFW.DLL" (ByVal hWnd%, ByVal nSFDrive%)

Function zzSFAssign (ByVal hWnd%, ByVal sSFDrive$, ByVal sSFSystem$, ByVal sSFName$) As Integer

 ' Description:
 '  Assign a folder

 ' Parameters:
 '  hWnd                     windows handle
 '  sSFDrive                 drive specification
 '  sSFSystem                AS/400 system name
 '  sSFName                  AS/400 folder

  ' handle DOS errors
  On Error Resume Next
  
  ' get ASCII value of drive letter
  nSFDriveASCII = Asc(Left$(UCase$(sSFDrive) & " ", 1))

  ' make call and return code
  zzSFAssign = EHNSF_AssignFlrDrive(hWnd, nSFDriveASCII, Trim$(sSFSystem) & gsCHR_NUL, Trim$(sSFName) & gsCHR_NUL)
  
  ' DOS error occurred
  If Err <> 0 Then zzSFAssign = gnSF_ENVIRONMENT_ERROR

End Function

Function zzSFGetAssignedDriveStatus (ByVal hWnd%, ByVal sSFDrive$, sSFSystem$, sSFName$) As Integer

 ' Description:
 '  Get system name and folder for assigned drive
 
 ' Parameters:
 ' Input:
 '  hWnd                     windows handle
 '  sSFDrive                 drive specification
 ' Output:
 '  sSFSystem                AS/400 system name
 '  sSFName                  AS/400 folder

  ' handle DOS errors
  On Error Resume Next
  
  ' get ASCII value of drive letter
  nSFDriveASCII = Asc(Left$(UCase$(sSFDrive) & " ", 1))

  ' make fields big enough
  sSFSystem = Space$(128)
  sSFName = Space$(128)

  ' get drive information
  nSFrc = EHNSF_QueryAssignedFlrDrive(hWnd, nSFDriveASCII, sSFSystem, sSFName)

  ' if no error
  If nSFrc = gnSF_SUCCESS And Err = 0 Then

    ' remove nulls
    If InStr(sSFSystem, gsCHR_NUL) > 0 Then
      sSFSystem = Left$(sSFSystem, InStr(sSFSystem, gsCHR_NUL) - 1)
    End If
    
    If InStr(sSFName, gsCHR_NUL) > 0 Then
      sSFName = Left$(sSFName, InStr(sSFName, gsCHR_NUL) - 1)
    End If

  ' DOS error
  ElseIf Err <> 0 Then
    nSFrc = gnSF_ENVIRONMENT_ERROR
  End If

  ' return error code as function value
  zzSFGetAssignedDriveStatus = nSFrc

End Function

Function zzSFGetAvailableDrive (ByVal hWnd%) As String

 ' Description:
 '  Get an available drive to assign to a folder
 
 ' Parameters:
 '  hWnd                     windows handle

 ' Variables:
  Dim sSFDrive As String

  ' handle DOS errors
  On Error Resume Next

  ' get drive
  sSFDrive = Space$(1)
  nSFrc = EHNSF_FindAvailDrive(hWnd, sSFDrive)
  If nSFrc <> gnSF_SUCCESS Or Err <> 0 Then sSFDrive = gsEMPTY
  zzSFGetAvailableDrive = sSFDrive

End Function

Function zzSFGetCapability (ByVal hWnd%) As Integer

 ' Description:
 '  Get Capability Of Folder DLL
 
 ' Parameters:
 '  hWnd                     windows handle
  
  ' handle DOS errors
  On Error Resume Next
  
  ' get capability
  zzSFGetCapability = EHNSF_GetCapability(hWnd)

End Function

Function zzSFGetDescription (ByVal hWnd%, ByVal sSFDrive$, ByVal sSFName$) As String

 ' Description:
 '  Get a folder description
 
 ' Parameters:
 '  hWnd                     windows handle
 '  sSFDrive                 drive specification
 '  sSFName                  AS/400 folder

 ' Constants:
  Const nFOLDER_DESCRIPTION_LENGTH = 46

 ' Variables:
  Dim sSFDescription As String

  ' handle DOS errors
  On Error Resume Next
  
  ' get ASCII value of drive letter
  nSFDriveASCII = Asc(Left$(UCase$(sSFDrive) & " ", 1))
  
  ' make string space available
  sSFDescription = Space$(nFOLDER_DESCRIPTION_LENGTH)
  
  ' get folder description
  nSFrc = EHNSF_GetFlrDesc(hWnd, nSFDriveASCII, Left$(RTrim$(sSFName) & gsCHR_NUL & Space$(128), 128), sSFDescription)
  If nSFrc <> gnSF_SUCCESS Or Err = 0 Then sSFDescription = gsEMPTY

  ' return value to caller
  zzSFGetDescription = RTrim$(sSFDescription)

End Function

Function zzSFGetStatusOfDrives (ByVal hWnd%) As String
  
 ' Description:
 '  Get status of all drives. Returns a 26 byte
 '  string which tells the status of each drive.
 
 ' Parameters:
 '  hWnd                     windows handle
  
 ' Variables:
  Dim sSFDriveStatusArray As String
  
  ' handle DOS errors
  On Error Resume Next
  
  ' create string space
  sSFDriveStatusArray = Space$(26)

  ' make the API call
  nSFrc = EHNSF_QueryDriveStatus(hWnd, sSFDriveStatusArray)
  
  ' if any error then all unavailable
  If nSFrc <> gnSF_SUCCESS Or Err <> 0 Then
    sSFDriveStatusArray = "00000000000000000000000000"
  End If
  
  ' return flags to caller
  zzSFGetStatusOfDrives = sSFDriveStatusArray

End Function

Function zzSFRelease (ByVal hWnd%, ByVal sSFDrive$) As Integer
  
 ' Description:
 '  Release a folder
 
 ' Parameters:
 '  hWnd                     windows handle
 '  sSFDrive                 drive specification
  
  ' handle DOS errors
  On Error Resume Next
  
  ' get ASCII value of drive letter
  nSFDriveASCII = Asc(Left$(UCase$(sSFDrive) & " ", 1))

  ' make call and return code
  zzSFRelease = EHNSF_ReleaseFlrDrive(hWnd, nSFDriveASCII)
  
  ' DOS error occurred
  If Err <> 0 Then zzSFRelease = gnSF_ENVIRONMENT_ERROR

End Function

