Option Explicit

 ' Description:
 '  This module contains constants and wrappers
 '  related to the remote command APIs.

 ' Constants:

  ' Remote Command: return codes
  Global Const gnSR_OK = &H0                                 ' command completed successfully
  Global Const gnSR_WARNING = &HA                            ' warning issued
  Global Const gnSR_ERROR = &H14                             ' error found
  Global Const gnSR_ACCESS_DAMAGE = &H15                     ' DDM access damaged file
  Global Const gnSR_SEVERE_ERROR = &H16                      ' severe error found
  Global Const gnSR_PERMANENT_DAMAGE = &H17                  ' damage occurred to permanent object(s)
  Global Const gnSR_TARGET_DAMAGE = &H18                     ' DDM damaged for this session
  Global Const gnSR_CMD_STRING_SYNTAX = &H1F                 ' command string greater than 2000 bytes
  Global Const gnSR_SYS_NAME_SYNTAX = &H29                   ' system name greater than 8 bytes
  Global Const gnSR_MSG_TRUNCATE = &H2B                      ' reply message buffer to small
  Global Const gnSR_INACTIVE_CONVERSATION = &H2D             ' no active conversation to remote system
  Global Const gnSR_INVALID_POINTER = &H30                   ' input pointer to data buffer is NULL
  Global Const gnSR_CMD_NOT_SUPPORT = &H31                   ' DDM command not supported
  Global Const gnSR_CONTACT_END = &H32                       ' contact with remote system ended
  Global Const gnSR_CONVERSATION_END = &H33                  ' conversation ended unexpectedly
  Global Const gnSR_TARGET_DDM_END = &H34                    ' DDM unexpectedly ended
  Global Const gnSR_NO_SESSION_AVAIL = &H35                  ' conversation allocation failed
  Global Const gnSR_SYS_NAME_NOT_FOUND = &H36                ' remote system name not found
  Global Const gnSR_NO_RESOURCE_AVAIL = &H37                 ' insufficient system resource
  Global Const gnSR_TEMP_INTERRUPT = &H38                    ' contact with system interrupted
  Global Const gnSR_NO_DDM_ON_SYSTEM = &H39                  ' DDM not on remote system
  Global Const gnSR_SECURITY_INVALID = &H3A                  ' user ID and/or password invalid
  Global Const gnSR_ROUTER_NOT_LOADED = &H46                 ' router not loaded
  Global Const gnSR_NO_MEMORY = &H48                         ' not enough memory
  Global Const gnSR_INVALID_MSG_BUFFER = &H49                ' reply message buffer not valid
  Global Const gnSR_INVALID_CMD_BUFFER = &H4A                ' command string buffer not valid
  Global Const gnSR_INVALID_SYSTEM_BUFFER = &H4B             ' remote system buffer not valid
  Global Const gnSR_REQUEST_INTERRUPT = &H52                 ' system interrupted the request
  Global Const gnSR_INVALID_CONV_STATE = &H53                ' conversation state not valid
  Global Const gnSR_PARM_CHECK_ERROR = &H54                  ' parameter checking error
  Global Const gnSR_ALLOCATION_ERROR = &H55                  ' allocation unexpected error
  Global Const gnSR_UNEXPECT_RETURN_CODE = &H56              ' unexpected primary return code
  Global Const gnSR_NOT_IN_REQUEST_STATE = &H57              ' conversation not in request state
  Global Const gnSR_INVALID_MSG_CODE_POINT = &H58            ' message code point not valid
  Global Const gnSR_WRONG_DATA_LENGTH = &H5B                 ' wrong object data length
  Global Const gnSR_UNEXPECTED_ERROR = &H63                  ' unexpected error
  Global Const gnSR_NO_MORE_MESSAGES = &HFF00                ' all messages processed
  Global Const gnSR_NO_ROUTER_TO_SYSTEM = &HFF01             ' router not connected to system
  Global Const gnSR_INVALID_WINDOW_HANDLE = &HFF02           ' window handle does not exist or NULL
  Global Const gnSR_INTERNAL_ERROR = &HFF03                  ' internal error
  Global Const gnSR_ENVIRONMENT_NOT_SUPPORT = &HFF04         ' environment not supported

 ' APIs:
  Declare Function EHNSR_SubmitCommand% Lib "EHNSRW.DLL" (ByVal hWnd%, ByVal sSRSystem$, ByVal sSRCmd$, ByVal sSRReply$, ByVal nSRReplyLen%, ByVal lSRCallBack&)
  Declare Function EHNSR_GetMessage% Lib "EHNSRW.DLL" (ByVal hWnd%, ByVal sSRSystem$, lSRMsgSeverity&, ByVal sSRMsgID$, ByVal sSRMsgText$, ByVal nSRMsgLen%, ByVal sSRReply$, ByVal nSRReplyLen%)
  Declare Function EHNSR_StopConversation% Lib "EHNSRW.DLL" (ByVal hWnd%, ByVal sSRSystem$)

Function zzSRCmd (ByVal hWnd%, ByVal sSRSystem$, ByVal sSRCmd$, sSRMsgsReturned$, nSRLenOfMsgsReturned%, ByVal lSRCallBack&) As Integer

 ' Description:
 '  Submits command to AS/400 specified. Returns messages received to the
 '  caller. If the command is the first one submitted it will opens a
 '  conversation with the AS/400 which must be closed with the
 '  "zzSREndConversation" wrapper.

 ' Parameters:
 '  hWnd                           windows handle
 '  sSRSystem                      AS/400 system name
 '  sSRCmd                         command to process
 '  sSRMsgsReturned                messages returned
 '  nSRLenOfMsgsReturned           length of messages returned
 '  lSRCallBack                    procedure call back address

 ' Constants:
  Const nMAX_MSGS_LENGTH = 1024    ' maximum length of messages returned
  
  ' turn on error handling
  On Error Resume Next
  
  ' field must be big enough to handle largest reply
  sSRMsgsReturned = String$(nMAX_MSGS_LENGTH, 0)
  nSRLenOfMsgsReturned = nMAX_MSGS_LENGTH
  
  ' execute remote command
  zzSRCmd = EHNSR_SubmitCommand(hWnd, Trim$(sSRSystem) & gsCHR_NUL, Trim$(sSRCmd) & gsCHR_NUL, sSRMsgsReturned, nSRLenOfMsgsReturned, lSRCallBack)

  ' if any DOS error then must assume DLL will not work
  If Err <> 0 Then zzSRCmd = gnSR_ERROR
                    
End Function

Function zzSRCmdAndFormatMsgs (ByVal hWnd%, sSRSystem$, sSRCmd$, sSRFormattedMsgs$) As Integer

 ' Description:
 '  This function will submit a command, retrieve any
 '  messages generated, and format those messages. This
 '  function does not support callbacks and will leave
 '  the conversation open.
  
 ' Parameters:
 '  hWnd                      windows handle
 '  sSRSystem                 AS/400 system name
 '  sSRCmd                    command to execute
 '  sSRFormattedMsgs          formatted messages returned

 ' Variables:
  Dim nSRLenOfMsgsReturned    As Integer  ' length of messages returned
  Dim sSRMsgsReturned         As String   ' messages returned

  ' submit the command
  zzSRCmdAndFormatMsgs = zzSRCmd(hWnd, sSRSystem, sSRCmd, sSRMsgsReturned, nSRLenOfMsgsReturned, 0&)
  
  ' get and format messages
  sSRFormattedMsgs = zzSRRetrieveAndFormatMsgs(hWnd, sSRSystem, sSRMsgsReturned, nSRLenOfMsgsReturned)

End Function

Function zzSRCmdAndFormatMsgsWithCB (ByVal hWnd%, ByVal sSRSystem$, sSRCmd$, sSRFormattedMsgs$, lSRCallBack&) As Integer

 ' Description:
 '  This function will submit a command, retrieve any
 '  messages generated, and format those messages.
 '  This function does support callbacks and will
 '  leave the conversation open.
  
 ' Parameters:
 '  hWnd                      windows handle
 '  sSRSystem                 AS/400 system name
 '  sSRCmd                    command to execute
 '  sSRFormattedMsgs          formatted messages returned
 '  lSRCallBack               procedure call back address

 ' Variables:
  Dim nSRLenOfMsgsReturned    As Integer  ' length of messages returned
  Dim sSRMsgsReturned         As String   ' messages returned

  ' submit the command
  zzSRCmdAndFormatMsgsWithCB = zzSRCmd(hWnd, sSRSystem, sSRCmd, sSRMsgsReturned, nSRLenOfMsgsReturned, lSRCallBack)
  
  ' get and format messages
  sSRFormattedMsgs = zzSRRetrieveAndFormatMsgs(hWnd, sSRSystem, sSRMsgsReturned, nSRLenOfMsgsReturned)
  
End Function

Function zzSRCmdFormatMsgsAndEnd (ByVal hWnd%, ByVal sSRSystem$, ByVal sSRCmd$, sSRFormattedMsgs$) As Integer

 ' Description:
 '  This function will submit a command, retrieve any
 '  messages generated, format those messages, and end
 '  the conversation. This function does not support
 '  callbacks.

 ' Parameters:
 '  hWnd                    windows handle
 '  sSRSystem               AS/400 system name
 '  sSRCmd                  command to execute
 '  sSRFormattedMsgs        formatted messages returned

 ' Variables:
  Dim nSRrc As Integer      ' return code for end conversation

  ' submit command and format messages
  zzSRCmdFormatMsgsAndEnd = zzSRCmdAndFormatMsgs(hWnd, sSRSystem, sSRCmd, sSRFormattedMsgs)
  
  ' end conversation
  nSRrc = zzSREndConversation(hWnd, sSRSystem)

End Function

Function zzSREndConversation (ByVal hWnd%, ByVal sSRSystem$) As Integer

 ' Description:
 '  Stop active AS/400 remote command conversation.

 ' Parameters:
 '  hWnd             windows handle
 '  sSRSystem        AS/400 system name

  ' handle DOS errors
  On Error Resume Next
  
  ' call the API
  zzSREndConversation = EHNSR_StopConversation(hWnd, RTrim$(sSRSystem) & gsCHR_NUL)

  ' if any DOS error then must assume DLL will not work
  If Err <> 0 Then zzSREndConversation = gnSR_ERROR

End Function

Function zzSRRetrieveAndFormatMsgs (ByVal hWnd%, ByVal sSRSystem$, ByVal sSRMsgsReturned$, ByVal nSRLenOfMsgsReturned%) As String

 ' Description:
 '  Retrieves and formats messages received during a
 '  remote command so they can be displayed in a Windows
 '  message box. The format is Message_ID: Message_TEXT.
 '  If more than one messages is retrieved each is
 '  seperated by a blank line. If any error occurs the
 '  function will return nothing.

 ' Parameters:
 '  hWnd                                windows handle
 '  sSRSystem                           AS/400 system name
 '  sSRMsgsReturned                     messages returned
 '  nSRLenOfMsgsReturned                length of messages returned

 ' Constants:
  Const bEND_OF_MSG_NOT_FOUND = -1      ' end of message not found
  Const nMAX_MSG_LENGTH = 1024          ' maximum length of messages returned
  Const nMAX_MSG_ID_LENGTH = 7          ' maximum length of message id
  Const sNO_MSGS_RETURNED = "5814"      ' no message returned

 ' Variables:
  Dim nSRrc            As Integer       ' API return code
  Dim sSRFormattedMsgs As String        ' all messages combined and formatted

  ' data retrieved for single message
  Dim lSRMsgSeverity   As Long          ' severity
  Dim nMsgEndsAt       As Integer       ' ending location
  Dim nSRMsgLen        As Integer       ' length
  Dim sSRMsgID         As String        ' message ID
  Dim sSRMsgText       As String        ' message text

  ' handle DOS errors
  On Error Resume Next
  
  ' field must be big enough to handle largest message possible
  sSRMsgText = Space$(nMAX_MSG_LENGTH)
  nSRMsgLen = nMAX_MSG_LENGTH
  
  ' setup message id field
  sSRMsgID = Space$(nMAX_MSG_ID_LENGTH)

  ' loop to get all pending messages
  Do

    ' get next message from buffer
    nSRrc = EHNSR_GetMessage(hWnd, Trim$(sSRSystem) & gsCHR_NUL, lSRMsgSeverity, sSRMsgID, sSRMsgText, nSRMsgLen, sSRMsgsReturned, nSRLenOfMsgsReturned)
    
    ' handle DOS errors
    If Err <> 0 Then nSRrc = gnSR_ERROR

    ' if API error or no messages returned then exit loop
    If nSRrc <> gnSR_OK Then Exit Do

    ' determine were message text ends
    nMsgEndsAt = InStr(sSRMsgText, gsCHR_NUL) - 1

    ' if no message found then exit loop
    If nMsgEndsAt = bEND_OF_MSG_NOT_FOUND Then Exit Do

    ' don't pass back empty message (IBM's 5814 mess)
    If Left$(sSRMsgID, 4) = sNO_MSGS_RETURNED Then Exit Do

    ' if message found then
    If nMsgEndsAt > 0 Then
      
      ' add to error message returned to caller in "ID: text" format
      sSRFormattedMsgs = sSRFormattedMsgs & sSRMsgID + ": " & Left$(sSRMsgText, nMsgEndsAt)
      
      ' add carriage return and line feed
      sSRFormattedMsgs = sSRFormattedMsgs & gsCHR_CRLF
    
    End If

  Loop

  ' return messages to caller as function value
  zzSRRetrieveAndFormatMsgs = sSRFormattedMsgs

End Function

