Option Explicit
 
 ' Description:
 '  This module contains data translation APIs
 '  and the associated wrappers.

 ' Variables:
  Dim nCV_rc As Integer           ' API return code

 ' APIs:

  Declare Function EHNDT_ASCII6ToBin2% Lib "EHNDTW.DLL" (ByVal hWnd%, ByVal sCV_BIN2$, ByVal sCV_ASCII$)
  Declare Function EHNDT_ASCII11ToBin4% Lib "EHNDTW.DLL" (ByVal hWnd%, ByVal sCV_BIN4$, ByVal sCV_ASCII$)
  Declare Function EHNDT_ASCIIToPacked% Lib "EHNDTW.DLL" (ByVal hWnd%, ByVal sCV_PACKED$, ByVal sCV_ASCII$, ByVal nCV_PACKEDSize%, ByVal nCV_DecimalLocation%)
  Declare Function EHNDT_ASCIIToZoned% Lib "EHNDTW.DLL" (ByVal hWnd%, ByVal sCV_ZONED$, ByVal sCV_ASCII$, ByVal nCV_ZONEDSize%, ByVal nCV_DecimalLocation%)
  Declare Function EHNDT_Bin2ToASCII6% Lib "EHNDTW.DLL" (ByVal hWnd%, ByVal sCV_ASCII$, ByVal sCV_BIN2$)
  Declare Function EHNDT_Bin4ToASCII11% Lib "EHNDTW.DLL" (ByVal hWnd%, ByVal sCV_ASCII$, ByVal sCV_BIN4$)
  Declare Function EHNDT_PackedToASCII% Lib "EHNDTW.DLL" (ByVal hWnd%, ByVal sCV_ASCII$, ByVal sCV_PACKED$, ByVal nCV_PACKEDSize%, ByVal nCV_DecimalLocation%)
  Declare Function EHNDT_ZonedToASCII% Lib "EHNDTW.DLL" (ByVal hWnd%, ByVal sCV_ASCII$, ByVal sCV_ZONED$, ByVal nCV_ZONEDSize%, ByVal nCV_DecimalLocation%)

Function zzCV_ASCIItoBIN2 (ByVal hWnd%, ByVal sCV_ASCII$) As String

 ' Description:
 '  Convert
 '   From:  6-character ASCII STRING (Representing a number)
 '   To:    2-byte BINARY number
  
 ' Parameters:
 '  hWnd                  windows handle
 '  sCV_ASCII             number to convert

 ' Variables:
  Dim sCV_BIN2 As String  ' work string

  ' handle error
  On Error Resume Next
  
  ' make sure big enough
  sCV_ASCII = Left$(sCV_ASCII & Space$(6), 6)

  ' make target string big enough
  sCV_BIN2 = Space$(2)
  
  ' call the API
  nCV_rc = EHNDT_ASCII6ToBin2(hWnd, sCV_BIN2, sCV_ASCII)
  
  ' return value to caller
  zzCV_ASCIItoBIN2 = sCV_BIN2
       
End Function

Function zzCV_ASCIItoBIN4 (ByVal hWnd%, ByVal sCV_ASCII$) As String

 ' Description:
 '  Convert
 '   From:  11-character ASCII STRING (Representing a number)
 '   To:    4-byte BINARY number
 
 ' Parameters:
 '  hWnd                  windows handle
 '  sCV_ASCII             number to convert

 ' Variables:
  Dim sCV_BIN4 As String  ' work string

  ' handle error
  On Error Resume Next
  
  ' make sure big enough
  sCV_ASCII = Left$(sCV_ASCII & Space$(11), 11)
  
  ' make target string big enough
  sCV_BIN4 = Space$(4)

  ' convert, if error then return nothing
  nCV_rc = EHNDT_ASCII11ToBin4(hWnd, sCV_BIN4, sCV_ASCII)

  ' return value to caller
  zzCV_ASCIItoBIN4 = sCV_BIN4
       
End Function

Function zzCV_ASCIItoPACKED (ByVal hWnd%, ByVal sCV_ASCII$, ByVal nCV_PACKEDSize%) As String

 ' Description:
 '  Convert
 '   From:  ASCII
 '   To:    PACKED
 
 ' Parameters:
 '  hWnd                              windows handle
 '  sCV_ASCII                         original ASCII number
 '  nCV_PACKEDSize                    size when packed
 
 ' Variables:
  Dim nPos                As Integer  ' location of decimal position
  Dim sCV_PACKED          As String   ' string to hold result
  
  ' handle error
  On Error Resume Next
  
  ' cannot handle empty string
  If sCV_ASCII = gsEMPTY Then Exit Function

  ' remove decimal location consideration
  nPos = InStr(sCV_ASCII, ".")
  If nPos > 0 Then
    sCV_ASCII = Left$(sCV_ASCII, nPos - 1) & Mid$(sCV_ASCII, nPos + 1)
  End If

  ' make target field big enough
  sCV_PACKED = Space$(nCV_PACKEDSize)
  
  ' call the API
  nCV_rc = EHNDT_ASCIIToPacked(hWnd, sCV_PACKED, sCV_ASCII, nCV_PACKEDSize, 0)

  ' return value to caller
  zzCV_ASCIItoPACKED = sCV_PACKED

End Function

Function zzCV_ASCIItoZONED (ByVal hWnd%, ByVal sCV_ASCII$, ByVal nCV_ZONEDSize%) As String

 ' Description:
 '  Convert
 '   From:  ASCII
 '   To:    ZONED
 
 ' Description:
 '  Converts ASCII number to EBCDIC zoned number.
  
 ' Parameters:
 '  hWnd                           windows handle
 '  sCV_ASCII                      original ASCII number
 '  nCV_ZONEDSize                  size when zoned

 ' Variables:
  Dim nPos             As Integer  ' location of decimal position
  Dim sCV_ZONED        As String   ' string to hold result

  ' handle error
  On Error Resume Next
  
  ' cannot handle empty string
  If sCV_ASCII = gsEMPTY Then Exit Function

  ' remove decimal location consideration
  nPos = InStr(sCV_ASCII, ".")
  If nPos > 0 Then
    sCV_ASCII = Left$(sCV_ASCII, nPos - 1) & Mid$(sCV_ASCII, nPos + 1)
  End If

  ' make target field big enough
  sCV_ZONED = Space$(nCV_ZONEDSize)

  ' call the API treating number if it has no decimals
  nCV_rc = EHNDT_ASCIIToZoned(hWnd, sCV_ZONED, sCV_ASCII, nCV_ZONEDSize, 0)
  
  ' return value to caller
  zzCV_ASCIItoZONED = sCV_ZONED

End Function

Function zzCV_BIN2toASCII (ByVal hWnd%, ByVal sCV_BIN2$) As String

 ' Description:
 '  Convert
 '   From:  2-byte BINARY
 '   To:    6-character ASCII STRING
  
 ' Parameters:
 '  hWnd                     windows handle
 '  sCV_BIN2                 2-byte BINARY number

 ' Variables:
  Dim sCV_ASCII As String   ' work string
  
  ' handle error
  On Error Resume Next
  
  ' make string bug enough
  sCV_ASCII = Space$(6)

  ' call the API
  nCV_rc = EHNDT_Bin2ToASCII6(hWnd, sCV_ASCII, sCV_BIN2)

  ' return value to caller
  zzCV_BIN2toASCII = zzCV_FormatNumber(sCV_ASCII, 0)

End Function

Function zzCV_BIN4toASCII (ByVal hWnd%, ByVal sCV_BIN4$) As String

 ' Description:
 '  Convert
 '   From:  4-byte BINARY
 '   To:    11-character ASCII STRING
  
 ' Parameters:
 '  hWnd                     windows handle
 '  sCV_BIN4                 4-byte BINARY number

 ' Variables:
  Dim sCV_ASCII As String  ' work string

  ' handle error
  On Error Resume Next
  
  ' make string big enough
  sCV_ASCII = Space$(11)
  
  ' call the API
  nCV_rc = EHNDT_Bin4ToASCII11(hWnd, sCV_ASCII, sCV_BIN4)

  ' return value to caller
  zzCV_BIN4toASCII = zzCV_FormatNumber(sCV_ASCII, 0)

End Function

Function zzCV_FormatNumber (ByVal sCV_ASCII$, ByVal nCV_DecimalLocation%) As String

 ' Description:
 '  Remove leading zeros from numbers
 '  and make sure decimal point is present

 ' Parameters:
 '  sCV_ASCII          number to format
 '  nDecPosition       decimal position

  ' trim excess
  sCV_ASCII = Trim$(sCV_ASCII)

  ' handle negative numbers
  If Left$(sCV_ASCII, 1) = "-" Then

    ' remove negative sign
    sCV_ASCII = Mid$(sCV_ASCII, 2)

    ' if decimal location specified
    If nCV_DecimalLocation > 0 Then
      
      ' if decimal not found
      If InStr(sCV_ASCII, ".") = 0 Then

       ' add leading zeros to prevent error
       sCV_ASCII = String$(nCV_DecimalLocation, "0") & sCV_ASCII
    
        ' add decimal point manually
        sCV_ASCII = Left$(sCV_ASCII, Len(sCV_ASCII) - nCV_DecimalLocation) & "." & Right$(sCV_ASCII, nCV_DecimalLocation)

      End If
  
    End If
    
    ' remove leading zeros
    Do While Left$(sCV_ASCII, 1) = "0"
      sCV_ASCII = Mid$(sCV_ASCII, 2)
    Loop
    
    ' replace negative sign
    sCV_ASCII = "-" & sCV_ASCII
  
  ' handle positive numbers
  Else
    
    ' if decimal location specified
    If nCV_DecimalLocation > 0 Then
      
      ' if decimal not found
      If InStr(sCV_ASCII, ".") = 0 Then
    
        ' add decimal point manually
        sCV_ASCII = Left$(sCV_ASCII, Len(sCV_ASCII) - nCV_DecimalLocation) & "." & Right$(sCV_ASCII, nCV_DecimalLocation)

      End If
  
    End If
    
    ' remove leading zeros
    Do While Left$(sCV_ASCII, 1) = "0"
      sCV_ASCII = Mid$(sCV_ASCII, 2)
    Loop
  
  End If

  zzCV_FormatNumber = sCV_ASCII

End Function

Function zzCV_PACKEDtoASCII (ByVal hWnd%, ByVal sCV_PACKED$, ByVal nCV_DecimalLocation%, ByVal nCV_PACKEDSize%) As String

 ' Description:
 '  Convert
 '   From:  PACKED
 '   To:    ASCII
  
 ' Parameters:
 '  hWnd                             windows handle
 '  sCV_PACKED                       EBCDIC packed field
 '  nCV_DecimalLocation              actual location of decimal place
 '  nCV_PACKEDSize                   size of packed field

 ' Variables:
  Dim sCV_ASCII         As String   ' string to hold unpacked numeric result
  Dim nCV_unPACKEDSize  As Integer  ' unpacked field length

  ' handle error
  On Error Resume Next
  
  ' cannot handle empty string
  If sCV_PACKED = gsEMPTY Then Exit Function
  
  ' get length of packed and unpacked field
  nCV_unPACKEDSize = nCV_PACKEDSize * 2
  
  ' handle decimal point
  If nCV_DecimalLocation > 0 Then
    nCV_unPACKEDSize = nCV_unPACKEDSize + 1
  End If

  ' make target field big enough
  sCV_ASCII = Space$(nCV_unPACKEDSize)

  ' convert, if error return nothing
  nCV_rc = EHNDT_PackedToASCII(hWnd, sCV_ASCII, sCV_PACKED, nCV_PACKEDSize, nCV_DecimalLocation)
  
  ' return value to caller
  zzCV_PACKEDtoASCII = zzCV_FormatNumber(sCV_ASCII, nCV_DecimalLocation)

End Function

Function zzCV_ZONEDtoASCII (ByVal hWnd%, ByVal sCV_ZONED$, ByVal nCV_DecimalLocation%, ByVal nCV_ZONEDSize%) As String

 ' Description:
 '  Convert
 '   From:  ZONED
 '   To:    ASCII
  
 ' Parameters:
 '  hWnd                          windows handle
 '  sCV_ZONED                     EBCDIC zoned number
 '  nCV_DecimalLocation           actual location of decimal place
 '  nCV_ZONEDSize                 size of zoned field
 
 ' Variables:
  Dim sCV_ASCII       As String   ' string to hold result
  Dim nCV_unZONEDSize As Integer  ' unpacked field length
  
  ' handle error
  On Error Resume Next
  
  ' cannot handle empty string
  If sCV_ZONED = gsEMPTY Then Exit Function
  
  ' get length of packed and unpacked field
  nCV_unZONEDSize = nCV_ZONEDSize + 1
  
  ' handle decimal point
  If nCV_DecimalLocation > 0 Then
    nCV_unZONEDSize = nCV_unZONEDSize + 1
  End If

  ' make target field big enough
  sCV_ASCII = Space$(nCV_unZONEDSize)

  ' call the API
  nCV_rc = EHNDT_ZonedToASCII(hWnd, sCV_ASCII, sCV_ZONED, nCV_ZONEDSize, nCV_DecimalLocation)

  ' return value to caller
  zzCV_ZONEDtoASCII = zzCV_FormatNumber(sCV_ASCII, nCV_DecimalLocation)

End Function

