Option Explicit

 ' Description:
 '  This module contains constants and wrappers related
 '  to the IBM PC Support/400 APPC action APIs.

 ' APIs:

  Declare Function EHNAPPC_Allocate% Lib "EHNAPPC.DLL" (ByVal hWnd%, ByVal nCABufSize%, ByVal nCAConvTyp%, ByVal nCASyncLvl%, ByVal sCASystem$, ByVal sCALibPgm$, ByVal nCAPipLen%, ByVal sCAPipData$, lCAConvId&)
  Declare Function EHNAPPC_Confirm% Lib "EHNAPPC.DLL" (ByVal hWnd%, ByVal lCAConvId&, nCARTSRcvd%)
  Declare Function EHNAPPC_Confirmed% Lib "EHNAPPC.DLL" (ByVal hWnd%, ByVal lCAConvId&)
  Declare Function EHNAPPC_Deallocate% Lib "EHNAPPC.DLL" (ByVal hWnd%, ByVal lCAConvId&, ByVal nCADeAllocTyp%)
  Declare Function EHNAPPC_ExtendedAllocate% Lib "EHNAPPC.DLL" (ByVal hWnd%, ByVal nCABufSize%, ByVal nCAConvTyp%, ByVal nCASyncLvl%, ByVal sCASystem$, ByVal sCALibPgm$, ByVal sCAModeName$, ByVal nCASecLvl%, ByVal sCAUserID$, ByVal sCAPassword$, ByVal nCAPipLen%, ByVal sCAPipData$, lCAConvId&)
  Declare Function EHNAPPC_Flush% Lib "EHNAPPC.DLL" (ByVal hWnd%, ByVal lCAConvId&)
  Declare Function EHNAPPC_GetAttributes% Lib "EHNAPPC.DLL" (ByVal hWnd%, ByVal lCAConvId&, nCASyncLvl%, ByVal sCAModeName$, ByVal sCALocalLU$, ByVal sCASystem$, ByVal sCAUserID$)
  Declare Function EHNAPPC_PrepareToReceive% Lib "EHNAPPC.DLL" (ByVal hWnd%, ByVal lCAConvId&)
  Declare Function EHNAPPC_QueryConvState% Lib "EHNAPPC.DLL" (ByVal hWnd%, ByVal lCAConvId&)
  Declare Function EHNAPPC_ReceiveAndWait% Lib "EHNAPPC.DLL" (ByVal hWnd%, ByVal lCAConvId&, ByVal nCAFill%, ByVal nCADataLenMax%, ByVal sCAData$, nCAWhatRcvd%, nCARTSRcvd%, lCADataLen&)
  Declare Function EHNAPPC_ReceiveImmediate% Lib "EHNAPPC.DLL" (ByVal hWnd%, ByVal lCAConvId&, ByVal nCAFill%, ByVal nCADataLenMax%, ByVal sCAData$, nCAWhatRcvd%, nCARTSRcvd%, lCADataLen&)
  Declare Function EHNAPPC_RqsToSend% Lib "EHNAPPC.DLL" (ByVal hWnd%, ByVal lCAConvId&)
  Declare Function EHNAPPC_SendData% Lib "EHNAPPC.DLL" (ByVal hWnd%, ByVal lCAConvId&, ByVal nCAData%, ByVal sCAData$, nCARTSRcvd%)
  Declare Function EHNAPPC_SendError% Lib "EHNAPPC.DLL" (ByVal hWnd%, ByVal lCAConvId&, nCARTSRcvd%)

Function zzCAConvStart (ByVal hWnd%, ByVal nCABuffSize%, ByVal sCASystem$, ByVal sCALibPgm$, ByVal sCAPipData$, nCArc%) As Long

 ' Description:
 '  Allocates a mapped conversion without confirmation.
 '  If successful will return the conversation ID as
 '  the function value. If not the error code will be
 '  returned in nCArc.

 ' Parameters:
 ' Input:
 '  hWnd                   windows handle
 '  nCABuffSize            desired buffer size
 '  sCASystem              AS/400 system name
 '  sCALibPgm              partner program "library/program"
 '  sCAPipData             PIP data
 ' Output:
 '  nCArc                  API return code

 ' Variables:
  Dim lCAConvId As Long    ' conversation ID

  ' handle DOS errors
  On Error Resume Next
 
  ' allocate conversation
  nCArc = EHNAPPC_Allocate(hWnd, nCABuffSize, gnCA_MAPPED, gnCA_SYNCLVL_NONE, sCASystem & gsCHR_NUL, sCALibPgm & gsCHR_NUL, Len(sCAPipData), sCAPipData, lCAConvId)

  ' if DOS then then return generic error number
  If Err <> 0 Then nCArc = gnCA_UNDEFINED

  ' return conversation ID as function value
  zzCAConvStart = lCAConvId

End Function

Function zzCAConvStartConfirm (ByVal hWnd%, ByVal nCABuffSize%, ByVal sCASystem$, ByVal sCALibPgm$, ByVal sCAPipData$, nCArc%) As Long

 ' Description:
 '  Allocates a mapped conversion with confirmation.
 '  If successful will return the conversation ID as
 '  the function value. If not the error code will be
 '  returned in nCArc.

 ' Parameters:
 ' Input:
 '  hWnd                   windows handle
 '  nCABuffSize            desired buffer size
 '  sCASystem              AS/400 system name
 '  sCALibPgm              library/program
 '  sCAPipData             PIP data
 ' Output:
 '  nCArc                  API return code

 ' Variables:
  Dim lCAConvId As Long    ' conversation ID

  ' handle DOS errors
  On Error Resume Next
 
  ' allocate conversation
  nCArc = EHNAPPC_Allocate(hWnd, nCABuffSize, gnCA_MAPPED, gnCA_SYNCLVL_CONFIRM, sCASystem & gsCHR_NUL, sCALibPgm & gsCHR_NUL, Len(sCAPipData), sCAPipData, lCAConvId)

  ' if DOS then then return generic error number
  If Err <> 0 Then nCArc = gnCA_UNDEFINED

  ' return conversation ID as function value
  zzCAConvStartConfirm = lCAConvId

End Function

Function zzCAConvStartWithSecurity (ByVal hWnd%, ByVal nCABuffSize%, ByVal sCASystem$, ByVal sCALibPgm$, ByVal sCAPipData$, ByVal sCAUserID$, ByVal sCAPassword$, nCArc%) As Long

 ' Description:
 '  Allocates a mapped conversion without confirmation.
 '  If successful will return the conversation ID as
 '  the function value. If not the error code will be
 '  returned in nCArc.

 ' Parameters:
 ' Input:
 '  hWnd                   windows handle
 '  nCABuffSize            desired buffer size
 '  sCASystem              AS/400 system name
 '  sCALibPgm              partner program "library/program"
 '  sCAPipData             PIP data
 '  sCAUserID              user id
 '  sCAPassword            password
 ' Output:
 '  nCArc                  API return code

 ' Variables:
  Dim lCAConvId As Long    ' conversation ID

  ' handle DOS errors
  On Error Resume Next
 
  ' allocate conversation
  nCArc = EHNAPPC_ExtendedAllocate(hWnd, nCABuffSize, gnCA_MAPPED, gnCA_SYNCLVL_NONE, sCASystem & gsCHR_NUL, sCALibPgm & gsCHR_NUL, "QPCSUPP" & gsCHR_NUL, gnCA_SECURITY_SAME, sCAUserID$, sCAPassword$, Len(sCAPipData), sCAPipData, lCAConvId)
  
  ' if DOS then then return generic error number
  If Err <> 0 Then nCArc = gnCA_UNDEFINED

  ' return conversation ID as function value
  zzCAConvStartWithSecurity = lCAConvId

End Function

Function zzCAConvStopAbEnd (ByVal hWnd%, ByVal lCAConvId&) As Integer

 ' Description:
 '  Deallocates a conversation with abnormal end.
 
 ' Parameters:
 '  hWnd                         windows handle
 '  lCAConvID                    conversation ID
 
  ' handle DOS errors
  On Error Resume Next
 
  ' deallocate conversation
  zzCAConvStopAbEnd = EHNAPPC_Deallocate(hWnd, lCAConvId, gnCA_DEALLOCATE_ABEND)

  ' if DOS then then return generic error number
  If Err <> 0 Then zzCAConvStopAbEnd = gnCA_UNDEFINED

End Function

Function zzCAConvStopConfirm (ByVal hWnd%, ByVal lCAConvId&, ByVal bCADoABEnd%) As Integer

 ' Description:
 '  Deallocate a conversation with confirmation.
 
 ' Parameters:
 '  hWnd                         windows handle
 '  lCAConvID                    conversation ID
 '  bCADoAbEnd                   do abnormal ending if this fails

 ' Variables:
  Dim nCArc As Integer           ' API return code

  ' handle DOS errors
  On Error Resume Next
 
  ' deallocate conversation with synchronization
  nCArc = EHNAPPC_Deallocate(hWnd, lCAConvId, gnCA_DEALLOCATE_SYNCLVL)
  
  ' if DOS then then return generic error number
  If Err <> 0 Then nCArc = gnCA_UNDEFINED

  ' if did not work and willing to do do abnormal
  If (nCArc <> gnCA_OK And nCArc <> gnCA_DEALLOC_ABEND_PROGRAM) And bCADoABEnd Then
    nCArc = zzCAConvStopAbEnd(hWnd, lCAConvId)
  End If

  ' return value to caller
  zzCAConvStopConfirm = nCArc

End Function

Function zzCAConvStopFlush (ByVal hWnd%, ByVal lCAConvId&) As Integer

 ' Description:
 '  Deallocate a conversation with flush of buffer.
 
 ' Parameters:
 '  hWnd                         windows handle
 '  lCAConvID                    conversation ID
 
  ' handle DOS errors
  On Error Resume Next
 
  ' deallocate conversation with flush
  zzCAConvStopFlush = EHNAPPC_Deallocate(hWnd, lCAConvId, gnCA_DEALLOCATE_FLUSH)

  ' if DOS then then return generic error number
  If Err <> 0 Then zzCAConvStopFlush = gnCA_UNDEFINED

End Function

Function zzCAFlush (ByVal hWnd%, ByVal lCAConvId&) As Integer

 ' Description:
 '  Flushes buffer causing all data to be sent.

 ' Parameters:
 '  hWnd                         windows handle
 '  lCAConvID                    conversation ID
  
  ' handle DOS errors
  On Error Resume Next
 
  ' flush the buffer
  zzCAFlush = EHNAPPC_Flush(hWnd, lCAConvId)
  
  ' if DOS then then return generic error number
  If Err <> 0 Then zzCAFlush = gnCA_UNDEFINED
  
End Function

Function zzCAFormattedPIP (ByVal hWnd%, asCApip$()) As String

 ' Description:
 '  Formats the program initialization
 '  parameters to be sent to the AS/400.
 '  The parameters must be passed in thru
 '  the string array (asCApip) with each
 '  element containing the value to pass
 '  plus any blank spaces required to make
 '  the element the size of the parameter.

 ' Example:

 '       *ENTRY    PLIST
 '                 PARM           XPARM1 10
 '                 PARM           XPARM2  5
 '                 PARM           XPARM3  1

 '  would require the following if wanting
 '  to pass the values "HELLO", "BYE", and
 '  "I".

 '   asCApip(0) = "HELLO     "
 '   asCApip(1) = "BYE  "
 '   asCApip(2) = "I"
 
 ' Parameters:
 '  hWnd                    windows handle
 '  asCApip                 parameters

 ' Constants:
  
  ' general data stream identifiers
  Const nGDSID_1 = &H12     ' data stream ID part 1
  Const nGDSID_2 = &HF5     ' data stream ID part 2
  Const nGDSID_3 = &HE2     ' data stream ID part 3
 
 ' Variables:
  Dim nBL   As Integer      ' lower array boundary
  Dim nBU   As Integer      ' upper array boundary
  Dim nDx   As Integer      ' loop index
  Dim nLen  As Integer      ' length of parameter
  Dim sPIP  As String       ' completed PIP
  Dim sW1   As String       ' work string
  Dim sW2   As String       ' work string

  ' get array boundaries
  nBL = LBound(asCApip)
  nBU = UBound(asCApip)

  ' do each element
  For nDx = nBL To nBU

    ' get length of field plus header length
    sW1 = asCApip(nDx)
    nLen = Len(sW1) + gnCA_MAPPED_HEADER_LEN

    ' if something to pass
    If nLen > gnCA_MAPPED_HEADER_LEN Then

      ' put in required "LL" field format
      sW2 = sW2 & Chr$(nLen \ 256) & Chr$(nLen Mod 256)
  
      ' put in general data stream ID
      sW2 = sW2 & Chr$(nGDSID_1) & Chr$(nGDSID_3)
  
      ' put in variable as EBCDIC data
      sW2 = sW2 & zzCV_ASCIItoEBCDIC(hWnd, sW1)
    
    End If

  Next nDx

  ' get length of entire PIP
  nLen = Len(sW2) + gnCA_MAPPED_HEADER_LEN

  ' put in required "LL" field format
  sPIP = Chr$(nLen \ 256) & Chr$(nLen Mod 256)

  ' put in general data stream ID
  sPIP = sPIP & Chr$(nGDSID_1) & Chr$(nGDSID_2)

  ' put in variable as EBCDIC data
  sPIP = sPIP & sW2

  ' return value to caller
  zzCAFormattedPIP = sPIP

End Function

Function zzCAQueryAmIReceiving (ByVal hWnd%, ByVal lCAConvId&) As Integer

 ' Description:
 '  Returns "True" if conversation state is in "receive".

 ' Parameters:
 '  hWnd                         windows handle
 '  lCAConvID                      conversation ID
  
  ' return value to caller as "real" True or False
  zzCAQueryAmIReceiving = (zzCAQueryState(hWnd, lCAConvId) = gnCA_RECEIVE_STATE)
  
End Function

Function zzCAQueryAmIReset (ByVal hWnd%, ByVal lCAConvId&) As Integer

 ' Description:
 '  Returns "True" if conversation state is in "reset state".

 ' Parameters:
 '  hWnd                         windows handle
 '  lCAConvID                      conversation ID
  
  ' return value to caller as "real" True or False
  zzCAQueryAmIReset = (zzCAQueryState(hWnd, lCAConvId) = gnCA_RESET_STATE)
  
End Function

Function zzCAQueryAmISending (ByVal hWnd%, ByVal lCAConvId&) As Integer

 ' Description:
 '  Returns "True" if conversation state is in "send".

 ' Parameters:
 '  hWnd                         windows handle
 '  lCAConvID                      conversation ID
  
  ' return value to caller as "real" True or False
  zzCAQueryAmISending = (zzCAQueryState(hWnd, lCAConvId) = gnCA_SEND_STATE)

End Function

Function zzCAQueryAttributes (ByVal hWnd%, ByVal lCAConvId&, nCASyncLvl%, sCAModeName$, sCALocalLU$, sCASystem$, sCAUserID$) As Integer

 ' Description:
 '  Retrieve attributes of current APPC conversation

 ' Parameters:
 ' Input:
 '  hWnd                  windows handle
 '  lCAConvId             conversation ID
 ' Output:
 '  nCASyncLvl            synchronization level
 '  sCAModeName           mode name
 '  sCALocalLU            local device name
 '  sCASystem             AS/400 system name
 '  sCAUserID$            User ID

 ' Variables:
  Dim nPos As Integer     ' position of null terminator

  ' handle DOS errors
  On Error Resume Next

  ' make result fields big enough
  sCAModeName = Space$(10)
  sCALocalLU = Space$(10)
  sCASystem = Space$(10)
  sCAUserID = Space$(10)

  ' call the API
  zzCAQueryAttributes = EHNAPPC_GetAttributes(hWnd, lCAConvId, nCASyncLvl, sCAModeName, sCALocalLU, sCASystem, sCAUserID)

  ' if DOS then then return generic error number
  If Err <> 0 Then zzCAQueryAttributes = gnCA_UNDEFINED

  ' remove null terminators
  nPos = InStr(sCAModeName, gsCHR_NUL)
  If nPos = 0 Then nPos = Len(sCAModeName) + 1
  sCAModeName = RTrim$(Left$(sCAModeName, nPos - 1))
  
  nPos = InStr(sCALocalLU, gsCHR_NUL)
  If nPos = 0 Then nPos = Len(sCALocalLU) + 1
  sCALocalLU = RTrim$(Left$(sCALocalLU, nPos - 1))
  
  nPos = InStr(sCASystem, gsCHR_NUL)
  If nPos = 0 Then nPos = Len(sCASystem) + 1
  sCASystem = RTrim$(Left$(sCASystem, nPos - 1))

  nPos = InStr(sCAUserID, gsCHR_NUL)
  If nPos = 0 Then nPos = Len(sCAUserID) + 1
  sCAUserID = RTrim$(Left$(sCAUserID, nPos - 1))

End Function

Function zzCAQueryState (ByVal hWnd%, ByVal lCAConvId&) As Integer

 ' Description:
 '  Determine conversation state.
 
 ' Parameters:
 '  hWnd                         windows handle
 '  lCAConvID                    conversation ID
  
  ' handle DOS errors
  On Error Resume Next
 
  ' query the conversation state
  zzCAQueryState = EHNAPPC_QueryConvState(hWnd, lCAConvId)

  ' if DOS then then return generic error number
  If Err <> 0 Then zzCAQueryState = gnCA_UNDEFINED

End Function

Function zzCAReceive (ByVal hWnd%, ByVal lCAConvId&, ByVal nCADataLen%, sCAData$, nCAWhatRcvd%, bCAPartnerWishesToSend%) As Integer

 ' Description:
 '  Receive data immediately, but do not wait for it.
 
 ' Parameters:
 '  hWnd                           windows handle
 '  lCAConvID                      conversation ID
 '  nCADataLen                     length of data to be received
 '  sCAData                        data received
 '  nCAWhatRcvd                    what received
 '  bCAPartnerWishesToSend         partner wishes to send

 ' Variables:
  Dim lCADataLen  As Long          ' length of data received
  Dim nCArc       As Integer       ' API return code

  ' add "header" bytes
  sCAData = Space$(nCADataLen + gnCA_MAPPED_HEADER_LEN)

  ' attempt receive of data from partner
  nCArc = EHNAPPC_ReceiveImmediate(hWnd, lCAConvId, gnCA_FILL_BUFFER, nCADataLen + gnCA_MAPPED_HEADER_LEN, sCAData, nCAWhatRcvd, bCAPartnerWishesToSend, lCADataLen)

  ' set to a "real" True if not zero (False)
  bCAPartnerWishesToSend = bCAPartnerWishesToSend <> 0

  ' if no error encountered
  If nCArc = gnCA_OK Then
    
    ' get data from the buffer
    sCAData = Left$(sCAData, lCADataLen)
    
  ' if error then nothing to return
  Else
    sCAData = gsEMPTY
  End If
  
  ' return code to caller
  zzCAReceive = nCArc

End Function

Function zzCAReceiveBlock (ByVal hWnd%, ByVal lCAConvId&, ByVal nCADataLen%, nCArc%, bCAPartnerWishesToSend%, nCAWhatRcvd%) As String
 
 ' Description:
 '  Receive a block of data records until request to
 '  send is received from partner. This routine uses
 '  immediate receive to allows Windows a chance to
 '  do something while program is waiting for data.
 
 ' Parameters:
 ' Input:
 '  hWnd                         windows handle
 '  lCAConvID                    conversation ID
 '  nCADataLen                   length of data to be received
 ' Output:
 '  nCArc                        return code
 '  bCAPartnerWishesToSend       partner wishes to send
 '  nCAWhatRcvd                  what received
  
 ' Variables:
  Dim sCABlock      As String    ' block of data to send back to caller
  Dim sCAData       As String    ' individual records received

  ' clear data buffers
  sCABlock = gsEMPTY
  zzCAReceiveBlock = gsEMPTY

  ' inform partner than I am ready to receive
  If zzCATellReadyToReceive(hWnd, lCAConvId) = gnCA_OK Then
  
    ' loop to get records
    Do

      ' loop until record found
      Do
  
        ' try to get next record
        nCArc = zzCAReceive(hWnd, lCAConvId, nCADataLen, sCAData, nCAWhatRcvd, bCAPartnerWishesToSend)

        ' give windows time
        DoEvents
  
        ' if error on receipt of data
        If nCArc <> gnCA_OK Then
        
          ' if not "unsuccessful read" then exit loop
          If nCArc <> gnCA_UNSUCCESSFUL Then Exit Function
      
        ' if no error
        Else

          ' if send requested by partner signals end of transmission
          If nCAWhatRcvd = gnCA_RCVD_SEND Then
          
            ' return data to caller after removing headers
            zzCAReceiveBlock = zzCARemoveHeadersFromBlock(sCABlock, nCADataLen)
            Exit Function

          End If
      
        End If
      
      Loop Until Len(sCAData) > 0

      ' save record into data block
      If nCArc = gnCA_OK Then
        sCABlock = sCABlock & sCAData
        sCAData = gsEMPTY
      End If

    Loop

  End If

End Function

Function zzCAReceiveOrWait (ByVal hWnd%, ByVal lCAConvId&, ByVal nCADataLen%, sCAData$, bCAPartnerWishesToSend%, nCAWhatRcvd%) As Integer

 ' Description:
 '  Receive data or wait until some gets there if not there.

 ' Parameters:
 ' Input:
 '  hWnd                         windows handle
 '  lCAConvID                    conversation ID
 '  nCADataLen                   length of data to be received
 ' Output:
 '  sCAData                      data received
 '  bCAPartnerWishesToSend       partner wishes to send
 '  nCAWhatRcvd                  what received

 ' Variables:
  Dim lCADataLen  As Long        ' length of data received
  Dim nCArc       As Integer     ' API return code
  
  ' make field big enough for header and data
  sCAData = Space$(nCADataLen + gnCA_MAPPED_HEADER_LEN)

  ' wait until data is found, then receive it
  nCArc = EHNAPPC_ReceiveAndWait(hWnd, lCAConvId, gnCA_FILL_BUFFER, nCADataLen + gnCA_MAPPED_HEADER_LEN, sCAData, nCAWhatRcvd, bCAPartnerWishesToSend, lCADataLen)

  ' set to a "real" True if not zero (False)
  bCAPartnerWishesToSend = bCAPartnerWishesToSend <> 0
  
  ' if no error encountered
  If nCArc = gnCA_OK Then
    
    ' get data from the buffer after 4 byte header removed
    sCAData = Left$(Mid$(sCAData, 5), lCADataLen)
  
  ' if error then nothing to return
  Else
    sCAData = gsEMPTY
  End If

  ' return code to caller
  zzCAReceiveOrWait = nCArc

End Function

Function zzCARemoveHeadersFromBlock (ByVal sCABlock$, ByVal nCARecordLen%) As String

 ' Description:
 '  Removes the 4 byte headers from a block of data.

 ' Parameters:
 '  sCABlock                     data block
 '  nCARecordLen                 length of single record in block
  
 ' Variables:
  Dim nI             As Integer  ' generic loop index
  Dim sCAData        As String   ' 4 byte header & data
  Dim sCANewBlock    As String   ' clean data repository

  ' loop through each record in block
  For nI = 1 To Len(sCABlock) Step nCARecordLen + gnCA_MAPPED_HEADER_LEN
    sCAData = Mid$(sCABlock, nI, nCARecordLen + gnCA_MAPPED_HEADER_LEN)
    sCANewBlock = sCANewBlock & Mid$(sCAData, gnCA_MAPPED_HEADER_LEN + 1)
  Next nI

  ' return data without headers
  zzCARemoveHeadersFromBlock = sCANewBlock

End Function

Function zzCASend (ByVal hWnd%, ByVal lCAConvId&, ByVal sCAData$, ByVal nCADataLen%, bCAPartnerWishesToSend%) As Integer

 ' Description:
 '  This procedure will send data to
 '  the partner program after placing
 '  the required general data stream
 '  identifiers on the front of the
 '  data.

 ' Parameters:
 ' Input:
 '  hWnd                            windows handle
 '  lCAConvID                       conversation ID
 '  sCAData                         data to be sent
 '  nCADataLen                      length of data to be sent
 ' Output:
 '  bCAPartnerWishesToSend          partner wishes to send
 
 ' Constants:
  
  ' general data stream identifiers
  Const nGDSID_1 = &H12             ' data stream ID part 1
  Const nGDSID_2 = &HFF             ' data stream ID part 2

 ' Variables:
  Dim nCARealDataLen As Integer     ' length of data actually sent
  Dim sCARealData    As String      ' data actually sent
  
  ' setup data length
  nCARealDataLen = nCADataLen + gnCA_MAPPED_HEADER_LEN

  ' put in required "LL" field information
  sCARealData = Chr$(nCARealDataLen \ 256) & Chr$(nCARealDataLen Mod 256)
  
  ' put in general data stream ID
  sCARealData = sCARealData & Chr$(nGDSID_1) & Chr$(nGDSID_2)

  ' put in data with trailing nulls
  sCARealData = sCARealData & Left$(sCAData & String$(nCADataLen, 0), nCADataLen)

  ' send data
  zzCASend = EHNAPPC_SendData(hWnd, lCAConvId, nCARealDataLen, sCARealData, bCAPartnerWishesToSend)

  ' set to a "real" True if not zero
  bCAPartnerWishesToSend = bCAPartnerWishesToSend <> 0

End Function

Function zzCASendConfirmReply (ByVal hWnd%, ByVal lCAConvId&) As Integer

 ' Description:
 '  Send confirmation reply.

 ' Parameters:
 '  hWnd                         windows handle
 '  lCAConvID                    conversation ID

  ' handle DOS errors
  On Error Resume Next
  
  ' send confirmation reply
  zzCASendConfirmReply = EHNAPPC_Confirmed(hWnd, lCAConvId)

  ' if DOS error then setup return code
  If Err <> 0 Then zzCASendConfirmReply = gnCA_UNDEFINED

End Function

Function zzCASendConfirmRequest (ByVal hWnd%, ByVal lCAConvId&, bCAPartnerWishesToSend%) As Integer

 ' Description:
 '  Send confirmation request

 ' Parameters:
 ' Input:
 '  hWnd                         windows handle
 '  lCAConvID                    conversation ID
 ' Output:
 '  bCAPartnerWishesToSend       partner wishes to send
  
  ' handle DOS errors
  On Error Resume Next
  
  ' send confirm request
  zzCASendConfirmRequest = EHNAPPC_Confirm(hWnd, lCAConvId, bCAPartnerWishesToSend)

  ' if DOS error then setup return code
  If Err <> 0 Then zzCASendConfirmRequest = gnCA_UNDEFINED
  
  ' set to a "real" True if not zero (False)
  bCAPartnerWishesToSend = bCAPartnerWishesToSend <> 0
  
End Function

Function zzCASendError (ByVal hWnd%, ByVal lCAConvId&, bCAPartnerWishesToSend%) As Integer

 ' Description:
 '  Send error notification.
  
 ' Parameters:
 ' Input:
 '  hWnd                         windows handle
 '  lCAConvID                      conversation ID
 ' Output:
 '  bCAPartnerWishesToSend         partner wishes to send

  ' handle DOS errors
  On Error Resume Next
  
  ' attempt to tell partner of error
  zzCASendError = EHNAPPC_SendError(hWnd, lCAConvId, bCAPartnerWishesToSend)

  ' if DOS error then setup return code
  If Err <> 0 Then zzCASendError = gnCA_UNDEFINED
  
  ' set to a "real" True if not zero (False)
  bCAPartnerWishesToSend = bCAPartnerWishesToSend <> 0

End Function

Function zzCASendReceive (ByVal hWnd%, ByVal lCAConvId&, ByVal sCAData$, ByVal nCADataLen%, nCArc%, bCAPartnerWishesToSend%, nCAWhatRcvd%) As String

 ' Description:
 '  Send data and wait for response.
 
 ' Parameters:
 ' Input:
 '  hWnd                         windows handle
 '  lCAConvID                    conversation ID
 '  sCAData                      data to send
 '  sCADataLen                   length of data to send
 ' Output:
 '  nCArc                        return code
 '  bCAPartnerWishesToSend       partner wishes to send
 '  nCAWhatRcvd                  what received

  ' send data
  nCArc = zzCASend(hWnd, lCAConvId, sCAData, nCADataLen, bCAPartnerWishesToSend)

  ' if any error then
  If nCArc <> gnCA_OK Then
    
    ' return nothing
    zzCASendReceive = gsEMPTY
    
  ' if no error then
  Else
    
    ' get block of data back
    zzCASendReceive = zzCAReceiveBlock(hWnd, lCAConvId, nCADataLen, nCArc, bCAPartnerWishesToSend, nCAWhatRcvd)
  
  End If

End Function

Function zzCATellReadyToReceive (ByVal hWnd%, ByVal lCAConvId&) As Integer

 ' Description:
 '  Change conversation state to prepare to receive.
 
 ' Parameters:
 '  hWnd                         windows handle
 '  lCAConvID                    conversation ID

  ' handle DOS errors
  On Error Resume Next
  
  ' tell partner ready to receive
  zzCATellReadyToReceive = EHNAPPC_PrepareToReceive(hWnd, lCAConvId)

  ' if DOS error then setup return code
  If Err <> 0 Then zzCATellReadyToReceive = gnCA_UNDEFINED

End Function

Function zzCATellWantToSend (ByVal hWnd%, ByVal lCAConvId&) As Integer
  
 ' Description:
 '  Request to send data.
 
 ' Parameters:
 '  hWnd                         windows handle
 '  lCAConvID                    conversation ID
 
  ' handle DOS errors
  On Error Resume Next
  
  ' tell partner ready to send
  zzCATellWantToSend = EHNAPPC_RqsToSend(hWnd, lCAConvId)
  
  ' if DOS error then setup return code
  If Err <> 0 Then zzCATellWantToSend = gnCA_UNDEFINED

End Function

